package ru.yandex.calendar.logic.contact.addressbook;

import java.net.URLEncoder;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;

import javax.annotation.PreDestroy;

import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.io.Charsets;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.ContentType;
import org.dom4j.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.calendar.frontend.caldav.proto.ETag;
import ru.yandex.calendar.frontend.caldav.proto.PutResponse;
import ru.yandex.calendar.frontend.caldav.proto.facade.ContactEtag;
import ru.yandex.calendar.frontend.caldav.proto.facade.ContactVcard;
import ru.yandex.calendar.logic.contact.Contact;
import ru.yandex.calendar.logic.contact.addressbook.HttpClientCallbacks.ByteCallback;
import ru.yandex.calendar.logic.contact.addressbook.HttpClientCallbacks.StatusCallback;
import ru.yandex.calendar.logic.ics.iv5j.vcard.VcfVCard;
import ru.yandex.calendar.tvm.TvmClient;
import ru.yandex.calendar.tvm.TvmHeaders;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.commune.json.JsonArray;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.commune.json.serialize.JsonParser;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClient4;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.client.HttpResponseCallback;
import ru.yandex.misc.io.http.client.entity.StringEntity;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;
import ru.yandex.misc.xml.stream.builder.XmlBuilder;
import ru.yandex.misc.xml.stream.builder.XmlBuilders;

public final class AddressBook {
    public static final String V_1_USERS = "/v1/users/";

    private final ApacheHttpClient4 collieClient;
    private final ApacheHttpClient4 searchClient;

    private final String collieUrl;
    private final int collieTvmId;
    private final String searchUrl;

    private final TvmClient tvmClient;

    public AddressBook(String collieUrl, int collieTvmId, String searchUrl,
                       HttpClient collieClient, HttpClient searchClient, TvmClient tvmClient) {

        this.collieUrl = collieUrl;
        this.collieTvmId = collieTvmId;
        this.searchUrl = searchUrl;
        this.collieClient = ApacheHttpClient4.wrap(collieClient);
        this.searchClient = ApacheHttpClient4.wrap(searchClient);

        this.tvmClient = tvmClient;
    }

    private static boolean is2xx(int statusCode) {
        return (statusCode / 100) == 2;
    }

    private static String encode(String value) {
        return URLEncoder.encode(value, Charsets.UTF_8);
    }

    private static org.apache.http.entity.StringEntity convertEntity(StringEntity entity) {
        return new org.apache.http.entity.StringEntity(
                entity.getContent(),
                ContentType.create(
                        entity.getMimeType().getOrElse("text/plain"),
                        entity.getCharset().getOrElse("ISO-8859-1")
                )
        );
    }

    private static void doPost(ApacheHttpClient4 client, String url, Tuple2List<String, String> headers,
                               org.apache.http.HttpEntity entity, HttpResponseCallback callback) {
        val post = new HttpPost(url);
        headers.forEach((Function2V<String, String>) post::addHeader);
        post.setEntity(entity);
        client.post(post, callback);
    }

    private static void doPost(ApacheHttpClient4 client, String url, Tuple2List<String, String> headers, byte[] payload,
                               HttpResponseCallback callback) {
        doPost(client, url, headers, new org.apache.http.entity.ByteArrayEntity(payload), callback);
    }

    private static byte[] doPost(ApacheHttpClient4 client, String url, Tuple2List<String, String> headers,
                                 byte[] payload) {
        val callback = new ByteCallback();
        doPost(client, url, headers, payload, callback);
        return callback.getStream().toByteArray();
    }

    private Tuple2List<String, String> tvmHeaders(int tvmId) {
        val ticket = tvmClient.getServiceTicketFor(tvmId);
        return Tuple2List.fromPairs(TvmHeaders.SERVICE_TICKET, ticket);
    }

    private Tuple2List<String, String> collieTvmHeaders() {
        return tvmHeaders(collieTvmId);
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(collieClient.getImpl());
        ApacheHttpClientUtils.stopQuietly(searchClient.getImpl());
    }

    public ListF<ContactEtag> getCarddavContactEtags(PassportUid uid) {
        val response = collieClient.get(collieUrl(V_1_USERS + uid + "/carddav/propfind"), collieTvmHeaders());

        return parseContactEtags(response);
    }

    public Tuple2List<String, Option<ContactVcard>> getCarddavContactsByFileNames(PassportUid uid,
                                                                                  List<String> fileNames) {
        val url = collieUrl(V_1_USERS + uid + "/carddav/multiget");
        val response = doPost(collieClient, url, collieTvmHeaders(), serializeFilenames(fileNames));

        return parseMultigetResponse(response);
    }

    public PutResponse putCarddavContact(PassportUid uid, String fileName, Optional<ETag> ifMatchEtag, VcfVCard vcard) {
        val etagValue = ifMatchEtag.map(ETag::getValue).orElse("*");

        val url = collieUrl(V_1_USERS + uid + "/carddav/" + encode(fileName) + "/put/" + encode(etagValue));
        val headers = collieTvmHeaders();
        val response = doPost(collieClient, url, headers, vcard.serializeToBytes());

        return parsePutResponse(response);
    }

    public void deleteCarddavContact(PassportUid uid, String fileName) {
        val callback = new StatusCallback();

        val url = collieUrl(V_1_USERS + uid + "/carddav/" + encode(fileName) + "/delete");
        doPost(collieClient, url, collieTvmHeaders(), new byte[0], callback);


        val status = callback.getStatus();
        if (!is2xx(status)) {
            throw new RuntimeException("bad http status " + status + " for " + url);
        }
    }

    public ListF<Contact> findUserContactsByEmails(PassportUid uid, List<Email> emails) {
        if (emails.isEmpty()) {
            return Cf.list();
        }

        val emailsStr = StreamEx.of(emails)
                .map(Email::getEmail)
                .joining(",");


        val url = searchUrl("/v1/search_by_email?user_type=passport_user&shared=include" +
                "&user_id=" + uid +
                "&email=" + emailsStr);
        val responseBytes = searchClient.get(url);
        return parseCollieSearchContactsResponse(responseBytes);
    }

    public ListF<Contact> findUserContacts(PassportUid uid, String query, int limit) {
        return findUserContacts(uid, query, OptionalInt.of(limit));
    }

    List<Email> suggestContacts(PassportUid uid) {
        val url = collieUrl("/v1/searchContacts?uid=" + uid + "&group=yes");
        val responseBytes = collieClient.get(url, collieTvmHeaders());
        val jsonObject = getJsonObject(responseBytes);
        val contactArray = (JsonArray) jsonObject.get("contact");
        return StreamEx.of(contactArray.getArray())
                .map(val -> getString((JsonObject) val, "email"))
                .flatCollection(Emails::punycodeSafe)
                .toImmutableList();

    }

    private ListF<Contact> findUserContacts(PassportUid uid, String query, OptionalInt limitO) {
        if (StringUtils.isBlank(query)) {
            return Cf.list();
        }

        val limit = limitO.orElse(0);

        val url = searchUrl("/v1/suggest?user_type=passport_user&shared=include&has_telephone_number=false" +
                "&user_id=" + uid +
                "&query=" + query +
                "&limit=" + limit);
        val body = new StringEntity("{\"exclude_emails\":[]}", Option.of("application/json"), Option.empty());
        val responseBytes = searchClient.post(url, body);
        return parseCollieSuggestContactsResponse(responseBytes);

    }

    void cleanAddressBook(PassportUid uid) {
        StreamEx.of(getCarddavContactEtags(uid))
                .map(ContactEtag::getFileName)
                .forEach(fileName -> deleteCarddavContact(uid, fileName));
    }

    public void exportUserContactsEmails(PassportUid uid, List<Email> contacts) {
        val data = "to=" + URLEncoder.encode(StreamEx.of(contacts).joining(","), Charsets.UTF_8);
        exportUserContactsImpl(uid, data);
    }

    private void exportUserContactsImpl(PassportUid uid, String data) {
        val client = collieClient;
        val host = collieUrl;
        val url = url(host, "/compat/colabook_feed_addrdb", Tuple2List.fromPairs("uid", uid.toString()));
        val headers = collieTvmHeaders();

        doPost(client, url, headers, convertEntity(urlEncodedStringEntity(data)), new ByteCallback());
    }

    private static byte[] serializeFilenames(List<String> filenames) {
        XmlBuilder xb = XmlBuilders.defaultBuilder();
        xb.startDocument();
        xb.startElement("request");
        for (String filename : filenames) {
            xb.textElement("href", filename);
        }
        xb.endElement();
        xb.endDocument();

        return xb.toByteArray();
    }

    private static ListF<ContactEtag> parseContactEtags(byte[] bytes) {
        ListF<ContactEtag> result = Cf.arrayList();
        @SuppressWarnings("unchecked")
        ListF<Element> contactElements = Cf.x(Dom4jUtils.readRootElement(bytes).selectNodes("/*/contact"));
        for (Element contactElement : contactElements) {
            String filename = contactElement.elementText("uri");
            ETag etag = new ETag(contactElement.elementText("etag"));
            result.add(new ContactEtag(filename, etag));
        }
        return result;
    }

    private static Tuple2List<String, Option<ContactVcard>> parseMultigetResponse(byte[] bytes) {
        Tuple2List<String, Option<ContactVcard>> result = Tuple2List.arrayList();
        @SuppressWarnings("unchecked")
        ListF<Element> contactElements = Cf.x(Dom4jUtils.readRootElement(bytes)
                .selectNodes("/multiget-result/contact"));

        for (Element contactElement : contactElements) {
            String fileName = contactElement.elementText("uri");
            Option<ContactVcard> contactVcard = Option.empty();

            int httpStatus = Integer.parseInt(contactElement.elementText("status"));
            if (httpStatus == HttpStatus.SC_200_OK) {
                ETag etag = new ETag(contactElement.elementText("etag"));
                VcfVCard vcard = VcfVCard.parse(contactElement.elementText("vcard"));
                contactVcard = Option.of(new ContactVcard(fileName, etag, vcard));
            }

            result.add(fileName, contactVcard);
        }
        return result;
    }

    private static PutResponse parsePutResponse(byte[] bytes) {
        Element root = Dom4jUtils.readRootElement(bytes);
        int statusCode = Integer.parseInt(root.elementText("status"));
        String etag = root.elementText("etag");
        String description = root.elementText("description");
        return new PutResponse(statusCode, etag, description);
    }

    private static ListF<Contact> parseCollieSearchContactsResponse(byte[] bytes) {
        final String text = CharsetUtils.decodeUtf8(bytes);
        return parseCollieSearchContactsResponse(text);
    }

    static ListF<Contact> parseCollieSearchContactsResponse(String text) {
        final MapF<String, JsonValue> map = JsonParser.getInstance().parseObject(text).getObject().toMap();
        return EntryStream.of(map)
                .mapKeys(Emails::punycode)
                .mapValues(o -> convertCollieContacts(List.of((JsonObject) o)))
                .filterValues(CollectionF::isNotEmpty)
                .map(e -> new Contact(e.getKey(), e.getValue().first().getName()))
                .collect(CollectorsF.toList());
    }

    private static JsonObject getJsonObject(byte[] bytes) {
        return JsonParser.getInstance().parseObject(CharsetUtils.decodeUtf8(bytes));
    }

    private static ListF<Contact> parseCollieSuggestContactsResponse(byte[] bytes) {
        val response = getJsonObject(bytes);
        return convertCollieContacts((JsonArray) response.get("contacts"));
    }

    private static String getString(JsonObject object, String property) {
        final Option<JsonValue> properyValueOpt = object.getO(property);
        return properyValueOpt.isPresent() ? ((JsonString) properyValueOpt.get()).getString() : "";
    }

    private static ListF<Contact> convertCollieContacts(JsonArray array) {
        val objects = StreamEx.of(array.getArray())
                .map(JsonObject.class::cast)
                .toImmutableList();
        return convertCollieContacts(objects);
    }

    private static ListF<Contact> convertCollieContacts(List<JsonObject> objects) {
        return StreamEx.of(objects)
                .map(object -> (JsonObject) object.get("vcard"))
                .flatMap(object -> {
                    val emailOpt = parseEmail((JsonArray) object.get("emails"));
                    final Option<JsonValue> names = object.getO("names");
                    final Optional<String> nameOpt = names.isPresent() ? parseName((JsonArray) names.get()) :
                            Optional.empty();
                    return emailOpt
                            .flatMap(mail -> nameOpt.map(name -> new Contact(mail, name)))
                            .stream();
                })
                .collect(CollectorsF.toList());
    }

    private static Optional<Email> parseEmail(JsonArray emails) {
        return StreamEx.of(emails.getArray())
                .map(JsonObject.class::cast)
                .flatMap(object -> Emails.punycodeSafe(getString(object, "email")).stream())
                .findFirst();
    }

    private static Optional<String> parseName(JsonArray names) {
        return StreamEx.of(names.getArray())
                .map(JsonObject.class::cast)
                .map(object -> {
                    String first = getString(object, "first");
                    String last = getString(object, "last");
                    return first + (!last.isEmpty() ? ' ' + last : "");
                })
                .findFirst();
    }

    private String url(String host, String path, Tuple2List<String, String> params) {
        return host + path + "?" + UrlUtils.listMapToUrlParameters(params);
    }

    private String collieUrl(String path) {
        return collieUrl + path;
    }

    private String searchUrl(String path) {
        return searchUrl + path;
    }

    private static StringEntity urlEncodedStringEntity(String urlEncodedData) {
        return new StringEntity(urlEncodedData, Option.of(URLEncodedUtils.CONTENT_TYPE), Option.of("utf-8"));
    }
}
