package ru.yandex.calendar.logic.contact.directory;

import java.util.Collection;
import java.util.stream.Stream;

import lombok.val;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.contact.Contact;
import ru.yandex.calendar.logic.contact.addressbook.AddressBook;
import ru.yandex.calendar.logic.contact.directory.search.DirectorySearchImpl;
import ru.yandex.calendar.logic.contact.directory.search.DirectorySearchPredicate;
import ru.yandex.calendar.logic.contact.directory.search.FieldPredicateMatcher;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.user.AvatarManager;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.l10n.Language;
import ru.yandex.calendar.micro.yt.StaffCache;
import ru.yandex.calendar.micro.yt.entity.YtDepartment;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.inside.passport.PassportDomain;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser.Phone;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser.PhoneType;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class DirectoryManager {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryManager.class);

    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private StaffCache staffCache;
    @Autowired
    private AddressBook addressBook;
    @Autowired
    private AvatarManager avatarManager;

    public Option<DirectoryType> getDirectory(PassportUid uid) {
        return isNotExternalYandexTeamUser(uid) ?
                Option.of(DirectoryType.YANDEX_TEAM) :
                Option.empty();
    }

    public ListF<DirectoryEntry> findContacts(
            PassportUid uid, DirectorySearchPredicate predicate, boolean includeAvatars, int limit)
    {
        if (isNotExternalYandexTeamUser(uid)) {
            return findYtContacts(predicate, includeAvatars, limit);
        }
        return findUserContacts(uid, predicate, limit);
    }

    private ListF<DirectoryEntry> findYtContacts(
            DirectorySearchPredicate predicate, boolean includeAvatars, int limit)
    {
        FieldPredicateMatcher matcher = DirectorySearchImpl.matcher(predicate);

        ListF<DirectoryEntry> r = Cf.arrayList();
        for (val user : userManager.getAllYtUsers()) {
            if (r.length() >= limit) {
                break;
            }

            try {
                if (matcher.userMatches(user)) {
                    r.add(convertUser(user, includeAvatars));
                }
            } catch (Exception e) {
                logger.warn(e, e);
            }
        }

        if (r.length() >= limit) {
            return r;
        }

        for (Resource resource : resourceRoutines.findActiveRooms()) {
            if (r.length() >= limit)
                break;

            try {
                if (matcher.resourceMatches(resource)) {
                    r.add(convertResource(resource));
                }
            } catch (Exception e) {
                logger.warn(e, e);
            }
        }

        return r;
    }

    private ListF<DirectoryEntry> findUserContacts(PassportUid uid, DirectorySearchPredicate predicate, int limit) {
        ListF<String> queries = predicate.getLeavesFieldPredicatesValues();

        return addressBook.findUserContacts(uid, queries.first(), limit).map(this::convertContact);
    }

    public Option<DirectoryEntry> getEntry(PassportUid uid, String id, boolean includeAvatar) {
        if (isNotExternalYandexTeamUser(uid)) {
            return getYtEntry(id, includeAvatar);
        }
        return Option.empty();
    }

    private Option<DirectoryEntry> getYtEntry(String id, boolean includeAvatar) {
        val email = Email.parseSafe(id).toOptional();

        val user = email.map(userManager::getYtUsersByEmail)
            .map(Collection::stream)
            .flatMap(Stream::findFirst)
            .or(() -> userManager.getYtUserByLogin(id));

        if (user.isPresent()) {
            return Option.of(convertUser(user.get(), includeAvatar));
        }

        val resource = resourceRoutines.findResourceByDomainAndExchangeName(
            email.map(Email::getDomain).orElse(PassportDomain.YANDEX_TEAM_RU.getDomain()),
            email.map(Email::getLocalPart).orElse(id));

        if (resource.isPresent()) {
            return Option.of(convertResource(resource.get()));
        }

        return Option.empty();
    }

    private DirectoryEntry convertUser(YtUser user, boolean includeAvatar) {
        val info = user.getInfo();
        DirectoryEntry.Builder builder = new DirectoryEntry.Builder();
        builder.setType(DirectoryEntryType.PERSON);
        Email email = new Email(info.getWorkEmail());
        builder.setEmail(email);
        builder.setExternalId(info.getWorkEmail());
        builder.setJabber(email);

        val firstName = user.extractFirstNameL10n().get(Language.RUSSIAN);
        val lastName = user.extractLastNameL10n().get(Language.RUSSIAN);
        builder.setFirstName(firstName.orElseGet(() -> lastName.isEmpty() ? user.getLogin() : ""));
        builder.setLastName(lastName.orElse(""));

        builder.setCellPhones(
            StreamEx.of(info.getPhones())
                .filterBy(Phone::getType, PhoneType.MOBILE)
                .map(Phone::getNumber)
                .collect(CollectorsF.toList())
        );
        builder.setWorkPhone(info.getWorkPhone().map(Object::toString).orElse(""));

        val jobTitle = info.getPositionRu();
        builder.setJobTitle(StringUtils.capitalize(jobTitle));

        builder.setJobOrganization(
            info.getDepartmentId()
                .flatMap(staffCache::getDepartmentById)
                .map(YtDepartment::extractFullNameL10n)
                .flatMap(name -> name.get(Language.RUSSIAN))
                .orElse("")
        );

        if (includeAvatar) {
            builder.setAvatar(avatarManager.getAvatarSafe(user.getLogin()));
        }

        return builder.build();
    }

    private DirectoryEntry convertResource(Resource resource) {
        Email email = ResourceRoutines.getResourceEmail(resource);

        DirectoryEntry.Builder builder = new DirectoryEntry.Builder();
        builder.setType(DirectoryEntryType.CONF_ROOM);
        builder.setEmail(email);
        builder.setExternalId(email.getEmail());

        Option<String> name = ResourceRoutines.getNameWithAlterNameI18n(resource).map(NameI18n.getNameF(ru.yandex.inside.utils.Language.RUSSIAN));
        builder.setFirstName(name.getOrElse(email.getLocalPart()));
        builder.setWorkPhone(resource.getPhone().map(Cf.Integer.toStringF()).getOrElse(""));

        Office office = officeManager.getOfficeById(resource.getOfficeId());
        Option<String> floorNum = resource.getFloorNum().map(Cf.Integer.toStringF());
        Option<String> floor = floorNum.map(s -> s + " " + UStringLiteral.FLOOR_LC);

        builder.setJobOrganization(office.getCityName().plus(office.getName()).stableUnique().plus(floor).mkString(", "));

        return builder.build();
    }

    private DirectoryEntry convertContact(Contact contact) {
        DirectoryEntry.Builder builder = new DirectoryEntry.Builder();
        builder.setType(DirectoryEntryType.PERSON);
        builder.setEmail(contact.getEmail());
        builder.setExternalId(contact.getEmail().getEmail());

        builder.setFirstName(StringUtils.notBlankO(contact.getName()).getOrElse(contact.getEmail().getLocalPart()));

        return builder.build();
    }

    private boolean isNotExternalYandexTeamUser(PassportUid uid) {
        return uid.isYandexTeamRu() && !userManager.isExternalYtUser(uid);
    }
}
