package ru.yandex.calendar.logic.contact.directory.search;

import lombok.val;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsCuType;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser.Car;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;

import static java.util.function.Predicate.not;

public class DirectorySearchImpl {
    public static FieldPredicateMatcher matcher(DirectorySearchPredicate predicate) {
        if (predicate.isAny()) {
            DirectorySearchPredicate.AnyPredicate anyPredicate = (DirectorySearchPredicate.AnyPredicate) predicate;
            return FieldPredicateMatcher.any(anyPredicate.getArgs().map(matcherF()));
        } else if (predicate.isAll()) {
            DirectorySearchPredicate.AllPredicate anyPredicate = (DirectorySearchPredicate.AllPredicate) predicate;
            return FieldPredicateMatcher.all(anyPredicate.getArgs().map(matcherF()));

        } else if (predicate.isField()) {
            return fieldPredicateMatcher((DirectorySearchPredicate.FieldPredicate) predicate);
        } else {
            throw new IllegalArgumentException("unknown predicate: " + predicate);
        }
    }

    private static Function<DirectorySearchPredicate, FieldPredicateMatcher> matcherF() {
        return DirectorySearchImpl::matcher;
    }

    private static FieldPredicateMatcher fieldPredicateMatcher(DirectorySearchPredicate.FieldPredicate fieldPredicate) {
        val operatorF = operatorF(fieldPredicate.getOperator());

        return new FieldPredicateMatcher(user -> {
            val fieldValues = getUserFields(user, fieldPredicate.getField());
            return fieldValues.findFirst(operatorF.bind2(fieldPredicate.getValue())).isPresent();
        }, resource -> {
            val fieldValues = getResourceFields(resource, fieldPredicate.getField());
            return fieldValues.findFirst(operatorF.bind2(fieldPredicate.getValue())).isPresent();
        });
    }

    private static Function2B<String, String> operatorF(DirectorySearchFieldOperator operator) {
        switch (operator) {
        case STARTS_WITH:
            return (a, b) -> a.toLowerCase().startsWith(b.toLowerCase());
        case CONTAINS:
            return (a, b) -> a.toLowerCase().contains(b.toLowerCase());
        default:
            throw new IllegalArgumentException("unknown operator: " + operator);
        }
    }

    private static StreamEx<String> getUserFields(YtUser user, DirectorySearchField field) {
        switch (field) {
        case EMAIL:
            return StreamEx.of(user.getInfo().getWorkEmail());
        case DISPLAY_NAME:
            return StreamEx.of(UserManager.getFullName(user, Language.RUSSIAN))
                .append(UserManager.getFullName(user, Language.ENGLISH).stream());
        case FIRST_NAME:
            return user.extractFirstNameL10n().getAllValues();
        case LAST_NAME:
            return user.extractLastNameL10n().getAllValues();
        case NICKNAME:
            return StreamEx.of(user.getLogin());
        case CAR_NUMBER:
            val numbers = StreamEx.of(user.getInfo().getCars())
                .map(Car::getPlate)
                .filter(not(String::isBlank))
                .toImmutableList();
            return StreamEx.of(numbers)
                .append(
                    StreamEx.of(numbers)
                        .map(s -> StringUtils.substring(s, 1))
                );
        case CALENDAR_USER_TYPE:
            return StreamEx.of(IcsCuType.INDIVIDUAL.getValue());
        case CALENDAR_USER_ADDRESS_SET:
            return getUserFields(user, DirectorySearchField.EMAIL)
                .map(e -> new Email(e).toMailto());
        default:
            throw new IllegalArgumentException("unknown field: " + field);
        }
    }

    private static StreamEx<String> getResourceFields(Resource resource, DirectorySearchField field) {
        switch (field) {
        case EMAIL:
            return StreamEx.of(ResourceRoutines.getResourceEmail(resource).getEmail());
        case DISPLAY_NAME:
        case FIRST_NAME:
        case LAST_NAME:
        case NICKNAME:
            return StreamEx.of(resource.getName())
                .append(resource.getNameEn())
                .append(resource.getAlterName())
                .append(resource.getAlterNameEn());
        case CAR_NUMBER:
            return StreamEx.empty();
        case CALENDAR_USER_TYPE:
            return StreamEx.of(IcsCuType.ROOM.getValue());
        case CALENDAR_USER_ADDRESS_SET:
            return StreamEx.of(ResourceRoutines.getResourceEmail(resource).toMailto());
        default:
            throw new IllegalArgumentException("unknown field: " + field);
        }
    }
}
