package ru.yandex.calendar.logic.domain;

import java.util.Optional;

import org.apache.commons.lang.NotImplementedException;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser.PhoneType;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class DomainManager {
    private static final Logger logger = LoggerFactory.getLogger(DomainManager.class);

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private UserManager userManager;

    private boolean disablePhones;

    public boolean hasMobilePhone(PassportUid uid) {
        return !disablePhones &&
                (!uid.isYandexTeamRu() || ytHasMobilePhone(uid)); // skip check for non-yandex-team uid
    }

    /**
     * @return mobile phone number +71234567890 in format ***7890
     */
    public Option<String> getPhoneNumber(PassportUid uid) {
        if (disablePhones) {
            return Option.empty();
        }
        if (uid.isYandexTeamRu()) {
            return Option.x(ytGetPhoneNumber(uid));
        } else {
            // nga: we can get full phone from sms passport,
            // but do we really need it? What for?
            // ssytnik: we don't
            throw new NotImplementedException();
        }
    }

    private boolean ytHasMobilePhone(PassportUid uid) {
        return ytGetPhoneNumber(uid).isPresent();
    }

    private Optional<String> ytGetPhoneNumber(PassportUid uid) {
        String userLogin = settingsRoutines.getSettingsByUid(uid).getUserLogin().getOrNull();
        Validate.notEmpty(userLogin);
        Optional<YtUser> ytUser = userManager.getYtUserByLogin(userLogin);
        if (ytUser.isPresent()) {
            return getPhone(ytUser);
        } else {
            logger.debug("Couldn't get user details: " + userLogin);
            return Optional.empty();
        }
    }

    @NotNull
    static Optional<String> getPhone(Optional<YtUser> ytUser) {
        return ytUser.get().getInfo().getPhones().stream()
                .filter(phone -> phone.getType() == PhoneType.MOBILE)
                .filter(StaffUser.Phone::isMain)
                .findFirst()
                .map(StaffUser.Phone::getNumber);
    }


    @Required
    @Value("${sms.passport.disable:-false}")
    public void setDisablePhones(boolean value) {
        disablePhones = value;
    }

}
