package ru.yandex.calendar.logic.event;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.Period;
import org.joda.time.PeriodType;
import org.joda.time.chrono.ISOChronology;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.util.CharsetUtils;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author akirakozov
 */
public abstract class AbstractEventDataConverter {
    public static final int MAX_EVENT_LENGTH_DAYS = 31;
    // XXX: kill this field, converter should convert data
    // independently from time zone
    protected final DateTimeZone eventTz;

    protected AbstractEventDataConverter(DateTimeZone eventTz) {
        this.eventTz = eventTz;
    }

    protected abstract EventData doConvert();

    protected final EventData convert() {
        EventData eventData = doConvert();
        Event event = eventData.getEvent();

        Cf.list(EventFields.NAME, EventFields.DESCRIPTION, EventFields.LOCATION, EventFields.URL).forEach(field -> {
            if (event.isFieldSet(field) && event.getFieldValue(field) != null) {
                String useful = CharsetUtils.cutUselessAsciiControls(event.getFieldValue(field)); // CAL-7422
                event.setFieldValue(field, useful);
            }
        });
        if (event.isFieldSet(EventFields.NAME)) {
            event.setName(event.getName().replace('\n', ' '));
        }
        cutStartEndTime(eventData);
        moveStartToDayOfRepetitionIfNeeded(eventData);
        validate(eventData);
        return eventData;
    }

    private void cutStartEndTime(EventData eventData) {
        DateTime start = eventData.getEvent().getStartTs().toDateTime(eventTz);
        DateTime end = eventData.getEvent().getEndTs().toDateTime(eventTz);

        // CAL-6159
        eventData.getEvent().setStartTs(start.minuteOfHour().roundFloorCopy().toInstant());
        eventData.getEvent().setEndTs(end.minuteOfHour().roundFloorCopy().toInstant());

        Option<Instant> recurrenceId = eventData.getEvent().getFieldValueO(EventFields.RECURRENCE_ID).filterNotNull();
        if (recurrenceId.isPresent()) {
            DateTime recurrenceIdDt = recurrenceId.get().toDateTime(eventTz);
            eventData.getEvent().setRecurrenceId(recurrenceIdDt.minuteOfHour().roundFloorCopy().toInstant());
        }

        if (eventData.getEvent().getFieldValueO(EventFields.IS_ALL_DAY).isSome(true)) {
            LocalDate startDate = new LocalDate(eventData.getEvent().getStartTs(), eventTz);
            LocalDate endDate = new LocalDate(eventData.getEvent().getEndTs(), eventTz);

            if (startDate.equals(endDate)) {
                endDate = startDate.plusDays(1);
            }
            eventData.getEvent().setStartTs(startDate.toDateTimeAtStartOfDay(eventTz).toInstant());
            eventData.getEvent().setEndTs(eventData.getEvent().getStartTs()
                    .toDateTime(eventTz).plus(new Period(startDate, endDate)).toInstant());
        }
    }

    private void moveStartToDayOfRepetitionIfNeeded(EventData eventData) {

        InstantInterval moved = RepetitionUtils.moveEventIntervalToDayOfRepetition(
                new InstantInterval(eventData.getEvent().getStartTs(), eventData.getEvent().getEndTs()),
                eventData.getRepetition(), eventTz);

        eventData.getEvent().setStartTs(moved.getStart());
        eventData.getEvent().setEndTs(moved.getEnd());
    }

    // CAL-7141
    protected void moveDueTsAfterEventStart(EventData eventData) {
        if (!RepetitionRoutines.isNoneRepetition(eventData.getRepetition())) {
            Instant startTs = eventData.getEvent().getStartTs();
            Repetition repetition = eventData.getRepetition();
            Option<Instant> rDueTs = repetition.getFieldValueO(RepetitionFields.DUE_TS).filterNotNull();

            if (rDueTs.isPresent() && !rDueTs.get().isAfter(startTs)) {
                repetition = repetition.copy();
                repetition.setDueTs(
                        EventRoutines.calculateDueTsFromUntilDate(startTs, new LocalDate(startTs, eventTz), eventTz));
                eventData.setRepetition(repetition);
            }
        }
    }

    protected final void validate(EventData eventData) {
        Instant startTs = eventData.getEvent().getStartTs();
        Instant endTs = eventData.getEvent().getEndTs();
        final boolean isAllDay = eventData.getEvent().getFieldValueO(EventFields.IS_ALL_DAY).isSome(true);
        if (isAllDay) {
            if (!(startTs.compareTo(endTs) < 0)) {
                final String msg =
                    "start time " + startTs + " must be strictly before the end time " + endTs + " for the all day";
                throw new CommandRunException(msg);
            }
        } else {
            if (!(startTs.compareTo(endTs) <= 0)) {
                final String msg =
                    "start time " + startTs + " must not be after the end time " + endTs + " for not the all day";
                throw new CommandRunException(msg);
            }
        }

        // 3. compare (start ? r.dueTs)
        if (!RepetitionRoutines.isNoneRepetition(eventData.getRepetition())) {
            final Repetition repetition = eventData.getRepetition();
            Instant rDueTs = repetition.getFieldValueO(RepetitionFields.DUE_TS).getOrNull();
            if (rDueTs != null) {
                if (startTs.compareTo(rDueTs) >= 0) {
                    throw CommandRunException.createSituation(
                            "event start " + startTs +
                            " should be less than repetition due time " + rDueTs, Situation.EVENT_STARTS_AFTER_DUE);
                }
            }
        }
        // 4. length [startTs; endTs)
        if (!RepetitionRoutines.isNoneRepetition(eventData.getRepetition())) {
            // Another possibility: compare Duration(startMs, endMs) ? Duration(Days.days(...))
            Instant startMs = startTs;
            Instant endMs = DateTimeFormatter.equalizeLtMsSafe(startMs, endTs, 1, eventTz);
            int eDays = new Period(startMs.getMillis(), endMs.getMillis(),
                    PeriodType.days(), ISOChronology.getInstance(eventTz)).getDays();

            int rApproxMinDays = RepetitionUtils.getApproxMinLengthDays(eventData.getRepetition());
            if (eDays > Math.min(MAX_EVENT_LENGTH_DAYS, rApproxMinDays)) {
                final String msg =
                    "ER.cutCheckNewStartEnd(): " + "verify: new event uses " + eDays + " day(s) " +
                    "(diff between " + startMs + " and " + endMs + "), while: " +
                    "a) min. approx. interval between two repeating events is " + rApproxMinDays +
                    " day(s); b) max. allowed event length is " + MAX_EVENT_LENGTH_DAYS + " day(s)";
                final Situation s = (eDays > MAX_EVENT_LENGTH_DAYS ?
                    Situation.TOO_LONG_EVENT :
                    Situation.EVENT_LONGER_THAN_REP
                );
                CommandRunException e = CommandRunException.createSituation(msg, s);
                e.setAdmissibleError(true);
                throw e;
            }
        }
    }
}
