package ru.yandex.calendar.logic.event;

import java.util.Date;
import java.util.OptionalLong;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.With;
import org.joda.time.Instant;
import org.joda.time.ReadableInstant;

import ru.yandex.calendar.logic.ics.TestDateTimes;
import ru.yandex.misc.annotation.FillWithSomething;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.net.HostnameUtils;

@AllArgsConstructor(access = AccessLevel.PRIVATE)
public class ActionInfo {
    @Getter private final String action;
    @JsonProperty("action_source")
    @Getter private final ActionSource actionSource;
    private final String rid;
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd HH:mm:ss.SSS", timezone="Europe/Moscow")
    private final Date datetime;
    private final Long unixtime;
    @JsonIgnore
    private final String host;
    @JsonIgnore
    @Getter private final Instant now;
    @JsonIgnore
    private final boolean freezedNowForTest;
    @With
    @Getter private final OptionalLong tvmId;

    public ActionInfo(ActionSource actionSource, String rid, Instant now) {
        this("", actionSource, rid, now);
    }

    @FillWithSomething
    public ActionInfo(String action, ActionSource actionSource, String rid, Instant now) {
        this(action, actionSource, rid, now, false);
    }

    public ActionInfo(String action, ActionSource actionSource, String rid, Instant now, boolean freezedNowForTest) {
        this(action, actionSource, rid, now, freezedNowForTest, OptionalLong.empty());
    }

    public ActionInfo(String action, ActionSource actionSource, String rid, Instant now, boolean freezedNowForTest, OptionalLong tvmId) {
        this.action = action;
        this.actionSource = actionSource;
        this.rid = rid;
        this.now = now;
        this.datetime = now.toDate();
        this.unixtime = now.getMillis();
        this.host = HostnameUtils.localHostname();
        this.freezedNowForTest = freezedNowForTest;
        this.tvmId = tvmId;
    }

    public String getRequestId() {
        return rid;
    }

    public String getRequestIdWithHostId() {
        return rid + "_" + StringUtils.substringBefore(host, ".qloud-c.yandex.net");
    }

    public static ActionInfo webTest() {
        return webTest(TestDateTimes.moscow(2010, 12, 25, 2, 28));
    }

    public static ActionInfo webTest(ReadableInstant now) {
        return ActionInfo.test(ActionSource.WEB, "web-test", now);
    }

    public static ActionInfo exchangeTest() {
        return exchangeTest(TestDateTimes.moscow(2010, 12, 25, 2, 28));
    }

    public static ActionInfo exchangeTest(Instant now) {
        return test(ActionSource.EXCHANGE, "exchange-test", now);
    }

    public static ActionInfo test(ActionSource actionSource, String requestId, ReadableInstant instant) {
        return new ActionInfo("test", actionSource, requestId, instant.toInstant(), true);
    }

    public static ActionInfo adminManager() {
        // XXX ssytnik 2011-07-06: temporarily switch back to PERFORMER,
        // because db is not updated now, and don't want to update it
        // until yt calendar finishes merge with public one.
        return new ActionInfo(ActionSource.PERFORMER /*ADMIN_MANAGER*/, "admin-manager", new Instant());
    }

    public static ActionInfo adminManagerForEwsSynchronizer() {
        return new ActionInfo(ActionSource.EXCHANGE_SYNCH, "ews-synch-admin-manager", new Instant());
    }

    public ActionInfo withActionSource(ActionSource actionSource) {
        return new ActionInfo(action, actionSource, rid, now).withTvmId(tvmId);
    }

    public ActionInfo withNow(Instant now) {
        return freezedNowForTest ? this : new ActionInfo(action, actionSource, rid, now).withTvmId(tvmId);
    }

    public ActionInfo freezeNowForTest() {
        return new ActionInfo(action, actionSource, rid, now, true).withTvmId(tvmId);
    }

    public ActionInfo unfreezeNowForTest() {
        return new ActionInfo(action, actionSource, rid, now, false).withTvmId(tvmId);
    }
}
