package ru.yandex.calendar.logic.event;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.beans.generated.EventInvitationFields;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventLayerFields;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventResourceFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerFields;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.OfficeFields;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.inside.passport.PassportUid;

/**
 * @author Stepan Koltsov
 */
public class DatabaseDump {
    private final ListF<MainEvent> mainEvents;
    private final ListF<Event> events;
    private final ListF<EventUser> eventUsers;
    private final ListF<EventLayer> eventLayers;
    private final ListF<EventResource> eventResources;
    private final ListF<EventInvitation> invitations;
    private final ListF<Layer> layers;
    private final ListF<LayerUser> layerUsers;
    private final ListF<SettingsInfo> settings;
    private final ListF<Resource> resources;

    private final MapF<Long, Layer> layersById;
    private final MapF<Long, Event> eventsById;
    private final MapF<Long, MainEvent> mainEventsById;
    private final MapF<Long, Office> officesById;
    private final MapF<PassportUid, SettingsInfo> settingsByUid;

    public DatabaseDump(
            ListF<MainEvent> mainEvents, ListF<Event> events, ListF<EventUser> eventUsers, ListF<EventLayer> eventLayers,
            ListF<EventInvitation> invitations, ListF<Layer> layers, ListF<LayerUser> layerUsers,
            ListF<SettingsInfo> settings, ListF<Resource> resources, ListF<Office> offices, ListF<EventResource> eventResources)
    {
        this.mainEvents = mainEvents;
        this.events = events;
        this.eventUsers = eventUsers;
        this.eventLayers = eventLayers;
        this.invitations = invitations;
        this.layers = layers;
        this.layerUsers = layerUsers;
        this.settings = settings;
        this.resources = resources;
        this.eventResources = eventResources;

        layersById = layers.toMapMappingToKey(LayerFields.ID.getF());
        eventsById = events.toMapMappingToKey(EventFields.ID.getF());
        mainEventsById = mainEvents.toMapMappingToKey(MainEventFields.ID.getF());
        officesById = offices.toMapMappingToKey(OfficeFields.ID.getF());
        settingsByUid = settings.toMapMappingToKey(SettingsInfo.getUidF());
    }

    public ListF<EventInvitation> getInvitations() {
        return invitations;
    }

    public ListF<Resource> getResources() {
        return resources;
    }

    private ListF<ResourceInfo> getResourceInfos() {
        return resources
            .zipWith(r -> officesById.getOrThrow(r.getId()))
            .map(ResourceInfo.consF())
        ;
    }

    private ListF<EventUserWithRelations> getEventUsersWithRelations() {
        return eventUsers.map(eu -> new EventUserWithRelations(eu, settingsByUid.getOrThrow(eu.getUid())));
    }

    public ListF<EventInvitation> getEventInvitations() {
        return invitations.filter(EventInvitationFields.EVENT_ID.getF().andThen(Function1B.notNullF()));
    }

    public ListF<MainEvent> getMainEvents() {
        return mainEvents;
    }

    public ListF<Settings> getSettingsCommon() {
        return settings.map(SettingsInfo.getCommonF());
    }

    public ListF<Layer> getLayers() {
        return layers;
    }

    public ListF<LayerUser> getLayerUsers() {
        return layerUsers;
    }

    public ListF<EventLayer> getEventLayers() {
        return eventLayers;
    }

    public ListF<EventLayerWithRelations> getEventLayerWithRelations() {
        return eventLayers.map(new Function<EventLayer, EventLayerWithRelations>() {
            public EventLayerWithRelations apply(EventLayer eventLayer) {
                return new EventLayerWithRelations(eventLayer, layersById.getOrThrow(eventLayer.getLayerId()));
            }
        });
    }

    public ListF<MainEventWithRelations> getMainEventsWithRelations() {
        final MapF<Long, ListF<EventWithRelations>> eventsWithRelationsByMainEventId =
            getEventsWithRelations().groupBy(EventWithRelations::getMainEventId);

        return mainEvents.map(new Function<MainEvent, MainEventWithRelations>() {
            public MainEventWithRelations apply(MainEvent mainEvent) {
                return new MainEventWithRelations(mainEvent, eventsWithRelationsByMainEventId.getO(mainEvent.getId()).getOrElse(Cf.<EventWithRelations>list()));
            }
        });
    }

    public ListF<EventWithRelations> getEventsWithRelations() {
        final MapF<Long, ListF<EventUserWithRelations>> eventUsers = this.getEventUsersWithRelations().groupBy(EventUserWithRelations::getEventId);
        final MapF<Long, ListF<EventLayerWithRelations>> eventLayersByEventId = this.getEventLayerWithRelations().groupBy(EventLayerWithRelations::getEventId);
        final MapF<Long, ListF<EventInvitation>> eventInvitations = this.getEventInvitations().groupBy(EventInvitationFields.EVENT_ID.getF());
        final MapF<Long, ListF<EventResource>> eventResources = this.getEventResources().groupBy(EventResourceFields.EVENT_ID.getF());
        final MapF<Long, ResourceInfo> resourcesByIds = this.getResourceInfos().toMapMappingToKey(ResourceInfo.resourceIdF());

        ListF<EventWithRelations> eventsWithRelations = events.map(new Function<Event, EventWithRelations>() {
            public EventWithRelations apply(Event event) {
                ListF<EventLayerWithRelations> eventLayers = eventLayersByEventId.getO(event.getId()).getOrElse(Cf.<EventLayerWithRelations>list());
                MainEvent mainEvent = mainEventsById.getOrThrow(event.getMainEventId());
                ListF<EventUserWithRelations> eventEventUsers = eventUsers.getO(event.getId()).getOrElse(Cf.<EventUserWithRelations>list());
                ListF<EventInvitation> eventEventInvitations = eventInvitations.getO(event.getId()).getOrElse(Cf.list());
                ListF<EventResource> eventEventResources = eventResources.getO(event.getId()).getOrElse(Cf.<EventResource>list());
                ListF<ResourceInfo> resources = eventEventResources.map(
                        EventResourceFields.RESOURCE_ID.getF()).filterMap(resourcesByIds::getO);

                EventParticipants participants = new EventParticipants(
                        event.getId(), eventEventUsers, eventEventInvitations, resources, eventEventResources);

                return new EventWithRelations(event, mainEvent, participants, new EventLayers(eventLayers));
            }
        });
        return eventsWithRelations;
    }

    public ListF<MainEvent> getOrphanedMainEvents() {
        return getMainEventsWithRelations()
            .filter(MainEventWithRelations::isOrphaned)
            .map(MainEventWithRelations::getMainEvent);
    }

    public Tuple2List<Tuple2<Long, PassportUid>, ListF<EventLayer>> getNonuniqueEventLayers() {
        return eventLayers.groupBy(new Function<EventLayer, Tuple2<Long, PassportUid>>() {
            public Tuple2<Long, PassportUid> apply(EventLayer eventLayer) {
                return Tuple2.tuple(eventLayer.getEventId(), eventLayer.getLCreatorUid());
            }
        }).entries().filterBy2(Cf.List.sizeF().andThen(Cf.Integer.gtF(1)));
    }

    public ListF<MainEvent> getMainEventsOnLayer(long layerId) {
        return eventLayers
            .map(EventLayerFields.EVENT_ID.getF())
            .map(eventsById::getOrThrow)
            .map(EventFields.MAIN_EVENT_ID.getF())
            .unique()
            .map(mainEventsById::getOrThrow)
            ;
    }

    public ListF<EventResource> getEventResources() {
        return eventResources;
    }

} //~
