package ru.yandex.calendar.logic.event;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.frontend.ews.exp.EventToCalendarItemConverter;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventHelper;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserHelper;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionHelper;
import ru.yandex.calendar.logic.event.EventChangesInfo.EventChangesInfoFactory;
import ru.yandex.calendar.logic.event.attachment.EventAttachmentChangesFinder;
import ru.yandex.calendar.logic.event.attachment.EventAttachmentChangesInfo;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.event.model.ParticipantsOrInvitationsData;
import ru.yandex.calendar.logic.event.repetition.RdateChangesInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.notification.EventNotifications;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.notification.NotificationChangesFinder;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

public class EventChangesFinder {
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private UserManager userManager;

    public EventChangesInfo getEventChangesInfo(
            EventInstanceForUpdate eventInstance, EventData newEventData,
            NotificationsData.Update notificationsUpdate,
            Option<PassportUid> clientUid, boolean updateFromWeb, boolean eventDataCouldBeChanged)
    {
        EventChangesInfoFactory factory = new EventChangesInfoFactory();

        Event eventChanges = EventHelper.INSTANCE.findChanges(eventInstance.getEvent(), newEventData.getEvent()).copy();
        eventChanges.unsetField(EventFields.ID);
        if (!eventDataCouldBeChanged) {
            eventChanges.unsetField(EventFields.DATA);
        }

        factory.setEventChanges(eventChanges);
        factory.setEventInstanceStartEndTsChanged(eventInstanceStartEndTsChanges(eventInstance, newEventData));
        factory.setEventUserChanges(eventUserChanges(eventInstance, newEventData));
        factory.setRepetitionChanges(repetitionChanges(eventInstance, newEventData));
        factory.setNewLayerIdAndChanged(eventLayerChanges(eventInstance, newEventData));
        factory.setExchangeData(newEventData.getExchangeData());

        boolean notificationsChanged = NotificationChangesFinder.changes(
                eventInstance.getEventUserWithNotifications()
                        .map(EventUserWithNotifications::getNotifications)
                        .getOrElse(EventNotifications::empty), notificationsUpdate).wasChanges();

        factory.setNotificationsUpdateAndChanged(notificationsUpdate, notificationsChanged);

        RdateChangesInfo rdateChangesInfo = !updateFromWeb ?
                repetitionRoutines.rdatesChanges(newEventData.getRdates(), eventInstance.getRepetitionInstanceInfo()) :
                RdateChangesInfo.EMPTY;
        factory.setRdateChangesInfo(rdateChangesInfo);

        ParticipantsOrInvitationsData newInvData = newEventData.getInvData();

        if (!eventInstance.getEventWithRelations().isMeeting() && !newInvData.getOrganizerEmail().isPresent()) {
            final var primaryLayerCreator = eventInstance.getEventWithRelations()
                    .getPrimaryLayerCreatorUid()
                    .flatMap(uid -> userManager.getEmailByUid(uid));
            newInvData = newInvData.withOrganizerIfInvitation(Option.x(primaryLayerCreator));
        }

        factory.setEventParticipantsChangesInfo(
                eventInvitationManager.participantsChanges(
                        clientUid, eventInstance.getEventWithRelations().getParticipants(), newInvData));

        EventAttachmentChangesInfo attachmentChangesInfo = newEventData.getAttachmentsO().isPresent() ?
                EventAttachmentChangesFinder.changes(eventInstance.getAttachments(), newEventData.getAttachmentsO().get()) :
                EventAttachmentChangesInfo.NO_CHANGES;
        factory.setAttachmentChangesInfo(attachmentChangesInfo);

        return factory.create();
    }

    /**
     * @return changes info enough for {@link EventToCalendarItemConverter#convertToChangeDescriptions}
     *
     * @url https://jira.yandex-team.ru/browse/CAL-3256
     * @url http://wiki.yandex-team.ru/Calendar/exchange/exportRecurrenceId
     */
    public EventChangesInfoForExchange getEventChangesInfoForExchangeForNewRecurrence(Event master, Event recurrence) {
        Validate.some(recurrence.getRecurrenceId());
        Validate.none(master.getRecurrenceId());
        Validate.equals(master.getMainEventId(), recurrence.getMainEventId());

        return new EventChangesInfoForExchange(
                EventHelper.INSTANCE.findChanges(master, recurrence),
                new Repetition(), true, Cf.list());
    }

    private boolean eventInstanceStartEndTsChanges(
            EventInstanceForUpdate eventInstance, EventData newEventData)
    {
        if (!eventInstance.getInterval().isPresent()) {
            Event master = eventInstance.getEvent();
            Event update = newEventData.getEvent();

            return (update.isFieldSet(EventFields.START_TS) && !update.getStartTs().equals(master.getStartTs())) ||
                   (update.isFieldSet(EventFields.END_TS) && !update.getEndTs().equals(master.getEndTs()));
        }
        InstantInterval oldEventInstanceInterval = eventInstance.getInterval().get();
        Event newEvent = newEventData.getEvent();
        if (newEvent.isFieldSet(EventFields.START_TS)) {
            if (!ObjectUtils.equals(oldEventInstanceInterval.getStart(), newEvent.getStartTs()))
                return true;
        }
        if (newEvent.isFieldSet(EventFields.END_TS)) {
            if (!ObjectUtils.equals(oldEventInstanceInterval.getEnd(), newEvent.getEndTs())) {
                return true;
            }
        }
        return false;
    }

    private EventUser eventUserChanges(
            EventInstanceForUpdate eventInstance, EventData newEventData)
    {
        final EventUserData eventUserData = newEventData.getEventUserData();
        Option<EventUserWithNotifications> oldEventUserWithNotifications = eventInstance.getEventUserWithNotifications();
        if (!oldEventUserWithNotifications.isPresent()) {
            return eventUserData.getEventUser().copy();
        } else {
            return EventUserHelper.INSTANCE.findChanges(
                    oldEventUserWithNotifications.get().getEventUser(),
                    eventUserData.getEventUser());
        }
    }

    private Repetition repetitionChanges(
            EventInstanceForUpdate eventInstance, EventData newEventData)
    {
        return repetitionChanges(Option.of(eventInstance.getRepetitionInstanceInfo()), newEventData.getRepetition());
    }

    public static Repetition repetitionChanges(Option<RepetitionInstanceInfo> oldInfo, Repetition newRepetition) {
        Repetition oldRepetition = oldInfo
                .map(RepetitionInstanceInfo::getRepetitionOrNone)
                .getOrElse(RepetitionRoutines::createNoneRepetition);

        if (oldRepetition.getType() != newRepetition.getType()) {
            return newRepetition;
        } else {
            return RepetitionHelper.INSTANCE.findChanges(oldRepetition, newRepetition);
        }
    }

    private Tuple2<Option<Long>, Boolean> eventLayerChanges(EventInstanceForUpdate eventInstance, EventData newEventData) {
        final Option<EventLayer> oldEventLayerO = eventInstance.getEventLayer();
        final Option<Long> newLayerId = newEventData.getLayerId();
        if (newLayerId.isPresent()
            && (!oldEventLayerO.isPresent() || changed(oldEventLayerO.get().getLayerId(), newLayerId.get())))
        {
            return Tuple2.tuple(newLayerId, true);
        } else {
            return Tuple2.tuple(Option.empty(), !oldEventLayerO.isPresent());
        }
    }

    private static <A, B extends A> boolean changed(A oldValue, B newValue) {
        return !ObjectUtils.equals(oldValue, newValue);
    }
}
