package ru.yandex.calendar.logic.event;

import java.util.EnumSet;
import java.util.List;
import java.util.function.Function;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.frontend.ews.ExchangeData;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.event.attachment.EventAttachmentChangesInfo;
import ru.yandex.calendar.logic.event.meeting.MeetingMailRecipients;
import ru.yandex.calendar.logic.event.repetition.RdateChangesInfo;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.EventParticipantsChangesInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.misc.lang.Validate;

import static java.util.function.Predicate.not;

/**
 * @see EventChangesInfoForExchange
 */
@Slf4j
public class EventChangesInfo {
    private final Tuple2<Option<Long>, Boolean> newLayerIdAndChanged;
    private final Option<ExchangeData> exchangeData;
    @Getter
    private final boolean isEventInstanceStartEndTsChanged;

    private final Event eventChanges;
    private final Repetition repetitionChanges;
    private final EventParticipantsChangesInfo eventParticipantsChangesInfo;
    private final RdateChangesInfo rdateChangesInfo;
    private final EventUser eventUserChanges;
    private final Tuple2<NotificationsData.Update, Boolean> notificationsUpdateAndChanged;
    private final EventAttachmentChangesInfo attachmentChangesInfo;

    public final static EventChangesInfo EMPTY = new EventChangesInfoFactory().create();

    public static class EventChangesInfoFactory {
        private Tuple2<Option<Long>, Boolean> newLayerIdAndChanged = Tuple2.tuple(Option.empty(), false);
        private Option<ExchangeData> exchangeData = Option.empty();
        private boolean isEventInstanceStartEndTsChanged = false;
        private Event eventChanges = new Event();
        private Repetition repetitionChanges = new Repetition();
        private EventParticipantsChangesInfo eventParticipantsChangesInfo = EventParticipantsChangesInfo.EMPTY;
        private RdateChangesInfo rdateChangesInfo = RdateChangesInfo.EMPTY;
        private EventUser eventUserChanges = new EventUser();
        private EventAttachmentChangesInfo attachmentChangesInfo = EventAttachmentChangesInfo.NO_CHANGES;
        private Tuple2<NotificationsData.Update, Boolean> notificationsUpdateAndChanged =
                Tuple2.tuple(NotificationsData.notChanged(), false);

        public void setNewLayerIdAndChanged(Tuple2<Option<Long>, Boolean> newLayerIdAndChanged) {
            this.newLayerIdAndChanged = newLayerIdAndChanged;
        }
        public void setExchangeData(Option<ExchangeData> exchangeData) {
            this.exchangeData = exchangeData;
        }
        public void setEventInstanceStartEndTsChanged(boolean isEventInstanceStartEndTsChanged) {
            this.isEventInstanceStartEndTsChanged = isEventInstanceStartEndTsChanged;
        }
        public void setEventChanges(Event eventChanges) {
            this.eventChanges = eventChanges;
        }
        public void setRepetitionChanges(Repetition repetitionChanges) {
            this.repetitionChanges = repetitionChanges;
        }
        public void setEventParticipantsChangesInfo(
                EventParticipantsChangesInfo eventParticipantsChangesInfo) {
            this.eventParticipantsChangesInfo = eventParticipantsChangesInfo;
        }
        public void setRdateChangesInfo(RdateChangesInfo rdateChangesInfo) {
            this.rdateChangesInfo = rdateChangesInfo;
        }
        public void setEventUserChanges(EventUser eventUserChanges) {
            this.eventUserChanges = eventUserChanges;
        }
        public void setAttachmentChangesInfo(EventAttachmentChangesInfo attachmentChangesInfo) {
            this.attachmentChangesInfo = attachmentChangesInfo;
        }
        public void setNotificationsUpdateAndChanged(NotificationsData.Update update, boolean changed) {
            this.notificationsUpdateAndChanged = Tuple2.tuple(update, changed);
        }

        public EventChangesInfo create() {
            return new EventChangesInfo(
                    eventParticipantsChangesInfo,
                    rdateChangesInfo, eventChanges,
                    eventUserChanges, repetitionChanges,
                    newLayerIdAndChanged, exchangeData,
                    attachmentChangesInfo,
                    isEventInstanceStartEndTsChanged,
                    notificationsUpdateAndChanged);
        }
    }

    private EventChangesInfo(
            EventParticipantsChangesInfo eventParticipantsChangesInfo,
            RdateChangesInfo rdateChangesInfo, Event eventChanges,
            EventUser eventUserChanges, Repetition repetitionChanges,
            Tuple2<Option<Long>, Boolean> newLayerIdAndChanged,
            Option<ExchangeData> exchangeData,
            EventAttachmentChangesInfo attachmentChangesInfo,
            boolean isEventInstanceStartEndTsChanged,
            Tuple2<NotificationsData.Update, Boolean> notificationsUpdateAndChanged)
    {
        this.eventParticipantsChangesInfo = eventParticipantsChangesInfo;
        this.rdateChangesInfo = rdateChangesInfo;
        this.eventChanges = eventChanges;
        this.eventUserChanges = eventUserChanges;
        this.repetitionChanges = repetitionChanges;
        this.newLayerIdAndChanged = newLayerIdAndChanged;
        this.exchangeData = exchangeData;
        this.isEventInstanceStartEndTsChanged = isEventInstanceStartEndTsChanged;
        this.notificationsUpdateAndChanged = notificationsUpdateAndChanged;
        this.attachmentChangesInfo = attachmentChangesInfo;


        Validate.isTrue(!eventChanges.isFieldSet(EventFields.ID));
        Validate.isTrue(!repetitionChanges.isFieldSet(RepetitionFields.ID));
    }

    public EventChangesInfoForExchange toEventChangesInfoForExchange() {
        val newExdates = StreamEx.of(getRdateChangesInfo().getNewRdates())
                .remove(Rdate::getIsRdate)
                .map(Rdate::getStartTs)
                .collect(CollectorsF.toList());
        return new EventChangesInfoForExchange(
                getEventChanges(), getRepetitionChanges(),
                getEventParticipantsChangesInfo().wasChange(),
                newExdates);
    }

    public EventChangesInfoForMails toEventChangesInfoForMails() {
        boolean nameChanged = eventChanges.isFieldSet(EventFields.NAME);
        boolean descriptionChanged = eventChanges.isFieldSet(EventFields.DESCRIPTION);
        boolean timeOrRepetitionChanged = timeOrRepetitionChanges();

        EventParticipantsChangesInfo participantsChanges = eventParticipantsChangesInfo;
        boolean locationChanged = eventChanges.isFieldSet(EventFields.LOCATION) || participantsChanges.wasResourcesChange();

        return new EventChangesInfoForMails(
                nameChanged, descriptionChanged, locationChanged, timeOrRepetitionChanged, participantsChanges);
    }

    public boolean wasChange() {
        return wasNonPerUserFieldsChange() || wasPerUserFieldsChange();
    }

    public boolean wasPerUserFieldsChange() {
        return eventUserChanges.isNotEmpty() ||
            notificationsUpdateAndChanged.get2() ||
            eventLayerChanges() ||
            exchangeIdChanges();
    }

    public EventChangesInfo withoutRepetitionChanges() {
        return new EventChangesInfo(
                eventParticipantsChangesInfo, rdateChangesInfo, eventChanges, eventUserChanges,
                new Repetition(), newLayerIdAndChanged, exchangeData, attachmentChangesInfo,
                isEventInstanceStartEndTsChanged, notificationsUpdateAndChanged);
    }

    public EventChangesInfo withoutDeletedExdates() {
        return new EventChangesInfo(
                eventParticipantsChangesInfo, rdateChangesInfo.withoutDeleted(), eventChanges, eventUserChanges,
                repetitionChanges, newLayerIdAndChanged, exchangeData, attachmentChangesInfo,
                isEventInstanceStartEndTsChanged, notificationsUpdateAndChanged);
    }

    public EventChangesInfo onlyPerUserChanges() {
        logIngoredChanges(EnumSet.noneOf(EventChangesPart.class));
        return new EventChangesInfo(EventParticipantsChangesInfo.EMPTY, RdateChangesInfo.EMPTY, new Event(),
                eventUserChanges, new Repetition(), newLayerIdAndChanged,
                exchangeData, EventAttachmentChangesInfo.NO_CHANGES, false, notificationsUpdateAndChanged);
    }

    public EventChangesInfo onlyInvitationPermittedChanges() {
        logIngoredChanges(EnumSet.of(EventChangesPart.ADDED_PARTICIPANTS, EventChangesPart.UPDATED_PARTICIPANTS));
        return new EventChangesInfo(eventParticipantsChangesInfo.withoutRemovedParticipants(),
                RdateChangesInfo.EMPTY, new Event(), eventUserChanges, new Repetition(), newLayerIdAndChanged,
                exchangeData, EventAttachmentChangesInfo.NO_CHANGES, false, notificationsUpdateAndChanged);
    }

    public EventChangesInfo onlyPerUserAndEventInstanceChanges() {
        logIngoredChanges(EnumSet.of(EventChangesPart.EVENT_CHANGES, EventChangesPart.EVENT_START_AND_END_CHANGES));
        return new EventChangesInfo(EventParticipantsChangesInfo.EMPTY, RdateChangesInfo.EMPTY, eventChanges,
                eventUserChanges, new Repetition(), newLayerIdAndChanged,
                exchangeData, EventAttachmentChangesInfo.NO_CHANGES,
                isEventInstanceStartEndTsChanged, notificationsUpdateAndChanged);
    }

    @AllArgsConstructor
    private enum EventChangesPart {
        ADDED_PARTICIPANTS("addedParticipants", EventChangesInfo::getAddedParticipants),
        UPDATED_PARTICIPANTS("updatedParticipants", EventChangesInfo::getUpdatedParticipants),
        REMOVED_PARTICIPANTS("removedParticipants", EventChangesInfo::getRemovedParticipants),
        EVENT_START_AND_END_CHANGES("isStartOrEndChanged", EventChangesInfo::isEventInstanceStartEndTsChanged),
        EVENT_CHANGES("eventChanges", EventChangesInfo::getEventChanges),
        RDATE_CHANGES("rdateChanges", EventChangesInfo::getRdateChangesInfo),
        REPETITION_CHANGES("repetitionChanges", EventChangesInfo::getRepetitionChanges),
        ATTACHMENT_CHANGES("attachementChanges", EventChangesInfo::getAttachmentChangesInfo);

        private final String name;
        private final Function<EventChangesInfo, Object> getter;

        public String generateLogString(EventChangesInfo eventChangesInfo) {
            return name + "=" + getter.apply(eventChangesInfo);
        }
    }

    private String generateStringWithIgnoredChanges(EnumSet<EventChangesPart> ignoredChanges) {
        return StreamEx.of(ignoredChanges)
                .map(changesPart -> changesPart.generateLogString(this))
                .joining(", ");
    }

    private void logIngoredChanges(EnumSet<EventChangesPart> notIgnoredChanges) {
        log.info("Ignored the following changes: {}.",
                generateStringWithIgnoredChanges(EnumSet.complementOf(notIgnoredChanges)));
    }

    private List<ParticipantId> getUpdatedParticipants() {
        return eventParticipantsChangesInfo.getUpdatedInvitations().get1();
    }

    private List<ParticipantId> getAddedParticipants() {
        return eventParticipantsChangesInfo.getNewParticipants().get1();
    }

    private List<ParticipantId> getRemovedParticipants() {
        return StreamEx.of(eventParticipantsChangesInfo.getRemovedParticipants())
                .map(ParticipantInfo::getId)
                .toImmutableList();
    }

    public EventChangesInfo withParticipantDecision(UidOrResourceId participant, Decision decision) {
        return withParticipantsChangesInfo(eventParticipantsChangesInfo.applyParticipantDecision(participant, decision));
    }

    public EventChangesInfo withOnlySubjectMightBeRemoved(UidOrResourceId subjectId) {
        return withParticipantsChangesInfo(eventParticipantsChangesInfo.withOnlySubjectMightBeRemoved(subjectId));
    }

    public EventChangesInfo withParticipantsChangesInfo(EventParticipantsChangesInfo participantsChangesInfo) {
        return new EventChangesInfo(
                participantsChangesInfo,
                rdateChangesInfo, eventChanges, eventUserChanges, repetitionChanges, newLayerIdAndChanged,
                exchangeData, attachmentChangesInfo, isEventInstanceStartEndTsChanged, notificationsUpdateAndChanged);
    }

    public EventChangesInfo withAttachmentChangesInfo(EventAttachmentChangesInfo newAttachmentChangesInfo) {
        return new EventChangesInfo(
                eventParticipantsChangesInfo,
                rdateChangesInfo, eventChanges, eventUserChanges, repetitionChanges, newLayerIdAndChanged,
                exchangeData, newAttachmentChangesInfo, isEventInstanceStartEndTsChanged, notificationsUpdateAndChanged);
    }

    public EventChangesInfo withoutEventStartEndTsChanges() {
        Event changes = eventChanges.copy();
        changes.unsetField(EventFields.START_TS);
        changes.unsetField(EventFields.END_TS);

        return new EventChangesInfo(
                eventParticipantsChangesInfo, rdateChangesInfo, changes, eventUserChanges,
                repetitionChanges, newLayerIdAndChanged, exchangeData, attachmentChangesInfo, false,
                notificationsUpdateAndChanged);
    }

    public boolean wasNonPerUserFieldsChange() {
        return eventInstanceChanges() ||
            repetitionChanges.isNotEmpty() ||
            eventParticipantsChangesInfo.wasChange() ||
            rdateChangesInfo.wasChange() ||
            attachmentChangesInfo.wasChanges();
    }

    public boolean wasCuttingChange() {
        return isEventInstanceStartEndTsChanged
                || repetitionRuleChanges()
                || eventParticipantsChangesInfo.wasNewResources();
    }

    public boolean wasOnlyUserAttendeesNonPerUserFieldsChange() {
        return eventParticipantsChangesInfo.wasOnlyUserAttendeesChange()
                && !withParticipantsChangesInfo(EventParticipantsChangesInfo.EMPTY).wasNonPerUserFieldsChange();
    }

    public boolean wasOnlyAttendeesNonPerUserFieldsChange() {
        return eventParticipantsChangesInfo.wasOnlyAttendeesChange()
                && !withParticipantsChangesInfo(EventParticipantsChangesInfo.EMPTY).wasNonPerUserFieldsChange();
    }

    public boolean wasEnoughToIncrementSequenceChange() {
        return wasEnoughToIncrementSequenceChange(false);
    }

    public boolean wasEnoughToIncrementSequenceChange(boolean ignoreRepetition) {
        return !ignoreRepetition && repetitionChanges()
            || eventInstanceSomethingExceptStampsChanges()
            || eventParticipantsChangesInfo.wasParticipantsChange();
    }

    // to prevent notification participants about every little change (CAL-4417)
    public MeetingMailRecipients getMeetingMailRecipients() {
        if (isEventInstanceStartEndTsChanged || repetitionRuleChanges()) {
            return MeetingMailRecipients.ALL_PARTICIPANTS_AND_SUBSCRIBERS;
        }
        if (repetitionChanges()
            || eventChanges.isAnyFieldSet(EventFields.LOCATION, EventFields.NAME, EventFields.DESCRIPTION)
            || eventParticipantsChangesInfo.wasNewOrRemovedResources()
            || eventParticipantsChangesInfo.wasOrganizerChange())
        {
            return MeetingMailRecipients.NOT_REJECTED_PARTICIPANTS_AND_SUBSCRIBERS;
        }
        return MeetingMailRecipients.INVITED_AND_REMOVED;
    }

    public boolean isNewRecurrenceWithChangedParticipation(UidOrResourceId subjectId) {
        if (!repetitionChanges.getFieldValueO(RepetitionFields.TYPE).isSome(RegularRepetitionRule.NONE)) {
            return false;
        }
        if (!rdateChangesInfo.getNewRdates().isEmpty()) {
            return false;
        }

        if (eventParticipantsChangesInfo.exceptSubject(subjectId).wasParticipantsOrDecisionChange()) {
            return false;
        }
        if (isEventInstanceStartEndTsChanged) {
            return false;
        }

        ListF<MapField<?>> changedEventFields = Cf.toArrayList(EventFields.OBJECT_DESCRIPTION.getFields().filter(eventChanges.isFieldSetF()));
        changedEventFields.removeTs(EventFields.START_TS);
        changedEventFields.removeTs(EventFields.END_TS);
        changedEventFields.removeTs(EventFields.SEQUENCE);
        changedEventFields.removeTs(EventFields.LAST_UPDATE_TS);
        changedEventFields.removeTs(EventFields.DTSTAMP);

        return changedEventFields.length() == 1 && changedEventFields.single().equals(EventFields.RECURRENCE_ID);
    }

    public boolean isSameWithNewExdate() {
        return !eventParticipantsChangesInfo.wasParticipantsChange()
                && !repetitionChanges.isNotEmpty()
                && !eventInstanceSomethingExceptStampsChanges()
                && rdateChangesInfo.getNewRdates().size() == 1
                && rdateChangesInfo.getRemRdateIds().isEmpty()
                && rdateChangesInfo.getNewRdates().stream()
                    .anyMatch(not(Rdate::getIsRdate));
    }

    public boolean wasNonPerUserAndNonPartcipantsFieldsChange() {
        return eventParticipantsChangesInfo.participantsRemoved() ||
            eventInstanceChanges() ||
            repetitionChanges.isNotEmpty() ||
            rdateChangesInfo.wasChange();
    }

    private boolean eventInstanceChanges() {
        if (isEventInstanceStartEndTsChanged) {
            return true;
        }
        return StreamEx.of(eventChanges.getSetFields())
                    .without(EventFields.START_TS, EventFields.END_TS)
                    .findAny().isPresent();
    }

    private boolean eventInstanceSomethingExceptStampsChanges() {
        SetF<MapField<?>> ignored = Cf.<MapField<?>>set(
                EventFields.SEQUENCE, EventFields.LAST_UPDATE_TS, EventFields.DTSTAMP);

        return eventChanges.getMapObjectDescription().getFields()
                .exists(ignored.containsF().notF().andF(eventChanges.isFieldSetF()));
    }

    public boolean repetitionRuleChanges() {
        return RepetitionUtils.repetitionRuleChanged(repetitionChanges);
    }

    public boolean repetitionChanges() {
        return rdateChangesInfo.wasChange() || repetitionChanges.isNotEmpty();
    }

    public boolean repetitionChangesIgnoreRdates() {
        return repetitionChanges.isNotEmpty();
    }

    public boolean eventLayerChanges() {
        return newLayerIdAndChanged.get2();
    }

    // XXX: create real changes method for this check
    private boolean exchangeIdChanges() {
        return exchangeData.isPresent();
    }

    public boolean timeOrRepetitionChanges() {
        return repetitionChanges() || timeChanges();
    }

    public boolean timeOrRepetitionRuleChanges() {
        return repetitionRuleChanges() || timeChanges();
    }

    public boolean timeOrRepetitionRuleOrDueTsChanges() {
        return repetitionChanges.isNotEmpty() || timeChanges();
    }

    public boolean timeChanges() {
        return isEventInstanceStartEndTsChanged || eventChanges.isFieldSet(EventFields.IS_ALL_DAY);
    }

    public Option<Long> getNewLayerId() {
        return newLayerIdAndChanged.get1();
    }

    public EventParticipantsChangesInfo getEventParticipantsChangesInfo() {
        return eventParticipantsChangesInfo;
    }

    public Event getEventChanges() {
        return eventChanges;
    }

    public RdateChangesInfo getRdateChangesInfo() {
        return rdateChangesInfo;
    }

    public Repetition getRepetitionChanges() {
        return repetitionChanges;
    }

    public EventUser getEventUserChanges() {
        return eventUserChanges;
    }

    public EventAttachmentChangesInfo getAttachmentChangesInfo() {
        return attachmentChangesInfo;
    }

    public Option<ExchangeData> getExchangeData() {
        return exchangeData;
    }

    public NotificationsData.Update getNotificationsUpdate() {
        return notificationsUpdateAndChanged.get1();
    }

}
