package ru.yandex.calendar.logic.event;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventAttachmentFields;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventNotification;
import ru.yandex.calendar.logic.beans.generated.EventNotificationFields;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventResourceFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.event.attachment.EventAttachmentChangesInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventResourceUncheckinDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionConfirmationManager;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.logic.layer.LayerDbManager;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerUserDao;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.notification.NotificationDao;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.user.UserDao;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.base.UidGen;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Stepan Koltsov
 * @see LayerDbManager
 */
public class EventDbManager {

    private static final Logger logger = LoggerFactory.getLogger(EventDbManager.class);


    @Autowired
    private EventDao eventDao;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventInvitationDao eventInvitationDao;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private RepetitionConfirmationManager repetitionConfirmationManager;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private LayerDao layerDao;
    @Autowired
    private LayerUserDao layerUserDao;
    @Autowired
    private UserDao userDao;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private NotificationDao notificationDao;
    @Autowired
    private EventResourceUncheckinDao eventResourceUncheckinDao;

    @Value("${updateEvent.bigEventsParticipants}")
    private Long bigEventsThreshold;

    /** Excluding resource ones */
    private final Cache<Tuple2<Long, PassportUid>, Option<EventLayer>> eventLayerByEventIdAndLayerCreatorUidCache = TlCache.asCache(LayerRoutines.class.getName() + ".eventLayerByEventIdAndLayerCreatorUid");

    void putEventLayersToCache(ListF<EventLayer> eventLayers) {
        for (EventLayer eventLayer : eventLayers) {
            eventLayerByEventIdAndLayerCreatorUidCache.putInCache(
                    new Tuple2<Long, PassportUid>(eventLayer.getEventId(), eventLayer.getLCreatorUid()),
                    Option.of(eventLayer));
        }
    }

    /**
     * @see LayerRoutines#getLayersById(ListF)
     */
    public ListF<Event> getEventsByIds(ListF<Long> ids) {
        return eventDao.findEventsByIds(ids);
    }

    public ListF<Event> getEventsByIdsSafe(ListF<Long> ids) {
        return eventDao.findEventsByIdsSafe(ids);
    }

    /**
     * @see LayerRoutines#getLayerById(long)
     */
    public Event getEventById(long id) {
        return getEventsByIds(Cf.list(id)).single();
    }

    public Event getEventByIdForUpdate(long id) {
        return getEventByIdSafeForUpdate(id).single();
    }

    public Option<Event> getEventByIdSafe(long id) {
        return getEventByIdSafe(id, false);
    }

    public Option<Event> getEventByIdSafeForUpdate(long id) {
        return getEventByIdSafe(id, true);
    }

    public Option<Event> getEventByIdSafe(long id, boolean forUpdate) {
        return eventDao.findEventsByIdsSafe(Cf.list(id), forUpdate).singleO();
    }

    public ListF<Event> getEventsByMainEventIdForUpdate(long mainEventId) {
        return eventDao.findEventsByMainId(mainEventId, true);
    }

    public EventAndRepetition getEventAndRepetitionById(long eventId) {
        return getEventAndRepetitionById(eventId, false);
    }

    public EventAndRepetition getEventAndRepetitionByIdForUpdate(long eventId) {
        return getEventAndRepetitionById(eventId, true);
    }

    public EventAndRepetition getEventAndRepetitionById(long eventId, boolean lockForUpdate) {
        return getEventsAndRepetitionsByEvents(
                Option.of(eventDao.findEventById(eventId, lockForUpdate))).single();
    }

    public EventAndRepetition getEventAndRepetitionByEvent(Event event) {
        return getEventsAndRepetitionsByEvents(Cf.list(event)).single();
    }

    public ListF<EventAndRepetition> getEventsAndRepetitionsByEventIds(ListF<Long> eventIds) {
        return getEventsAndRepetitionsByEventIds(eventIds, false);
    }

    public ListF<EventAndRepetition> getEventsAndRepetitionsByEventIdsForUpdate(ListF<Long> eventIds) {
        return getEventsAndRepetitionsByEventIds(eventIds, true);
    }

    public ListF<EventAndRepetition> getEventsAndRepetitionsByEventIds(ListF<Long> eventIds, boolean lockForUpdate) {
        return getEventsAndRepetitionsByEvents(eventDao.findEventsByIds(eventIds, lockForUpdate));
    }

    public ListF<EventAndRepetition> getEventsAndRepetitionsByEvents(ListF<Event> events) {
        MapF<Long, RepetitionInstanceInfo> repetitionById =
                repetitionRoutines.getRepetitionInstanceInfosByEvents(events);

        return events.map(e -> new EventAndRepetition(e, repetitionById.getOrThrow(e.getId())));
    }

    public ListF<EventAndRepetition> getEventsAndRepetitionsByMainEventId(long mainEventId) {
        return getEventsAndRepetitionsByMainEventId(mainEventId, false);
    }

    public ListF<EventAndRepetition> getEventsAndRepetitionsByMainEventIdForUpdate(long mainEventId) {
        return getEventsAndRepetitionsByMainEventId(mainEventId, true);
    }

    public ListF<EventAndRepetition> getEventsAndRepetitionsByMainEventId(long mainEventId, boolean lockForUpdate) {
        ListF<Event> events = eventDao.findEventsByMainId(mainEventId, lockForUpdate);
        int usersInMainEvent = eventUserDao.findEventUsersByEventId(mainEventId).size();
        if (usersInMainEvent > bigEventsThreshold) {
            events = events.filter(ev -> ev.getStartTs().isAfter(Instant.now().minus(org.joda.time.Duration.standardDays(1))));
        }
        MapF<Long, RepetitionInstanceInfo> repetitionById =
                repetitionRoutines.getRepetitionInstanceInfosAmongEvents(events);

        return events.map(e -> new EventAndRepetition(e, repetitionById.getOrThrow(e.getId())));
    }

    public ListF<EventWithRelations> getEventsWithRelationsByIds(ListF<Long> eventIds, boolean withSubscribers) {
        ListF<Event> events = eventDao.findEventsByIds(eventIds);
        return getEventsWithRelationsByEvents(events, withSubscribers);
    }

    public ListF<EventWithRelations> getEventsWithRelationsByIds(ListF<Long> eventIds) {
        return getEventsWithRelationsByIds(eventIds, true);
    }

    public ListF<EventWithRelations> getEventsWithRelationsByIdsSafe(ListF<Long> eventIds) {
        ListF<Event> events = eventDao.findEventsByIdsSafe(eventIds);
        return getEventsWithRelationsByEvents(events);
    }

    public EventWithRelations getEventWithRelationsById(long eventId) {
        return getEventsWithRelationsByIds(Cf.list(eventId)).single();
    }

    public ListF<EventWithRelations> getEventsWithRelationsByEvents(ListF<Event> events) {
        return getEventsWithRelationsByEvents(events, true);
    }

    public ListF<EventWithRelations> getEventsWithRelationsByEvents(ListF<Event> events, boolean withSubscribers) {
        return getEventsWithRelationsByEvents(events, Option.empty(), withSubscribers);
    }

    public ListF<EventWithRelations> getEventsWithRelationsByEvents(ListF<Event> events, Option<Long> layerIdO, boolean withSubscribers) {
        if (events.isEmpty()) return Cf.list();

        ListF<Long> eventIds = events.map(EventFields.ID.getF());

        MapF<Long, MainEvent> mainEventsById = mainEventDao.findMainEventsByIds(events.map(Event.getMainEventIdF()))
                .toMapMappingToKey(MainEvent.getIdF());
//findEventLayersWithRelations
        MapF<Long, ListF<EventLayerWithRelations>> eventLayersByEventId =
                eventLayerDao.findEventLayersWithRelationsByEventIds(eventIds, layerIdO)
                        .groupBy(EventLayerWithRelations::getEventId);

        MapF<Long, EventParticipants> participantsByEventId = getParticipantsByEventIds(eventIds, withSubscribers)
                .toMap(EventParticipants::getEventId, Function.identityF());

        return events.map(event -> new EventWithRelations(
                event, mainEventsById.getOrThrow(event.getMainEventId()),
                participantsByEventId.getOrThrow(event.getId()),
                new EventLayers(eventLayersByEventId.getOrElse(event.getId(), Cf.list()))));
    }

    public ListF<EventParticipants> getParticipantsByEventIds(ListF<Long> eventIds) {
        return getParticipantsByEventIds(eventIds, true);
    }

    public ListF<EventParticipants> getParticipantsByEventIds(ListF<Long> eventIds, boolean withSubscribers) {
        MapF<Long, ListF<EventUserWithRelations>> eventUsersByEventId = eventUserRoutines
                .findEventUsersWithRelationsByEventIds(eventIds, withSubscribers)
                .groupBy(EventUserWithRelations::getEventId);
        MapF<Long, ListF<EventInvitation>> invitationsByEventId =
                eventInvitationDao.findEventInvitationsByEventIds(eventIds).groupBy(EventInvitation::getEventId);

        ListF<EventResource> eventResources = eventResourceDao.findEventResourcesByEventIds(eventIds);

        MapF<Long, ListF<EventResource>> eventResourcesByEventId = eventResources.groupBy(EventResource::getEventId);
        MapF<Long, ResourceInfo> resourcesByIds = resourceDao.findResourceInfosByIds(
                eventResources.map(EventResource::getResourceId)).toMapMappingToKey(ResourceInfo.resourceIdF());

        return eventIds.map(id -> new EventParticipants(id,
                eventUsersByEventId.getOrElse(id, Cf.list()),
                invitationsByEventId.getOrElse(id, Cf.list()),
                eventResourcesByEventId.getOrElse(id, Cf.list())
                        .filterMap(er -> resourcesByIds.getO(er.getResourceId())),
                eventResourcesByEventId.getOrElse(id, Cf.list())));
    }

    public EventWithRelations getEventWithRelationsByEvent(Event event) {
        return getEventsWithRelationsByEvents(Cf.list(event)).single();
    }

    public MainEventWithRelations getMainEventWithRelationsById(long mainEventId) {
        return getMainEventWithRelationsByMainEvent(mainEventDao.findMainEventById(mainEventId));
    }

    public MainEventWithRelations getMainEventWithRelationsByMainEvent(MainEvent mainEvent) {
        ListF<Event> events = eventDao.findEventsByMainId(mainEvent.getId());
        return new MainEventWithRelations(mainEvent, getEventsWithRelationsByEvents(events));
    }

    public MainEventWithRelations getMainEventWithRelationsByMainEventForUpdate(MainEvent mainEvent) {
        ListF<Event> events = eventDao.findEventsByMainId(mainEvent.getId(), true);
        return new MainEventWithRelations(mainEvent, getEventsWithRelationsByEvents(events));
    }

    public Tuple2List<Long, ListF<ResourceInfo>> findResourcesByEventIds(ListF<Long> eventIds) {
        ListF<EventResource> eventResources = eventResourceDao.findEventResourcesByEventIds(eventIds);

        MapF<Long, ResourceInfo> resourceById = resourceDao
                .findResourceInfosByIds(eventResources.map(EventResource.getResourceIdF()).stableUnique())
                .toMapMappingToKey(ResourceInfo.resourceIdF());

        Tuple2List<Long, ResourceInfo> eventIdsAndResources = eventResources.toTuple2List(
                EventResource.getEventIdF(), er -> resourceById.getOrThrow(er.getResourceId()));

        MapF<Long, ListF<ResourceInfo>> resourcesByEventId = eventIdsAndResources.groupBy1();

        return eventIds.zipWith(id -> resourcesByEventId.getOrElse(id, Cf.list()));
    }

    /**
     * @see #updateMainEventAndLayerTimestamps(long, ActionInfo)
     */
    public void updateEvent(Event event, ActionInfo actionInfo) {
        Validate.isTrue(event.isFieldSet(EventFields.ID));
        if (event.cardinality() > 1) {
            eventDao.updateEvent(event);
            if (actionInfo.getActionSource().isWebOrApi()) {
                eventDao.updateEventLastUpdateTsAndEventModificationInfo(
                        event.getId(), actionInfo.getNow(), actionInfo);
            } else {
                if (event.isFieldSet(EventFields.LAST_UPDATE_TS)) {
                    eventDao.updateEventLastUpdateTsAndEventModificationInfo(
                        event.getId(), event.getLastUpdateTs(), actionInfo);
                } else {
                    logger.warn("No last update ts from " + actionInfo.getActionSource());
                    eventDao.updateEventModificationInfo(event.getId(), actionInfo);
                }
            }
        }
    }

    public Event incrementEventSequenceAndModificationInfo(Event event, ActionInfo actionInfo) {
        Event temp = new Event();
        temp.setSequence(event.getSequence() + 1);
        temp.setLastUpdateTs(actionInfo.getNow());
        temp.setModificationSource(actionInfo.getActionSource());
        temp.setModificationReqId(actionInfo.getRequestIdWithHostId());
        temp.setId(event.getId());

        eventDao.updateEvent(temp);

        Event updated = event.copy();
        updated.setFields(temp);
        return updated;
    }

    public Option<EventLayer> getEventLayerForEventAndUser(final long eventId, final PassportUid layerCreatorUid) {
        return getEventLayersForEventsAndUser(Cf.list(eventId), layerCreatorUid).singleO();
    }

    public ListF<EventLayer> getEventLayersForEventsAndUser(ListF<Long> eventIds, final PassportUid layerCreatorUid) {
        return getEventLayersForEventsAndUsers(eventIds, Cf.list(layerCreatorUid));
    }

    public ListF<EventLayer> getEventLayersForEventsAndUsers(ListF<Long> eventIds, ListF<PassportUid> layerCreatorUids) {
        Function<ListF<Tuple2<Long, PassportUid>>, ListF<Option<EventLayer>>> populateF = missingIds -> {
            ListF<EventLayer> found = eventLayerDao.findEventLayersByEventIdsAndLayerCreatorUids(
                    missingIds.map(Tuple2::get1), missingIds.map(Tuple2::get2));

            return missingIds.map(found.toMapMappingToKey(el -> Tuple2.tuple(el.getEventId(), el.getLCreatorUid()))::getO);
        };

        ListF<Tuple2<Long, PassportUid>> ids = Cf2.join(eventIds, layerCreatorUids);
        return Cf2.flatBy2(eventLayerByEventIdAndLayerCreatorUidCache.getFromCacheSomeBatch(ids, populateF)).get2();
    }

    // Obtains start millis of the first event instance for given event information.
    // Includes rdates / excludes exdates, recurrence-ids.
    // But NOTE that recurrence-id instances are not put to the output, even for mainEvent.
    public Instant getFirstEventInstanceStart(long id) {
        Event event = getEventById(id);
        if (!event.getRepetitionId().isPresent()) { // includes recurrence-id case
            // TODO support strange case: event without repetition, with single recurrence-id at it
            // XXX: rdates without repetition are not supported
            return event.getStartTs();
        }
        // Here we deal with main event with repetition
        RepetitionInstanceInfo repetitionInstanceInfo = repetitionRoutines.getRepetitionInstanceInfoByEvent(event);
        Instant lookupStart = event.getStartTs();
        // TODO if exists rdate: rdate.start < event.start, then use it
        Option<InstantInterval> intervalO = repetitionRoutines.getSingleInterval(event, repetitionInstanceInfo, lookupStart, false);
        // XXX: none possible?
        return intervalO.get().getStart();
    }

    // NOTE subjectId == eventData.getExchangeData().get().getSubjectId(). For now, get() always succeeds // ssytnik@
    public ListF<UidOrResourceId> getSubjectsAffectedByUpdateData(UidOrResourceId subjectId, EventData eventData) {
        ListF<UidOrResourceId> r = Cf.arrayList();

        ListF<Email> emailsFromEventData = eventData.getInvData().getParticipantEmails();
        r.addAll(eventInvitationManager.getSubjectsFromEmails(emailsFromEventData));
        r.add(subjectId);

        return r;
    }

    public void saveEventLayer(EventLayer eventLayer, ActionInfo actionInfo) {
        saveEventLayers(Cf.list(eventLayer), actionInfo);
    }

    public void saveEventLayers(ListF<EventLayer> eventLayers, ActionInfo actionInfo) {
        eventLayerDao.saveEventLayersBatch(eventLayers.map(el -> {
            el = el.copy();
            el.setCreationSource(actionInfo.getActionSource());
            el.setCreationReqId(actionInfo.getRequestIdWithHostId());

            return el;
        }));
        removeEventLayersFromCache(eventLayers);
    }

    public void updateEventLayer(EventLayer eventLayer, ActionInfo actionInfo) {
        updateEventLayerByEventIdAndLayerId(eventLayer, eventLayer.getEventId(), eventLayer.getLayerId(), actionInfo);
    }

    public void updateEventLayerByEventIdAndLayerId(
            EventLayer eventLayer, long eventId, long layerId, ActionInfo actionInfo)
    {
        EventLayer data = eventLayer.copy();

        data.setModificationSource(actionInfo.getActionSource());
        data.setModificationReqId(actionInfo.getRequestIdWithHostId());

        eventLayerDao.updateEventLayerByEventIdAndLayerId(data, eventId, layerId);
        removeEventLayersFromCache(Cf.list(eventLayer));
    }

    void removeEventLayersFromCache(ListF<EventLayer> eventLayers) {
        ListF<Long> eventIds = eventLayers.map(EventLayer.getEventIdF());
        ListF<PassportUid> lCreatorUids = eventLayers.map(EventLayer.getLCreatorUidF());
        for (Tuple2<Long, PassportUid> key : eventIds.zip(lCreatorUids)) {
            eventLayerByEventIdAndLayerCreatorUidCache.removeFromCache(key);
        }
    }

    public void saveEventResource(EventResource eventResource, ActionInfo actionInfo) {
        saveEventResources(Cf.list(eventResource), actionInfo);
    }

    public void saveEventResources(ListF<EventResource> eventResources, ActionInfo actionInfo) {
        eventResourceDao.saveEventResourcesBatch(eventResources.map(er -> {
            er = er.copy();
            er.setCreationSource(actionInfo.getActionSource());
            er.setCreationReqId(actionInfo.getRequestIdWithHostId());

            return er;
        }));
        // TODO: cache
    }

    public EventAttachedLayerId saveEventLayerIfAbsentForUser(
            PassportUid uid, EventAndRepetition event, ActionInfo actionInfo)
    {
        Option<Tuple2<EventLayer, Layer>> eventLayerO =
                eventLayerDao.findEventLayerWithLayerByEventIdAndLayerCreatorUid(event.getEventId(), uid);
        if (!eventLayerO.isPresent()) {
            ListF<Long> ids = eventDao.findEventIdsByMainEventId(event.getMainEventId());
            long layerId = eventLayerDao
                    .findEventLayersByEventIdsAndLayerCreatorUid(ids, uid).firstO().map(EventLayer.getLayerIdF())
                    .getOrElse(layerRoutines.getOrCreateDefaultLayer(uid));

            saveEventLayer(uid, event, layerId, actionInfo);
            return EventAttachedLayerId.attached(layerId);

        } else {
            return EventAttachedLayerId.alreadyAttached(eventLayerO.get()._1.getLayerId());
        }
    }

    public void saveEventLayerForDefaultLayer(PassportUid uid, EventAndRepetition event, ActionInfo actionInfo) {
        saveEventLayer(uid, event, layerRoutines.getOrCreateDefaultLayer(uid), actionInfo);
    }

    private void saveEventLayer(PassportUid uid, EventAndRepetition event, long layerId, ActionInfo actionInfo) {
        event.validateIsLockedForUpdate();

        EventLayer eventLayer = new EventLayer();
        eventLayer.setEventId(event.getEventId());
        eventLayer.setLayerId(layerId);

        setIndentsData(eventLayer, event);

        eventLayer.setLCreatorUid(uid);
        eventLayer.setCreationSource(actionInfo.getActionSource());
        eventLayer.setCreationReqId(actionInfo.getRequestIdWithHostId());
        eventLayerDao.saveEventLayer(eventLayer);
    }

    public void updateEventLayersAndResourcesIndents(EventAndRepetition event, ActionInfo actionInfo) {
        event.validateIsLockedForUpdate();

        EventLayer eventLayerData = new EventLayer();
        setIndentsData(eventLayerData, event);

        eventLayerData.setModificationReqId(actionInfo.getRequestIdWithHostId());
        eventLayerData.setModificationSource(actionInfo.getActionSource());

        eventLayerDao.updateEventLayersByEventId(eventLayerData, event.getEventId());

        EventResource eventResourceData = new EventResource();
        setIndentsData(eventResourceData, event);

        eventResourceData.setModificationReqId(actionInfo.getRequestIdWithHostId());
        eventResourceData.setModificationSource(actionInfo.getActionSource());

        eventResourceDao.updateEventResourcesByEventId(eventResourceData, event.getEventId());
    }

    public void deleteUserCreatedEvents(ListF<PassportUid> uids, ActionInfo actionInfo) {
        ListF<Event> events = eventDao.findEvents(EventFields.CREATOR_UID.column().inSet(uids));
        deleteEventsByIds(events.map(EventFields.ID.getF()), true, actionInfo);
    }

    private void deleteEventUsersByEventIds(ListF<Long> eventIds) {
        ListF<EventUser> eventUsers = eventUserDao.findEventUsersByEventIds(eventIds);

        notificationDbManager.deleteNotificationsByEventUserIds(eventUsers.map(EventUser.getIdF()));
        eventUserDao.deleteEventUsersByIds(eventUsers.map(EventUserFields.ID.getF()));
    }

    public void deleteEventUsersByUid(ListF<PassportUid> uids) {
        ListF<EventUser> eventUsers = eventUserDao.findEventUsers(EventUserFields.UID.column().inSet(uids));

        notificationDbManager.deleteNotificationsByEventUserIds(eventUsers.map(EventUser.getIdF()));
        eventUserDao.deleteEventUsersByIds(eventUsers.map(EventUserFields.ID.getF()));
    }


    public void deleteNotOrganizerEventUsersByUid(ListF<PassportUid> uids) {
        ListF<EventUser> eventUsers = eventUserDao.findEventUsers(
                EventUserFields.UID.column().inSet(uids).and(EventUserFields.IS_ORGANIZER.eq(false)));

        notificationDbManager.deleteNotificationsByEventUserIds(eventUsers.map(EventUser.getIdF()));
        eventUserDao.deleteEventUsersByIds(eventUsers.map(EventUserFields.ID.getF()));
    }


    // https://jira.yandex-team.ru/browse/CAL-3293
    // use EventRoutines#deleteEventFromDbAndExchange(long) instead // ssytnik@
    @Deprecated
    public void deleteEventById(long eventId) {
        deleteEventsByIds(Cf.list(eventId), true, ActionInfo.adminManager());
    }

    public void deleteEventsByIds(
            ListF<Long> eventIds, boolean deleteMainEventsIfOrphaned, ActionInfo actionInfo)
    {
        ListF<EventResource> eventResources = eventResourceDao.findEventResourcesByEventIds(eventIds);
        ListF<Long> resourcesIds = eventResources.map(EventResourceFields.RESOURCE_ID.getF()).stableUnique();
        resourceScheduleManager.invalidateCachedScheduleForResources(resourcesIds);

        ListF<Long> mainEventIds = eventDao.findMainEventIdsByEventIds(eventIds);

        deleteEventUsersByEventIds(eventIds);

        lastUpdateManager.updateMainEventAndLayerTimestamps(eventIds, actionInfo);

        eventLayerDao.deleteEventLayersByEventIds(eventIds);
        eventLayerByEventIdAndLayerCreatorUidCache.flush();
        eventResourceDao.deleteEventResourcesByEventIds(eventIds);

        eventInvitationDao.deleteInvitationsByEventIds(eventIds);
        repetitionConfirmationManager.deleteConfirmationsByEventIds(eventIds);

        eventDao.deleteRdatesByEventIds(eventIds);
        eventDao.deleteRepetitionsByIds(eventDao.findRepetitionIdsByEventIds(eventIds));
        eventDao.deleteEventAttachmentsByEventIds(eventIds);
        eventDao.deleteEventsByIds(eventIds);

        if (deleteMainEventsIfOrphaned) {
            mainEventDao.deleteTimezoneInfosByMainEventIdsIfOrphaned(mainEventIds);
            mainEventDao.deleteMainEventsIfOrphaned(mainEventIds);
            eventResourceUncheckinDao.deleteByMainEventIds(mainEventIds);
        }
    }

    public DatabaseDump loadDatabase() {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_SM, () -> new DatabaseDump(
                mainEventDao.findMainEvents(),
                eventDao.findEvents(),
                eventUserDao.findEventUsers(),
                eventLayerDao.findEventLayers(),
                eventInvitationDao.findEventInvitations(),
                layerDao.findLayers(),
                layerUserDao.findLayerUsers(),
                userDao.findSettings(),
                resourceDao.findResources(),
                resourceDao.findOffices(PassportDomain.YANDEX_TEAM_RU),
                eventResourceDao.findEventResources()
        ));
    }

    public long cloneEventWithDependents(
            EventWithRelations event,
            ListF<EventAttachment> attachments,
            Event eventOverrides,
            EventAttachmentChangesInfo attachmentChangesInfo,
            ActionInfo actionInfo) {
        long newEventId = cloneEvent(event.getEvent(), eventOverrides, actionInfo);

        Event newEvent = eventOverrides.copy();
        newEvent.setId(newEventId);

        ListF<EventUserWithNotifications> eventUsers = notificationDbManager
                .getEventUserWithNotificationsByEventUsers(event.getEventUsers());

        cloneEventDependentInstances(
                eventUsers,
                event.getInvitations(),
                event.getEventResources(),
                event.getEventLayers(),
                attachments,
                newEvent,
                attachmentChangesInfo,
                actionInfo);

        return newEventId;
    }

    private long cloneEvent(Event event, Event eventOverrides, ActionInfo actionInfo) {
        Event cloneData = event.copy();
        cloneData.setFields(eventOverrides);

        cloneData.unsetField(EventFields.ID);
        cloneData.unsetField(EventFields.MODIFICATION_SOURCE);
        cloneData.unsetField(EventFields.MODIFICATION_REQ_ID);
        cloneData.unsetField(EventFields.CREATION_TS);
        cloneData.unsetField(EventFields.LAST_UPDATE_TS);

        return eventDao.saveEvent(cloneData, actionInfo);
    }

    private void cloneEventUsersWithNotifications(
            ListF<EventUserWithNotifications> eventUsersNotifications, Event overrideEvent, ActionInfo actionInfo)
    {
        ListF<EventUser> eventUserClones = Cf.arrayList();

        for (EventUserWithNotifications eventUserNotifications : eventUsersNotifications) {
            EventUser clone = eventUserNotifications.getEventUser().copy();
            clone.unsetField(EventUserFields.ID);

            clone.setEventId(overrideEvent.getId());
            clone.setExchangeIdNull();
            clone.setPrivateToken(UidGen.createPrivateToken());
            clone.setLastSentSmsIdNull();
            clone.setLastSentTsNull();

            if (overrideEvent.isFieldSet(EventFields.SEQUENCE)) {
                clone.setSequence(overrideEvent.getSequence());
            }

            clone.setCreationSource(actionInfo.getActionSource());
            clone.setCreationReqId(actionInfo.getRequestIdWithHostId());
            clone.setLastUpdate(actionInfo.getNow());
            clone.setModificationReqIdNull();
            clone.setModificationSource(ActionSource.UNKNOWN);

            clone.setIcalXMozLastackNull();
            clone.setIcalXMozSnoozeTimeNull();

            eventUserClones.add(clone);
        }
        ListF<Long> clonedEventUserIds = eventUserDao.saveEventUsersBatch(eventUserClones, actionInfo);

        MapF<PassportUid, Long> clonedEventUserIdByOldId =
                eventUserClones.map(EventUser::getUid).zip(clonedEventUserIds).toMap();

        ListF<EventNotification> notificationClones = Cf.arrayList();

        for (EventUserWithNotifications eventUserNotifications : eventUsersNotifications) {
            for (EventNotification notification : eventUserNotifications.getNotifications().getEventNotifications()) {
                EventNotification clone = notification.copy();
                clone.unsetField(EventNotificationFields.ID);

                clone.setEventUserId(clonedEventUserIdByOldId.getOrThrow(eventUserNotifications.getEventUser().getUid()));
                clone.setNextSendTsNull();

                notificationClones.add(clone);
            }
        }
        notificationDao.saveEventNotificationsBatch(notificationClones);
    }

    private void cloneEventInvitations(ListF<EventInvitation> srcBeans, long eventId, ActionInfo actionInfo) {
        ListF<EventInvitation> clones = Cf.arrayList();

        for (EventInvitation bean : srcBeans) {
            EventInvitation clone = bean.copy();

            clone.setEventId(eventId);
            clone.setPrivateToken(UidGen.createPrivateToken());

            clone.setCreationTs(actionInfo.getNow());
            clone.setCreationSource(actionInfo.getActionSource());
            clone.setCreationReqId(actionInfo.getRequestIdWithHostId());

            clones.add(clone);
        }
        eventInvitationDao.saveEventInvitationsBatch(clones, actionInfo);
    }

    private void cloneEventLayers(ListF<EventLayer> eventLayers, Event overrideEvent, ActionInfo actionInfo) {
        ListF<EventLayer> clones = Cf.arrayList();

        for (EventLayer eventLayer : eventLayers) {
            EventLayer clone = eventLayer.copy();

            clone.setEventId(overrideEvent.getId());
            clone.setEventStartTs(overrideEvent.getStartTs());
            clone.setEventEndTs(overrideEvent.getEndTs());

            clone.setRepetitionDueTsNull();
            clone.setRdatesMinTsNull();
            clone.setRdatesMaxTsNull();

            clone.setCreationSource(actionInfo.getActionSource());
            clone.setCreationReqId(actionInfo.getRequestIdWithHostId());
            clone.setModificationReqIdNull();
            clone.setModificationSource(ActionSource.UNKNOWN);

            clones.add(clone);
        }
        eventLayerDao.saveEventLayersBatch(clones);
    }

    private void cloneEventResources(ListF<EventResource> eventResources, Event overrideEvent, ActionInfo actionInfo) {
        ListF<EventResource> clones = Cf.arrayList();

        for (EventResource eventResource : eventResources) {
            EventResource clone = eventResource.copy();

            clone.setEventId(overrideEvent.getId());
            clone.setEventStartTs(overrideEvent.getStartTs());
            clone.setEventEndTs(overrideEvent.getEndTs());

            clone.setRepetitionDueTsNull();
            clone.setRdatesMinTsNull();
            clone.setRdatesMaxTsNull();

            clone.setExchangeIdNull();

            clone.setCreationSource(actionInfo.getActionSource());
            clone.setCreationReqId(actionInfo.getRequestIdWithHostId());
            clone.setModificationReqIdNull();
            clone.setModificationSource(ActionSource.UNKNOWN);

            clones.add(clone);
        }
        eventResourceDao.saveEventResourcesBatch(clones);
    }

    private void cloneEventAttachments(
            ListF<EventAttachment> attachments,
            EventAttachmentChangesInfo attachmentChangesInfo,
            Event overrideEvent) {
        ListF<EventAttachment> clones = Cf.arrayList();

        for (EventAttachment attachment : attachments) {
            if (!attachmentChangesInfo.getRemoveAttachmentUrls().contains(attachment.getUrl())) {
                EventAttachment clone = attachment.copy();
                clone.unsetField(EventAttachmentFields.ID);
                clone.setEventId(overrideEvent.getId());
                clones.add(clone);
            }
        }

        for (EventAttachment attachment : attachmentChangesInfo.getNewAttachments()) {
            EventAttachment clone = attachment.copy();
            clone.setEventId(overrideEvent.getId());
            clones.add(clone);
        }

        eventDao.insertEventAttachments(clones);
    }

    private void cloneEventDependentInstances(
            ListF<EventUserWithNotifications> eventUsersWithNotifications, ListF<EventInvitation> invitations,
            ListF<EventResource> eventResources, ListF<EventLayer> eventLayers, ListF<EventAttachment> attachments,
            Event overrideEvent, EventAttachmentChangesInfo attachmentChangesInfo, ActionInfo actionInfo)
    {
        cloneEventUsersWithNotifications(eventUsersWithNotifications, overrideEvent, actionInfo);
        cloneEventInvitations(invitations, overrideEvent.getId(), actionInfo);
        cloneEventLayers(eventLayers, overrideEvent, actionInfo);
        cloneEventResources(eventResources, overrideEvent, actionInfo);
        cloneEventAttachments(attachments, attachmentChangesInfo, overrideEvent);
    }

    public boolean getIsExportedWithEws(Event event) {
        return mainEventDao.findMainEventById(event.getMainEventId()).getIsExportedWithEws().getOrElse(false);
    }

    public boolean isEventAttached(PassportUid uid, long eventId) {
        return getEventLayersForEventsAndUser(Cf.list(eventId), uid).isNotEmpty();
    }

    public Option<Layer> getPrimaryLayer(long eventId) {
        return eventLayerDao.findPrimaryLayerByEventId(eventId);
    }

    public static void setIndentsData(EventLayer target, EventAndRepetition event) {
        target.setEventStartTs(event.getEvent().getStartTs());
        target.setEventEndTs(event.getEvent().getEndTs());

        target.setRepetitionDueTs(RepetitionUtils.dueTsForEventIndent(event.getRepetitionInfo()));
        target.setRdatesMinTs(event.getRepetitionInfo().getRdatesMinTs());
        target.setRdatesMaxTs(event.getRepetitionInfo().getRdatesMaxTs());
    }

    public static void setIndentsData(EventResource target, EventAndRepetition event) {
        target.setEventStartTs(event.getEvent().getStartTs());
        target.setEventEndTs(event.getEvent().getEndTs());

        target.setRepetitionDueTs(RepetitionUtils.dueTsForEventIndent(event.getRepetitionInfo()));
        target.setRdatesMinTs(event.getRepetitionInfo().getRdatesMinTs());
        target.setRdatesMaxTs(event.getRepetitionInfo().getRdatesMaxTs());
    }
} //~
