package ru.yandex.calendar.logic.event;

import org.joda.time.Duration;
import org.joda.time.Hours;
import org.joda.time.Instant;
import org.joda.time.Minutes;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.notification.SmsNotificationManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.time.InstantInterval;

/**
 * CAL-6662
 *
 * @author dbrylev
 */
public class EventDeletionSmsHandler {

    private static final Duration MIN_BEFORE_EVENT_START = Minutes.minutes(-5).toStandardDuration();
    private static final Duration MAX_BEFORE_EVENT_START = Hours.hours(23).toStandardDuration();

    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private SmsNotificationManager smsNotificationManager;


    public void handleClosestEventDeletion(
            PassportUid actorUid,
            Tuple2List<EventWithRelations, RepetitionInstanceInfo> events, final ActionInfo actionInfo)
    {
        Function2<EventWithRelations, RepetitionInstanceInfo, Option<EventInstanceParameters>> firstSuitableF =
                new Function2<EventWithRelations, RepetitionInstanceInfo, Option<EventInstanceParameters>>() {
                    public Option<EventInstanceParameters> apply(EventWithRelations e, RepetitionInstanceInfo r) {
                        return findFirstSuitableInstance(e.getEvent(), r, actionInfo);
                    }
                };
        Tuple2List<EventWithRelations, EventInstanceParameters> occurrences =
                Cf2.flatBy2(events.zip3With(firstSuitableF.asFunction()).get13());

        handleEventClosestOccurrenceDeletion(actorUid, occurrences, actionInfo);
    }

    public void handleEventClosestOccurrenceDeletion(
            PassportUid actorUid,
            Tuple2List<EventWithRelations, EventInstanceParameters> occurrences, ActionInfo actionInfo)
    {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) return;

        InstantInterval interval = suitableInterval(actionInfo.getNow());

        Option<Tuple2<EventWithRelations, EventInstanceParameters>> closestSuitable = Option.empty();
        Function<Tuple2<EventWithRelations, EventInstanceParameters>, Instant> instanceStartF =
                Tuple2.<EventWithRelations, EventInstanceParameters>get2F().andThen(EventInstanceParameters.getStartTsF());

        for (Tuple2<EventWithRelations, EventInstanceParameters> occurrence : occurrences) {
            Instant instanceStart = instanceStartF.apply(occurrence);
            if (interval.contains(instanceStart)
                    && !closestSuitable.exists(instanceStartF.andThen(Cf2.f1B(ts -> ts.isBefore(instanceStart)))))
            {
                closestSuitable = Option.of(occurrence);
            }
        }
        for (Tuple2<EventWithRelations, EventInstanceParameters> closestOccurrence: closestSuitable) {
            handleEventDeletion(actorUid, closestOccurrence.get1(), closestOccurrence.get2(), actionInfo);
        }
    }

    public void handleEventClosestOccurrenceDeletion(
            PassportUid actorUid, long mainEventId, final ActionInfo actionInfo)
    {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) return;

        InstantInterval interval = suitableInterval(actionInfo.getNow());

        ListF<Event> events = eventDao.findEventsByMainEventIdAndStartTsIn(
                mainEventId, interval.getStart(), interval.getEnd());

        events = events.plus(eventDao.findMasterEventByMainId(mainEventId)).stableUniqueBy(Event.getIdF());

        MapF<Long, RepetitionInstanceInfo> repByEventId = repetitionRoutines.getRepetitionInstanceInfosByEvents(events);

        Option<Tuple2<Event, EventInstanceParameters>> closestSuitable = Option.empty();
        Option<Instant> closestSuitableStart = Option.empty();

        for (Event event : events) {
            Option<EventInstanceParameters> suitableInstance = findFirstSuitableInstance(
                    event, repByEventId.getOrThrow(event.getId()), actionInfo);

            if (suitableInstance.isPresent()
                    && !closestSuitableStart.exists(ts -> ts.isBefore(suitableInstance.get().getStartTs())))
            {
                closestSuitableStart = Option.of(suitableInstance.get().getStartTs());
                closestSuitable = Option.of(Tuple2.tuple(event, suitableInstance.get()));
            }
        }
        for (Tuple2<Event, EventInstanceParameters> closestOccurrence : closestSuitable) {
            EventWithRelations event = eventDbManager.getEventWithRelationsByEvent(closestOccurrence.get1());
            handleEventDeletion(actorUid, event, closestOccurrence.get2(), actionInfo);
        }
    }

    public void handleEventDeletion(
            PassportUid actorUid,
            final EventWithRelations event, final EventInstanceParameters instance, final ActionInfo actionInfo)
    {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) return;

        if (!suitableInterval(actionInfo.getNow()).contains(instance.getStartTs())) return;

        ListF<EventUserWithRelations> notifiableEventUsers = eventUserRoutines
                .filterNotDismissedNotDeclinedEventUsersHaveSmsNotification(event.getEventUsersWithRelation())
                .filterNot(eu -> eu.getUid().sameAs(actorUid));

        smsNotificationManager.submitEventDeletionSmss(
                actorUid, event.getEvent(), event.getMainEvent(), instance.getStartTs(),
                notifiableEventUsers.map(EventUserWithRelations::getSettings), actionInfo);
    }

    private static InstantInterval suitableInterval(Instant now) {
        return new InstantInterval(now.plus(MIN_BEFORE_EVENT_START), now.plus(MAX_BEFORE_EVENT_START));
    }

    private static Option<EventInstanceParameters> findFirstSuitableInstance(
            Event event, RepetitionInstanceInfo repetitionInfo, ActionInfo actionInfo)
    {
        InstantInterval interval = suitableInterval(actionInfo.getNow());

        if (!event.getRecurrenceId().isPresent()) {
            Option<InstantInterval> firstInterval = RepetitionUtils.getInstanceIntervalStartingAfter(
                    repetitionInfo, interval.getStart());

            if (firstInterval.isPresent() && interval.contains(firstInterval.get().getStart())) {
                return Option.of(new EventInstanceParameters(
                        firstInterval.get().getStart(), firstInterval.get().getEnd(),
                        Option.of(firstInterval.get().getStart())));
            }
        } else if (interval.contains(event.getStartTs())) {
            return Option.of(EventInstanceParameters.fromEvent(event));
        }
        return Option.empty();
    }
}
