package ru.yandex.calendar.logic.event;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.BeanHelper;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventHelper;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.beans.generated.MainEventHelper;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.RdateFields;
import ru.yandex.calendar.logic.beans.generated.RdateHelper;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.beans.generated.RepetitionHelper;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public abstract class EventIndent {

    private final long eventId;
    private final long mainEventId;
    private final Instant start;
    private final Instant end;
    private final boolean isAllDay;
    private final DateTimeZone tz;
    private final long layerOrResourceId;

    public boolean isSingle() {
        return this instanceof Single;
    }

    public boolean isRdate() {
        return this instanceof RDate;
    }

    public boolean isRepeating() {
        return this instanceof Repeating;
    }

    public Single asSingle() {
        return (Single) this;
    }

    public RDate asRdate() {
        return (RDate) this;
    }

    public Repeating asRepeating() {
        return (Repeating) this;
    }

    public Option<Instant> getRecurrenceId() {
        return isSingle() ? asSingle().getRecurrenceId() : Option.empty();
    }

    public Duration getDuration() {
        return new Duration(start, end);
    }

    public static class Single extends EventIndent {
        private static final ListF<MapField<?>> EVENT_FIELDS = Cf.list(
                EventFields.ID, EventFields.START_TS, EventFields.END_TS, EventFields.IS_ALL_DAY,
                EventFields.MAIN_EVENT_ID, EventFields.RECURRENCE_ID);

        private static final ListF<MapField<?>> MAIN_EVENT_FIELDS = Cf.list(MainEventFields.TIMEZONE_ID);

        private final Option<Instant> recurrenceId;

        public Single(
                long eventId, long mainEventId, Instant start, Instant end,
                Option<Instant> recurrenceId, boolean isAllDay, String tzId, long layerOrResourceId)
        {
            super(eventId, mainEventId, start, end, isAllDay, tzId, layerOrResourceId);
            this.recurrenceId = recurrenceId;
        }

        public static String columns(String eventPrefix, String mainEventPrefix, String targetColumn) {
            return EVENT_FIELDS.map(BeanHelper.columnNameF(eventPrefix))
                    .plus(MAIN_EVENT_FIELDS.map(BeanHelper.columnNameF(mainEventPrefix)))
                    .plus1(targetColumn).mkString(", ");
        }

        public static RowMapper<Single> rowMapper(int columnOffset) {
            BeanRowMapper<Event> eventRm =
                    EventHelper.INSTANCE.offsetRowMapper(columnOffset, EVENT_FIELDS);
            BeanRowMapper<MainEvent> mainEventRm =
                    MainEventHelper.INSTANCE.offsetRowMapper(eventRm.nextOffset(), MAIN_EVENT_FIELDS);
            RowMapper<Long> targetIdRm = (rs, numRow) -> rs.getLong(mainEventRm.nextOffset() + 1);

            return (rs, numRow) -> {
                Event event = eventRm.mapRow(rs, numRow);
                MainEvent mainEvent = mainEventRm.mapRow(rs, numRow);

                long targetId = targetIdRm.mapRow(rs, numRow);

                return new Single(
                        event.getId(), event.getMainEventId(),
                        event.getStartTs(), event.getEndTs(), event.getRecurrenceId(),
                        event.getIsAllDay(), mainEvent.getTimezoneId(), targetId);
            };
        }

        public Option<Instant> getRecurrenceId() {
            return recurrenceId;
        }
    }

    public static class RDate extends EventIndent {
        private static final ListF<MapField<?>> EVENT_FIELDS = Cf.list(
                EventFields.ID, EventFields.MAIN_EVENT_ID, EventFields.IS_ALL_DAY);

        private static final ListF<MapField<?>> MAIN_EVENT_FIELDS = Cf.list(MainEventFields.TIMEZONE_ID);

        private static final ListF<MapField<?>> RDATE_FIELDS = Cf.list(
                RdateFields.ID, RdateFields.START_TS, RdateFields.END_TS);

        private final Rdate rdate;

        public RDate(
                long eventId, long mainEventId, Rdate rdate,
                boolean isAllDay, String tzId, long layerOrResourceId)
        {
            super(eventId, mainEventId, rdate.getStartTs(),
                    rdate.getEndTs().getOrElse(rdate.getStartTs()), isAllDay, tzId, layerOrResourceId);
            this.rdate = rdate;
        }

        public static String columns(
                String eventPrefix, String rdatePrefix, String mainEventPrefix, String targetColumn)
        {
            return EVENT_FIELDS.map(BeanHelper.columnNameF(eventPrefix))
                    .plus(RDATE_FIELDS.map(BeanHelper.columnNameF(rdatePrefix)))
                    .plus(MAIN_EVENT_FIELDS.map(BeanHelper.columnNameF(mainEventPrefix)))
                    .plus1(targetColumn).mkString(", ");
        }

        public static RowMapper<RDate> rowMapper(int columnOffset) {
            BeanRowMapper<Event> eventRm =
                    EventHelper.INSTANCE.offsetRowMapper(columnOffset, EVENT_FIELDS);
            BeanRowMapper<Rdate> rdateRm =
                    RdateHelper.INSTANCE.offsetRowMapper(eventRm.nextOffset(), RDATE_FIELDS);
            BeanRowMapper<MainEvent> mainEventRm =
                    MainEventHelper.INSTANCE.offsetRowMapper(rdateRm.nextOffset(), MAIN_EVENT_FIELDS);
            RowMapper<Long> targetIdRm = (rs, numRow) -> rs.getLong(mainEventRm.nextOffset() + 1);

            return (rs, numRow) -> {
                Event event = eventRm.mapRow(rs, numRow);
                Rdate rdate = rdateRm.mapRow(rs, numRow);
                MainEvent mainEvent = mainEventRm.mapRow(rs, numRow);

                long targetId = targetIdRm.mapRow(rs, numRow);

                return new RDate(
                        event.getId(), event.getMainEventId(), rdate,
                        event.getIsAllDay(), mainEvent.getTimezoneId(), targetId);
            };
        }

        public Rdate getRdate() {
            return rdate;
        }
    }

    public static class Repeating extends EventIndent {
        private static final ListF<MapField<?>> EVENT_FIELDS = Cf.list(
                EventFields.ID, EventFields.MAIN_EVENT_ID,
                EventFields.START_TS, EventFields.END_TS, EventFields.IS_ALL_DAY);

        private static final ListF<MapField<?>> REPETITION_FIELDS = Cf.list(
                RepetitionFields.ID, RepetitionFields.TYPE, RepetitionFields.R_EACH,
                RepetitionFields.R_MONTHLY_LASTWEEK, RepetitionFields.R_WEEKLY_DAYS, RepetitionFields.DUE_TS);

        private static final ListF<MapField<?>> MAIN_EVENT_FIELDS = Cf.list(MainEventFields.TIMEZONE_ID);

        private final Repetition repetition;

        public Repeating(
                long eventId, long mainEventId,
                Instant start, Instant end, boolean isAllDay,
                Repetition repetition, String tzId, long layerOrResourceId)
        {
            super(eventId, mainEventId, start, end, isAllDay, tzId, layerOrResourceId);
            this.repetition = repetition;
        }

        public static String columns(
                String eventPrefix, String repetitionPrefix, String mainEventPrefix, String targetColumn)
        {
            return EVENT_FIELDS.map(BeanHelper.columnNameF(eventPrefix))
                    .plus(REPETITION_FIELDS.map(BeanHelper.columnNameF(repetitionPrefix)))
                    .plus(MAIN_EVENT_FIELDS.map(BeanHelper.columnNameF(mainEventPrefix)))
                    .plus1(targetColumn).mkString(", ");
        }

        public static RowMapper<Repeating> rowMapper(int columnOffset) {
            BeanRowMapper<Event> eventRm =
                    EventHelper.INSTANCE.offsetRowMapper(columnOffset, EVENT_FIELDS);
            BeanRowMapper<Repetition> repetitionRm =
                    RepetitionHelper.INSTANCE.offsetRowMapper(eventRm.nextOffset(), REPETITION_FIELDS);
            BeanRowMapper<MainEvent> mainEventRm =
                    MainEventHelper.INSTANCE.offsetRowMapper(repetitionRm.nextOffset(), MAIN_EVENT_FIELDS);
            RowMapper<Long> targetIdRm = (rs, numRow) -> rs.getLong(mainEventRm.nextOffset() + 1);

            return (rs, numRow) -> {
                Event event = eventRm.mapRow(rs, numRow);
                Repetition repetition = repetitionRm.mapRow(rs, numRow);
                MainEvent mainEvent = mainEventRm.mapRow(rs, numRow);

                long targetId = targetIdRm.mapRow(rs, numRow);

                return new Repeating(
                        event.getId(), event.getMainEventId(),
                        event.getStartTs(), event.getEndTs(), event.getIsAllDay(), repetition,
                        mainEvent.getTimezoneId(), targetId);
            };
        }

        public Repetition getRepetition() {
            return repetition;
        }
    }

    public EventIndent(
            long eventId, long mainEventId,
            Instant start, Instant end, boolean isAllDay, String tzId, long layerOrResourceId)
    {
        this.eventId = eventId;
        this.mainEventId = mainEventId;
        this.start = start;
        this.end = end;
        this.isAllDay = isAllDay;
        this.tz = AuxDateTime.getVerifyDateTimeZone(tzId);
        this.layerOrResourceId = layerOrResourceId;
    }

    public InstantInterval getInterval() {
        return new InstantInterval(getStart(), getEnd());
    }

    public long getEventId() {
        return eventId;
    }

    public long getMainEventId() {
        return mainEventId;
    }

    public Instant getStart() {
        return start;
    }

    public Instant getEnd() {
        return end;
    }

    public boolean isAllDay() {
        return isAllDay;
    }

    public DateTimeZone getTz() {
        return tz;
    }

    public long getLayerOrResourceId() {
        return layerOrResourceId;
    }

    public EventTime getTime() {
        return new EventTime(start, end, isAllDay, tz);
    }
}
