package ru.yandex.calendar.logic.event;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.EventInfoForPermsCheck;
import ru.yandex.commune.mapObject.MapField;

/**
 * @author gutman
 */
public class EventInfo {
    private final EventInfoForPermsCheck infoForPermsCheck;
    private final EventIndentAndRepetition indent;
    private final Event partialEvent;
    private final MainEvent partialMainEvent;
    private final Option<RepetitionInstanceInfo> repetitionInfo;
    private final Option<EventParticipants> participants;
    private final Option<ListF<ResourceInfo>> resources;
    private final Option<EventWithRelations> eventWithRelations;

    private final Option<EventUserWithNotifications> eventUserWithNotifications;
    private final Option<ListF<EventAttachment>> attachments;

    private final Option<Long> layerId;
    private final Option<Long> resourceId;

    private final boolean mayView;

    protected EventInfo(
            EventInfoForPermsCheck infoForPermsCheck,
            EventIndentAndRepetition indent,
            Event partialEvent, MainEvent partialMainEvent,
            Option<RepetitionInstanceInfo> repetitionInfo,
            Option<EventParticipants> participants,
            Option<ListF<ResourceInfo>> resources,
            Option<EventWithRelations> eventWithRelations,
            Option<EventUserWithNotifications> eventUserWithNotifications,
            Option<ListF<EventAttachment>> attachments,
            Option<Long> layerId, Option<Long> resourceId, boolean mayView)
    {
        this.infoForPermsCheck = infoForPermsCheck;
        this.indent = indent;
        this.partialEvent = partialEvent;
        this.partialMainEvent = partialMainEvent;
        this.repetitionInfo = repetitionInfo;
        this.participants = participants;
        this.resources = resources;
        this.eventWithRelations = eventWithRelations;
        this.eventUserWithNotifications = eventUserWithNotifications;
        this.attachments = attachments;
        this.mayView = mayView;
        this.layerId = layerId;
        this.resourceId = resourceId;
    }

    protected static EventInfo cons(
            EventInfoForPermsCheck infoForPermsCheck, EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
            Option<EventUserWithNotifications> eventUser, ListF<EventAttachment> attachments,
            Option<Long> layerId, Option<Long> resourceId, boolean mayView)
    {
        return new EventInfo(infoForPermsCheck,
                EventIndentAndRepetition.fromEventAndRepetition(event.getEvent(), repetitionInfo),
                event.getEvent(), event.getMainEvent(),
                Option.of(repetitionInfo),
                Option.of(event.getEventParticipants()),
                Option.of(event.getResources()),
                Option.of(event), eventUser, Option.of(attachments),
                layerId, resourceId, mayView);
    }

    public EventInfo withRepetitionInfo(RepetitionInstanceInfo repetitionInfo) {
        return new EventInfo(
                infoForPermsCheck, indent, partialEvent, partialMainEvent, Option.of(repetitionInfo),
                participants, resources, eventWithRelations,
                eventUserWithNotifications, attachments, layerId, resourceId, mayView);
    }

    public EventInfo plusExdates(ListF<Instant> exdates) {
        return withRepetitionInfo(getRepetitionInstanceInfo().plusExdates(exdates));
    }

    public <T> Option<T> getEventFieldValueO(MapField<T> field) {
        return partialEvent.getFieldValueO(field);
    }

    public Event getEvent() {
        return partialEvent;
    }

    public <T> Option<T> getMainEventFieldValueO(MapField<T> field) {
        return partialMainEvent.getFieldValueO(field);
    }

    public MainEvent getMainEvent() {
        return partialMainEvent;
    }

    public Option<EventWithRelations> getEventWithRelationsO() {
        return eventWithRelations;
    }

    public EventWithRelations getEventWithRelations() {
        return eventWithRelations.getOrThrow("event with relations not loaded");
    }

    public Option<EventParticipants> getEventParticipantsO() {
        return participants;
    }

    public EventParticipants getEventParticipants() {
        return participants.getOrThrow("participants not loaded");
    }

    public Participants getParticipants() {
        return getEventParticipants().getParticipants();
    }

    public Option<ListF<ResourceInfo>> getResourcesO() {
        return resources;
    }

    public ListF<ResourceInfo> getResources() {
        return resources.getOrThrow("resources not loaded");
    }

    public Option<ListF<EventAttachment>> getAttachmentO() {
        return attachments;
    }

    public ListF<EventAttachment> getAttachments() {
        return attachments.getOrThrow("attachments not loaded");
    }

    public Option<ListF<EventResource>> getEventResourcesO() {
        return participants.map(EventParticipants::getEventResources);
    }

    public ListF<EventResource> getEventResources() {
        return getEventResourcesO().getOrThrow("event resources not loaded");
    }

    public Option<RepetitionInstanceInfo> getRepetitionInfoO() {
        return repetitionInfo;
    }

    public EventIndent getIndent() {
        return indent.getIndent();
    }

    public EventIndentAndRepetition getIndentAndRepetition() {
        return indent;
    }

    public EventInfoForPermsCheck getInfoForPermsCheck() {
        return infoForPermsCheck;
    }

    public EventAndRepetition getEventAndRepetition() {
        return new EventAndRepetition(getEvent(), getRepetitionInstanceInfo());
    }

    public DateTimeZone getTimezone() {
        return getIndent().getTz();
    }

    public Duration getDuration() {
        return new Duration(getIndent().getStart(), getIndent().getEnd());
    }

    public long getEventId() {
        return indent.getEventId();
    }

    public RepetitionInstanceInfo getRepetitionInstanceInfo() {
        return repetitionInfo.getOrElse(indent.getRepetitionInfo());
    }

    public Option<EventUser> getEventUser() {
        return getEventUserWithNotifications().map(EventUserWithNotifications.getEventUserF());
    }

    public Option<EventUserWithNotifications> getEventUserWithNotifications() {
        return eventUserWithNotifications;
    }

    public ListF<EventLayerWithRelations> getEventLayersWithRelations() {
        return getEventWithRelations().getEventLayersWithRelations();
    }

    public long getMainEventId() {
        return getIndent().getMainEventId();
    }

    public Option<Instant> getRecurrenceId() {
        return getIndent().isSingle() ? getIndent().asSingle().getRecurrenceId() : Option.empty();
    }

    public boolean isMaster() {
        return !getRecurrenceId().isPresent();
    }

    public boolean isRecurrence() {
        return getRecurrenceId().isPresent();
    }

    public Option<ResourceParticipantInfo> findResourceParticipant(long resourceId) {
        return getEventParticipants().getParticipants()
                .getByIdSafeWithInconsistent(ParticipantId.resourceId(resourceId)).uncheckedCast();
    }

    public boolean isRejected() {
        return eventUserWithNotifications.exists(eu -> eu.getEventUser().getDecision() == Decision.NO);
    }

    public Option<Layer> findLayerById(long layerId) {
        return getEventWithRelations().findLayerById(layerId);
    }

    public Option<Long> getLayerId() {
        return layerId;
    }

    public Option<Long> getResourceId() {
        return resourceId;
    }

    public boolean mayView() {
        return mayView;
    }

    public boolean goesOnAfter(Instant ts) {
        return getRepetitionInstanceInfo().goesOnAfter(ts);
    }
}
