package ru.yandex.calendar.logic.event;

import java.util.Optional;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventLayerFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.RdateFields;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetitionAndPerms;
import ru.yandex.calendar.logic.event.repetition.RecurrenceTimeInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.layer.UserLayersSharing;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.EventFieldsForPermsCheck;
import ru.yandex.calendar.logic.sharing.perm.EventInfoForPermsCheck;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.EventAction;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.db.CalendarJdbcTemplate;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.ConditionUtils;
import ru.yandex.misc.db.q.ConditionUtils.Column;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.time.InstantInterval;

public class EventInfoDbLoader {
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private CalendarJdbcTemplate jdbcTemplate;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private UserManager userManager;

    private CalendarJdbcTemplate getJdbcTemplate() {
        return jdbcTemplate;
    }

    private ListF<EventInfo> loadEventInfos(
            Option<UserInfo> userO, EventGetProps egp, EventsSource source, ActionSource actionSource)
    {
        if (source.isEmpty()) return Cf.list();

        ListF<EventIndentAndRepetition> indentAndRepetitions =
                source.isIndentsOrIndentsAndPerms() ? source.getIndentsDeduplicated() : Cf.list();
        ListF<EventAndRepetition> eventAndRepetitions = source.isEvents() ? source.getEvents() : Cf.list();

        boolean isByIndents = source.isIndentsOrIndentsAndPerms();

        ListF<Event> events = eventAndRepetitions.map(EventAndRepetition::getEvent);
        ListF<Long> indentEventIds = indentAndRepetitions.map(EventIndentAndRepetition::getEventId);

        Option<PassportUid> uidO = userO.map(UserInfo.getUidF());

        MapF<Long, RepetitionInstanceInfo> indentRepetitionInfoByEventId = isByIndents
                ? indentAndRepetitions.toMap(EventIndentAndRepetition::getEventId, EventIndentAndRepetition::getRepetitionInfo)
                : Cf.map();
        MapF<Long, RepetitionInstanceInfo> eventRepetitionByEventId = isByIndents
                ? Cf.map()
                : eventAndRepetitions.toMap(EventAndRepetition::getEventId, EventAndRepetition::getRepetitionInfo);

        ListF<EventInfoForPermsCheck> eventsForPermsCheck;

        MapF<Long, Event> eventById;
        MapF<Long, MainEvent> mainEventById;
        MapF<Long, RepetitionInstanceInfo> repetitionInfoByEventId;
        MapF<Long, EventParticipants> participantsByEventId;
        MapF<Long, ListF<ResourceInfo>> resourcesByEventId;
        MapF<Long, EventWithRelations> eventWithRelationsById;
        MapF<Long, EventUserWithNotifications> eventUserWithNotificationsByEventId;
        MapF<Long, ListF<EventAttachment>> attachmentsByEventId;

        if (egp.isWithEventWithRelations()) {
            Function<EventWithRelations, Long> idF = EventWithRelations::getId;
            ListF<EventWithRelations> eventsWr;
            if (isByIndents) {
                eventsWr = eventDbManager.getEventsWithRelationsByIds(
                        indentEventIds, egp.isIncludeSubscribers());
            } else {
                eventsWr = eventDbManager.getEventsWithRelationsByEvents(
                        events, egp.getLayerId(), egp.isIncludeSubscribers());
            }

            Optional<UserLayersSharing> layersSharing = uidO.map(authorizer::loadLayerSharing).toOptional();
            eventsForPermsCheck = eventsWr.map(e ->
                    userO.map(user -> authorizer.loadEventInfoForPermsCheck(userO.toOptional(), e, Optional.empty(), layersSharing))
                            .getOrElse(() -> authorizer.loadEventInfoForPermsCheck(e)));
            eventById = eventsWr.toMap(idF, EventWithRelations::getEvent);
            mainEventById = eventsWr.toMap(EventWithRelations::getMainEventId, EventWithRelations::getMainEvent);

            if (isByIndents) {
                repetitionInfoByEventId = egp.isWithFullRepetitionInfo()
                        ? repetitionRoutines.getRepetitionInstanceInfos(eventsWr)
                        : indentRepetitionInfoByEventId;
            } else {
                repetitionInfoByEventId = eventRepetitionByEventId;
            }
            participantsByEventId = eventsWr.toMap(idF, EventWithRelations::getEventParticipants);
            resourcesByEventId = eventsWr.toMap(idF, EventWithRelations::getResources);
            eventWithRelationsById = eventsWr.toMapMappingToKey(idF);

        } else {
            ListF<Event> es;
            if (isByIndents) {
                ListF<MapField<?>> fields = egp.eventFieldsToLoad();

                if (!source.isIndentAndPerms()) {
                    fields = fields.plus(EventFieldsForPermsCheck.FIELDS);
                }
                if (egp.isWithFullRepetitionInfo()) {
                    fields = fields.plus(
                            EventFields.RECURRENCE_ID, EventFields.REPETITION_ID, EventFields.MAIN_EVENT_ID);
                    es = eventDao.findEventsFieldsByIdsSafe(indentEventIds, fields);

                    MapF<Long, DateTimeZone> tzByEventId = indentAndRepetitions.toMap(
                            EventIndentAndRepetition::getEventId, e -> e.getIndent().getTz());

                    repetitionInfoByEventId = repetitionRoutines.getRepetitionInstanceInfosByEventsAndTimezones(
                            es.zipWith(e -> tzByEventId.getOrThrow(e.getId())));

                } else if (fields.isNotEmpty()) {
                    es = eventDao.findEventsFieldsByIdsSafe(indentEventIds, fields);
                    repetitionInfoByEventId = indentRepetitionInfoByEventId;

                } else {
                    es = indentEventIds.map(Function.constF(new Event()));
                    repetitionInfoByEventId = indentRepetitionInfoByEventId;
                }
            } else {
                es = events;
                repetitionInfoByEventId = eventRepetitionByEventId;
            }
            ListF<Long> mainEventIds = isByIndents
                    ? indentAndRepetitions.map(e -> e.getIndent().getMainEventId())
                    : eventAndRepetitions.map(EventAndRepetition::getMainEventId);

            if (egp.mainEventFieldsToLoad().isNotEmpty()) {
                mainEventById = mainEventDao.findMainEventsFieldsByIdsSafe(mainEventIds, egp.mainEventFieldsToLoad())
                        .toMapMappingToKey(MainEvent.getIdF());
            } else {
                mainEventById = mainEventIds.toMapMappingToValue(Function.constF(new MainEvent()));
            }
            ListF<EventParticipants> participants = egp.isWithEventParticipants()
                    ? eventDbManager.getParticipantsByEventIds(es.map(Event::getId), egp.isIncludeSubscribers())
                    : Cf.list();

            if (!source.isIndentAndPerms()) {
                val fields = es.map(EventFieldsForPermsCheck::fromEvent);
                eventsForPermsCheck = Cf.toList(userO.map(user -> authorizer.loadInfoForPermsCheck(user, fields, participants))
                        .getOrElse(() -> authorizer.loadInfoForPermsCheck(fields, participants)));
            } else {
                eventsForPermsCheck = source.getPermsInfoFromIndents();
            }

            if (egp.isWithEventParticipants()) {
                resourcesByEventId = participants.toMap(EventParticipants::getEventId, EventParticipants::getResources);

            } else if (egp.isWithEventResources()) {
                MapF<Long, ResourceInfo> resourceById = resourceDao.findResourceInfosByIds(
                        eventsForPermsCheck.flatMap(EventInfoForPermsCheck::getResourceIds))
                        .toMapMappingToKey(ResourceInfo::getResourceId);

                resourcesByEventId = eventsForPermsCheck.toMap(
                        EventInfoForPermsCheck::getEventId, e -> e.getResourceIds().map(resourceById::getOrThrow));

            } else {
                resourcesByEventId = Cf.map();
            }

            eventById = es.toMapMappingToKey(Event.getIdF());
            participantsByEventId = participants.toMapMappingToKey(EventParticipants::getEventId);
            eventWithRelationsById = Cf.map();

        }
        if (eventsForPermsCheck.isNotEmpty() && userO.isPresent() && egp.isWithEventUserWithNotifications()) {
            if (egp.isWithEventParticipants()) {
                ListF<EventUser> eventUsers = participantsByEventId.values()
                        .filterMap(p -> p.getEventUser(uidO.get()).map(EventUserWithRelations::getEventUser));

                eventUserWithNotificationsByEventId = notificationDbManager
                        .getEventUserWithNotificationsByEventUsers(eventUsers)
                        .toMapMappingToKey(EventUserWithNotifications.getEventIdF());

            } else {
                eventUserWithNotificationsByEventId = notificationDbManager
                        .getEventUsersWithNotificationsByUidAndEventIds(
                                uidO.get(), eventsForPermsCheck.map(EventInfoForPermsCheck::getEventId))
                        .toMapMappingToKey(EventUserWithNotifications.getEventIdF());
            }
        } else {
            eventUserWithNotificationsByEventId = Cf.map();
        }
        if (egp.isWithEventAttachments()) {
            ListF<Long> eventIds = isByIndents
                    ? indentAndRepetitions.map(e -> e.getIndent().getEventId()).stableUnique()
                    : events.map(Event::getId).stableUnique();

            attachmentsByEventId = eventDao.findEventAttachmentsByEventIds(eventIds).toMap();
        } else {
            attachmentsByEventId = Cf.map();
        }

        MapF<Long, EventInfoForPermsCheck> permsInfoByEventId = eventsForPermsCheck
                .toMapMappingToKey(EventInfoForPermsCheck::getEventId);

        if (!isByIndents) {
            indentAndRepetitions = eventAndRepetitions.map(
                    er -> EventIndentAndRepetition.fromEventAndRepetition(er.getEvent(), er.getRepetitionInfo()));
        }
        return indentAndRepetitions.map(indent -> {
            long eventId = indent.getEventId();
            long mainEventId = indent.getIndent().getMainEventId();

            Option<Long> layerOrResourceId = Option.of(indent.getIndent().getLayerOrResourceId()).filter(i -> i > 0);

            EventInfoForPermsCheck permsInfo = permsInfoByEventId.getOrThrow(indent.getEventId());

            return new EventInfo(permsInfo, indent,
                    eventById.getOrThrow(eventId), mainEventById.getOrThrow(mainEventId),
                    Option.when(egp.isWithFullRepetitionInfo(), () -> repetitionInfoByEventId.getOrThrow(eventId)),
                    Option.when(egp.isWithEventParticipants(), () -> participantsByEventId.getOrThrow(eventId)),
                    Option.when(egp.isWithEventResources(), () -> resourcesByEventId.getOrThrow(eventId)),
                    Option.when(egp.isWithEventWithRelations(), () -> eventWithRelationsById.getOrThrow(eventId)),
                    eventUserWithNotificationsByEventId.getO(eventId),
                    Option.when(egp.isWithEventAttachments(), () -> attachmentsByEventId.getOrThrow(eventId)),
                    layerOrResourceId.filter(i -> source.isLayers.isSome(true)),
                    layerOrResourceId.filter(i -> source.isLayers.isSome(false)),
                    userO.map(user -> authorizer.canViewEvent(user, permsInfo, actionSource))
                            .getOrElse(() -> authorizer.canViewEvent(permsInfo, actionSource)));
        });
    }

    private EventInfo loadEventInfo(Option<UserInfo> userO, EventWithRelations event, RepetitionInstanceInfo repetition,
                                    ActionSource actionSource) {
        Option<EventUserWithNotifications> eventUser = userO.isPresent()
                ? notificationDbManager.getEventUserWithNotificationsByUidAndEventId(userO.get().getUid(), event.getId())
                : Option.empty();

        EventInfoForPermsCheck infoForPermsCheck = userO.map(user -> authorizer.loadEventInfoForPermsCheck(user, event))
                .getOrElse(() -> authorizer.loadEventInfoForPermsCheck(event));

        ListF<EventAttachment> attachments = eventDao.findEventAttachmentsByEventId(event.getId());

        final var mayView = userO.map(user -> {
            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, event);
            return authorizer.canViewEvent(user, eventAuthInfo, actionSource);
        })
        .getOrElse(() -> {
            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(event);
            return authorizer.canViewEvent(eventAuthInfo, actionSource);
        });

        return EventInfo.cons(
                infoForPermsCheck, event, repetition, eventUser, attachments, Option.empty(), Option.empty(), mayView);
    }

    private class Do {
        private final EventLoadLimits limits;

        private final Option<Duration> maxAllowedEventDuration;
        private final ListF<Long> targetIds;
        private final boolean onLayer;

        private final String targetTableName;
        private final String targetIdColumn;
        private final SqlCondition targetIdsCondition;

        private Do(EventLoadLimits limits, LayerIdsOrResourceIds targetIds) {
            this.limits = limits;
            this.targetIds = targetIds.get();
            this.onLayer = targetIds.isLayers();

            if (!onLayer) {
                this.maxAllowedEventDuration = resourceDao.findResourceTypesByIds(targetIds.get())
                        .map(t -> t.get2().getMaxAllowedEventDuration()).maxO();
            } else {
                this.maxAllowedEventDuration = Option.empty();
            }
            this.targetTableName = onLayer ? "event_layer" : "event_resource";
            this.targetIdColumn = onLayer ? "layer_id" : "resource_id";
            this.targetIdsCondition = ConditionUtils.column(targetIdColumn).inSet(this.targetIds);
        }

        public ListF<EventAndRepetition> getEventsOnTarget() {
            ListF<Long> eventIds = getIndentsOnTarget().map(e -> e.getIndent().getEventId()).stableUnique();

            return eventDbManager.getEventsAndRepetitionsByEventIds(eventIds);
        }

        public ListF<EventIndentAndRepetition> getIndentsOnTarget() {
            return filterEventsByLimits(loadEvents());
        }

        private ListF<EventIndentAndRepetition> filterEventsByLimits(ListF<EventIndentAndRepetition> events) {
            if (!limits.hasTimeLimits() && !limits.hasResultSizeLimit()) {
                return events;
            }
            Function<EventIndentAndRepetition, Option<Instant>> firstInstanceStartF = firsInstanceStartF()
                    .compose(EventIndentAndRepetition::getRepetitionInfo);

            if (limits.hasResultSizeLimit()) {
                int limit = limits.getResultSizeLimit().get();
                Tuple2List<EventIndentAndRepetition, Instant> filtered = events.zipWithFlatMapO(firstInstanceStartF);

                return filtered.takeSortedBy2(limit * targetIds.size()).get1()
                        .stableUniqueBy(e -> e.getIndent().getEventId()).take(limit);
            } else {
                return events.filter(e -> firstInstanceStartF.apply(e).isPresent());
            }
        }

        private ListF<EventIndentAndRepetition> loadEvents() {
            if (targetIds.isEmpty() || limits.getHaveId().isSome(Cf.list())) return Cf.list();

            ListF<EventIndent.Single> singleEvents = loadSingleEvents();
            ListF<EventIndent.RDate> rDates = loadRdateEvents();
            ListF<EventIndent.Repeating> repeatingEvents = loadRepeatingEvents();

            MapF<Long, ListF<EventIndent.RDate>> rDatesByEventId = rDates
                    .groupBy(EventIndent::getEventId);

            ListF<EventIndent> singleAndRepeatingEvents = Cf.arrayList();
            singleAndRepeatingEvents.addAll(singleEvents);
            singleAndRepeatingEvents.addAll(repeatingEvents);

            SetF<Long> singleAndRepeatingEventIds = singleAndRepeatingEvents.map(EventIndent::getEventId).unique();

            ListF<EventIndent.RDate> orphanedRDates = rDates.filter(
                    r -> !singleAndRepeatingEventIds.containsTs(r.getEventId()));

            MapF<Long, ListF<Instant>> exDatesByEventId = loadExdates(singleAndRepeatingEvents);

            MapF<Long, ListF<RecurrenceTimeInfo>> recurrenceIdsByMainEventId =
                    loadRecurrences(repeatingEvents.uncheckedCast());

            Function<EventIndent, EventIndentAndRepetition> consF = indent -> {
                DateTimeZone tz = indent.getTz();

                if (indent.isSingle() && indent.asSingle().getRecurrenceId().isPresent()) {
                    Interval interval = new Interval(indent.getStart().getMillis(), indent.getEnd().getMillis(), tz);
                    return new EventIndentAndRepetition(indent, RepetitionInstanceInfo.create(interval, Option.empty()));
                }
                Option<EventIndent.Repeating> repeating = Option.when(indent.isRepeating(), indent::asRepeating);
                Option<Repetition> repetition = repeating.map(EventIndent.Repeating::getRepetition);

                ListF<Rdate> exdates = exDatesByEventId.getOrElse(indent.getEventId(), Cf.list())
                        .map(Cf2.f2(RepetitionUtils::consExdateEventId).bind1(indent.getEventId()));

                ListF<Rdate> rdates = Cf.list();
                if (!indent.isRdate()) {
                    ListF<EventIndent.RDate> indents = rDatesByEventId.getOrElse(indent.getEventId(), Cf.list());
                    rdates = indents.map(EventIndent.RDate::getRdate).stableUniqueBy(Rdate::getId);
                }
                ListF<RecurrenceTimeInfo> recurrences =
                        recurrenceIdsByMainEventId.getOrElse(indent.getMainEventId(), Cf.list());

                return new EventIndentAndRepetition(indent, new RepetitionInstanceInfo(
                        indent.getInterval(), tz, repetition, rdates, exdates, recurrences));
            };

            ListF<EventIndentAndRepetition> result = Cf.arrayListWithCapacity(
                    singleEvents.size() + repeatingEvents.size());

            result.addAll(singleEvents.map(consF));
            result.addAll(repeatingEvents.map(consF));
            result.addAll(orphanedRDates.map(consF));

            return result;
        }

        private String targetColumn() {
            return et() + "." + targetIdColumn;
        }

        private String et() {
            return onLayer ? "el" : "er";
        }

        private ListF<EventIndent.Single> loadSingleEvents() {
            SqlCondition eventCondition = createEventIdCondition(EventFields.ID.column());

            SqlCondition eventTargetCondition = SqlCondition.trueCondition()
                    .and(EventLayerFields.REPETITION_DUE_TS.column().isNull())
                    .and(createTimeConditionsForColumns(EventLayerFields.EVENT_START_TS, EventLayerFields.EVENT_END_TS))
                    .and(targetIdsCondition);

            String q = "SELECT " + EventIndent.Single.columns("e.", "me.", targetColumn()) + " FROM event e" +
                    " INNER JOIN " + targetTableName + " et ON et.event_id = e.id".replace("et", et()) +
                    " INNER JOIN main_event me ON me.id = e.main_event_id" +
                    " WHERE (" + eventCondition.sqlForTable("e") + ")" +
                    " AND (" + eventTargetCondition.sqlForTable(et()) + ")" +
                    andModifiedSinceSqlForMeTable();

            if (limits.getResultSizeLimit().isPresent()) {
                q += " ORDER BY e.start_ts LIMIT " + limits.getResultSizeLimit().get();
            }
            return getJdbcTemplate().query(q, EventIndent.Single.rowMapper(0),
                    eventCondition.args(), eventTargetCondition.args(), limits.getModifiedSince());
        }

        private ListF<EventIndent.RDate> loadRdateEvents() {
            SqlCondition rdateCondition = RdateFields.IS_RDATE.eq(true)
                    .and(createTimeConditionsForColumns(RdateFields.START_TS, RdateFields.END_TS));

            SqlCondition eventTargetCondition = SqlCondition.trueCondition()
                    .and(createEventIdCondition(EventLayerFields.EVENT_ID.column()))
                    .and(createTimeConditionForRepeating(
                            EventLayerFields.RDATES_MIN_TS,
                            EventLayerFields.RDATES_MAX_TS))
                    .and(targetIdsCondition);

            String q = "SELECT " + EventIndent.RDate.columns("e.", "rd.", "me.", targetColumn()) +
                    " FROM " + targetTableName + " <et>" +
                    " INNER JOIN event e ON e.id = <et>.event_id" +
                    " INNER JOIN main_event me ON me.id = e.main_event_id" +
                    " INNER JOIN rdate rd ON rd.event_id = + <et>.event_id" + // "+" prevents scan rdate first
                    " WHERE (" + rdateCondition.sqlForTable("rd") + ")" +
                    " AND (" + eventTargetCondition.sqlForTable(et()) + ")" +
                    andModifiedSinceSqlForMeTable();

            q = q.replace("<et>", et());

            if (limits.getResultSizeLimit().isPresent()) {
                q += " ORDER BY rd.start_ts LIMIT " + limits.getResultSizeLimit().get();
            }
            return getJdbcTemplate().query(q, EventIndent.RDate.rowMapper(0),
                    rdateCondition.args(), eventTargetCondition.args(), limits.getModifiedSince());
        }

        private ListF<EventIndent.Repeating> loadRepeatingEvents() {
            SqlCondition eventCondition = createEventIdCondition(EventFields.ID.column());

            SqlCondition eventTargetCondition = SqlCondition.trueCondition()
                    .and(createTimeConditionForRepeating(
                            EventLayerFields.EVENT_START_TS,
                            EventLayerFields.REPETITION_DUE_TS))
                    .and(targetIdsCondition);

            String q = "SELECT " + EventIndent.Repeating.columns("e.", "r.", "me.", targetColumn()) + " FROM event e" +
                    " INNER JOIN " + targetTableName + " et ON et.event_id = e.id".replace("et", et()) +
                    " INNER JOIN repetition r ON r.id = e.repetition_id" +
                    " INNER JOIN main_event me ON me.id = e.main_event_id" +
                    " WHERE (" + eventCondition.sqlForTable("e") + ")" +
                    " AND (" + eventTargetCondition.sqlForTable(et()) + ")" +
                    andModifiedSinceSqlForMeTable();

            ListF<?> args = eventCondition.args().plus(eventTargetCondition.args()).plus(limits.getModifiedSince());
            RowMapper<EventIndent.Repeating> eventRowMapper = EventIndent.Repeating.rowMapper(0);

            if (limits.hasTimeLimits()) {
                Function<EventIndent.Repeating, RepetitionInstanceInfo> repetitionF = r -> new RepetitionInstanceInfo(
                        r.getInterval(), r.getTz(), Option.of(r.getRepetition()), Cf.list(), Cf.list(), Cf.list());

                Function1B<EventIndent.Repeating> hasInstanceF = r ->
                        firsInstanceStartF().apply(repetitionF.apply(r)).isPresent();

                RowMapper<EventIndent.Repeating> rm = (rs, num) -> {
                    EventIndent.Repeating event = eventRowMapper.mapRow(rs, num);
                    return hasInstanceF.apply(event) ? event : null;
                };
                return getJdbcTemplate().query(q, rm, args).filterNotNull();

            } else {
                return getJdbcTemplate().query(q, eventRowMapper, args);
            }
        }

        private MapF<Long, ListF<Instant>> loadExdates(ListF<EventIndent> indents) {
            if (indents.isEmpty()) return Cf.map();

            ListF<Long> eventIds = indents.map(EventIndent::getEventId);
            Duration maxDuration = indents.map(EventIndent::getDuration).max();

            SqlCondition condition = RdateFields.IS_RDATE.eq(false)
                    .and(RdateFields.EVENT_ID.column().inSet(eventIds))
                    .and(createTimeConditionForExdate(RdateFields.START_TS, maxDuration));

            String q = "SELECT event_id, start_ts FROM rdate" + condition.whereSql();

            return getJdbcTemplate().query2(q,
                    RdateFields.EVENT_ID.getDatabaser().rowMapper(),
                    RdateFields.START_TS.getDatabaser().rowMapper(), condition.args()).groupBy1();
        }

        private MapF<Long, ListF<RecurrenceTimeInfo>> loadRecurrences(ListF<EventIndent> indents) {
            if (indents.isEmpty()) return Cf.map();

            ListF<Long> mainEventIds = indents.map(EventIndent::getMainEventId);
            Duration maxDuration = indents.map(EventIndent::getDuration).max();

            return eventDao.findRecurrenceInstantInfosByMainEventIds(mainEventIds,
                    Option.of(createTimeConditionForExdate(EventFields.RECURRENCE_ID, maxDuration))).groupBy1();
        }

        private SqlCondition createEventIdCondition(Column eventIdColumn) {
            SqlCondition condition = eventIdColumn.inSet(limits.getExceptIds()).not();
            return limits.getHaveId().isPresent()
                    ? condition.and(eventIdColumn.inSet(limits.getHaveId().get()))
                    : condition;
        }

        private SqlCondition createTimeConditionsForColumns(MapField<?> startTsColumn, MapField<?> endTsColumn) {
            return createTimeConditionsForColumns(startTsColumn, Option.of(endTsColumn), maxAllowedEventDuration);
        }

        private SqlCondition createTimeConditionForExdate(MapField<?> column, Duration maxEventLength) {
            maxEventLength = maxAllowedEventDuration.plus(maxEventLength).max();
            maxEventLength = maxEventLength.plus(Duration.standardHours(1)); // timezone transition
            return createTimeConditionsForColumns(
                    column, Option.empty(), maxAllowedEventDuration.plus(maxEventLength).maxO());
        }

        private SqlCondition createTimeConditionsForColumns(
                MapField<?> startTsColumn, Option<MapField<?>> endTsColumn, Option<Duration> maxDuration)
        {
            Option<Instant> startsInOrAfterO = limits.getStartsInOrAfterLimit();
            Option<Instant> startsInOrBeforeO = limits.getStartsInOrBeforeLimit();
            Option<Instant> endsInOrAfterO = limits.getEndsInOrAfterLimit();

            Option<SqlCondition> endTsIsNull = endTsColumn
                    .map(c -> c.isNotNull() ? SqlCondition.falseCondition() : c.column().isNull());

            SqlCondition condition = SqlCondition.trueCondition();
            if (startsInOrAfterO.isPresent()) {
                condition = condition.and(startTsColumn.ge(startsInOrAfterO.get()));
                if (endTsIsNull.isPresent()) {
                    condition = condition.and(endTsIsNull.get().or(endTsColumn.get().ge(startsInOrAfterO.get())));
                }
            }
            if (startsInOrBeforeO.isPresent()) {
                condition = condition.and(startTsColumn.le(startsInOrBeforeO.get()));
            }
            if (endsInOrAfterO.isPresent() && endTsIsNull.isPresent()) {
                condition = condition.and(endTsIsNull.get().or(endTsColumn.get().ge(endsInOrAfterO.get())));
            }
            if (startsInOrBeforeO.isPresent() && maxDuration.isPresent() && endTsIsNull.isPresent()) {
                condition = condition.and(endTsIsNull.get().or(
                        endTsColumn.get().le(startsInOrBeforeO.get().plus(maxDuration.get()))));
            }
            if (endsInOrAfterO.isPresent() && maxDuration.isPresent()) {
                condition = condition.and(startTsColumn.ge(endsInOrAfterO.get().minus(maxDuration.get())));
            }
            return condition;
        }

        private SqlCondition createTimeConditionForRepeating(MapField<?> startTsColumn, MapField<?> dueTsColumn) {
            Option<Instant> since = limits.getStartsInOrBeforeLimit();
            Option<Instant> due = limits.getEndsInOrAfterLimit().orElse(limits.getStartsInOrAfterLimit());

            return SqlCondition.trueCondition()
                    .and(due.map(dueTsColumn.column()::ge).orElseGet(dueTsColumn.column()::isNotNull)
                    .and(since.map(startTsColumn::le)));
        }

        private String andModifiedSinceSqlForMeTable() {
            SqlCondition condition = limits.getModifiedSince()
                    .map(MainEventFields.LAST_UPDATE_TS::ge)
                    .orElseGet(SqlCondition::trueCondition);

            return condition.andSqlForTable("me");
        }

        private Function<RepetitionInstanceInfo, Option<Instant>> firsInstanceStartF() {
            Option<Instant> startO = limits.getStartsInOrAfterLimit().orElse(limits.getEndsInOrAfterLimit());
            Option<Instant> endO = limits.getStartsInOrBeforeLimit();
            boolean overlap = !limits.getStartsInOrAfterLimit().isPresent();

            Function<InstantInterval, Instant> startF = InstantInterval::getStart;

            return info -> {
                Instant start = startO.getOrElse(info.getEventInterval().getStart());
                Option<InstantInterval> interval =
                        RepetitionUtils.getIntervals(info, start, endO, overlap, 1).singleO();

                return interval.map(startF);
            };
        }
    }

    /**
     * <p>
     * One of the core event-getting methods. Obtains 'pure (raw) db events'
     * (event + repetition + related stuff, but not calculated event instances).
     * </p><p>
     * Layers to watch at can be specified by layerIds at {@link EventGetProps}.
     * If so, then uid can be null. Otherwise, uid must be specified, and selected
     * events will be bounded to layers created or shared by the corresponding user.
     * </p><p>
     * Note that 'startMs', 'endMs' can be null (this means selection time interval
     * is not bounded by the past or the future). In practice, we give null values
     * for BOTH of them at once, so, though they can be null separately, this is
     * NOT well-tested by now (2009-09-10).
     * </p><p>
     * If 'startMs' is given, then 'overlap' parameter matters. If specifies whether
     * events need to begin in given interval, or just overlap its earlier bound.
     * Note that the latter (specified by 'endMs') bound is always overlapped.
     * Also, please NOTE that repeating events returned may not even overlap given
     * interval (in this sense, repeating events part in the result is APPROXIMATE).
     * </p><p>
     * The following permissions are required:
     * - user must have {@link LayerAction#LIST} for each layer;
     * - events that do not have at least {@link EventAction#VIEW}, are discarded.
     * Note that method always check these permissions. Therefore, if you want to
     * implement omit-check-behavior, you must specify some fake, powerful, uid
     * (e.g. uid of the layer creator, or uid of the person who shared the
     * layer to someone else, for the layer specified by 'privateToken'.)
     * </p>
     * @param uidO uid (possibly null/ANYONE) looks at the layers
     * specified at EventGetProps (by layerIds).
     * If uid specified, layerIds can be non-set
     * (in this case, all uid's layers are viewed).
     * Also is used for checking event permissions.
     * @return {@link ListF<EventInfo>} that contains information
     * about 'raw db events' and related entities.
     * or both uid and layerIds not given,
     * or some of the given layers cannot be found
     *
     * @see #getEventInfosByEvents
     */
    public ListF<EventInfo> getEventInfosByIds(
            Option<PassportUid> uidO, ListF<Long> eventIds, ActionSource actionSource)
    {
        return getEventInfosByEvents(uidO, eventDbManager.getEventsByIds(eventIds), actionSource);
    }

    public ListF<EventInfo> getEventInfosByIds(
            Option<UserInfo> userO, EventGetProps egp, ListF<Long> eventIds, ActionSource actionSource)
    {
        return getEventInfosByEvents(userO, egp, eventDbManager.getEventsByIds(eventIds), actionSource);
    }

    public ListF<EventInfo> getEventInfosByIdsSafe(
            Option<UserInfo> userO, EventGetProps egp, ListF<Long> eventIds, ActionSource actionSource)
    {
        return getEventInfosByEvents(userO, egp, eventDbManager.getEventsByIdsSafe(eventIds), actionSource);
    }

    public ListF<EventInfo> getEventInfosByIndentsAndPerms(
            Option<UserInfo> userO, EventGetProps egp,
            ListF<EventIndentAndRepetitionAndPerms> indents, ActionSource actionSource)
    {
        return loadEventInfos(userO, egp, EventsSource.indentsAndPerms(indents), actionSource);
    }

    public ListF<EventInfo> getEventInfosByIndents(
            Option<UserInfo> userO, EventGetProps egp,
            ListF<EventIndentAndRepetition> indents, ActionSource actionSource)
    {
        return loadEventInfos(userO, egp, EventsSource.indents(Option.empty(), indents), actionSource);
    }

    public EventInfo getEventInfoById(Option<PassportUid> uidO, long eventId, ActionSource actionSource) {
        return getEventInfosByIds(uidO, Cf.list(eventId), actionSource).single();
    }

    public EventInfo getEventInfoByEvent(Option<PassportUid> uidO, Event event, ActionSource actionSource) {
        return getEventInfosByEvents(uidO, Cf.list(event), actionSource).single();
    }

    public EventInfo getEventInfo(
            Option<PassportUid> uidO, EventWithRelations event, RepetitionInstanceInfo repetition, ActionSource actionSource)
    {
        return loadEventInfo(userInfo(uidO), event, repetition, actionSource);
    }

    public ListF<EventInfo> getEventInfosByEvents(
            Option<PassportUid> uidO, ListF<Event> events, ActionSource actionSource)
    {
        return getEventInfosByEvents(uidO, events, EventGetProps.any(), actionSource);
    }

    public ListF<EventInfo> getEventInfosByEvents(
            Option<PassportUid> uidO, ListF<Event> events, EventGetProps egp, ActionSource actionSource)
    {
        return getEventInfosByEvents(userInfo(uidO), egp, events, actionSource);
    }

    public ListF<EventInfo> getEventInfosByEvents(
            Option<UserInfo> userO, EventGetProps egp, ListF<Event> events, ActionSource actionSource)
    {
        Tuple2List<Event, RepetitionInstanceInfo> pairs = events.zipWith(Event.getIdF()
                .andThen(Cf2.f(repetitionRoutines.getRepetitionInstanceInfosByEvents(events)::getOrThrow)));
        final ListF<EventAndRepetition> eventAndRepetitions = pairs.map(EventAndRepetition.consF().asFunction());
        return loadEventInfos(userO, egp, EventsSource.events(eventAndRepetitions), actionSource);
    }

    public ListF<EventInfo> getEventInfosByEventsAndRepetitions(
            Option<UserInfo> userO, ListF<EventAndRepetition> events, ActionSource actionSource)
    {
        return loadEventInfos(userO, EventGetProps.any(), EventsSource.events(events), actionSource);
    }

    public ListF<EventInfo> getEventInfosByMainEventIds(
            Option<UserInfo> userO, ListF<Long> mainEventIds, EventGetProps egp, ActionSource actionSource)
    {
        ListF<EventAndRepetition> events = eventDbManager.getEventsAndRepetitionsByEvents(
                eventDao.findEventsByMainIds(mainEventIds));
        return loadEventInfos(userO, egp, EventsSource.events(events), actionSource);
    }

    public MainEventInfo getMainEventInfoById(Option<PassportUid> uidO, long mainEventId, ActionSource actionSource) {
        return getMainEventInfos(uidO, mainEventDao.findMainEventsByIds(Cf.list(mainEventId)), actionSource).single();
    }

    public ListF<MainEventInfo> getMainEventInfos(
            Option<PassportUid> uidO, ListF<MainEvent> mainEvents, ActionSource actionSource)
    {
       return getMainEventInfos(uidO, mainEvents, EventGetProps.any(), actionSource);
    }

    public ListF<MainEventInfo> getMainEventInfos(
            Option<PassportUid> uidO, ListF<MainEvent> mainEvents, EventGetProps egp, ActionSource actionSource)
    {
        ListF<Long> mainEventIds = mainEvents.map(MainEvent.getIdF());
        ListF<EventInfo> events = getEventInfosByEvents(uidO, eventDao.findEventsByMainIds(mainEventIds),
                egp, actionSource);

        return mainEvents
                .zipWith(MainEvent.getIdF().andThen(Cf2.f(events.groupBy(EventInfo::getMainEventId)::getOrThrow)))
                .map(MainEventInfo::new);
    }

    public ListF<EventInfo> getEventInfosOnLayers(
            Option<PassportUid> uidO, EventGetProps eventProps,
            ListF<Long> layerIds, EventLoadLimits limits, ActionSource actionSource)
    {
        return loadEventInfos(userInfo(uidO), eventProps, EventsSource.indents(Option.of(true),
            getEventIndentsOnLayers(layerIds, limits)), actionSource);
    }

    public ListF<EventInfo> getEventInfosOnLayer(
            Option<PassportUid> uidO, EventGetProps eventProps,
            long layerId, EventLoadLimits limits, ActionSource actionSource)
    {
        return getEventInfosOnLayers(uidO, eventProps, Cf.list(layerId), limits, actionSource);
    }

    public ListF<MainEvent> getMainEventsOnLayer(long layerId, EventLoadLimits limits) {
        ListF<EventIndentAndRepetition> events = getEventIndentsOnLayers(Cf.list(layerId), limits);

        return mainEventDao.findMainEventsByIds(events.map(e -> e.getIndent().getMainEventId()).stableUnique());
    }

    public ListF<EventInfo> getEventInfosOnResources(
            Option<PassportUid> uidO, EventGetProps eventProps,
            ListF<Long> resourceIds, EventLoadLimits limits, ActionSource actionSource)
    {
        return loadEventInfos(userInfo(uidO), eventProps,
                EventsSource.indents(Option.of(false), getEventIndentsOnResources(resourceIds, limits)), actionSource);
    }

    public ListF<EventAndRepetition> getEventsOnLayers(
            ListF<Long> layerIds, EventLoadLimits limits)
    {
        return new Do(limits, LayerIdsOrResourceIds.layerIds(layerIds)).getEventsOnTarget();
    }

    public ListF<EventAndRepetition> getEventsOnResources(
            ListF<Long> resourceIds, EventLoadLimits limits)
    {
        return new Do(limits, LayerIdsOrResourceIds.resourceIds(resourceIds)).getEventsOnTarget();
    }

    public ListF<EventIndentAndRepetition> getEventIndentsOnLayers(
            ListF<Long> layerIds, EventLoadLimits limits)
    {
        return new Do(limits, LayerIdsOrResourceIds.layerIds(layerIds)).getIndentsOnTarget();
    }

    public ListF<EventIndentAndRepetition> getEventIndentsOnResources(
            ListF<Long> resourceIds, EventLoadLimits limits)
    {
        return new Do(limits, LayerIdsOrResourceIds.resourceIds(resourceIds)).getIndentsOnTarget();
    }

    private Option<UserInfo> userInfo(Option<PassportUid> uidO) {
        return userManager.getUserInfos(uidO).singleO();
    }

    private static class EventsSource {
        private Option<ListF<EventIndentAndRepetition>> indents;
        private Option<ListF<EventIndentAndRepetitionAndPerms>> indentAndPerms;
        private Option<ListF<EventAndRepetition>> events;
        private Option<Boolean> isLayers;

        private EventsSource(
                Option<ListF<EventIndentAndRepetition>> indents,
                Option<ListF<EventIndentAndRepetitionAndPerms>> indentAndPerms,
                Option<ListF<EventAndRepetition>> events,
                Option<Boolean> isLayers)
        {
            this.indents = indents;
            this.indentAndPerms = indentAndPerms;
            this.events = events;
            this.isLayers = isLayers;
        }

        public static EventsSource indents(Option<Boolean> isLayers, ListF<EventIndentAndRepetition> indents) {
            return new EventsSource(Option.of(indents), Option.empty(), Option.empty(), isLayers);
        }

        public static EventsSource indentsAndPerms(ListF<EventIndentAndRepetitionAndPerms> indents) {
            return new EventsSource(Option.empty(), Option.of(indents), Option.empty(), Option.empty());
        }

        public static EventsSource events(ListF<EventAndRepetition> events) {
            return new EventsSource(Option.empty(), Option.empty(), Option.of(events), Option.empty());
        }

        public boolean isEvents() {
            return events.isPresent();
        }

        public boolean isIndentsOrIndentsAndPerms() {
            return !isEvents();
        }

        public boolean isIndentAndPerms() {
            return indentAndPerms.isPresent();
        }

        public ListF<EventAndRepetition> getEvents() {
            return events.get();
        }

        public ListF<EventIndentAndRepetition> getIndentsDeduplicated() {
            if (isLayers.isPresent()) {
                return indents.get().stableUniqueBy(Tuple2.join(
                        EventIndentAndRepetition::getEventId, e -> e.getIndent().getLayerOrResourceId()));

            } else if (indents.isPresent()) {
                return indents.get().stableUniqueBy(EventIndentAndRepetition::getEventId);

            } else {
                return indentAndPerms.get().map(EventIndentAndRepetitionAndPerms::getIndentAndRepetition)
                        .stableUniqueBy(EventIndentAndRepetition::getEventId);
            }
        }

        public ListF<EventInfoForPermsCheck> getPermsInfoFromIndents() {
            return indentAndPerms.get().map(EventIndentAndRepetitionAndPerms::getPermInfo);
        }

        public boolean isEmpty() {
            return indents.getOrElse(Cf.list()).isEmpty()
                    && indentAndPerms.getOrElse(Cf.list()).isEmpty()
                    && events.getOrElse(Cf.list()).isEmpty();
        }
    }
}
