package ru.yandex.calendar.logic.event;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.sharing.perm.EventInfoForPermsCheck;
import ru.yandex.misc.time.InstantInterval;


/**
 * @author Stepan Koltsov
 */
public class EventInstanceForUpdate {
    private final Option<InstantInterval> interval;
    private final EventInfoForPermsCheck infoForPermsCheck;
    private final EventWithRelations eventWithRelations;
    private final RepetitionInstanceInfo repetitionInstanceInfo;
    private final Option<EventLayer> eventLayer;
    private final Option<EventUserWithNotifications> eventUserWithNotifications;
    private final ListF<EventAttachment> attachments;

    public EventInstanceForUpdate(
            Option<InstantInterval> interval,
            EventInfoForPermsCheck infoForPermsCheck,
            EventWithRelations eventWithRelations,
            RepetitionInstanceInfo repetitionInstanceInfo,
            Option<EventUserWithNotifications> eventUserWithNotifications,
            Option<EventLayer> eventLayer,
            ListF<EventAttachment> attachments)
    {
        this.interval = interval;
        this.infoForPermsCheck = infoForPermsCheck;
        this.eventWithRelations = eventWithRelations;
        this.repetitionInstanceInfo = repetitionInstanceInfo;
        this.eventUserWithNotifications = eventUserWithNotifications;
        this.eventLayer = eventLayer;
        this.attachments = attachments;
    }

    public EventInfo toEventInfo() {
        return EventInfo.cons(
                infoForPermsCheck, eventWithRelations, repetitionInstanceInfo,
                eventUserWithNotifications, attachments, Option.empty(), Option.empty(), true);
    }

    public Option<InstantInterval> getInterval() {
        return interval;
    }

    public InstantInterval getIntervalOrEventInterval() {
        return interval.getOrElse(() -> new InstantInterval(getEvent().getStartTs(), getEvent().getEndTs()));
    }

    public EventWithRelations getEventWithRelations() {
        return eventWithRelations;
    }

    public Event getEvent() {
        return eventWithRelations.getEvent();
    }

    public long getEventId() {
        return getEvent().getId();
    }

    public MainEvent getMainEvent() {
        return eventWithRelations.getMainEvent();
    }

    public long getMainEventId() {
        return getEvent().getMainEventId();
    }

    public boolean isExportedWithEws() {
        return getMainEvent().getIsExportedWithEws().isSome(true);
    }

    public RepetitionInstanceInfo getRepetitionInstanceInfo() {
        return repetitionInstanceInfo;
    }

    public EventAndRelations getEventAndRelations() {
        return new EventAndRelations(getEventWithRelations(), getRepetitionInstanceInfo());
    }

    public Option<EventUserWithNotifications> getEventUserWithNotifications() {
        return eventUserWithNotifications;
    }

    public Option<EventUser> getEventUser() {
        return eventUserWithNotifications.map(EventUserWithNotifications.getEventUserF());
    }

    public Option<EventLayer> getEventLayer() {
        return eventLayer;
    }

    public ListF<EventAttachment> getAttachments() {
        return attachments;
    }

    public EventInstanceForUpdate withEmptyInterval() {
        return new EventInstanceForUpdate(
                Option.empty(), infoForPermsCheck, eventWithRelations, repetitionInstanceInfo,
                eventUserWithNotifications, eventLayer, attachments);
    }

    public EventInstanceForUpdate withNoRepetition() {
        return new EventInstanceForUpdate(
                interval, infoForPermsCheck, eventWithRelations,
                repetitionInstanceInfo.withoutAnyRepeating(),
                eventUserWithNotifications, eventLayer, attachments);
    }

    public EventInstanceForUpdate withIntervalInsteadOfRecurrence(Instant start) {
        RepetitionInstanceInfo repetitionInfo = repetitionInstanceInfo.withoutRecurrence(start);

        InstantInterval interval = RepetitionUtils.getInstanceIntervalStartingAt(repetitionInfo, start)
                .getOrThrow(CommandRunException.createSituationF(
                        "not a valid start of event instance: " + start, Situation.EVENT_NOT_FOUND));

        return new EventInstanceForUpdate(Option.of(interval),
                infoForPermsCheck, eventWithRelations, repetitionInfo,
                eventUserWithNotifications, eventLayer, attachments);
    }
} //~
