package ru.yandex.calendar.logic.event;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.notification.Channel;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.perm.EventInfoForPermsCheck;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.misc.time.InstantInterval;

/**
 * Used for transmitting information about event instances
 * (for both non-repeating and repeating events) as well as
 * auxiliary environment information related to them.
 * Main fields: interval and event-layer well define the
 * other fields, assuming that the user is the same.
 */
public class EventInstanceInfo {
    private final EventInterval interval;
    private final EventInfoForPermsCheck infoForPermsCheck;
    private final EventIndentAndRepetition indent;
    private final Event partialEvent;
    private final MainEvent partialMainEvent;
    private final Option<RepetitionInstanceInfo> repetitionInfo;
    private final Option<EventParticipants> participants;
    private final Option<ListF<ResourceInfo>> resources;
    private final Option<EventWithRelations> eventWithRelations;
    private final Option<EventUserWithNotifications> eventUserWithNotifications;
    private final Option<ListF<EventAttachment>> attachments;
    private final Option<Long> layerId;
    private final Option<Long> resourceId;

    private final boolean mayView;
    private final long eventId;

    public EventInstanceInfo(
            InstantInterval interval,
            EventInfoForPermsCheck infoForPermsCheck,
            EventIndentAndRepetition indent,
            Event partialEvent, MainEvent partialMainEvent,
            Option<RepetitionInstanceInfo> repetitionInfo,
            Option<EventParticipants> participants,
            Option<ListF<ResourceInfo>> resources,
            Option<EventWithRelations> eventWithRelations,
            Option<EventUserWithNotifications> eventUserWithNotifications,
            Option<ListF<EventAttachment>> attachments,
            Option<Long> layerId, Option<Long> resourceId, boolean mayView)
    {
        this.interval = indent.getEventInterval(interval);
        this.infoForPermsCheck = infoForPermsCheck;
        this.indent = indent;
        this.partialEvent = partialEvent;
        this.partialMainEvent = partialMainEvent;
        this.repetitionInfo = repetitionInfo;
        this.participants = participants;
        this.resources = resources;
        this.eventWithRelations = eventWithRelations;
        this.eventUserWithNotifications = eventUserWithNotifications;
        this.attachments = attachments;
        this.layerId = layerId;
        this.resourceId = resourceId;
        this.mayView = mayView;
        this.eventId = indent.getEventId();
    }

    public <T> Option<T> getEventFieldValueO(MapField<T> field) {
        return partialEvent.getFieldValueO(field);
    }

    public Event getEvent() {
        return partialEvent;
    }

    public <T> Option<T> getMainEventFieldValueO(MapField<T> field) {
        return partialMainEvent.getFieldValueO(field);
    }

    public MainEvent getMainEvent() {
        return partialMainEvent;
    }

    public EventWithRelations getEventWithRelations() {
        return eventWithRelations.getOrThrow("event with relations not loaded");
    }

    public EventParticipants getEventParticipants() {
        return participants.getOrThrow("participants not loaded");
    }

    public Participants getParticipants() {
        return getEventParticipants().getParticipants();
    }

    public ListF<ResourceInfo> getResources() {
        return resources.getOrThrow("resources not loaded");
    }

    public Option<ListF<EventAttachment>> getAttachmentsO() {
        return attachments;
    }

    public ListF<EventAttachment> getAttachments() {
        return attachments.getOrThrow("attachments not loaded");
    }

    public EventInfoForPermsCheck getInfoForPermsCheck() {
        return infoForPermsCheck;
    }

    public InstantInterval getInterval() {
        return interval.toInstantInterval(indent.getIndent().getTz());
    }

    public EventInterval getEventInterval() {
        return interval;
    }

    public long getEventId() {
        return eventId;
    }

    public Option<EventUser> getEventUser() {
        return eventUserWithNotifications.map(EventUserWithNotifications.getEventUserF());
    }

    public ListF<Notification> getNotifications(SetF<Channel> channels) {
        return eventUserWithNotifications
                .map(eu -> eu.getNotifications().getNotifications())
                .getOrElse(Cf.<Notification>list())
                .filter(n -> channels.containsTs(n.getChannel()));
    }

    public Option<Repetition> getRepetition() {
        return getRepInstInfo().getRepetition();
    }

    public Option<Long> getLayerId() {
        return layerId;
    }

    public Option<Long> getResourceId() {
        return resourceId;
    }

    public EventIndent getIndent() {
        return indent.getIndent();
    }

    public EventInfo toEventInfo() {
        return new EventInfo(
                infoForPermsCheck, indent, partialEvent, partialMainEvent, repetitionInfo,
                participants, resources, eventWithRelations, eventUserWithNotifications, attachments,
                layerId, resourceId, mayView);
    }

    public RepetitionInstanceInfo getRepInstInfo() {
        return repetitionInfo.getOrElse(indent.getRepetitionInfo());
    }

    public Option<EventUserWithNotifications> getEventUserWithNotifications() {
        return eventUserWithNotifications;
    }

    public boolean getMayView() {
        return mayView;
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + "["
            + eventId + ","
            + interval
            + ",...]";
    }

    public static Function1B<EventInstanceInfo> notRejectedF() {
        return eventInstanceInfo -> !eventInstanceInfo.getEventUser().map(EventUser.getDecisionF()).isSome(Decision.NO);
    }

    public static Function<EventInstanceInfo, Instant> startF() {
        return eventInstanceInfo -> eventInstanceInfo.getInterval().getStart();
    }

    public static Function<EventInstanceInfo, InstantInterval> getIntervalF() {
        return EventInstanceInfo::getInterval;
    }

    public static Function<EventInstanceInfo, EventWithRelations> getEventWithRelationsF() {
        return EventInstanceInfo::getEventWithRelations;
    }

    public static Function<EventInstanceInfo, Event> getEventF() {
        return EventInstanceInfo::getEvent;
    }

    public static Function<EventInstanceInfo, Option<EventUser>> getEventUserF() {
        return EventInstanceInfo::getEventUser;
    }

    public static Function<EventInstanceInfo, Long> eventIdF() {
        return EventInstanceInfo::getEventId;
    }
}
