package ru.yandex.calendar.logic.event;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.ExternalIdChangeLogEventJson;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author ssytnik
 */
public class EventInstanceStatusChecker {
    private static final Logger logger = LoggerFactory.getLogger(EventInstanceStatusChecker.class);

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventsLogger eventsLogger;


    /**
     * tries to find event instance at user and then all specified subjects,
     * then calls {{@link #getStatusBySubjectAndSyncData(UidOrResourceId, EventSynchData, EventWithRelations, ActionSource)}}
     */
    public EventInstanceStatusInfo getStatusByParticipantsAndIcsSyncData(
            UidOrResourceId subject, IcsEventSynchData icsSynchData, ListF<UidOrResourceId> participantSubjects, ActionSource actionSource)
    {
        if (icsSynchData.externalId.isPresent()) {
            Option<Event> eventO = eventRoutines.findEvent(
                     Cf.list(subject).plus(participantSubjects),
                     icsSynchData.externalId.get(), icsSynchData.recurrenceId);
            if (eventO.isPresent()) {
                EventWithRelations eventWithRelations = eventDbManager.getEventWithRelationsByEvent(eventO.get());
                return getStatusBySubjectAndSyncData(subject, icsSynchData, eventWithRelations, actionSource);
            }
            // XXX else look it up in deleted_event
        }
        return EventInstanceStatusInfo.notFound();
    }

    public EventInstanceStatusInfo getStatusByExchangeSynchData(
            UidOrResourceId subject, ExchangeEventSynchData exchangeSynchData,
            Option<String> externalId, ActionInfo actionInfo)
    {
        ActionSource actionSource = actionInfo.getActionSource();
        String exchangeId = exchangeSynchData.exchangeId;
        Option<Event> eventO = eventRoutines.findEventByExchangeId(exchangeId);
        if (eventO.isPresent()) {
            final Event event = eventO.get();

            if (externalId.isPresent()) {
                MainEvent mainEvent = mainEventDao.findMainEventById(event.getMainEventId());
                String currentExternalId = mainEvent.getExternalId();
                boolean isExportedWithEws = mainEvent.getIsExportedWithEws().getOrElse(false);

                if (!externalId.get().equals(currentExternalId)) {
                    logger.debug("External-id: exchange [" + externalId.get() + ", calendar [" + currentExternalId + "]");
                    eventsLogger.log(new ExternalIdChangeLogEventJson(currentExternalId, externalId.get()), actionInfo);
                    return EventInstanceStatusInfo.needToUpdateWithNewExternalId(
                            event.getId(), externalId.get(), isExportedWithEws);
                }
            }

            if (actionSource == ActionSource.EXCHANGE_SYNCH) {
                // optimization: exchange synch always works with organizer or resource (as this check goes first),
                // so check status for organizer - this is quick. XXX note case 'not organizer in calendar' // ssytnik@
                return getStatusByEventAndSyncData(exchangeSynchData, event);
            } else {
                EventWithRelations eventWithRelations = eventDbManager.getEventWithRelationsByEvent(eventO.get());
                return getStatusBySubjectAndSyncData(subject, exchangeSynchData, eventWithRelations, actionSource);
            }
        } else {
            Instant exchangeLastModifiedTs = exchangeSynchData.lastModified;
            Option<Tuple3<Instant, Long, ActionSource>> deletionO = eventRoutines.findLatestEventDeletionInfoByExchangeId(exchangeId);
            logger.debug("Exchange [" + exchangeLastModifiedTs + "], calendar [deleted: " + deletionO + "]");

            if (!deletionO.isPresent()) {
                return EventInstanceStatusInfo.notFound();
            }

            if (actionSource == ActionSource.EXCHANGE_SYNCH) {
                return EventInstanceStatusInfo.notFound();
            }

            boolean deletionIsNewerThenUpdate = deletionO.get()._1.isAfter(exchangeLastModifiedTs);

            if (deletionIsNewerThenUpdate && !deletionO.get()._3.isByExchangeNotification()) {
                return EventInstanceStatusInfo.alreadyUpdated(deletionO.get()._2);
            } else {
                return EventInstanceStatusInfo.notFound();
            }
        }
    }


    /**
     * check status using event's fields for event creator/organizer, or event-user's fields otherwise
     */
    public EventInstanceStatusInfo getStatusBySubjectAndSyncData(
            UidOrResourceId subject, EventSynchData eventSynchData, EventWithRelations eventWithRelations, ActionSource actionSource) {

        EventInstanceStatusInfo status;
        if (subject.isResource() || eventWithRelations.ownerIs(subject.getUid()) || actionSource.isFromMailOrHook()) {
            status = getStatusByEventAndSyncData(eventSynchData, eventWithRelations.getEvent());
        } else {
            Option<EventUser> eventUserO = eventWithRelations.getEventUsers()
                    .find(EventUserFields.UID.getF().andThenEquals(subject.getUid()));
            // XXX if empty, look it up in deleted_event_user
            status = !eventUserO.isPresent() ?
                    EventInstanceStatusInfo.needToUpdate(eventWithRelations.getId()) :
                    getStatusByEventUserAndSyncData(eventSynchData, eventUserO.get());
        }

        if (status.isAlreadyUpdated() && actionSource == ActionSource.CALDAV) {
            return EventInstanceStatusInfo.fromCaldavAlreadyUpdated(status.getEventId());
        } else {
            return status;
        }
    }

    private EventInstanceStatusInfo getStatusByEventAndSyncData(EventSynchData eventSynchData, Event event) {
        logger.info("Checking version by event (" + eventSynchData.getClass().getSimpleName() + ")");
        return getStatusByTwoVersions(event.getId(),
                eventSynchData.sequence, eventSynchData.lastModified, event.getSequence(), event.getLastUpdateTs());
    }

    private EventInstanceStatusInfo getStatusByEventUserAndSyncData(EventSynchData eventSynchData, EventUser eventUser) {
        logger.info("Checking version by event-user (" + eventSynchData.getClass().getSimpleName() + ")");
        return getStatusByTwoVersions(eventUser.getEventId(),
                eventSynchData.sequence, eventSynchData.dtstamp, eventUser.getSequence(), eventUser.getDtstamp().getOrElse(new Instant(0)));
    }

    private EventInstanceStatusInfo getStatusByTwoVersions(long eventId, int seq1, Instant instant1, int seq2, Instant instant2) {
        return isVersionGreater(seq1, instant1, seq2, instant2) ?
                EventInstanceStatusInfo.needToUpdate(eventId) : EventInstanceStatusInfo.alreadyUpdated(eventId);
    }

    /**
     * Compares version of two event version by theirs sequence and dtstamp
     * @param seq1 first (new) event sequence number
     * @param instant1 first (new) event last-modified or event-user dtstamp
     * @param seq2 second (existing) event sequence number
     * @param instant2 second (existing) event last-modified or event-user dtstamp
     * @return true, if first version is latest version and false otherwise
     */
    public static boolean isVersionGreater(int seq1, Instant instant1, int seq2, Instant instant2) {
        // ssytnik@: version should be strictly greater to avoid needless ews synch. event updates.
        // It should work for recurrence ids, proof: IcsImporterFromFileNonMeetingTest.repEvent2NeedToUpdate() test.
        // TODO check if we need to ignore millis (see TestManager#withNoMillis)
        boolean res = seq1 > seq2 || (seq1 == seq2 && instant1.getMillis() > instant2.getMillis());
        // XXX diag // ssytnik@ 2011-04-06
        logger.info("IsVersionGreater: seq1 = " + seq1 + ", instant1 = " + instant1 + ", seq2 = " + seq2 + ", instant2 = " + instant2 + ", res = " + res);
        return res;
    }

} //~
