package ru.yandex.calendar.logic.event;

import java.util.Collection;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.beans.generated.EventInvitationFields;
import ru.yandex.calendar.logic.beans.generated.EventInvitationHelper;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ExternalUserParticipantInfo;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Stepan Koltsov
 */
public class EventInvitationDao extends CalendarJdbcDaoSupport {
    private static final Logger logger = LoggerFactory.getLogger(EventInvitationDao.class);

    private static EventInvitation withDecisionSourceIfNeeded(EventInvitation eventInvitation, ActionInfo actionInfo) {
        EventInvitation newEventInvitation = eventInvitation.clone();
        if (eventInvitation.isFieldSet(EventInvitationFields.DECISION)) {
            newEventInvitation.setDecisionSource(actionInfo.getActionSource());
        }
        return newEventInvitation;
    }

    @Autowired
    private GenericBeanDao genericBeanDao;

    @RunWithRandomTest
    public ListF<EventInvitation> findEventInvitationsByEventId(long eventId) {
        return findEventInvitationsByEventIds(Cf.list(eventId));
    }

    @RunWithRandomTest
    public ListF<EventInvitation> findEventInvitationsByEventIds(ListF<Long> eventIds) {
        String q = "SELECT * FROM event_invitation WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, EventInvitation.class);
    }

    @RunWithRandomTest
    public Option<EventInvitation> findEventInvitationByEventIdAndEmail(long eventId, Email email) {
        return findEventInvitationsByEventIdsAndEmails(Cf.list(eventId), Cf.list(email)).singleO();
    }

    @RunWithRandomTest
    public ListF<EventInvitation> findEventInvitationsByEventIdsAndEmails(ListF<Long> eventIds, ListF<Email> emails) {
        SqlCondition c = EventInvitationFields.EVENT_ID.column().inSet(eventIds)
                .and(EventInvitationFields.EMAIL.column().inSet(emails));

        String q = "SELECT * FROM event_invitation" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, EventInvitation.class, c.args());
    }

    @RunWithRandomTest
    public ListF<EventInvitation> findEventInvitationsByEmails(ListF<Email> emails) {
        SqlCondition emailIn = EventInvitationFields.EMAIL.column().inSet(emails);
        String q = "SELECT * FROM event_invitation WHERE " + emailIn.sql();

        if (skipQuery(emailIn, q, emailIn.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, EventInvitation.class, emailIn.args());
    }

    @RunWithRandomTest
    public Option<String> findInvitationPrivateTokenByEventIdAndEmail(long eventId, Email email) {
        return findEventInvitationByEventIdAndEmail(eventId, email).map(EventInvitationFields.PRIVATE_TOKEN.getF());
    }

    @RunWithRandomTest
    public Option<EventInvitation> findInvitationByPrivateToken(String privateToken) {
        String q = "SELECT * FROM event_invitation WHERE private_token = ?";
        return getJdbcTemplate().queryForOption(q, EventInvitation.class, privateToken);
    }

    @RunWithRandomTest
    public void deleteInvitationByEmailAndEventIdAndIsOrganizer(Email email, long eventId, Option<Boolean> isOrganizer) {
        String q = "DELETE FROM event_invitation WHERE email = ? AND event_id = ?"
                + isOrganizer.map(o -> " AND is_organizer = ?").getOrElse("");
        getJdbcTemplate().update(q, email, eventId, isOrganizer);
    }

    @RunWithRandomTest
    public void deleteInvitationsByEventIds(ListF<Long> eventIds) {
        String q = "DELETE FROM event_invitation WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void deleteInvitationById(EventInvitationId id) {
        genericBeanDao.deleteBeanById(EventInvitationHelper.INSTANCE, id.asTuple());
    }

    @RunWithRandomTest
    public ListF<ExternalUserParticipantInfo> findExternalUserEventInvitees(Collection<Long> eventIds) {
        SqlCondition c = SqlCondition.all(
                EventInvitationFields.EVENT_ID.column().inSet(eventIds));
        return genericBeanDao.loadBeans(EventInvitationHelper.INSTANCE, c).map(ExternalUserParticipantInfo.consF());
    }

    @RunWithRandomTest
    public Tuple2List<Long, Option<Email>> findOrganizersByEventIds(ListF<Long> eventIds) {
        if (eventIds.isEmpty()) return Tuple2List.tuple2List();

        SqlCondition c = EventInvitationFields.IS_ORGANIZER.eq(true).and(EventInvitationFields.EVENT_ID.column().inSet(eventIds));
        String q = "SELECT event_id, email FROM event_invitation" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Tuple2List.tuple2List();

        MapF<Long, Email> x = getJdbcTemplate().queryForList2(q, Long.class, Email.class, c.args()).toMap();
        return eventIds.zipWith(x::getO);
    }

    @RunWithRandomTest
    public int updateEventInvitationSetOrganizerByEventIdAndEmail(long eventId, Email email, boolean organizer) {
        String q = "UPDATE event_invitation SET is_organizer = ? WHERE event_id = ? AND email = ?";
        return getJdbcTemplate().update(q, organizer, eventId, email);
    }

    @RunWithRandomTest
    public int updateEventInvitationSetOrganizerByInvitationId(EventInvitationId invitationId, boolean organizer) {
        SqlCondition c = invitationId.eq();
        String q = "UPDATE event_invitation SET is_organizer = ?" + c.whereSql();
        return getJdbcTemplate().update(q, organizer, c.args());
    }

    public int updateEventInvitationByEventIdAndEmail(
            long eventId, Email email, EventInvitation invitationData, ActionInfo actionInfo)
    {
        SqlCondition c = EventInvitationFields.EVENT_ID.eq(eventId).and(EventInvitationFields.EMAIL.eq(email));
        return genericBeanDao.updateBeans(withDecisionSourceIfNeeded(invitationData, actionInfo), c);
    }

    public void updateEventInvitation(EventInvitation invitation, ActionInfo actionInfo) {
        genericBeanDao.updateBean(withDecisionSourceIfNeeded(invitation, actionInfo));
    }

    @RunWithRandomTest
    public int updateEventInvitationsByEventIdsAndEmail(
            ListF<Long> eventIds, Email email, Decision decision, ActionInfo actionInfo)
    {
        final String sql = "UPDATE event_invitation SET decision = ?, decision_source = ? WHERE email = ? AND event_id "
                + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, sql, decision, email)) return 0;

        return getJdbcTemplate().update(sql, decision, actionInfo.getActionSource(), email);
    }

    public void saveEventInvitation(EventInvitation invitationData, ActionInfo actionInfo) {
        EventInvitation tmp = invitationData.clone();

        tmp.setCreationReqId(actionInfo.getRequestIdWithHostId());
        tmp.setCreationTs(actionInfo.getNow());
        tmp.setCreationSource(actionInfo.getActionSource());

        genericBeanDao.insertBean(withDecisionSourceIfNeeded(tmp, actionInfo));
    }

    public void saveEventInvitationsBatch(ListF<EventInvitation> invitations, ActionInfo actionInfo) {
        genericBeanDao.insertBeans(invitations.map(ei -> withDecisionSourceIfNeeded(ei, actionInfo)));
    }

    @RunWithRandomTest
    public Option<Layer> findLayerByUidAndMainEventId(PassportUid uid, long mainEventId) {
        String q = "SELECT l.*" +
                " FROM event_layer el" +
                " INNER JOIN layer l ON el.layer_id = l.id" +
                " INNER JOIN event e ON el.event_id = e.id" +
                " WHERE e.main_event_id = ? AND l.creator_uid = ?" +
                " LIMIT 1";
        return getJdbcTemplate().queryForOption(q, Layer.class, mainEventId, uid);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateEventInvitationSetDecisionAndReasonById(
            Decision decision, String reason, EventInvitationId id, ActionInfo actionInfo)
    {
        SqlCondition c = id.eq();
        String q = "UPDATE event_invitation SET decision = ?, reason = ?, decision_source = ? " + c.whereSql();
        getJdbcTemplate().updateRow(q, decision, reason, actionInfo.getActionSource(), c.args());
    }

    public void updateInvitation(EventInvitation invitation, ActionInfo actionInfo) {
        genericBeanDao.updateBean(withDecisionSourceIfNeeded(invitation, actionInfo));
    }

    @RunWithRandomTest
    public ListF<EventInvitation> findEventInvitations() {
        String q = "SELECT * FROM event_invitation WHERE event_id IS NOT NULL";
        return getJdbcTemplate().queryForList(q, EventInvitation.class);
    }

    public ListF<EventInvitation> findEventInvitations(SqlCondition condition) {
        String q = "SELECT * FROM event_invitation WHERE " + condition.sql();
        return getJdbcTemplate().queryForList(q, EventInvitation.class, condition.args());
    }

    @RunWithRandomTest
    public ListF<EventInvitation> findInvitations() {
        String q = "SELECT * FROM event_invitation";
        return getJdbcTemplate().queryForList(q, EventInvitation.class);
    }

    @RunWithRandomTest
    public int lowercaseInvitationEmails() {
        String q = "UPDATE event_invitation SET email = LOWER(email)";
        return getJdbcTemplate().update(q);
    }

} //~
