package ru.yandex.calendar.logic.event;

import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.calendar.CalendarException;
import ru.yandex.calendar.boot.EwsAliveHandler;
import ru.yandex.calendar.frontend.ews.exp.EwsExportRoutines;
import ru.yandex.calendar.frontend.ews.imp.ExchangeEventDataConverter;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.beans.generated.EventInvitationFields;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.meeting.EventInvitationResults;
import ru.yandex.calendar.logic.event.meeting.UpdateMode;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventInvitationData;
import ru.yandex.calendar.logic.event.model.EventInvitationUpdateData;
import ru.yandex.calendar.logic.event.model.EventInvitationsData;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.event.model.EventUserUpdate;
import ru.yandex.calendar.logic.event.model.ParticipantsOrInvitationsData;
import ru.yandex.calendar.logic.event.model.WebReplyData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.ics.exp.IcsEventExporter;
import ru.yandex.calendar.logic.ics.exp.IcsExportMode;
import ru.yandex.calendar.logic.ics.exp.IcsExportParameters;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsProperty;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.PropertiesMeta;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.resource.RejectedResources;
import ru.yandex.calendar.logic.resource.ResourceAccessRoutines;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sending.EventInvitationOrCancelMessageXmlCreator;
import ru.yandex.calendar.logic.sending.EventSendingInfo;
import ru.yandex.calendar.logic.sending.InvitationOrCancelEmailData;
import ru.yandex.calendar.logic.sending.bazinga.MessageExtraDao;
import ru.yandex.calendar.logic.sending.param.ApartmentMessageParameters;
import ru.yandex.calendar.logic.sending.param.CommonEventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventInviteeKey;
import ru.yandex.calendar.logic.sending.param.EventInviteeNames;
import ru.yandex.calendar.logic.sending.param.EventInviteeNamesI18n;
import ru.yandex.calendar.logic.sending.param.EventMessageInfo;
import ru.yandex.calendar.logic.sending.param.EventMessageInfoCreator;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventOnLayerChangeMessageParameters;
import ru.yandex.calendar.logic.sending.param.MessageDestination;
import ru.yandex.calendar.logic.sending.param.MessageExtra;
import ru.yandex.calendar.logic.sending.param.MessageOverrides;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.sending.param.Recipient;
import ru.yandex.calendar.logic.sending.param.ReplyMessageParameters;
import ru.yandex.calendar.logic.sending.param.Sender;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.sending.so.SoChecker;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.EventParticipantsChangesInfo;
import ru.yandex.calendar.logic.sharing.InvAcceptingType;
import ru.yandex.calendar.logic.sharing.InvitationCreature;
import ru.yandex.calendar.logic.sharing.InvitationData;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.sharing.Nick;
import ru.yandex.calendar.logic.sharing.ParticipantChangesInfo;
import ru.yandex.calendar.logic.sharing.ReplyInfo;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.participant.ExternalUserParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantBasics;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantKind;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.LayerActionClass;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.user.KarmaCheckAction;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.LoginOrEmail;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.NameI18nWithOptionality;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.EventAction;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.Environment;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.base.UidGen;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.mail.cerberus.yt.staff.dto.StaffUser.Gender;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

@Slf4j
public class EventInvitationManager {
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private MailSender mailSender;
    @Autowired
    private EventInvitationOrCancelMessageXmlCreator eventInvitationOrCancelMessageXmlCreator;
    @Autowired
    private EventInvitationDao eventInvitationDao;
    @Autowired
    private EventInvitationRoutines eventInvitationRoutines;
    @Autowired
    private IcsEventExporter icsEventExporter;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private EwsExportRoutines ewsExportRoutines;
    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private ResourceAccessRoutines resourceAccessRoutines;

    @Autowired
    private EventMessageInfoCreator eventMessageInfoCreator;
    @Autowired
    private SoChecker soChecker;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private NotificationRoutines notificationRoutines;
    @Autowired
    private MessageExtraDao messageExtraDao;
    @Autowired
    private EwsAliveHandler ewsAliveHandler;


    private boolean isAutoAccept(InvitationData invitation) {
        ParticipantId participantId = invitation.participantId;

        if (participantId.isResource()) {
            return true;
        } else if (participantId.isExternalUser()) {
            return false;
        } else if (participantId.isYandexUser()) {
            return invitation.settings.get().getCommon().getInvAcceptType() != InvAcceptingType.MANUAL;
        } else {
            throw new IllegalStateException();
        }
    }

    public void createAndSendEventInvitationOrCancelMails(
            ActorId actorId, ListF<EventSendingInfo> sendingInfoList, ActionInfo actionInfo) {
        sendEventMails(createEventInvitationOrCancelMails(actorId, sendingInfoList, actionInfo), actionInfo);
    }

    public ListF<EventMessageParameters> createEventInvitationOrCancelMails(
            ActorId actorId, EventWithRelations event, RepetitionInstanceInfo repetition,
            ListF<EventSendingInfo> sendingInfos, final ActionInfo actionInfo) {
        ListF<Long> eventIds = eventRoutines.findMasterAndSingleEventIds(event.getId());

        Validate.forAll(sendingInfos, EventSendingInfo.getEventIdF().andThen(eventIds::containsTs));

        final ListF<EventInfo> events = Cf.arrayList(eventInfoDbLoader.getEventInfo(
                Option.empty(), event, repetition, actionInfo.getActionSource()));

        if (!repetition.isEmpty() && !event.getRecurrenceId().isPresent()) {
            ListF<Long> recurrenceEventIds = eventRoutines.findMasterAndSingleEventIds(event.getId())
                    .filter(Function1B.equalsF(event.getId()).notF());

            if (recurrenceEventIds.isNotEmpty()) {
                events.addAll(eventInfoDbLoader.getEventInfosByIds(
                        Option.empty(), recurrenceEventIds, actionInfo.getActionSource()));
            }
        }
        return createEventInvitationOrCancelMails(actorId, sendingInfos.zipWith(Function.constF(events)), actionInfo);
    }

    public ListF<EventMessageParameters> createEventInvitationOrCancelMails(
            ActorId actorId, ListF<EventSendingInfo> sendingInfoList, ActionInfo actionInfo) {
        if (sendingInfoList.isEmpty()) return Cf.list();

        ListF<Long> eventIds = sendingInfoList.map(EventSendingInfo.getEventIdF()).stableUnique();
        ListF<Event> events = eventDao.findEventsByIds(eventIds);

        ListF<Event> recurrences = events.filter(e -> e.getRecurrenceId().isPresent());
        ListF<Event> masters = events.filterNot(e -> e.getRecurrenceId().isPresent());

        ListF<Event> mastersWithRecurrences = eventDao.findEventsByMainIds(masters.map(Event.getMainEventIdF()));

        ListF<EventInfo> eventInfos = eventInfoDbLoader.getEventInfosByEvents(Option.empty(),
                recurrences.plus(mastersWithRecurrences).stableUniqueBy(Event.getIdF()), actionInfo.getActionSource());

        MapF<Long, ListF<EventInfo>> masterWithRecurrencesById = masters.toMap(Event.getIdF(), Event.getMainEventIdF()
                .andThen(Cf2.f(eventInfos.groupBy(EventInfo::getMainEventId)::getOrThrow)));

        MapF<Long, ListF<EventInfo>> recurrenceById = recurrences.toMap(Event.getIdF(), Event.getIdF()
                .andThen(Cf2.f(eventInfos.toMapMappingToKey(EventInfo::getEventId)::getOrThrow))
                .andThen(Cf.List.consFrom1F()));

        Function<Long, ListF<EventInfo>> eventsByIdF = recurrenceById.plus(masterWithRecurrencesById)::getOrThrow;

        return createEventInvitationOrCancelMails(
                actorId, sendingInfoList.zipWith(EventSendingInfo.getEventIdF().andThen(eventsByIdF)), actionInfo);
    }

    public EventOnLayerChangeMessageParameters createEventOnLayerChangeMail(
            Option<Sender> senderO, LayerUser layerUser, String layerName, SettingsInfo recipientSettings,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition, EventInstanceParameters instance,
            MailType mailType, Option<EventChangesInfoForMails> changesInfo, ActionInfo actionInfo) {
        Email emailFrom = svcRoutines.getCalendarInfoEmail(updatedEvent.getEvent().getCreatorUid());
        Email recipientEmail = recipientSettings.getEmail();

        PassportUid recipientUid = recipientSettings.getUid();
        Language language = settingsRoutines.chooseMailLanguage(senderO.flatMapO(Sender::getUid), Option.of(recipientUid));
        DateTimeZone userTz = recipientSettings.getTz();

        NameI18n calName = new NameI18n(UStringLiteral.YA_CALENDAR, Option.empty());
        Email calFrom = svcRoutines.getCalendarInfoEmail(updatedEvent.getEvent().getCreatorUid());
        Sender calendarSender = new Sender(Option.empty(), Option.empty(), calName, Option.empty(), calFrom);

        CommonEventMessageParameters commonEventMessageParameters = new CommonEventMessageParameters(
                recipientSettings.getCommon().getLanguage(),
                new LocalDateTime(userTz),
                senderO.getOrElse(calendarSender), Recipient.of(recipientSettings), emailFrom,
                svcRoutines.getCalendarUrlBySettings(recipientSettings.getCommon()), false, MessageOverrides.EMPTY);

        EventMessageInfo eventMessageInfo = eventMessageInfoCreator.create(
                updatedEvent, updatedRepetition, instance, recipientEmail,
                Option.of(recipientUid), language, userTz);

        EventInviteeNamesI18n guestNames = getInviteeNames(changesInfo, updatedEvent);

        return new EventOnLayerChangeMessageParameters(
                commonEventMessageParameters, eventMessageInfo,
                layerUser, layerName, senderO.isPresent(), mailType, changesInfo, guestNames);
    }

    public void createAndSendEventInvitationMailIfOutlooker(
            PassportUid uid, Event event, ActionInfo actionInfo) {
        Option<EventSendingInfo> sendingInfo = prepareSendingInfoForOutlookerSubscriber(uid, event);

        sendEventMails(createEventInvitationOrCancelMails(ActorId.yaCalendar(), sendingInfo, actionInfo), actionInfo);
    }

    public void sendEventMails(ListF<? extends MessageParameters> eventMails, ActionInfo actionInfo) {
        soChecker.checkNoSpam(eventMails.cast(), actionInfo);

        mailSender.sendEmailsViaTask(eventMails, actionInfo);
    }

    private ListF<EventMessageParameters> createEventInvitationOrCancelMails(
            ActorId actorId, Tuple2List<EventSendingInfo, ListF<EventInfo>> sendingEvents, ActionInfo actionInfo) {
        final Sender sender = getSender(actorId);
        final Instant now = actionInfo.getNow();

        final MapF<String, String> apartmentRulesHtmlByExchangeName =
                toMapOfHtmlApartmentRulesByExchangeNames(sendingEvents);

        Map<Long, Participants> eventParticipants = sendingEvents.get2()
                .flatMap(events -> events)
                .toMap(EventInfo::getEventId, eventInfo -> eventInfo.getEventWithRelations().getParticipants());
        var eventIdToExtraId = saveAttendeesToExtra(eventParticipants);

        var usersSettings = settingsRoutines.getOrCreateSettingsByUidBatch(sendingEvents.filterMap(t -> t.get1().getUnivContact().getUid()));

        Map<EventInviteeKey, EventInviteeNames> inviteesCache = new HashMap<>();

        Function2<EventSendingInfo, ListF<EventInfo>, Option<EventMessageParameters>> createF =
                (sendingInfo, events) -> {
                    Validate.hasSize(1, events.map(EventInfo::getMainEventId).unique());
                    Option<SettingsInfo> recipientSettings = sendingInfo.getUnivContact().getUid().flatMapO(usersSettings::getO);

                    if (events.first().getEventWithRelations().isParkingOccupation()) {
                        return Option.empty();
                    } else if (events.first().getEventWithRelations().isApartmentOccupation()) {
                        return Option.of(createInvitationOrCancelMailXmlForApartment(
                                sender, events, sendingInfo, apartmentRulesHtmlByExchangeName, now));
                    } else if (events.first().getEventWithRelations().isHotelOccupation()) {
                        return Option.of(createInvitationOrCancelMailXmlForHotel(sender, events, sendingInfo, now));
                    } else if (events.first().getEventWithRelations().isCampusOccupation()) {
                        return Option.of(createInvitationOrCancelMailXmlForCampus(sender, events, sendingInfo, now));
                    } else {
                        return Option.of(createInvitationOrCancelMailXmlForEvent(sender, events, eventIdToExtraId, sendingInfo, recipientSettings, inviteesCache, now));
                    }
                };
        return sendingEvents.filterMap(createF.asFunction());
    }

    private Map<Long, Long> saveAttendeesToExtra(Map<Long, Participants> eventParticipants) {
        Map<Long, Long> eventIdToExtraId = new HashMap<>();
        eventParticipants.forEach((eventId, participants) -> {
            if (participants.isMeeting()) {
                ListF<IcsProperty> icsAttendees = IcsEventExporter
                        .exportAttendees(participants.getAllAttendees())
                        .map(PropertiesMeta.M::fromIcal4j);
                var extraId = messageExtraDao.save(new MessageExtra(new IcsCalendar(Cf.list(), icsAttendees)));
                eventIdToExtraId.put(eventId, extraId);
            }
        });
        return eventIdToExtraId;
    }

    private MapF<String, String> toMapOfHtmlApartmentRulesByExchangeNames(
            Tuple2List<EventSendingInfo, ListF<EventInfo>> sendingEvents) {
        return sendingEvents
                .filterBy1(EventSendingInfo.isMeetingMailTypeInF(MailType.EVENT_INVITATION, MailType.EVENT_UPDATE))
                .flatMap(Tuple2::get2)
                .map(EventInfo::getEventWithRelations)
                .filter(EventWithRelations::isApartmentOccupation)
                .flatMap(EventWithRelations::getResources)
                .filter(ResourceInfo.isTypeF(ResourceType.APARTMENT))
                .filterMap(ResourceInfo.exchangeNameF())
                .unique().toMapMappingToValue(resourceRoutines.getApartmentRulesHtmlF());
    }

    private EventMessageParameters createInvitationOrCancelMailXmlForEvent(
            Sender sender, ListF<EventInfo> events,
            Map<Long, Long> eventIdToExtraId,
            EventSendingInfo eventSendingInfo,
            Option<SettingsInfo> recipientSettings,
            Map<EventInviteeKey, EventInviteeNames> inviteesCache,
            Instant now
    ) {
        Validate.hasSize(1, events.map(EventInfo::getMainEventId).unique());
        EventInfo eventInfo = events.find(EventInfo::isMaster).getOrElse(events.first());

        EventWithRelations event = eventInfo.getEventWithRelations();
        RepetitionInstanceInfo repetitionInfo = eventInfo.getRepetitionInstanceInfo();

        UnivContact recipientContact = eventSendingInfo.getUnivContact();
        Option<String> privateTokenO = eventSendingInfo.getPrivateTokenO();

        Option<PassportUid> recipientUid = recipientContact.getUid();

        DateTimeZone userTz = recipientSettings.map(SettingsInfo::getTz).getOrElse(event.getTimezone());

        PassportUid someUid = recipientUid.orElse(sender.getUid()).getOrElse(event.getEvent().getCreatorUid());
        EventInstanceParameters instance = eventSendingInfo.getEventInstanceParameters();

        IcsMethod method = eventSendingInfo.getMeetingMailType().getIcsMethod().getOrElse(IcsMethod.PUBLISH);

        Option<Settings> settings = recipientSettings.map(SettingsInfo::getCommon);
        IcsExportParameters exportParams = new IcsExportParameters(IcsExportMode.EMAIL, method, true, now, settings, eventIdToExtraId);

        Option<IcsCalendar> calendar = Option.empty();
        if (!eventSendingInfo.isOmitIcs() && eventSendingInfo.getMeetingMailType() != MailType.RESOURCE_UNCHECKIN) {
            calendar = Option.of(!instance.getOccurrenceId().isPresent()
                    ? icsEventExporter.exportEvents(someUid, events, exportParams)
                    : icsEventExporter.exportEvent(someUid, event, repetitionInfo, Option.empty(), instance, exportParams));
        }

        Language language = settingsRoutines.chooseMailLanguageBySettings(sender.getUid(), settings);
        var changes = eventSendingInfo.getChangesInfo().map(ChangedEventInfoForMails::getChanges);
        var key = new EventInviteeKey(event.getId(), changes, language);
        var inviteeNames = inviteesCache.computeIfAbsent(key, k -> new EventInviteeNames(getInviteeNames(changes, event), language));

        InvitationOrCancelEmailData invitationOrCancelEmailData = new InvitationOrCancelEmailData(
                sender, Recipient.of(recipientContact.getMailAddress(), recipientSettings),
                userTz, privateTokenO, calendar,
                inviteeNames,
                eventSendingInfo.getChangesInfo(),
                eventSendingInfo.getDecision(),
                eventSendingInfo.getMeetingMailType(),
                instance, eventSendingInfo.getReorganizedEventId(), eventSendingInfo.getSubscriptionResource(),
                eventSendingInfo.getDestination());

        return eventInvitationOrCancelMessageXmlCreator.createXml(invitationOrCancelEmailData, event, repetitionInfo, settings);
    }

    private EventMessageParameters createInvitationOrCancelMailXmlForApartment(
            Sender sender, ListF<EventInfo> events, EventSendingInfo sendingInfo,
            MapF<String, String> apartmentRulesHtmlByExchangeName, Instant now) {
        return createInvitationOrCancelMailXmlForApartmentOrCampus(
                sender, events, sendingInfo,
                sendingInfo.getMeetingMailType().forResource(ResourceType.APARTMENT), e -> Option.empty(),
                e -> getApartmentRulesHtml(e, apartmentRulesHtmlByExchangeName, sendingInfo), now);
    }

    private EventMessageParameters createInvitationOrCancelMailXmlForHotel(
            Sender sender, ListF<EventInfo> events, EventSendingInfo sendingInfo, Instant now) {
        return createInvitationOrCancelMailXmlForApartmentOrCampus(
                sender, events, sendingInfo,
                sendingInfo.getMeetingMailType().forResource(ResourceType.HOTEL),
                e -> e.getResources().filterMap(r -> Option.when(
                        r.getResource().getType() == ResourceType.HOTEL, r.getResource().getDescription())).firstO(),
                e -> Option.empty(), now);
    }

    private EventMessageParameters createInvitationOrCancelMailXmlForCampus(
            Sender sender, ListF<EventInfo> events, EventSendingInfo sendingInfo, Instant now) {
        return createInvitationOrCancelMailXmlForApartmentOrCampus(
                sender, events, sendingInfo,
                sendingInfo.getMeetingMailType().forResource(ResourceType.CAMPUS),
                e -> Option.empty(), e -> Option.empty(), now);
    }

    private EventMessageParameters createInvitationOrCancelMailXmlForApartmentOrCampus(
            Sender sender, ListF<EventInfo> events, EventSendingInfo sendingInfo, MailType mailType,
            Function<EventWithRelations, Option<String>> addressF,
            Function<EventWithRelations, Option<String>> apartmentRulesF, Instant now) {
        EventInfo eventInfo = events.find(EventInfo::isMaster).getOrElse(events.first());
        EventWithRelations event = eventInfo.getEventWithRelations();

        Language language = Language.RUSSIAN;
        DateTimeZone tz = event.getTimezone();
        LocalDateTime nowLocal = new LocalDateTime(now, tz);

        Recipient recipient = Recipient.of(sendingInfo.getUnivContact(), Option.empty());
        Participants participants = event.getParticipants();

        Email emailFrom = svcRoutines.getCalendarInfoEmail(sender.getEmail());
        String calendarUrl = svcRoutines.getCalendarUrlForEmail(sender.getEmail());

        CommonEventMessageParameters commonParameters = new CommonEventMessageParameters(
                language, nowLocal, sender, recipient, emailFrom, calendarUrl, false, MessageOverrides.EMPTY);

        EventMessageInfo messageInfo = eventMessageInfoCreator.create(
                event, eventInfo.getRepetitionInstanceInfo(), sendingInfo.getEventInstanceParameters(),
                recipient.getEmail(), recipient.getUid(), language, tz);

        UnivContact organizer = UnivContact.formParticipantInfo(participants.getOrganizer());
        ListF<UnivContact> customers = participants.getAllUserAttendeesButNotOrganizer()
                .map(UnivContact::fromParticipantInfoWithHiddenName);

        return new ApartmentMessageParameters(
                commonParameters, messageInfo, mailType, organizer, customers,
                addressF.apply(event), apartmentRulesF.apply(event));
    }

    private Option<String> getApartmentRulesHtml(EventWithRelations event,
                                                 MapF<String, String> apartmentRulesHtmlByExchangeName, EventSendingInfo sendingInfo) {
        if (!EnumSet.of(MailType.EVENT_INVITATION, MailType.EVENT_UPDATE).contains(sendingInfo.getMeetingMailType())) {
            return Option.empty();
        }
        ListF<String> apartmentExchangeNames = event.getResources()
                .filter(ResourceInfo.isTypeF(ResourceType.APARTMENT))
                .filterMap(ResourceInfo.exchangeNameF());
        if (apartmentExchangeNames.isEmpty()) {
            throw new RuntimeException("Apartment with exchange name not found, can't get habitation rules from wiki");
        }

        return Option.of(apartmentRulesHtmlByExchangeName
                .filterKeys(apartmentExchangeNames.containsF())
                .entries()
                .firstO()
                .getOrThrow("Rules not found for apartments with exchange names=" + apartmentExchangeNames)
                .get2());
    }

    public Sender getSender(ActorId actorId) {
        final NameI18n name;
        final Optional<Gender> gender;
        final Email email;

        Option<String> login = Option.empty();

        if (actorId.isYaCalendar()) {
            name = new NameI18n(UStringLiteral.YA_CALENDAR, UStringLiteral.YA_CALENDAR_EN);
            gender = Optional.of(Gender.MALE);
            email = svcRoutines.getCalendarInfoEmail(passportAuthDomainsHolder.getPassportAuthDomains().toList().first());

        } else if (actorId.isUser()) {
            Settings settings = settingsRoutines.getSettingsByUid(actorId.getUid()).getCommon();

            email = settings.getYandexEmail();
            login = settings.getUserLogin();

            name = userManager.getYtUserNameByUid(actorId.getUid())
                    .orElseGet(() -> new NameI18n(Nick.getSender(settingsRoutines, actorId.getUid()), Option.empty()));

            gender = userManager.getYtUserGender(actorId.getUid());
        } else {
            Resource resource = resourceRoutines.loadById(actorId.getResourceId());

            email = ResourceRoutines.getResourceEmail(resource);
            gender = Optional.of(Gender.FEMALE);

            Option<NameI18n> simpleName = ResourceRoutines.getNameWithAlterNameI18n(resource);

            String exchangeOrIdName = resource.getExchangeName().getOrElse(" " + resource.getId());
            Option<String> ruName = simpleName.map(NameI18n.getNameF(Language.RUSSIAN));
            Option<String> enName = simpleName.map(NameI18n.getNameF(Language.ENGLISH)).orElse(ruName);
            name = new NameI18n(
                    UStringLiteral.MEETING_ROOM_RU + " " + ruName.getOrElse(exchangeOrIdName),
                    UStringLiteral.MEETING_ROOM_EN + " " + enName.getOrElse(exchangeOrIdName));
        }
        return new Sender(actorId.getUidO(), login, name, Option.x(gender), email);
    }

    private EventInviteeNamesI18n getInviteeNames(Option<EventChangesInfoForMails> changesInfo, EventWithRelations event) {
        ListF<UserParticipantInfo> nowUserAttendees = Cf.list();
        ListF<UserParticipantInfo> removedUserAttendees = Cf.list();
        ListF<UserParticipantInfo> newUserAttendees = Cf.list();

        if (event.getParticipants().isMeeting()) {
            nowUserAttendees = event.getParticipants()
                    .getUserParticipants().filter(ParticipantInfo.isOrganizerF().notF());
        }
        if (changesInfo.isPresent()) {
            removedUserAttendees = changesInfo.get().getRemovedGuests();
            newUserAttendees = nowUserAttendees.filter(UserParticipantInfo.getIdF()
                    .andThen(changesInfo.get().getNewGuestsParticipantIds().unique().containsF()));
        }
        return new EventInviteeNamesI18n(
                getParticipantNameI18ns(nowUserAttendees),
                getParticipantNameI18ns(newUserAttendees),
                getParticipantNameI18ns(removedUserAttendees));
    }

    public ListF<NameI18nWithOptionality> getParticipantNameI18ns(ListF<? extends ParticipantBasics> participants) {
        return participants.map((Function<ParticipantBasics, NameI18nWithOptionality>) a -> {
            Email email = a.getEmail();
            String ru = StringUtils.defaultIfEmpty(a.getName(), Emails.getUnicoded(email));
            Optional<String> enO = Optional.empty();
            if (passportAuthDomainsHolder.containsYandexTeamRu()) {
                val users = userManager.getYtUsersByEmail(email);
                if (!users.isEmpty()) {
                    val user = users.iterator().next();
                    ru = UserManager.getFullName(user, Language.RUSSIAN).orElse(ru);
                    enO = UserManager.getFullNameEn(user);
                }
            }
            return new NameI18nWithOptionality(new NameI18n(ru, Option.x(enO)), a.isOptional());
        });
    }

    /**
     * Temporary, until verstka will send participants instead new and rem invitations lists
     */
    private EventParticipantsChangesInfo participantsChanges(
            PassportUid clientUid, Participants participants,
            EventInvitationUpdateData invUpdData, boolean makePseudoLocalCopy) {
        SetF<Email> newEmails = invUpdData.getNewEmails().unique();
        SetF<Email> remEmails = invUpdData.getRemEmails().unique().minus(newEmails);

        ListF<Email> survivedEmails = participants.getParticipantsSafeWithInconsistent()
                .filter(ParticipantInfo.getEmailF().andThen(remEmails.containsF().notF()))
                .map(ParticipantInfo.getEmailF());
        newEmails = newEmails.minus(survivedEmails);

        return participantsChanges(clientUid, participants, new EventInvitationsData(
                invUpdData.getOrganizerEmail(),
                survivedEmails.plus(newEmails).map(EventInvitationData.consWithNoNameF())), makePseudoLocalCopy);
    }

    private EventParticipantsChangesInfo participantsChanges(
            PassportUid clientUid, Participants participants,
            EventInvitationsData eventInvitationsData, boolean makePseudoLocalCopy) {
        ListF<ParticipantData> participantDatas = Cf.arrayList();

        for (EventInvitationData invitation : eventInvitationsData.getNowInvitations()) {
            Email email = invitation.getEmail();
            Option<ParticipantInfo> participantByEmail = participants.getParticipantByEmailSafe(email);

            if (participantByEmail.isPresent()) {
                ParticipantInfo existing = participantByEmail.get();
                participantDatas.add(new ParticipantData(
                        invitation.getEmail(),
                        invitation.getName().getOrElse(existing.getName()),
                        existing.getDecision(), true, false, invitation.isOptional()));
            } else {
                String participantName = invitation.getName().getOrElse("");
                participantDatas.add(new ParticipantData(email, participantName, Decision.UNDECIDED, true, false, invitation.isOptional()));
            }
        }

        Function2B<ParticipantData, Email> emailIsF = Email.equalsIgnoreCaseF2().compose1(ParticipantData.getEmailF());
        ParticipantsData participantsData;

        if (participantDatas.isEmpty()) {
            participantsData = ParticipantsData.notMeeting();
        } else {
            final ParticipantData organizer;
            final Option<Email> organizerEmailO = eventInvitationsData.getOrganizerEmail();

            if (participants.isMeeting()
                    && !organizerEmailO.exists(participants.getOrganizer().getEmail().equalsIgnoreCaseF().notF())) {
                organizer = participants.getOrganizer().toData();
            } else {
                Email organizerEmail = organizerEmailO.getOrElse(() -> userManager.getEmailByUid(clientUid).get());
                organizer = new ParticipantData(organizerEmail, "", Decision.YES, true, true, false);
            }
            participantDatas = participantDatas.filter(emailIsF.notF().bind2(organizer.getEmail()));
            participantsData = ParticipantsData.merge(organizer, participantDatas);
        }

        return participantsChanges(Option.of(clientUid), participants, participantsData, makePseudoLocalCopy);
    }

    private EventParticipantsChangesInfo participantsChanges(
            Option<PassportUid> clientUid, Participants participants,
            ParticipantsData participantsData, boolean makePseudoLocalCopy) {
        Tuple2List<ParticipantId, ParticipantChangesInfo> updatedInvitations = Tuple2List.arrayList();
        Tuple2List<ParticipantId, ParticipantData> newInvitations = Tuple2List.arrayList();
        SetF<ParticipantId> toRemove = Cf.hashSet();

        ListF<ParticipantInfo> existingParticipants = participants.getParticipantsSafeWithInconsistent();
        toRemove.addAll(existingParticipants.map(ParticipantInfo.getIdF()));

        SetF<ParticipantId> processedParticipantIds = Cf.hashSet();

        MapF<Email, ParticipantId> participantIdByEmail = getParticipantIdsByEmails(
                participantsData.getParticipantEmailsSafe()).toMap();

        for (ParticipantData participant : participantsData.getParticipantsSafe()) {
            ParticipantId participantId = makePseudoLocalCopy && participant.isOrganizer()
                    ? ParticipantId.invitationIdForExternalUser(participant.getEmail())
                    : participantIdByEmail.getOrThrow(participant.getEmail());

            if (processedParticipantIds.containsTs(participantId)) {
                log.warn("Participant {} ({}) specified more than once, ignoring latter", participantId, participant);
                continue;
            }
            processedParticipantIds.add(participantId);

            Option<ParticipantInfo> existingParticipantO = participants.getByIdSafe(participantId);

            boolean useExternal = makePseudoLocalCopy && !clientUid.exists(participantId::isYandexUserWithUid); // CAL-7933

            if (useExternal && !existingParticipantO.isPresent()) {
                existingParticipantO = participants.getByIdSafe(
                        ParticipantId.invitationIdForExternalUser(participant.getEmail()));
            }

            if (existingParticipantO.isPresent()) {
                ParticipantInfo existingParticipant = existingParticipantO.get();

                if (wasChange(existingParticipant, participant) && !existingParticipant.isResource()) {
                    Decision decision = existingParticipant.getDecision() != Decision.NO
                            ? participant.getDecision()
                            : Decision.NO;

                    ParticipantChangesInfo invitationChangesInfo = new ParticipantChangesInfo(
                            (UserParticipantInfo) existingParticipant, participant.getName(),
                            decision, participant.isOrganizer(), participant.isOptional());

                    updatedInvitations.add(existingParticipant.getId(), invitationChangesInfo);
                }
                toRemove.removeTs(existingParticipant.getId());

            } else {
                ParticipantId newId = useExternal
                        ? ParticipantId.invitationIdForExternalUser(participant.getEmail())
                        : participantId;

                newInvitations.add(newId, participant);

                toRemove.removeTs(newId);
            }
        }
        if (participants.isMeeting()
                && toRemove.containsAllTs(participants.getAllAttendeesButNotOrganizer().map(ParticipantInfo.getIdF()).unique())
                && newInvitations.isEmpty()) {
            return EventParticipantsChangesInfo.changedToNotMeeting(participants.getParticipants());
        }
        if (!participants.isMeeting() && newInvitations.size() == 1 && newInvitations.get2().single().isOrganizer()) {
            return EventParticipantsChangesInfo.EMPTY;
        }
        ListF<ParticipantInfo> removedParticipants = existingParticipants
                .filter(ParticipantInfo.getIdF().andThen(toRemove.containsF()));

        return EventParticipantsChangesInfo.changes(newInvitations, updatedInvitations, removedParticipants);
    }

    private final Cache<Email, ParticipantId> participantIdByEmailCache =
            TlCache.asCache(EventInvitationManager.class.getName() + ".participantIdByEmail");

    public Tuple2List<Email, ParticipantId> getParticipantIdsByEmails(ListF<Email> emails) {
        return participantIdByEmailCache.getFromCacheSomeBatch(emails, this::doGetParticipantIdsByEmails);
    }

    public Tuple2List<Email, Option<UidOrResourceId>> getSubjectIdsByEmails(ListF<Email> emails) {
        return getParticipantIdsByEmails(emails).map2(ParticipantId.getSubjectIdF());
    }

    public Tuple2List<Email, Option<PassportUid>> getUidsByEmails(ListF<Email> emails) {
        return getParticipantIdsByEmails(emails).map2(ParticipantId::getUidIfYandexUser);
    }

    // for tests
    public void removeParticipantIdFromCacheByEmail(Email email) {
        participantIdByEmailCache.removeFromCache(email);
    }

    public ParticipantId getParticipantIdByEmail(Email email) {
        return getParticipantIdsByEmails(Cf.list(email)).single()._2;
    }

    /**
     * Should populate cache before, or use bulk version
     *
     * @see #getParticipantIdsByEmails(ListF)
     */
    public Function<Email, ParticipantId> getParticipantIdByEmailF() {
        return this::getParticipantIdByEmail;
    }

    public Tuple2List<Long, Option<ParticipantId>> findOrganizersByEventIds(ListF<Long> eventIds) {
        MapF<Long, ParticipantId> userByEventId = Cf2.flatBy2(eventUserDao.findOrganizersByEventIds(eventIds))
                .map2(ParticipantId::yandexUid).toMap();

        ListF<Long> restIds = eventIds.filterNot(userByEventId::containsKeyTs);

        MapF<Long, ParticipantId> externalByEventId = Cf2.flatBy2(eventInvitationDao.findOrganizersByEventIds(restIds))
                .map2(ParticipantId::invitationIdForExternalUser).toMap();

        return eventIds.zipWith(id -> userByEventId.getO(id)
                .orElse(() -> externalByEventId.getO(id)));
    }

    /**
     * Non-subjects are thrown
     */
    public ListF<UidOrResourceId> getSubjectsFromEmails(ListF<Email> emails) {
        return getParticipantIdsByEmails(emails).get2().flatMap(ParticipantId.getSubjectIdF());
    }

    private ListF<ParticipantId> doGetParticipantIdsByEmails(ListF<Email> normalizedEmails) {
        SetF<Email> unclassifiedEmails = normalizedEmails.unique();

        final MapF<Email, ParticipantId> resourceParticipantIdsByEmail =
                Cf2.flatBy2(resourceRoutines.parseIdsFromEmails(unclassifiedEmails.toList()))
                        .map2(ParticipantId.consResourceIdF()).toMap();

        unclassifiedEmails = unclassifiedEmails.minus(resourceParticipantIdsByEmail.keys());

        final MapF<Email, ParticipantId> yandexUidParticipantIdsByEmail =
                Cf2.flatBy2(userManager.getUidsByEmails(unclassifiedEmails.toList()))
                        .map2(ParticipantId.consPassportUidIdF()).toMap();

        return normalizedEmails.map(email -> {
            Option<ParticipantId> r;

            r = resourceParticipantIdsByEmail.getO(email);
            if (r.isPresent()) {
                return r.get();
            }

            r = yandexUidParticipantIdsByEmail.getO(email);
            if (r.isPresent()) {
                return r.get();
            }

            return ParticipantId.invitationIdForExternalUser(email);
        });
    }

    public EventSendingInfo createSendingInfoForEventCreator(PassportUid creatorUid, Event event) {
        Email email = settingsRoutines.getSettingsByUid(creatorUid).getCommon().getYandexEmail();
        String name = Nick.getSender(settingsRoutines, creatorUid);

        return new EventSendingInfo(
                new UnivContact(email, name, Option.of(creatorUid)),
                Option.empty(), event.getId(),
                MailType.EVENT_INVITATION, EventInstanceParameters.fromEvent(event),
                Decision.YES /* doesn't matter for EVENT_INFO_FOR_ORGANIZER_YT MailType */);
    }

    public final DynamicProperty<ListF<String>> apartmentSubscribers = new DynamicProperty<ListF<String>>(
            "apartmentSubscribers", Environment.isProductionOrPre()
            ? Cf.list("apartments@yandex-team.ru", "reception@yandex-team.ru")
            : Cf.list());

    public final DynamicProperty<ListF<String>> campusSubscribers =
            new DynamicProperty<>("campusSubscribers", Cf.list());

    public final DynamicProperty<ListF<String>> hotelSubscribers =
            new DynamicProperty<>("hotelSubscribers", Cf.list());

    public final DynamicProperty<ListF<RoomsSubscribers>> resourceSubscribers =
            new DynamicProperty<>("resourceSubscribers", Cf.list());


    public ListF<EventSendingInfo> createSendingInfoForResourceSubscribers(
            Participants participants, Event event, MailType mailType,
            EventInstanceParameters instanceParameters, Option<ChangedEventInfoForMails> changes) {
        ListF<ResourceType> specialTypes = Cf.list(ResourceType.APARTMENT, ResourceType.HOTEL, ResourceType.CAMPUS);

        ListF<ResourceInfo> resources = participants.getResourceParticipantsSafeWithInconsistent()
                .map(ResourceParticipantInfo::getResourceInfo)
                .plus(resourceRoutines.getResourceInfosByIds(
                        changes.flatMap(c -> c.getChanges().getParticipantsChanges().getNewResources())))
                .plus(changes.flatMap(c -> c.getChanges().getRemovedResources()))
                .sortedByDesc(r -> specialTypes.containsTs(r.getResource().getType()));

        Tuple2List<Email, ResourceInfo> subscriptions = Tuple2List.tuple2List(resources.flatMap(r -> {
            Option<DynamicProperty<ListF<String>>> dp =
                    r.getResource().getType() == ResourceType.APARTMENT ? Option.of(apartmentSubscribers)
                            : r.getResource().getType() == ResourceType.CAMPUS ? Option.of(campusSubscribers)
                            : r.getResource().getType() == ResourceType.HOTEL ? Option.of(hotelSubscribers)
                            : Option.empty();

            ListF<LoginOrEmail> subs = dp.map(d -> d.get().map(LoginOrEmail::parse))
                    .getOrElse(() -> resourceSubscribers.get().flatMap(
                            rs -> rs.getRooms().containsTs(r.getExchangeName()) ? rs.getSubscribers() : Cf.list()));

            return subs.filterMap(l -> l.getEmail().orElse(l.getLogin().flatMapO(login -> Option.x(userManager.getYtUserEmailByLogin(login)))))
                    .zipWith(s -> r);
        }));

        subscriptions = subscriptions.stableUniqueBy(Tuple2::get1).filterBy1Not(
                participants.getParticipantsSafeWithInconsistent().map(ParticipantInfo::getEmail).unique()::containsTs);

        Function<ResourceInfo, MailType> getMailType = r -> {
            MailType type = mailType;

            if (!specialTypes.containsTs(r.getResource().getType()) && mailType == MailType.EVENT_UPDATE) {
                if (changes.exists(c -> c.getChanges().getRemovedResources().exists(rr -> rr.getResourceId() == r.getResourceId()))) {
                    type = MailType.EVENT_CANCEL;
                }
                if (changes.exists(c -> c.getChanges().getParticipantsChanges().getNewResources().containsTs(r.getResourceId()))) {
                    type = MailType.EVENT_INVITATION;
                }
            }
            return type.forResource(r.getResource().getType());
        };

        return subscriptions.map(t -> new EventSendingInfo(
                new UnivContact(t.get1(), "", Option.empty()), Option.empty(), event.getId(), true,
                getMailType.apply(t.get2()), instanceParameters, changes, Decision.YES, Option.empty(),
                Option.of(t.get2()), MessageDestination.ANYWHERE));
    }

    /**
     * Create invitations at the meeting and returns sending info. Sending info is used in
     * IR.sendMails() method.
     */
    private InvitationCreature createEventInvitationAndSendingInfo(
            ActorId sender, EventAndRepetition event, InvitationData invitation,
            boolean forceNoDecision, EventInstanceParameters icsEventExportParameters, ActionInfo actionInfo) {
        ParticipantId participantId = invitation.participantId;
        ParticipantData participantData = invitation.participantData;

        ActionSource actionSource = actionInfo.getActionSource();
        Option<PassportUid> participantUid = participantId.getUidIfYandexUser();

        Option<EventUser> eventUserO = invitation.user;

        long eventId = event.getEventId();
        Decision decision;

        if (forceNoDecision) {
            decision = Decision.NO;
        } else if (sender.isUserWithSomeOfUids(participantUid) && actionSource.isWeb()) { // CAL-5456
            decision = Decision.YES;
        } else if (isAutoAccept(invitation)) {
            decision = Decision.YES;
        } else if (participantUid.exists(uid -> !uid.isYandexTeamRu()) && !participantData.isOrganizer()) { // CAL-7856
            decision = Decision.UNDECIDED;
        } else {
            decision = participantData.getDecision();
        }

        InvitationCreature result = InvitationCreature.empty();

        String privateToken = UidGen.createPrivateToken();
        if (participantId.isExternalUser()) {
            result = InvitationCreature.forExternal(eventInvitationRoutines.createEventInvitationData(
                    sender, event.getEventId(), participantData, decision, privateToken, actionInfo));

        } else if (participantId.isYandexUser()) {
            // for guest user, act from the name of the guest user:
            // * auto-accept can be done on his behalf;
            // * adding of non-confirmed events can be done on his behalf.
            // for resource subject, act from the name of sender (it is user, not cc)
            boolean isOptional = invitation.participantData.isOptional();
            EventUser eventUser = new EventUser();

            eventUser.setIsOrganizer(participantData.isOrganizer());
            eventUser.setIsAttendee(participantData.isAttendee());
            eventUser.setIsOptional(isOptional);
            eventUser.setPrivateToken(privateToken);
            eventUser.setDecision(decision);
            eventUser.setAvailability(eventUserO.map(EventUser::getAvailability).getOrElse(Availability.byDecision(decision)));

            Option<EventLayer> layer = Option.empty();
            ListF<Notification> notifications = Cf.list();

            boolean participantIsSender = sender.getUidO().isSome(participantId.getUid());

            if (decision != Decision.NO && !participantIsSender && invitation.layers.isEmpty()) {
                layer = Option.of(eventRoutines.createEventLayerData(
                        invitation.defaultLayerId.get(), participantUid.get(), event, actionInfo));

                layer.get().setIsPrimaryInst(participantData.isOrganizer());
                notifications = invitation.defaultNotifications;
            }

            result = InvitationCreature.forUser(layer, eventUserO.isPresent()
                    ? eventUserRoutines.eventUserUpdateData(eventUserO.get().getId(), eventUser, actionInfo)
                    : eventUserRoutines.eventUserSaveData(participantUid.get(), event, eventUser, notifications, actionInfo));

        } else if (participantId.isResource() && !invitation.resource.isPresent()) {
            result = InvitationCreature.forResource(eventRoutines.createEventResourceData(participantId.getResourceId(), event));
        }

        if (participantId.isAnyUser() && !sender.isUserWithSomeOfUids(participantId.getUidIfYandexUser())) {
            UnivContact univContact = new UnivContact(
                    participantData.getEmail(), participantData.getName(),
                    participantId.getUidIfYandexUser());

            return result.withSendingInfo(new EventSendingInfo(
                    univContact, Option.of(privateToken), eventId, MailType.EVENT_INVITATION,
                    icsEventExportParameters, decision));
        } else {
            return result;
        }
    }

    public EventInvitation createExternalUserInvitation(
            UidOrResourceId creator, long eventId, ParticipantData participantData, ActionInfo actionInfo) {
        return eventInvitationRoutines.createEventInvitation(
                creator, eventId, participantData, participantData.getDecision(),
                UidGen.createPrivateToken(), actionInfo);
    }

    public boolean isOrganizerOfEvent(PassportUid uid, long eventId) {
        return getParticipantsByEventId(eventId).isOrganizer(uid);
    }

    public Option<EventAttachedLayerId> removeAttendeeByEmail(long eventId, UserInfo client, Email attendeeEmail, ActionInfo actionInfo) {
        EventWithRelations event = eventDbManager.getEventWithRelationsById(eventId);
        val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(client, event);
        authorizer.ensureCanEditEventPermissions(client, eventAuthInfo, actionInfo.getActionSource());
        final ParticipantId participantId = getParticipantIdByEmail(attendeeEmail);
        return removeAttendeeByParticipantId(eventId, participantId, actionInfo);
    }

    public void updateParticipantsSetIsOrganizer(
            long eventId, ListF<ParticipantId> ids, boolean isOrganizer, ActionInfo actionInfo) {
        for (PassportUid uid : ids.filterMap(ParticipantId.getUidIfYandexUserF())) {
            eventUserDao.updateEventUserSetOrganizerByEventIdAndUid(eventId, uid, isOrganizer, actionInfo);
        }
        for (Email email : ids.filterMap(ParticipantId.getEmailIfExternalUserF())) {
            eventInvitationDao.updateEventInvitationSetOrganizerByEventIdAndEmail(eventId, email, isOrganizer);
        }
    }

    public EventsAttachedUser rejectMeetingsByYandexUser(
            ListF<Event> events, PassportUid uid, ActionInfo actionInfo) {
        ListF<Long> eventIds = events.map(Event::getId);

        ListF<EventUserUpdate> eventUsers = eventUserRoutines.updateEventUsersDecision(eventIds, uid, Decision.NO, actionInfo);

        ListF<Long> missingIds = eventIds.filter(eventUsers.map(eu -> eu.cur.get().getEventId()).unique().containsF().notF());

        if (missingIds.isNotEmpty()) {
            log.warn("Missing eventUsers, events: {}, user: {}", missingIds, uid);
        }

        ListF<Long> notAttendeeIds = eventUsers.filterMap(eu -> Option.when(!eu.cur.get().getIsAttendee(), eu.cur.get().getEventId()));

        ListF<EventLayer> eventLayers = eventDbManager.getEventLayersForEventsAndUser(notAttendeeIds, uid);

        eventRoutines.storeAndDeleteEventLayers(eventLayers, actionInfo);

        return EventsAttachedUser.detached(uid, events, eventUsers, eventLayers);
    }

    public Option<EventAttachedLayerId> removeAttendeeByParticipantId(
            long eventId, ParticipantId participantId, ActionInfo actionInfo) {
        return removeAttendeeByParticipantId(eventId, participantId, false, actionInfo);
    }

    public Option<EventAttachedLayerId> removeAttendeeButNotOrganizerByParticipantId(
            long eventId, ParticipantId participantId, ActionInfo actionInfo) {
        return removeAttendeeByParticipantId(eventId, participantId, true, actionInfo);
    }

    private Option<EventAttachedLayerId> removeAttendeeByParticipantId(
            long eventId, ParticipantId participantId, boolean keepOrganizer, ActionInfo actionInfo) {
        switch (participantId.getKind()) {
            case EXTERNAL_USER:
                deleteExternalUserEventParticipant(eventId, participantId.getInviteeEmailForExternalUser(), keepOrganizer);
                break;
            case RESOURCE:
                deleteResourceEventParticipant(eventId, participantId.getResourceId(), actionInfo);
                break;
            case YANDEX_USER:
                return deleteYandexUserEventParticipant(eventId, participantId.getUid(), keepOrganizer, actionInfo);
            default:
                throw new IllegalStateException("unknown participant id: " + participantId);
        }
        return Option.empty();
    }

    private void deleteExternalUserEventParticipant(long eventId, Email email, boolean keepOrganizer) {
        eventInvitationDao.deleteInvitationByEmailAndEventIdAndIsOrganizer(
                email, eventId, Option.when(keepOrganizer, false));
    }

    // https://jira.yandex-team.ru/browse/CAL-3353
    public void deleteYandexUserEventParticipant(long eventId, PassportUid uid) {
        deleteYandexUserEventParticipant(eventId, uid, false, ActionInfo.adminManager());
    }

    public boolean isDeletedYandexUserEventParticipant(long eventId, PassportUid uid) {
        Option<EventUser> eventUser = eventUserDao.findEventUserByEventIdAndUid(eventId, uid);

        return eventUser.exists(eu -> !eu.getIsAttendee() && !eu.getIsOrganizer() && !eu.getIsSubscriber())
                && eventUser.exists(eu -> eu.getDecision() == Decision.NO)
                && !eventDbManager.getEventLayerForEventAndUser(eventId, uid).isPresent();
    }

    private Option<EventAttachedLayerId> deleteYandexUserEventParticipant(
            long eventId, PassportUid uid, boolean keepOrganizer, ActionInfo actionInfo) {
        EventUser eventUser = eventUserDao.findEventUserByEventIdAndUid(eventId, uid).get();

        if (eventUser.getIsOrganizer() && keepOrganizer) {
            return Option.empty();
        }
        EventUser data = new EventUser();
        data.setId(eventUser.getId());
        data.setDecision(Decision.NO);
        data.setIsAttendee(false);
        data.setIsOptional(false);
        data.setIsOrganizer(false);
        data.setPrivateTokenNull();
        data.setModificationReqId(actionInfo.getRequestIdWithHostId());
        data.setModificationSource(actionInfo.getActionSource());
        data.setLastUpdate(actionInfo.getNow());

        eventUserDao.updateEventUser(data, actionInfo);

        return deleteExistingEventLayers(Cf.list(eventId), uid, actionInfo).getEvents().get2().singleO();
    }

    private EventsAttachedLayerIds deleteExistingEventLayers(
            ListF<Long> eventIds, PassportUid uid, ActionInfo actionInfo) {
        ListF<EventLayer> eventLayers = eventDbManager.getEventLayersForEventsAndUser(eventIds, uid);
        final MapF<Long, Long> layerIdByEventId = eventLayers.toMap(EventLayer.getEventIdF(), EventLayer.getLayerIdF());

        eventRoutines.storeAndDeleteEventLayers(eventLayers, actionInfo);

        return new EventsAttachedLayerIds(eventDao.findEventsByIds(layerIdByEventId.keys())
                .zipWith(e -> EventAttachedLayerId.detached(layerIdByEventId.getOrThrow(e.getId()))));
    }

    private void deleteResourceEventParticipant(long eventId, long resourceId, ActionInfo actionInfo) {
        EventResource eventResource = eventResourceDao.findEventResourceByEventIdAndResourceId(eventId, resourceId).get();
        eventRoutines.storeAndDeleteEventResource(eventResource, actionInfo);
    }

    private void fixOrganizerAsYandexUser(Event event,
                                          final PassportUid organizerUid, ActionInfo actionInfo) {
        int updated = eventUserDao.updateEventUserSetOrganizerByEventIdAndUid(event.getId(), organizerUid, true, actionInfo);
        if (updated == 0) {
            EventUser eventUser = new EventUser();
            eventUser.setDecision(Decision.YES);
            eventUser.setAvailability(Availability.MAYBE);
            eventUser.setIsOrganizer(true);
            eventUser.setIsAttendee(true);
            eventUser.setIsOptional(false);
            eventUserRoutines.saveEventUser(organizerUid, event.getId(),
                    eventUser, Cf.list(), actionInfo);
        }
    }

    private void fixMissingOrganizerForEvent(long eventId, ActionInfo actionInfo) {
        Event event = eventDao.findEventById(eventId);
        log.info("Fixing event {} by creatorUid {}", event.getId(), event.getCreatorUid());
        fixOrganizerAsYandexUser(event, event.getCreatorUid(), actionInfo);
    }

    private void makeRegularAttendee(ParticipantInfo participant, ActionInfo actionInfo) {
        if (participant.getKind() == ParticipantKind.YANDEX_USER) {
            int updatedEventUsers = eventUserDao.updateEventUserSetOrganizerByEventIdAndUid(
                    participant.getEventId(), participant.getUid().get(), false, actionInfo);
            if (updatedEventUsers != 1) {
                throw new CalendarException("wrong number of rows updated for yandex user, " +
                        "event_users: " + updatedEventUsers);
            }
        } else if (participant.getKind() == ParticipantKind.EXTERNAL_USER) {
            int updated = eventInvitationDao.updateEventInvitationSetOrganizerByInvitationId(
                    EventInvitationId.of(((ExternalUserParticipantInfo) participant).getInvitation()), false);
            if (updated != 1) {
                throw new CalendarException("wrong number of rows updated for ext user: " + updated);
            }
        } else if (participant.getKind() != ParticipantKind.RESOURCE) {
            throw new IllegalArgumentException();
        }
    }

    private void checkConsistent(long eventId) {
        Check.C.isTrue(!getParticipantsByEventId(eventId).isInconsistent());
    }

    public void fixParticipants(final long eventId, ActionInfo actionInfo) {
        Participants participants = getParticipantsByEventId(eventId);

        if (!participants.isInconsistent()) {
            return;
        }

        ListF<ParticipantInfo> organizers = participants.getParticipantsSafeWithInconsistent().filter(ParticipantInfo.isOrganizerF());

        if (organizers.size() == 1) {
            log.warn("{} has no attendees; making organizer attendee", LogMarker.EVENT_ID.format(eventId));
            Check.C.hasSize(0, participants.getAllAttendeesSafe());
            ParticipantInfo organizer = organizers.single();
            YandexUserParticipantInfo yandexUserOrganizer = (YandexUserParticipantInfo) organizer;
            EventUser eventUser = new EventUser();

            eventUser.setId(yandexUserOrganizer.getEventUser().getId());
            eventUser.setIsAttendee(true);
            eventUser.setIsOptional(yandexUserOrganizer.getEventUser().getIsOptional());
            eventUserDao.updateEventUser(eventUser, actionInfo);
        } else if (organizers.size() == 0) {
            log.warn("{} has no organizer", LogMarker.EVENT_ID.format(eventId));
            fixMissingOrganizerForEvent(eventId, actionInfo);
        } else {
            log.warn("{} has more than one organizer; dropping random", LogMarker.EVENT_ID.format(eventId));
            for (ParticipantInfo organizer : organizers.drop(1)) {
                makeRegularAttendee(organizer, actionInfo);
            }
        }

        Tuple2<ListF<ParticipantInfo>, ListF<ParticipantInfo>> partitionedEmails =
                participants.getParticipantsSafeWithInconsistent()
                        .partition(pi -> pi.getKind() == ParticipantKind.EXTERNAL_USER);
        SetF<Email> intersectingEmails = partitionedEmails._1.map(ParticipantInfo::getEmail).unique()
                .intersect(partitionedEmails._2.map(ParticipantInfo::getEmail).unique());

        if (intersectingEmails.isNotEmpty()) {  // CAL-8190
            log.warn("{} has same emails in Yandex users and External users; removing external users", LogMarker.EVENT_ID.format(eventId));
            intersectingEmails.forEach(email -> deleteExternalUserEventParticipant(eventId, email, true));
        }

        log.warn("Querying again for event {}, should return proper participants", eventId);
        checkConsistent(eventId);
    }

    public Tuple2List<Long, Participants> getParticipantsByEventIds(ListF<Long> eventIds) {
        return eventDbManager.getParticipantsByEventIds(eventIds)
                .toTuple2List(EventParticipants::getEventId, EventParticipants::getParticipants);
    }

    public Participants getParticipantsByEventId(long eventId) {
        return getParticipantsByEventIds(Cf.list(eventId)).single()._2;
    }

    public static ListF<ParticipantInfo> getInvitationsExceptSenderAndRejected(
            Participants participants, final Option<UidOrResourceId> senderIdO) {
        return participants.getParticipantsSafe()
                .filter(ParticipantInfo.isYandexUserWithSomeOfUidsF(senderIdO.filterMap(UidOrResourceId.getUidOF())).notF())
                .filter(ParticipantInfo.getDecisionF().andThenEquals(Decision.NO).notF());
    }

    public Option<EventSendingInfo> prepareSendingInfoForActorOrPrimaryLayerOwner(
            PassportUid actorUid, Participants participants, Event event, MailType mailType,
            EventInstanceParameters icsEventExportParameters, Option<ChangedEventInfoForMails> changes,
            ActionSource actionSource, boolean isExportedWithEws) {
        Validate.isTrue(mailType == MailType.EVENT_UPDATE || mailType == MailType.EVENT_CANCEL);
        Validate.isTrue((mailType == MailType.EVENT_UPDATE) == changes.isPresent());

        if (actionSource != ActionSource.WORKER && !actionSource.isWebOrApi()) {
            return Option.empty();
        }
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) {
            return Option.empty();
        }

        Function1B<ParticipantInfo> isOfUserF = ParticipantInfo.getUidF().andThen(Cf2.isSomeF(actorUid));

        Option<ParticipantInfo> actorParticipant = participants.getParticipantsSafeWithInconsistent().find(isOfUserF);

        ListF<Email> exchangeMails = isExportedWithEws ? participants.getParticipantEmailsSafe() : Cf.list();

        if (actorParticipant.isPresent()) {
            UnivContact univContact = UnivContact.formParticipantInfo(actorParticipant.get());
            Decision decision = actorParticipant.get().getDecision();

            if (exchangeMails.containsTs(univContact.getEmail())) {
                return Option.empty();
            }

            return Option.of(new EventSendingInfo(
                    univContact, Option.empty(), event.getId(), false,
                    mailType, icsEventExportParameters, changes, decision, MessageDestination.ANYWHERE));

        } else if (participants.isNotMeetingStrict()) {
            if (isExportedWithEws) {
                return Option.empty();
            }

            Option<Layer> primaryLayer = eventDbManager.getPrimaryLayer(event.getId());
            PassportUid uid = primaryLayer.map(Layer::getCreatorUid).getOrElse(actorUid);

            SettingsInfo settings = settingsRoutines.getSettingsByUid(uid);

            if (exchangeMails.containsTs(settings.getEmail()) || settings.isDismissed()
                    || !primaryLayer.map(Layer::getId).equals(layerRoutines.getDefaultLayerId(uid))) {
                return Option.empty();
            }

            Decision decision = participants.getSubscribers().find(isOfUserF).map(ParticipantInfo.getDecisionF())
                    .orElse(() -> eventUserRoutines.findEventUserDecision(uid, event.getId()))
                    .getOrElse(Decision.UNDECIDED);

            if (decision != Decision.NO && settings.isOutlookerOrEwser()
                    && eventDbManager.isEventAttached(uid, event.getId())) {
                return Option.of(new EventSendingInfo(
                        new UnivContact(settings.getEmail(), null, Option.of(uid)), Option.empty(), event.getId(),
                        false, mailType, icsEventExportParameters, changes, decision,
                        MessageDestination.ONLY_FOR_EXCHANGE));
            }
        }
        return Option.empty();
    }

    public Option<EventSendingInfo> prepareSendingInfoForOutlookerSubscriber(PassportUid uid, Event event) {
        if (passportAuthDomainsHolder.containsYandexTeamRu()) {
            SettingsInfo settings = settingsRoutines.getSettingsByUid(uid);
            Option<EventLayer> eventLayer = eventDbManager.getEventLayerForEventAndUser(event.getId(), uid);

            if (settings.isOutlookerOrEwser() && !settings.isDismissed()
                    && eventLayer.exists(el -> layerRoutines.getDefaultLayerId(uid).isSome(el.getLayerId()))) {
                return Option.of(new EventSendingInfo(
                        new UnivContact(settings.getEmail(), null, Option.of(uid)), Option.empty(), event.getId(),
                        false, MailType.EVENT_INVITATION, EventInstanceParameters.fromEvent(event), Option.empty(),
                        Decision.YES, MessageDestination.ONLY_FOR_EXCHANGE));
            }
        }
        return Option.empty();
    }

    public ListF<EventSendingInfo> prepareSendingInfoForParticipants(ListF<ParticipantInfo> guests, MailType mailType,
                                                                     EventInstanceParameters icsEventExportParameters, Option<ChangedEventInfoForMails> changesInfo,
                                                                     boolean isExportedWithEws) {
        ListF<EventSendingInfo> sendingInfoStore = Cf.arrayList();

        ListF<Email> participants =
                guests.filter(p -> p.isOrganizer() || p.isAttendee()).map(ParticipantInfo::getEmail);

        ListF<Email> exchangeMails = isExportedWithEws ? participants : Cf.list();

        for (UserParticipantInfo invitation : guests.filterByType(UserParticipantInfo.class)) {
            boolean omitIcs = invitation.isSubscriber() && !invitation.isOutlookerOrEwser();

            if (exchangeMails.containsTs(invitation.getEmail())
                    || invitation.getYtSettingsIfYtUser().exists(SettingsYt::getIsDismissed)) {
                continue;
            }

            EventSendingInfo eventSendingInfo = new EventSendingInfo(
                    UnivContact.formParticipantInfo(invitation),
                    invitation.getPrivateToken(),
                    invitation.getEventId(),
                    omitIcs, mailType, icsEventExportParameters, changesInfo,
                    invitation.getDecision(), MessageDestination.ANYWHERE);

            sendingInfoStore.add(eventSendingInfo);
        }
        return sendingInfoStore;
    }

    public Option<ParticipantInfo> getParticipantByEventIdAndEmail(long eventId, Email email) {
        return getParticipantsByEventId(eventId).getParticipantsSafe().find(
                ParticipantInfo.getEmailF().andThen(email.equalsIgnoreCaseF()));
    }

    public Option<ParticipantInfo> getParticipantByEventIdAndParticipantId(long eventId, ParticipantId participantId) {
        return getParticipantsByEventId(eventId).getByIdSafe(participantId);
    }

    public Option<UserParticipantInfo> getUserEventSharing(long eventId, ParticipantId participantId) {
        if (participantId.isYandexUser()) {
            return eventUserRoutines.findEventUser(eventId, participantId.getUid())
                    .map(eu -> new EventUserWithRelations(eu, settingsRoutines.getSettingsByUid(participantId.getUid())))
                    .filter(eu -> eu.isAttendee() || eu.getDecision() != Decision.NO)
                    .map(eu -> new YandexUserParticipantInfo(eu.getEventUser(), eu.getSettings()));
        }
        if (participantId.isExternalUser()) {
            return eventInvitationDao.findEventInvitationByEventIdAndEmail(
                    eventId, participantId.getInviteeEmailForExternalUser()).map(ExternalUserParticipantInfo::new);
        }
        return Option.empty();
    }

    // Checks whether invitation is not null, has proper target type,
    // and has corresponding target id set to a proper non-null value.
    // If 'uid' information is available, checks it, too.
    public static void checkInvUid(Option<PassportUid> invitationUid, Option<PassportUid> uid) {
        // Check authorization
        if (!uid.isPresent() || !invitationUid.isPresent()) {
            return;
        }
        if (!invitationUid.get().sameAs(uid.get())) {
            String msg =
                    "Non-corresponding uids: authorization uid: " + uid.get() + ", " +
                            "invitation uid: " + invitationUid.getOrNull();
            throw CommandRunException.createSituation(msg, Situation.NON_CORRESPONDING_UID);
        }
    }

    /**
     * At given layer, removes all FUTURE meetings:
     * <ul>
     * <li>that are (optionally) filtered by 'targetUid' event creator;</li>
     * <li>on behalf of 'uid' user (INCLUDING sent email messages).</li>
     * </ul>
     * for given layer.
     * Permission requirements:
     * <ul>
     * <li>{@link EventAction#VIEW}, {@link EventAction#EDIT_PERMISSIONS}
     * (keepEvents case) or {@link EventAction#DELETE} (not-keepEvents case)
     * for meetings in given layer<br/>(all of them can be obtained through
     * {@link LayerAction#VIEW_EVENT}, {@link LayerAction#EDIT_EVENT} and
     * {@link LayerAction#DELETE_EVENT});</li>
     * <li>{@link LayerAction#LIST}.</li>
     * </ul>
     * These permission groups are accessible for
     * {@link LayerActionClass#EDIT} and {@link LayerActionClass#ADMIN} users.
     *
     * @param actingUid  id of the user on whose behalf layer
     *                   delete/detach occurs + emails are sent
     * @param layerId    layer at which meetings are removed
     * @param targetUids (optional) users whose meetings should be deleted.
     *                   If unset, meetings for all users are affected.
     */
    public void removeMeetingsFromLayer(
            PassportUid actingUid, long layerId,
            ListF<PassportUid> targetUids, ActionInfo actionInfo) {
        // akirakozov: we don't decide, what to do with such events.
        // Probably we should delete all events, which organizers lost
        // all event_layers to that events.
    }

    // For given user, returns count of invitation that user created today [at the same UTC date]
    public long getInvitationCountTodayUtc(PassportUid uid) {
        // at index (uid + creation_ts), we want to use both fields,
        // so we use 'field BETWEEN...' instead of 'DATE(field)'
        return 0;
    }

    public EventsAttachedUser handleEventInvitationDecision(
            PassportUid uid, UserParticipantInfo participant,
            WebReplyData replyData, boolean applyToAllOccurrences, ActionInfo actionInfo) {
        ListF<Long> eventIds = Cf.list(participant.getEventId());
        if (applyToAllOccurrences) {
            ListF<Long> ids = eventRoutines.findMasterAndFutureSingleEventIds(participant.getEventId(), actionInfo);
            eventIds = eventIds.plus(findNotRejectedEventIds(participant.getId(), ids)).stableUnique();
        }
        InvitationProcessingMode invitationMode =
                !passportAuthDomainsHolder.containsYandexTeamRu() || participant.getId().isYandexUser()
                        ? InvitationProcessingMode.SAVE_ATTACH_SEND
                        : InvitationProcessingMode.SAVE_ONLY;

        if (replyData.getDecision().goes()) {
            return eventRoutines.acceptMeetings(
                    eventIds, uid, replyData, participant, actionInfo, invitationMode);

        } else {
            eventUserDao.updateEventUserReasonByEventIdsAndUserId(
                    eventIds, uid, replyData.getReason().getOrElse(""), actionInfo);

            return eventRoutines.rejectMeetings(eventIds, uid, replyData.getReason(), actionInfo, invitationMode);
        }
    }

    public ReplyMessageParameters createReplyMail(
            PassportUid senderUid, Option<PassportUid> recipientUid, Email orgEmail,
            Decision decision, Option<String> reason, IcsCalendar ics,
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo) {
        Validate.notNull(ics);

        Email emailFrom = userManager.getEmailByUid(senderUid)
                .orElseGet(() -> svcRoutines.getCalendarInfoEmail(PassportAuthDomain.byUid(senderUid)));

        Sender sender = getSender(ActorId.user(senderUid));
        Language language = settingsRoutines.chooseMailLanguage(Option.of(senderUid), recipientUid);

        Option<SettingsInfo> recipientSettings = recipientUid.map(settingsRoutines::getSettingsByUid);

        Recipient recipient = Recipient.of(new MailAddress(orgEmail), recipientSettings);

        DateTimeZone userTz = recipientSettings.map(SettingsInfo::getTz).getOrElse(event.getTimezone());

        CommonEventMessageParameters commonEventMessageParameters = new CommonEventMessageParameters(
                language, new LocalDateTime(userTz),
                sender, recipient, emailFrom,
                svcRoutines.getCalendarUrlForUid(recipientUid.getOrElse(senderUid)), false, MessageOverrides.EMPTY);

        EventMessageInfo eventMessageInfo = eventMessageInfoCreator.create(
                event, repetitionInfo, EventInstanceParameters.fromEvent(event.getEvent()),
                orgEmail, recipientUid, language, userTz);

        return new ReplyMessageParameters(
                commonEventMessageParameters, eventMessageInfo, decision, reason, Option.of(ics), MailType.EVENT_REPLY);
    }

    public ReplyMessageParameters createRejectMail(PassportUid uid, EventInfo event, Instant now) {
        return createRejectMail(uid, Cf.list(event), now);
    }

    public ReplyMessageParameters createRejectMail(
            PassportUid uid, ListF<EventInfo> events, Instant now) {
        return createReplyMail(uid, events, Decision.NO, now);
    }

    public ReplyMessageParameters createReplyMail(
            PassportUid uid, ListF<EventInfo> events, Decision decision, Instant now) {
        Validate.hasSize(1, events.map(EventInfo::getMainEventId).unique());
        EventInfo eventInfo = events.find(EventInfo::isMaster).getOrElse(events.first());

        EventWithRelations event = eventInfo.getEventWithRelations();
        RepetitionInstanceInfo repetitionInfo = eventInfo.getRepetitionInstanceInfo();

        Participants participants = event.getParticipants();
        Validate.isTrue(participants.isMeeting());

        Option<YandexUserParticipantInfo> attendee = participants
                .getAllAttendeesButNotOrganizer()
                .filterByType(YandexUserParticipantInfo.class)
                .find(YandexUserParticipantInfo.getUidSomeF().andThenEquals(uid));

        if (!attendee.isPresent()) {
            log.debug("attendeesButNotOrganizer: " + participants.getAllAttendeesButNotOrganizer());
            throw new CalendarException("Current user invitation is not found");
        }

        Email orgEmail = participants.getOrganizer().getEmail();

        IcsExportParameters exportParams = new IcsExportParameters(IcsExportMode.EMAIL, IcsMethod.REPLY, true, now);
        IcsCalendar ics = icsEventExporter.exportEvents(uid, events, exportParams);

        Option<PassportUid> recipientUid = participants.isExtMeeting()
                ? Option.empty()
                : event.getOrganizerIfMeetingOrElseCreator().getUidO();

        Option<String> reason = attendee.filterByType(YandexUserParticipantInfo.class).singleO()
                .map(YandexUserParticipantInfo.eventUserF())
                .filterMap(EventUser.getReasonF()).filter(StringUtils::isNotBlank);

        return createReplyMail(
                uid, recipientUid, orgEmail, decision, reason, ics, event, repetitionInfo
        );
    }

    private boolean wasChange(ParticipantInfo oldInfo, ParticipantData newData) {
        if (oldInfo.getDecision() != newData.getDecision()) {
            return true;
        }
        if (!oldInfo.getHiddenName().equals(newData.getName())) {
            return true;
        }
        return oldInfo.isOrganizer() != newData.isOrganizer() || oldInfo.isAttendee() != newData.isAttendee() || oldInfo.isOptional() != newData.isOptional();
    }

    public EventParticipantsChangesInfo participantsChanges(
            Option<PassportUid> clientUid, Participants participants, ParticipantsOrInvitationsData data) {
        if (data.isParticipantsData()) {
            return participantsChanges(
                    clientUid, participants, data.getParticipantsData(), data.isMakePseudoLocalCopy());
        } else if (data.isEventInvitationUpdateData()) {
            return participantsChanges(
                    clientUid.get(), participants, data.getEventInvitationUpdateData(), data.isMakePseudoLocalCopy());
        } else {
            return participantsChanges(
                    clientUid.get(), participants, data.getEventInvitationsData(), data.isMakePseudoLocalCopy());
        }
    }

    // user 'uid' invites 'emails' to 'target' having id 'targetId'
    // returns list of guests' names; if there is no name, returns e-mail
    public EventInvitationResults createNewEventInvitations(
            ActorId sender, Tuple2List<ParticipantId, ParticipantData> newInvitations,
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
            boolean isParkingOrApartmentOccupation, UpdateMode updateMode, ActionInfo actionInfo) {
        event.getEvent().validateIsLockedForUpdate();
        Validate.sameSize(newInvitations, newInvitations.get1().unique());

        if (newInvitations.isEmpty()) return EventInvitationResults.EMPTY;

        if (sender.isUser()) {
            UserInfo user = userManager.getUserInfo(sender.getUid());

            if (!actionInfo.getActionSource().isInvitationMailsFree()) {
                userManager.checkPublicUserKarma(KarmaCheckAction.invite(user.getUid(), event.getEvent(), actionInfo));
            }

            if (updateMode != UpdateMode.RECURRENCE_FOR_MASTER && updateMode != UpdateMode.MEETING_CREATION) {
                val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(Optional.of(user), event, Optional.empty(), Optional.empty());
                authorizer.ensureCanInviteToEvent(user, eventAuthInfo, actionInfo.getActionSource());
            }
        }

        ListF<ParticipantId> newParticipantIds = newInvitations.get1();
        ListF<PassportUid> uids = newParticipantIds.filterMap(ParticipantId::getUidIfYandexUser);

        MapF<PassportUid, SettingsInfo> settings = settingsRoutines.getOrCreateSettingsByUidBatch(uids);

        newInvitations = newInvitations.filterBy1Not(participantId ->
                participantId.getUidIfYandexUser().filterMap(settings::getO).exists(SettingsInfo::isDismissed));

        MapF<PassportUid, EventUser> eventUsers = eventUserDao.findEventUsersByEventIdsAndUids(
                Cf.list(event.getId()), uids).toMapMappingToKey(EventUser::getUid);

        MapF<PassportUid, ListF<EventLayer>> eventLayers = eventDbManager.getEventLayersForEventsAndUsers(
                Cf.list(event.getId()), uids).groupBy(EventLayer::getLCreatorUid);

        MapF<PassportUid, Long> defaultLayerIds = layerRoutines.getOrCreateDefaultLayers(settings.values());

        MapF<Long, ListF<Notification>> defaultNotifications =
                notificationDbManager.getLayerCreatorNotificationsByLayerIds(defaultLayerIds.values());

        MapF<Long, EventResource> eventResources = eventResourceDao.findEventResourcesByEventIdsAndResourceIds(
                        Cf.list(event.getId()), newParticipantIds.filterMap(ParticipantId::getResourceIdIfResource))
                .toMapMappingToKey(EventResource::getResourceId);

        MapF<Email, EventInvitation> eventInvitations = eventInvitationDao.findEventInvitationsByEventIdsAndEmails(
                        Cf.list(event.getId()), newParticipantIds.filterMap(ParticipantId::getEmailIfExternalUser))
                .toMapMappingToKey(EventInvitation::getEmail);

        EventAndRepetition eventAndRepetition = new EventAndRepetition(event.getEvent(), repetitionInfo);

        SetF<ParticipantId> oldParticipantIds = event.getParticipants()
                .getParticipantsSafe().map(ParticipantInfo.getIdF()).unique();

        ListF<InvitationCreature> creatures = newInvitations.filterMap(t -> {
            ParticipantData participantData = t._2;
            ParticipantId participantId = t._1;

            if (oldParticipantIds.containsTs(participantId)) {
                log.debug("Participant already invited to event: {}", participantId);
                return Option.empty();
            }
            Option<PassportUid> uid = participantId.getUidIfYandexUser();

            InvitationData invitation = new InvitationData(
                    participantId, participantData,
                    uid.flatMapO(eventUsers::getO), uid.map(defaultLayerIds::getOrThrow),
                    uid.map(defaultLayerIds::getOrThrow).map(defaultNotifications::getOrThrow).getOrElse(Cf.list()),
                    uid.map(settings::getOrThrow),
                    uid.map(u -> eventLayers.getOrElse(u, Cf.list())).getOrElse(Cf.list()),
                    participantId.getResourceIdIfResource().filterMap(eventResources::getO),
                    participantId.getEmailIfExternalUser().filterMap(eventInvitations::getO));

            return Option.of(createEventInvitationAndSendingInfo(
                    sender, eventAndRepetition, invitation, isParkingOrApartmentOccupation,
                    EventInstanceParameters.fromEvent(event.getEvent()), actionInfo));
        });

        ListF<EventUserData> eventUsersToSave =
                creatures.filterMap(c -> c.eventUser.flatMapO(eu -> eu.createOrUpdate.leftO()));
        ListF<EventUser> eventUsersToUpdate =
                creatures.filterMap(c -> c.eventUser.flatMapO(eu -> eu.createOrUpdate.rightO()));

        ListF<Long> eventUserIds = eventUserDao.saveEventUsersBatch(
                eventUsersToSave.map(EventUserData::getEventUser), actionInfo);

        eventUserDao.updateEventUsersBatch(eventUsersToUpdate, actionInfo);

        notificationDbManager.saveEventNotifications(eventUsersToSave.zip(eventUserIds).flatMap(t -> {
            DateTimeZone tz = settings.getOrThrow(t.get1().getEventUser().getUid()).getTz();

            return t.get1().getNotifications().map(n -> n.toEventNotification(t.get2()))
                    .map(en -> notificationRoutines.recalcNextSendTs(en, eventAndRepetition, tz, actionInfo));
        }));

        eventDbManager.saveEventLayers(creatures.filterMap(c -> c.eventLayer), actionInfo);
        eventDbManager.saveEventResources(creatures.filterMap(c -> c.eventResource), actionInfo);
        eventInvitationDao.saveEventInvitationsBatch(creatures.filterMap(c -> c.eventInvitation), actionInfo);

        ListF<Long> resourceIds = newInvitations.get1().filterMap(ParticipantId.getResourceIdIfResourceF());

        RejectedResources rejectedResources = resourceAccessRoutines.checkBookingPermissions(
                sender, event.getId(), resourceIds, actionInfo);

        return new EventInvitationResults(creatures.filterMap(c -> c.sendingInfo), rejectedResources);
    }

    public Option<UserParticipantInfo> getUserParticipantByPrivateToken(String privateToken) {
        Option<EventUser> eventUserO = eventUserDao.findEventUserByPrivateToken(privateToken);
        if (eventUserO.isPresent()) {
            SettingsInfo settings = settingsRoutines.getSettingsByUid(eventUserO.get().getUid());
            return Option.of(new YandexUserParticipantInfo(eventUserO.get(), settings));
        }

        Option<EventInvitation> invitationO = eventInvitationDao.findInvitationByPrivateToken(privateToken);
        if (invitationO.isPresent()) {
            return Option.of(new ExternalUserParticipantInfo(invitationO.get()));
        } else {
            return Option.empty();
        }
    }

    public void updateUserParticipantDecision(
            UserParticipantInfo participant, ReplyInfo replyInfo, ActionInfo actionInfo, EventInstanceStatusInfo status) {
        if (participant.getKind() == ParticipantKind.YANDEX_USER) {
            EventUser eventUser = new EventUser();
            eventUser.setDecision(replyInfo.getDecision());
            if (replyInfo.getReason().isPresent()) {
                eventUser.setReason(replyInfo.getReason());
            }
            eventUser.setModificationReqId(actionInfo.getRequestIdWithHostId());
            eventUser.setModificationSource(actionInfo.getActionSource());
            eventUser.setId(((YandexUserParticipantInfo) participant).getEventUser().getId());
            eventUserDao.updateEventUser(eventUser, actionInfo);

            if (!status.isFromCaldavAlreadyUpdated()) { // use old decision from caldav, but don't decrement version
                updateEventUserSequenceAndDtstamp(
                        participant.getUid().get(), participant.getEventId(),
                        replyInfo.getVersion(), actionInfo);
            }
        } else if (participant.getKind() == ParticipantKind.EXTERNAL_USER) {
            EventInvitation invitation = new EventInvitation();
            invitation.setDecision(replyInfo.getDecision());
            if (replyInfo.getReason().isPresent()) {
                invitation.setReason(replyInfo.getReason());
            }
            invitation.setDtstamp(replyInfo.getDtstamp());
            invitation.setSequence(replyInfo.getSequence());
            invitation.setModificationReqId(actionInfo.getRequestIdWithHostId());
            invitation.setModificationSource(actionInfo.getActionSource());
            invitation.setId(((ExternalUserParticipantInfo) participant).getInvitation().getId());
            eventInvitationDao.updateInvitation(invitation, actionInfo);
        } else {
            throw new IllegalStateException("Illegal participant kind: " + participant.getKind());
        }
    }

    public void updateUserDecision(
            long eventId, ActionInfo actionInfo, EventInstanceStatusInfo eventInstanceStatusInfo,
            SequenceAndDtStamp userVersion, ParticipantData participantData) {
        updateUserDecision(
                eventId, actionInfo, eventInstanceStatusInfo, userVersion, participantData, Option.empty());
    }

    public void updateUserDecision(
            long eventId, ActionInfo actionInfo, EventInstanceStatusInfo eventInstanceStatusInfo,
            SequenceAndDtStamp userVersion, ParticipantData participantData, Option<String> reason) {
        ParticipantId participantId = getParticipantIdByEmail(
                participantData.getEmail());
        Option<ParticipantInfo> participantO =
                getParticipantByEventIdAndParticipantId(eventId, participantId);
        if (participantO.isPresent() && participantO.get().getId().isAnyUser()) {
            Decision decision = participantData.getDecision();
            UserParticipantInfo userParticipant = (UserParticipantInfo) participantO.get();
            ReplyInfo replyInfo = new ReplyInfo(decision, reason, userVersion);
            updateUserParticipantDecision(userParticipant, replyInfo, actionInfo, eventInstanceStatusInfo);
        }
    }

    public void updateEventUserSequenceAndDtstamp(PassportUid uid, long eventId, SequenceAndDtStamp version, ActionInfo actionInfo) {
        eventUserDao.updateSequenceAndDtstamp(eventId, uid, version.getSequence(), version.getDtStamp(), actionInfo);
    }

    public ListF<Long> findNotRejectedFutureEventIds(
            ParticipantId participantId, long mainEventId, ActionInfo actionInfo) {
        return findNotRejectedEventIds(
                participantId, eventDao.findFutureOrMasterEventIdsByMainEventIds(Cf.list(mainEventId), actionInfo));
    }

    public ListF<Long> findNotRejectedFutureEventIds(
            ParticipantId participantId, ListF<Long> eventIds, ActionInfo actionInfo) {
        return findNotRejectedEventIds(participantId, eventDao.filterMasterAndFutureRecurrences(eventIds, actionInfo));
    }

    private ListF<Long> findNotRejectedEventIds(ParticipantId participantId, ListF<Long> eventIds) {
        if (participantId.isYandexUser()) {
            SqlCondition c = EventUserFields.EVENT_ID.column().inSet(eventIds)
                    .and(EventUserFields.UID.eq(participantId.getUid()))
                    .and(EventUserFields.DECISION.ne(Decision.NO));

            return eventUserDao.findEventUsers(c).map(EventUser.getEventIdF());
        } else if (participantId.isExternalUser()) {
            SqlCondition c = EventInvitationFields.EVENT_ID.column().inSet(eventIds)
                    .and(EventInvitationFields.EMAIL.eq(participantId.getEmailIfExternalUser().get().normalize()))
                    .and(EventInvitationFields.DECISION.ne(Decision.NO));

            return eventInvitationDao.findEventInvitations(c).map(EventInvitation.getEventIdF());
        } else {
            return eventResourceDao
                    .findEventResourcesByEventIdsAndResourceIds(eventIds, Cf.list(participantId.getResourceId()))
                    .map(EventResource.getEventIdF());
        }
    }

    public void sendDecisionFixingMailsIfNeeded(
            EventWithRelations event, RepetitionInstanceInfo repetitionInstanceInfo, ActionInfo actionInfo) {
        if (!event.isExportedWithEws() || !ewsAliveHandler.isEwsAlive()) {
            return;
        }

        ListF<PassportUid> nonEwsAttendees = event.getParticipants().getAllAttendeesButNotOrganizerSafe()
                .filterMap(ParticipantInfo::getSettingsIfYandexUser)
                .filterMap(s -> Option.when(!s.getYt().exists(SettingsYt::getIsEwser), s.getUid()));

        if (nonEwsAttendees.isEmpty()) {
            return;
        }

        Option<String> exchangeIdO = ewsExportRoutines.findIdInExchange(
                event.getOrganizerIfMeetingOrElseCreator(),
                event.getEvent(), event.getMainEvent());

        MapF<Email, Decision> exchangeDecisions = exchangeIdO.filterMap(ewsProxyWrapper::getEvent)
                .map(item -> ExchangeEventDataConverter.convertWithoutExchangeDataAndRealOrganizer(
                        item, resourceRoutines::selectResourceEmails))
                .map(EventData::getParticipantsDecisionsSafe).getOrElse(Cf::map);

        ListF<PassportUid> undecided = userManager.getUidsByEmails(
                        exchangeDecisions.filterValues(decision -> decision.equals(Decision.UNDECIDED)).keys())
                .get2().filterMap(uidO -> uidO);

        nonEwsAttendees = nonEwsAttendees.unique().intersect(undecided.unique()).toList();

        if (nonEwsAttendees.isEmpty()) {
            return;
        }

        EventInfo eventInfo = eventInfoDbLoader.getEventInfo(
                Option.empty(), event, repetitionInstanceInfo, actionInfo.getActionSource());

        MessageOverrides overrides = new MessageOverrides(MessageDestination.ONLY_FOR_EXCHANGE);

        ListF<MessageParameters> messages = nonEwsAttendees.filterMap(event::findUserEventUser)
                .filter(eventUser -> eventUser.getDecision() != Decision.UNDECIDED)
                .map(eventUser -> createReplyMail(
                        eventUser.getUid(), Cf.list(eventInfo), eventUser.getDecision(), actionInfo.getNow())
                        .withForceDelayedSending()
                        .withOverrides(overrides));

        sendEventMails(Cf.toList(messages), actionInfo);
    }

    public void restoreEventMissingInExchange(EventWithRelations event, PassportUid uid, ActionInfo actionInfo) {
        Option<YandexUserParticipantInfo> participantInfoO = event.getParticipants().getParticipantByUid(uid);

        if (!participantInfoO.isPresent()) {
            return;
        }

        EventSendingInfo sendingInfo = new EventSendingInfo(participantInfoO.get(),
                MailType.EVENT_INVITATION, EventInstanceParameters.fromEvent(event.getEvent()), Option.empty());

        sendingInfo = sendingInfo.withDestination(MessageDestination.ONLY_FOR_EXCHANGE);

        createAndSendEventInvitationOrCancelMails(ActorId.yaCalendar(), Cf.list(sendingInfo), actionInfo);
    }

    public void setEwsProxyWrapperForTests(EwsProxyWrapper newEwsProxyWrapper) {
        ewsProxyWrapper = newEwsProxyWrapper;
    }
}
