package ru.yandex.calendar.logic.event;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Daniel Brylev
 */
public class EventLoadLimits {
    private final Option<Instant> startsInOrAfterO;
    private final Option<Instant> startsInOrBeforeO;
    private final Option<Instant> endsInOrAfterO;
    private final Option<Integer> sizeO;

    private final Option<ListF<Long>> haveId;
    private final ListF<Long> exceptIds;

    private final Option<Instant> modifiedSince;

    private EventLoadLimits() {
        this(Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(), Cf.list(), Option.empty());
    }

    private EventLoadLimits(
            Option<Instant> startsInOrAfterO,
            Option<Instant> startsInOrBeforeO, Option<Instant> endsInOrAfterO,
            Option<Integer> sizeO,
            Option<ListF<Long>> haveId, ListF<Long> exceptIds,
            Option<Instant> modifiedSince)
    {
        this.startsInOrAfterO = startsInOrAfterO;
        this.startsInOrBeforeO = startsInOrBeforeO;
        this.endsInOrAfterO = endsInOrAfterO;
        this.sizeO = sizeO;
        this.haveId = haveId;
        this.exceptIds = exceptIds;
        this.modifiedSince = modifiedSince;
    }

    public static EventLoadLimits cons() {
        return new EventLoadLimits();
    }

    public static EventLoadLimits noLimits() {
        return cons();
    }

    public static EventLoadLimits limitResultSize(int size) {
        return cons().withResultSize(size);
    }

    public static EventLoadLimits intersectsInterval(Instant start, Instant end) {
        return intersectsInterval(new InfiniteInterval(start, Option.of(end)));
    }

    public static EventLoadLimits intersectsIntervalOptionalEnd(Instant start, Option<Instant> optionEnd) {
        return intersectsInterval(new InfiniteInterval(start, optionEnd));
    }

    public static EventLoadLimits intersectsInterval(InstantInterval interval) {
        return intersectsInterval(new InfiniteInterval(interval.getStart(), Option.of(interval.getEnd())));
    }

    public static EventLoadLimits intersectsInterval(InfiniteInterval interval) {
        EventLoadLimits limits = cons().withEndsInOrAfter(interval.getStart());
        return interval.getEnd().isPresent() ? limits.withStartsInOrBefore(interval.getEnd().get()) : limits;
    }

    public static EventLoadLimits startsInInterval(InstantInterval interval) {
        return startsInInterval(new InfiniteInterval(interval.getStart(), Option.of(interval.getEnd())));
    }

    public static EventLoadLimits startsInInterval(InfiniteInterval interval) {
        EventLoadLimits limits = cons().withStartsInOrAfter(interval.getStart());
        return interval.getEnd().isPresent() ? limits.withStartsInOrBefore(interval.getEnd().get()) : limits;
    }

    public boolean hasTimeLimits() {
        return startsInOrAfterO.isPresent() || startsInOrBeforeO.isPresent() || endsInOrAfterO.isPresent();
    }

    public boolean hasResultSizeLimit() {
        return getResultSizeLimit().isPresent();
    }

    public EventLoadLimits withStartsInOrAfter(Instant instant) {
        return new EventLoadLimits(
                Option.of(instant), startsInOrBeforeO, endsInOrAfterO, sizeO, haveId, exceptIds, modifiedSince);
    }

    public EventLoadLimits withStartsInOrBefore(Instant instant) {
        return new EventLoadLimits(
                startsInOrAfterO, Option.of(instant), endsInOrAfterO, sizeO, haveId, exceptIds, modifiedSince);
    }

    public EventLoadLimits withEndsInOrAfter(Instant instant) {
        return new EventLoadLimits(
                startsInOrAfterO, startsInOrBeforeO, Option.of(instant), sizeO, haveId, exceptIds, modifiedSince);
    }

    public EventLoadLimits withResultSize(int size) {
        return new EventLoadLimits(
                startsInOrAfterO, startsInOrBeforeO, endsInOrAfterO, Option.of(size), haveId, exceptIds, modifiedSince);
    }

    public EventLoadLimits withHaveId(ListF<Long> eventIds) {
        return new EventLoadLimits(
                startsInOrAfterO, startsInOrBeforeO, endsInOrAfterO, sizeO, Option.of(eventIds), exceptIds, modifiedSince);
    }

    public EventLoadLimits withExcludeIds(ListF<Long> eventIds) {
        return new EventLoadLimits(
                startsInOrAfterO, startsInOrBeforeO, endsInOrAfterO, sizeO, haveId, eventIds, modifiedSince);
    }

    public EventLoadLimits withModifiedSince(Option<Instant> since) {
        return new EventLoadLimits(
                startsInOrAfterO, startsInOrBeforeO, endsInOrAfterO, sizeO, haveId, exceptIds, since);
    }

    public Option<Instant> getStartsInOrAfterLimit() {
        return startsInOrAfterO;
    }

    public Option<Instant> getStartsInOrBeforeLimit() {
        return startsInOrBeforeO;
    }

    public Option<Instant> getEndsInOrAfterLimit() {
        return endsInOrAfterO;
    }

    public Option<Integer> getResultSizeLimit() {
        return sizeO;
    }

    public Option<ListF<Long>> getHaveId() {
        return haveId;
    }

    public ListF<Long> getExceptIds() {
        return exceptIds;
    }

    public Option<Instant> getModifiedSince() {
        return modifiedSince;
    }
}
