package ru.yandex.calendar.logic.event;

import org.joda.time.Duration;
import org.joda.time.Hours;
import org.joda.time.Instant;
import org.joda.time.Minutes;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.repetition.EventInstanceInterval;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.notification.SmsNotificationManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.time.InstantInterval;

/**
 * CAL-6876
 *
 * @author dbrylev
 */
public class EventMoveSmsHandler {
    private static final Duration MIN_BEFORE_EVENT_START = Minutes.minutes(-5).toStandardDuration();
    private static final Duration MAX_BEFORE_EVENT_START = Hours.hours(23).toStandardDuration();

    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private SmsNotificationManager smsNotificationManager;
    @Autowired
    private EventDao eventDao;

    public InstantInterval suitableInterval(Instant now) {
        return new InstantInterval(now.plus(MIN_BEFORE_EVENT_START), now.plus(MAX_BEFORE_EVENT_START));
    }

    public void handleEventUpdate(
            PassportUid actorUid, ListF<EventInstanceInterval> instancesBeforeUpdate,
            ListF<MainEvent> updatedAndNewMainEvents, ActionInfo actionInfo)
    {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) return;

        InstantInterval suitableInterval = suitableInterval(actionInfo.getNow());

        MapF<Long, Event> eventBeforeUpdateById = instancesBeforeUpdate.toMap(
                EventInstanceInterval.getEventIdF(), EventInstanceInterval.getEventF());

        MapF<Long, RepetitionInstanceInfo> updatedRepetitionInfosByEventId =
                repetitionRoutines.getRepetitionInstanceInfosAmongEvents(
                        eventDao.findEventsByMainIds(updatedAndNewMainEvents.map(MainEvent::getId)));

        Tuple2List<Long, InstantInterval> initialInstances = instancesBeforeUpdate
                .toTuple2List(EventInstanceInterval.getEventIdF(), EventInstanceInterval.getIntervalF());

        Tuple2List<Long, InstantInterval> updatedInstances = Cf2.flatBy2(updatedRepetitionInfosByEventId.entries()
                .map2(r -> RepetitionUtils.getInstanceIntervalStartingAfter(r, suitableInterval.getStart())));

        Option<Tuple3<Long, Long, Instant>> moved = initialInstances.sortedBy(t -> t.get2().getStart())
                .zip(updatedInstances.sortedBy(t -> t.get2().getStart()))
                .filterNot(tt -> tt.get1().get2().getStart().equals(tt.get2().get2().getStart()))
                .map(tt -> Tuple3.tuple(tt.get1().get1(), tt.get2().get1(), tt.get2().get2().getStart())).firstO();

        if (!moved.isPresent()) return;

        ListF<Long> eventIds = Cf.list(moved.get().get1(), moved.get().get2()).stableUnique();

        ListF<EventUserWithRelations> eventUsers =
                eventUserRoutines.filterNotDismissedNotDeclinedEventUsersHaveSmsNotification(
                        eventUserRoutines.findEventUsersWithRelationsByEventIds(eventIds)
                                .filterNot(eu -> eu.getEventUser().getUid().equalsTs(actorUid)));

        if (!moved.get().get1().equals(moved.get().get2())) {
            SetF<PassportUid> targetUsers = eventUsers.filterMap(eu -> Option.when(
                    eu.getEventUser().getEventId() == moved.get().get2(), eu.getEventUser().getUid())).unique();

            eventUsers = eventUsers.filter(eu -> eu.getEventUser().getEventId() == moved.get().get1()
                    && targetUsers.containsTs(eu.getEventUser().getUid()));
        }

        smsNotificationManager.submitEventMovingSmss(
                actorUid, eventBeforeUpdateById.getOrThrow(moved.get().get1()),
                updatedAndNewMainEvents.first(), moved.get().get3(),
                eventUsers.map(EventUserWithRelations::getSettings), actionInfo);
    }

    public void setSmsNotificationManagerForTest(SmsNotificationManager manager) {
        this.smsNotificationManager = manager;
    }
}
