package ru.yandex.calendar.logic.event;

import java.util.List;
import java.util.Optional;

import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.jdom.Element;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.Period;
import org.joda.time.chrono.ISOChronology;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.CalendarUtils;
import ru.yandex.calendar.boot.EwsAliveHandler;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.ExchangeData;
import ru.yandex.calendar.frontend.ews.WindowsTimeZones;
import ru.yandex.calendar.frontend.ews.exp.EwsExportRoutines;
import ru.yandex.calendar.frontend.ews.exp.OccurrenceId;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxy;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.ReadableErrorMessageException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.frontend.web.cmd.run.ui.event.EventXmlizer;
import ru.yandex.calendar.frontend.webNew.dto.out.LocalDateTimeInterval;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.DeletedEventResource;
import ru.yandex.calendar.logic.beans.generated.DeletedEventUser;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventHelper;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventLayerHelper;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventResourceFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerFields;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.beans.generated.LayerUserFields;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.contact.ContactRoutines;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.archive.ArchiveManager;
import ru.yandex.calendar.logic.event.archive.DeletedEventDao;
import ru.yandex.calendar.logic.event.attachment.EventAttachmentChangesInfo;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.avail.AvailabilityOverlap;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.meeting.CancelMeetingHandler;
import ru.yandex.calendar.logic.event.meeting.EventInvitationResults;
import ru.yandex.calendar.logic.event.meeting.ExchangeMails;
import ru.yandex.calendar.logic.event.meeting.UpdateMeetingHandler;
import ru.yandex.calendar.logic.event.meeting.UpdateMode;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.model.EventUserUpdate;
import ru.yandex.calendar.logic.event.model.ParticipantsOrInvitationsData;
import ru.yandex.calendar.logic.event.model.WebReplyData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetition;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetitionAndPerms;
import ru.yandex.calendar.logic.event.repetition.EventIndentInterval;
import ru.yandex.calendar.logic.event.repetition.EventIndentIntervalAndPerms;
import ru.yandex.calendar.logic.event.repetition.EventInstanceInterval;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.calendar.logic.event.repetition.RecurrenceTimeInfo;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionConfirmationManager;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.event.web.ModificationScope;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.logic.layer.LayerDbManager;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.layer.LayerUserDao;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.log.change.changes.EventFieldsChangesJson;
import ru.yandex.calendar.logic.notification.Channel;
import ru.yandex.calendar.logic.notification.ControlDataNotification;
import ru.yandex.calendar.logic.notification.ControlDataNotification.CalendarNotificationType;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.RejectedResources;
import ru.yandex.calendar.logic.resource.ResourceAccessRoutines;
import ru.yandex.calendar.logic.resource.ResourceInaccessibility;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.resource.SpecialResources;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sending.EventMarkupXmlCreator.EventXmlCreationInfo;
import ru.yandex.calendar.logic.sending.EventSendingInfo;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sending.param.EventOnLayerChangeMessageParameters;
import ru.yandex.calendar.logic.sending.param.ReplyMessageParameters;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.EventParticipantsChangesInfo;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.EventActionClass;
import ru.yandex.calendar.logic.sharing.perm.EventInfoForPermsCheck;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.logic.sharing.perm.PermXmlizer;
import ru.yandex.calendar.logic.suggest.SuggestUtils;
import ru.yandex.calendar.logic.svc.DbSvcRoutines;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.telemost.TelemostManager;
import ru.yandex.calendar.logic.user.Group;
import ru.yandex.calendar.logic.user.KarmaCheckAction;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.EventAction;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.data.DataProvider;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateInterval;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.dates.TimesInUnit;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.calendar.util.xml.TagReplacement;
import ru.yandex.calendar.util.xml.TagReplacement.Action;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.resultSet.TwoColumnRowMapper;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.time.InstantInterval;

import static java.util.Objects.requireNonNull;
import static java.util.function.Function.identity;

@Slf4j
public class EventRoutines extends EventOrLayerRoutines<Event, EventLayer> {
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private RepetitionConfirmationManager repetitionConfirmationManager;
    @Autowired
    private NotificationRoutines notificationRoutines;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private DbSvcRoutines dbSvcRoutines;
    @Autowired
    private EwsExportRoutines ewsExportRoutines;
    @Autowired
    private EwsProxy ewsProxy;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private ArchiveManager archiveManager;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private DeletedEventDao deletedEventDao;
    @Autowired
    private EventInvitationDao eventInvitationDao;
    @Autowired
    private LayerDao layerDao;
    @Autowired
    private LayerUserDao layerUserDao;
    @Autowired
    private ResourceRoutines resourceRoutines;

    @Autowired
    private EventChangesFinder eventChangesFinder;

    @Autowired
    private CancelMeetingHandler cancelMeetingHandler;
    @Autowired
    private UpdateMeetingHandler updateMeetingHandler;

    @Autowired
    private ControlDataNotification controlDataNotification;
    @Autowired
    private LayerDbManager layerDbManager;

    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private ContactRoutines contactRoutines;
    @Autowired
    private EventsOnLayerChangeHandler eventsOnLayerChangeHandler;
    @Autowired
    private EventDeletionSmsHandler eventDeletionSmsHandler;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private ResourceAccessRoutines resourceAccessRoutines;
    @Autowired
    private EventsLogger eventsLogger;
    @Autowired
    private TelemostManager telemostManager;
    @Autowired
    private EwsAliveHandler ewsAliveHandler;

    private final DynamicProperty<Boolean> ewsAutoFixDecisions = new DynamicProperty<>("ewsAutoFixDecisions", true);

    public static final String DEFAULT_NAME = UStringLiteral.NO_NAME;

    public static final TagReplacement[] TAGS = {TagReplacement.A, TagReplacement.B };

    public long createMainEvent(DateTimeZone eventTz, ActionInfo actionInfo) {
        return createMainEvent(CalendarUtils.generateExternalId(), eventTz, actionInfo);
    }

    public long createMainEvent(String externalId, DateTimeZone eventTz, ActionInfo actionInfo) {
        return createMainEvent(externalId, eventTz, false, actionInfo);
    }

    public long createMainEvent(
            String externalId, DateTimeZone eventTz, boolean isExportedWithEws, ActionInfo actionInfo)
    {
        log.info("Creating main event {ext-id={}, tz={}}", externalId, eventTz);
        return mainEventDao.saveMainEvent(new ExternalId(externalId), eventTz, actionInfo.getNow(), isExportedWithEws);
    }

    public long createMainEvent(PassportUid uid, EventData eventData, ActionInfo actionInfo) {
        return createMainEvent(UidOrResourceId.user(uid), eventData, actionInfo);
    }

    private static final List<UnindexedFieldURIType> FIELDS = List.of(
        UnindexedFieldURIType.CALENDAR_UID,
        UnindexedFieldURIType.CALENDAR_MY_RESPONSE_TYPE,
        UnindexedFieldURIType.CALENDAR_IS_MEETING
    );

    private boolean isExportedWithEws(UidOrResourceId subjectId, EventData eventData, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return false;
        val organizerEmail = eventData.getOrganizerEmail().toOptional();

        val login = organizerEmail.flatMap(userManager::getYtLoginByEmail)
            .or(() -> subjectId.getUidO()
                .toOptional()
                .flatMap(uid -> userManager.getYtUserLoginByUid(uid).toOptional())
            );

        boolean isEwsExportedLoginAndNotParkingOrApartment = !getIsParkingOrApartmentOccupation(eventData)
                && login.flatMap(userManager::getYtUserUidByLogin).stream().anyMatch(settingsRoutines::getIsEwser);

        if (!isEwsExportedLoginAndNotParkingOrApartment) {
            return false;
        }
        if (ewsExportRoutines.ensureActionSource(actionInfo.getActionSource())) {
            Option<Layer> layer = eventData.getLayerId().map(layerRoutines::getLayerById);
            Option<Long> defaultId = subjectId.getUidO().flatMapO(layerRoutines::getDefaultLayerId);

            return layer.forAll(l -> l.getType().isAbsence() || defaultId.isSome(l.getId()));
        }

        return login.map(userManager::getLdEmailByLogin)
            .stream()
            .flatMap(organizer -> ewsProxy.findMasterAndSingleEvents(organizer, eventData.getInterval(), FIELDS, Cf.list()).stream())
            .filter(item -> eventData.getExternalId().isSome(item.getUID()))
            .anyMatch(EwsUtils::isOrganizedOrNonMeeting);
    }

    public long createMainEvent(UidOrResourceId subjectId, EventData eventData, ActionInfo actionInfo) {
        val externalId = eventData.getExternalId().getOrElse(CalendarUtils.generateExternalId());

        val eventTz = eventData.getTimeZone();

        val isExportedWithEws = isExportedWithEws(subjectId, eventData, actionInfo);

        log.info("Creating main event {ext-id={}, tz={}, ews-export={}}", externalId, eventTz, isExportedWithEws);

        return mainEventDao.saveMainEvent(new ExternalId(externalId), eventTz, actionInfo.getNow(), isExportedWithEws);
    }

    public static boolean getShowIsAllDay(EventTime e, DateTimeZone tz) {
        return e.isAllDay() || DateTimeFormatter.isAtLeastOneDayDiff(e.getStartTs(), e.getEndTs(), tz);
    }

    public static String getDescriptionHtmlForVerstka(String description, Option<String> linkSignKey) {
        return TagReplacement.processText(description, Action.CONVERT_XML, linkSignKey, TAGS);
    }

    public static String getLocationHtmlForVerstka(String location, Option<String> linkSignKey) {
        return TagReplacement.processText(location, Action.CONVERT_XML, linkSignKey, TAGS);
    }

    public Element getElement(
            Option<UserInfo> userO, EventInstanceInfo ei,
            Option<Layer> layerO, Option<EventActions> actionsO,
            Option<LocalDateTimeInterval> showBoundsO, DateTimeZone tz,
            Option<String> linkSignKey, ActionSource actionSource)
    {
        EventInfoForPermsCheck permsInfo = ei.getInfoForPermsCheck();
        RepetitionInstanceInfo repInstInfo = ei.getRepInstInfo();
        Option<ListF<EventAttachment>> attachments = ei.getAttachmentsO();

        Option<Long> layerId = layerO.map(Layer::getId);
        Option<EventUser> eventUserO = ei.getEventUserWithNotifications().map(EventUserWithNotifications.getEventUserF());
        boolean mayView = ei.getMayView();

        // Event tags available both for VIEW_BASICS and VIEW
        Element eEvent = new Element("event");
        CalendarXmlizer.appendElm(eEvent, "id", ei.getEventId());
        PermXmlizer.appendPermElm(eEvent, EventAction.VIEW, mayView);

        Function1V<MapField<?>> appendEventFieldO = field -> {
            Option<?> value = ei.getEventFieldValueO(field);
            if (value.isPresent()) {
                CalendarXmlizer.appendElm(eEvent, CamelWords.parse(field.getName()).toXmlName(), value.get(), tz);
            }
        };
        EventInterval i = ei.getEventInterval();
        CalendarXmlizer.appendDtfElm(eEvent, "instance-start-ts", i.getInstanceStart(), DateTimeZone.UTC);
        CalendarXmlizer.appendDtfElm(eEvent, EventXmlizer.START_TS_NAME, i.getStart(), tz);
        CalendarXmlizer.appendDtfElm(eEvent, EventXmlizer.END_TS_NAME, i.getEnd(), tz);
        CalendarXmlizer.appendElm(eEvent, "is-all-day", ei.getIndent().isAllDay());

        CalendarXmlizer.appendElm(eEvent, "is-private", permsInfo.isPrivate());

        if (showBoundsO.isPresent()) {
            CalendarXmlizer.appendElm(eEvent, "show-start-ts", DateTimeFormatter.formatLocalDateTimeForMachines(showBoundsO.get().getStart()));
            CalendarXmlizer.appendElm(eEvent, "show-end-ts", DateTimeFormatter.formatLocalDateTimeForMachines(showBoundsO.get().getEnd()));
        } // if bounded event piece interval is set
        CalendarXmlizer.appendElm(eEvent, "show-is-all-day", getShowIsAllDay(ei.getIndent().getTime(), tz), null);

        Option<PassportUid> uidO = userO.map(UserInfo.getUidF());

        appendEventFieldO.apply(EventFields.SEQUENCE);
        appendEventFieldO.apply(EventFields.LAST_UPDATE_TS);

        if (mayView) {
            CalendarXmlizer.appendElm(eEvent, "type", permsInfo.getType().toDbValue());
            appendEventFieldO.apply(EventFields.CREATOR_UID);
            appendEventFieldO.apply(EventFields.SID);

            appendEventFieldO.apply(EventFields.NAME);
            if (ei.getEventFieldValueO(EventFields.LOCATION).isPresent()) {
                eEvent.addContent(TagReplacement.processText(
                        "location", ei.getEvent().getLocation(), Action.CONVERT_XML, linkSignKey, TAGS));
            }
            if (ei.getEventFieldValueO(EventFields.DESCRIPTION).isPresent()) {
                eEvent.addContent(TagReplacement.processText(
                        "description", ei.getEvent().getDescription(), Action.CONVERT_XML, linkSignKey, TAGS));
            }
            String url = ei.getEventFieldValueO(EventFields.URL).getOrNull();

            if (StringUtils.isNotEmpty(url)) { CalendarXmlizer.appendElm(eEvent, "url", url); }

            if (!repInstInfo.isEmpty()) {
                CalendarXmlizer.appendElm(eEvent, "reg-rep");
            }
            if (ei.getIndent().getRecurrenceId().isPresent()) { CalendarXmlizer.appendElm(eEvent, "recurrence"); }

            CalendarXmlizer.appendElm(eEvent, "is-recurrence", ei.getIndent().getRecurrenceId().isPresent());

            Option<String> extId = ei.getMainEventFieldValueO(MainEventFields.EXTERNAL_ID);
            if (extId.isPresent()) CalendarXmlizer.appendElm(eEvent, "external-id", extId.get());

            CalendarXmlizer.appendElm(eEvent, "is-meeting", permsInfo.getOrganizer().isPresent());

            if (userO.isPresent()) {
                EventActions actions = actionsO.getOrElse(
                        () -> getEventActions(userO.get(), permsInfo, eventUserO, layerO, actionSource));

                CalendarXmlizer.appendElm(eEvent, "should-accept", actions.canAccept()); // XXX: remove when verstka will stop to use it
                CalendarXmlizer.appendElm(eEvent, "can-accept", actions.canAccept());
                CalendarXmlizer.appendElm(eEvent, "can-reject", actions.canReject());
                CalendarXmlizer.appendElm(eEvent, "can-delete", actions.canDelete());
                CalendarXmlizer.appendElm(eEvent, "can-attach", actions.canAttach());
                CalendarXmlizer.appendElm(eEvent, "can-detach", actions.canDetach());
                CalendarXmlizer.appendElm(eEvent, "can-edit", actions.canEdit());
                CalendarXmlizer.appendElm(eEvent, "can-invite", actions.canInvite());
                PermXmlizer.appendPermElm(eEvent, EventAction.SPLIT,
                        authorizer.canSplitEvent(userO.get(), permsInfo, actionSource));
            } else {
                CalendarXmlizer.appendElm(eEvent, "should-accept", false); // XXX: remove when verstka will stop to use it
                PermXmlizer.appendPermElm(eEvent, EventAction.SPLIT,
                        authorizer.canSplitEvent(permsInfo, actionSource));
            }

            if (attachments.isPresent() && attachments.get().isNotEmpty()) {
                Element eColl = CalendarXmlizer.appendElm(eEvent, "attachments");
                eColl.setAttribute("type", "array");
                for (EventAttachment attachment : attachments.get()) {
                    Element element = CalendarXmlizer.appendElm(eColl, "attachment");
                    CalendarXmlizer.appendElm(element, "id", attachment.getUrl());
                    CalendarXmlizer.appendElm(element, "filename", attachment.getFilename());
                }
            }
        } // if canView
        // Event-layer tags
        Element eLayer = new Element("layer"); // tag must exist
        if (layerId.isPresent()) {
            CalendarXmlizer.appendElm(eLayer, "layer-id", layerId.get());
        } else if (uidO.isPresent()) {
            // XXX: hack to fix balloon of event on invite page
            CalendarXmlizer.appendElm(eLayer, "layer-id", "0");
        }
        if (mayView) {
            Option<LayerInfoForPermsCheck> layer = permsInfo.getPrimaryLayer();
            // XXX: return flags "delete" or "detach" instead of is-primary-inst
            CalendarXmlizer.appendElm(eLayer, "is-primary-inst", layer.isPresent()
                    && layerId.containsTs(layer.get().getId()));
            boolean goesToMeeting = eventUserO.isPresent() && eventUserO.get().getDecision().goes();
            CalendarXmlizer.appendElm(eLayer, "goes-to-meeting", goesToMeeting);

        }
        if (userO.isPresent() && layerO.isPresent()) {
            CalendarXmlizer.appendElm(eLayer, "user-created", userO.get().getUid().sameAs(layerO.get().getCreatorUid()));
        }
        eEvent.addContent(eLayer);

        if (eventUserO.isPresent()) {
            Element eUser = new Element("user"); // tag must exist if 'eu' exists
            eventUserO.get().appendXmlTo(eUser, tz,
                EventUserFields.AVAILABILITY, EventUserFields.COMPLETION,
                EventUserFields.PRIORITY, EventUserFields.DECISION
            );
            eEvent.addContent(eUser);
        }
        return eEvent;
    }

    public EventActions getEventActions(
            UserInfo user, EventWithRelations event, Option<Layer> layer, ActionSource actionSource)
    {
        Option<EventUser> eventUser = event.findUserEventUser(user.getUid());
        EventInfoForPermsCheck infoForPermsCheck = authorizer.loadEventInfoForPermsCheck(user, event);

        return getEventActions(user, infoForPermsCheck, eventUser, layer, actionSource);
    }

    public EventActions getEventActions(
            UserInfo user, EventInfoForPermsCheck event,
            Option<EventUser> eventUserO, Option<Layer> layerO, ActionSource actionSource)
    {
        val uid = user.getUid();

        if (layerO.exists(l -> l.getType().isAbsence() || l.getType() == LayerType.FEED)) {
            return EventActions.empty();
        }
        // dbrylev: can not accept or reject our invitation on someones layer
        val isLayerOwner = layerO.isPresent() && layerO.get().getCreatorUid().sameAs(uid);
        val isLayerOwnerOrNoLayer = isLayerOwner || !layerO.isPresent();

        val isPrimary = event.getPrimaryLayer().exists(pl -> layerO.exists(l -> l.getId() == pl.getId()));

        val isAttendee = eventUserO.isPresent() && eventUserO.get().getIsAttendee();
        val isOrganizer = eventUserO.isPresent() && eventUserO.get().getIsOrganizer();
        val rejectedNow = eventUserO.isPresent() && eventUserO.get().getDecision() == Decision.NO;
        val notAcceptedNow = eventUserO.isPresent() && eventUserO.get().getDecision() != Decision.YES;

        val canReply = isAttendee
                || !uid.isYandexTeamRu() && eventUserO.exists(eu -> eu.getDecision() == Decision.UNDECIDED);

        val canAccept = isLayerOwnerOrNoLayer && canReply && notAcceptedNow;
        val canReject = isLayerOwnerOrNoLayer && canReply && !isOrganizer && !rejectedNow;
        val eventPermissions = authorizer.getEventPermissions(user, event, actionSource);
        val canDelete = eventPermissions.contains(EventAction.DELETE);
        val canView = eventPermissions.contains(EventAction.VIEW);

        val alreadyAttached = event.isUserHasEventLayer() && !rejectedNow;

        val canAttach = !alreadyAttached && canView;

        val canPerformDetach = layerO.stream()
            .map(LayerInfoForPermsCheck::fromLayer)
            .anyMatch(layerPermInfo -> {
                return authorizer.canPerformLayerAction(user, layerPermInfo, Optional.of(event.getUserLayersSharing()),
                    LayerAction.DETACH_EVENT, actionSource);
            });

        val canDetachFromOwnLayer = isLayerOwner && !isAttendee && !rejectedNow
                && (!canDelete || !isPrimary && !isOrganizer);
        val canDetachFromSharedLayer = !isLayerOwner && !canDelete && canPerformDetach;

        val canDetach = canDetachFromOwnLayer || canDetachFromSharedLayer;

        val canEdit = eventPermissions.contains(EventAction.EDIT);
        val canInvite = eventPermissions.contains(EventAction.INVITE);

        val canMove = layerO.isPresent() && layerO.get().getType() == LayerType.USER && canPerformDetach;

        val canChangeOrganizer = eventPermissions.contains(EventAction.CHANGE_ORGANIZER);

        return new EventActions(
                canAccept, canReject, canDelete, canAttach, canDetach,
                canEdit, canInvite, canMove, canChangeOrganizer);
    }

    public Element getElementWithNotification(
            EventInstanceInfo eventInstanceInfo, Option<UserInfo> userO,
            Option<Layer> layerO, Option<EventActions> actions,
            LocalDateTime showStart, LocalDateTime showEnd,
            DateTimeZone tz, Option<String> linkSignKey, ActionSource actionSource)
    {
        Option<EventUserWithNotifications> pairO = eventInstanceInfo.getEventUserWithNotifications();
        Element eventElement = getElement(userO, eventInstanceInfo, layerO, actions,
                Option.of(new LocalDateTimeInterval(showStart, showEnd)), tz, linkSignKey, actionSource);
        // XXX ssytnik: these fields are user-specific. Since eventInstanceInfo can be obtained
        // for 'uid' only, it is not correct to check EventAction.VIEW. Please review and remove.
        if (pairO.isPresent() && eventInstanceInfo.getMayView()) {
            // Prepare some CmdGetEvents-specific notification data
            ListF<Channel> channels = pairO.get().getNotifications().getNotifications().map(Notification::getChannel);
            Element notificationElement = new Element("notification");
            CalendarXmlizer.appendElm(notificationElement, "is-email-notify", channels.containsTs(Channel.EMAIL));
            CalendarXmlizer.appendElm(notificationElement, "is-sms-notify", channels.containsTs(Channel.SMS));
            CalendarXmlizer.appendElm(notificationElement, "is-ical-display-notify", channels.containsTs(Channel.DISPLAY));
            // Append to existing user element
            eventElement.getChild("user").addContent(notificationElement);
        }
        return eventElement;
    }

    public Option<MainEvent> getMainEventBySubjectAndExternalId(UidOrResourceId subjectId, String externalId) {
        return getMainEventBySubjectsAndExternalId(Cf.list(subjectId.toParticipantId()), externalId);
    }

    public Long getMainEventIdBySubjectsIdAndExternalIdOrCreateNew(
            UidOrResourceId subjectId, ListF<Email> participantEmails,
            EventData eventData, ActionInfo actionInfo)
    {
        Option<MainEvent> mainEvent =
                getMainEventBySubjectIdAndParticipantEmailsAndExternalId(
                        subjectId, participantEmails, eventData.getExternalId().get());
        if (mainEvent.isPresent()) {
            return mainEvent.get().getId();
        } else {
            return createMainEvent(subjectId, eventData, actionInfo);
        }
    }

    public Option<MainEvent> getMainEventBySubjectsAndExternalId(ListF<ParticipantId> participantIds, String externalId) {
        ListF<Long> resourceIds = participantIds.filterMap(ParticipantId.getResourceIdIfResourceF());
        ListF<PassportUid> uids = participantIds.filterMap(ParticipantId.getUidIfYandexUserF());
        ListF<Email> invitationEmails = participantIds.filterMap(ParticipantId.getEmailIfExternalUserF());

        ListF<Long> eventIds = mainEventDao.findEventIdsByExternalId(new ExternalId(externalId));

        ListF<MainEvent> found = mainEventDao.findMainEventsByEventUsers(uids, eventIds);
        if (found.size() > 1) {
            log.warn("More than one main event found by event_user for {}, {}", uids, externalId);
        }
        if (found.isNotEmpty()) return found.firstO();

        found = mainEventDao.findMainEventsByLayerIds(layerUserDao.findLayerIdsByLayerUserUids(uids), eventIds);
        if (found.size() > 1) {
            log.warn("More than one main event found by layer_user for {}, {}",uids, externalId);
        }
        if (found.isNotEmpty()) return found.firstO();

        found = mainEventDao.findMainEventsByInvitationEmails(invitationEmails, eventIds);
        if (found.size() > 1) {
            log.warn("More than one main event found by invitation emails for {}, {}", invitationEmails, externalId);
        }
        if (found.isNotEmpty()) return found.firstO();

        found = mainEventDao.findMainEventsByResourceIds(resourceIds, eventIds);
        if (found.size() > 1) {
            log.warn("More than one main event found by resources for {}, {}", resourceIds, externalId);
        }
        return found.firstO();
    }

    public Option<MainEvent> getMainEventBySubjectIdAndParticipantEmailsAndExternalId(
            UidOrResourceId subjectId, ListF<Email> emails, String externalId)
    {
        Option<MainEvent> mainEvent = getMainEventBySubjectAndExternalId(subjectId, externalId);
        if (!mainEvent.isPresent()) {
            ListF<ParticipantId> participantIds = eventInvitationManager.getParticipantIdsByEmails(emails).get2();
            mainEvent = getMainEventBySubjectsAndExternalId(participantIds, externalId);
        }
        return mainEvent;
    }

    public Option<Event> getMainInstOfEvent(Event e) {
        return findMasterEventByMainId(e.getMainEventId());
    }

    public Option<Event> findMasterEventByMainId(long mainEventId) {
        ListF<Event> masterEventInstances = eventDao.findMasterEventByMainId(mainEventId);
        if (masterEventInstances.size() > 1) {
            log.warn("Inconsistent database: more than one master instance for main event id {}", mainEventId);
        }
        return masterEventInstances.firstO();
    }

    public Option<Event> findMasterEventBySubjectIdAndExternalId(UidOrResourceId subject, String extId) {
        return findMasterEventBySubjectIdAndExternalId(subject, extId, false);
    }

    public Option<Event> findMasterEventBySubjectIdAndExternalId(
            UidOrResourceId subject, String extId, boolean lockForUpdate)
    {
        Option<MainEvent> mainEvent = getMainEventBySubjectAndExternalId(subject, extId);
        if (mainEvent.isPresent()) {
            ListF<Event> masterEventInstances = eventDao.findMasterEventByMainId(mainEvent.get().getId(), lockForUpdate);
            if (masterEventInstances.size() > 1) {
                log.warn("Inconsistent database: more than one master instance " +
                        "for main event id {}", mainEvent.get().getId());
            }
            return masterEventInstances.firstO();
        } else {
            return Option.empty();
        }
    }

    public Option<Event> getRecurrenceEventInstanceBySubjectIdAndExternalId(
            UidOrResourceId subject, String extId, Instant recurrenceId)
    {
        Option<MainEvent> mainEvent = getMainEventBySubjectAndExternalId(subject, extId);
        if (mainEvent.isPresent()) {
            ListF<Event> recurrenceEventInstances = eventDao.findRecurrenceEventByMainId(mainEvent.get().getId(), recurrenceId);
            if (recurrenceEventInstances.size() > 1) {
                log.warn("Inconsistent database: more than one recurrence instance " +
                        "for main event id and recurrence id {}, {}", mainEvent.get().getId(), recurrenceId);
            }
            return recurrenceEventInstances.firstO();
        } else {
            return Option.empty();
        }
    }

    public boolean findDeletedMasterExistsByExternalIdAndParticipantEmails(String externalId, ListF<Email> emails) {
        ListF<UidOrResourceId> participants = eventInvitationManager.getSubjectsFromEmails(emails);

        ListF<PassportUid> uids = participants.filter(UidOrResourceId.isUserF()).map(UidOrResourceId.getUidF());
        ListF<Long> resourceIds = participants.filter(UidOrResourceId.isUserF().notF()).map(UidOrResourceId.getResourceIdF());

        return deletedEventDao.findDeletedMasterExistsByExternalIdAndEventUserUids(externalId, uids)
                || deletedEventDao.findDeletedMasterExistsByExternalIdAndResourceIds(externalId, resourceIds);
    }

    public boolean findDeletedMasterExistsByUid(String externalId, PassportUid uid) {
        return deletedEventDao.findDeletedMasterExistsByExternalIdAndEventUserUids(externalId, Cf.list(uid));
    }

    /**
     * Common part of event creating routine
     * @param creatorId event creator user id
     * @param sid for svc. events, service id. Otherwise, null.
     * @return event id and flag whether it was created or (if found
     * with same external id and sid for user) existed before
     */
    public CreateInfo createEventCommon(
            UidOrResourceId creatorId, long mainEventId, PassportSid sid, EventType eventType,
            EventData eventData, boolean forceNoDecision,
            NotificationsData.Create notificationsData, ActionInfo actionInfo)
    {
        val event = eventData.getEvent();
        val repetition = eventData.getRepetition();
        val eventUserData = eventData.getEventUserData();
        val rdates = eventData.getRdates();
        val layerIdO = eventData.getLayerId();
        // Check if event with given external id already exists
        val isRecurrenceIdSet = event.isFieldSet(EventFields.RECURRENCE_ID);

        validateEventStartTsEndTs(event);

        if (isRecurrenceIdSet) {
            Option<Instant> recurIdO = event.getRecurrenceId();

            RecurrenceIdOrMainEvent r;
            if (recurIdO.isPresent()) {
                r = RecurrenceIdOrMainEvent.recurrenceId(recurIdO.get());
            } else {
                r = RecurrenceIdOrMainEvent.mainEvent();
            }

            final Option<Event> existingEventO =
                findEventByMainEventIdAndRecurrence(mainEventId, r);

            if (existingEventO.isPresent()) {
                long eFoundId = existingEventO.get().getId();
                final String msg =
                    "event exists for main event id = " + mainEventId + ". Found event id = " + eFoundId;
                throw CommandRunException.createSituation(msg, Situation.EVENT_ALREADY_EXISTS);
            }
        }

        Option<Long> repetitionIdO;
        if (RepetitionRoutines.isNoneRepetition(repetition)) {
            repetitionIdO = Option.empty();
        } else {
            repetitionIdO = Option.of(repetitionRoutines.createRepetition(repetition));
        }

        final Event newEvent = event.copy();
        newEvent.unsetField(EventFields.ID);
        newEvent.setSid(sid);
        newEvent.setRepetitionId(repetitionIdO);
        newEvent.setFieldValueDefault(EventFields.SEQUENCE, 0);

        newEvent.setFieldValueDefault(EventFields.NAME, DEFAULT_NAME);

        if (passportAuthDomainsHolder.containsYandexTeamRu()) { // CAL-5097
            newEvent.setFieldValueDefault(EventFields.PARTICIPANTS_INVITE, true);
        }
        if (passportAuthDomainsHolder.containsYandexTeamRu()) {
            if (resourceRoutines.selectResourceEmails(eventData.getParticipantEmails()).isNotEmpty()) {
                newEvent.setLocation(""); // CAL-6280
            }
        }

        // XXX: make field nullable
        val creatorUid = resourceRoutines.getUidOrMasterOfResource(creatorId, eventData);

        newEvent.setFieldValueDefault(EventFields.CREATOR_UID, creatorUid);
        newEvent.setType(eventType);

        newEvent.setMainEventId(mainEventId);

        // Event, EventLayer
        final long eCreatedId = eventDao.saveEvent(newEvent, actionInfo);

        for (Rdate rdate : rdates) {
            rdate.setEventId(eCreatedId);
        }
        repetitionRoutines.createRdates(rdates, actionInfo);

        if (creatorId.isUser() && actionInfo.getActionSource() != ActionSource.MAIL) {
            svcRoutines.addCalendarSidToPassport(creatorId.getUid());
        }
        if (creatorId.isUser() && !creatorId.getUid().isYandexTeamRu()) {
            settingsRoutines.createSettingsIfNotExistsForUids(Cf.list(creatorId.getUid()));
        }
        if (eventData.getAttachmentsO().isPresent()) {
            saveEventAttachments(eCreatedId, eventData.getAttachmentsO().get());
        }

        EventAndRepetition createdEvent = eventDbManager.getEventAndRepetitionByIdForUpdate(eCreatedId);
        ParticipantId organizer = getOrganizerId(creatorId, eventData.getInvData());

        CreateInfo result;
        if (creatorId.isResource()) {
            boolean isOrganizer = organizer.isResource() && organizer.getResourceId() == creatorId.getResourceId();

            createEventResource(creatorId.getResourceId(), createdEvent, isOrganizer, actionInfo);
            result = new CreateInfo(createdEvent, Option.<Long>empty());
        } else {
            boolean isOrganizer = organizer.isYandexUserWithUid(creatorUid);
            UserInfo creatorUserInfo = userManager.getUserInfo(creatorUid);

            if (forceNoDecision) {
                EventUser eventUser = new EventUser();
                eventUser.setDecision(Decision.NO);

                eventUserRoutines.createOrUpdateEventUser(organizer.getUid(), createdEvent, eventUser, actionInfo);
                result = new CreateInfo(createdEvent, Option.empty());
            } else if (actionInfo.getActionSource().isWebOrApi() && !isOrganizer && !isParticipant(creatorId, eventData)) {
                attachEventOrMeetingToUser(
                        userManager.getUserInfo(organizer.getUid()), createdEvent, Option.empty(), Option.of(true),
                        Decision.YES, new EventUser(), NotificationsData.useLayerDefaultIfCreate(), actionInfo);

                result = new CreateInfo(createdEvent, Option.empty());
            } else {
                EventAttachedLayerId layerId = createOrUpdateEventLayer(
                        creatorUserInfo, layerIdO, sid, createdEvent, eventType, isOrganizer, actionInfo);

                eventUserRoutines.saveEventUserAndNotification(creatorUid, createdEvent,
                        eventUserData.getEventUser(), notificationsData, layerId.getCurrentLayerId(), actionInfo);

                result = new CreateInfo(createdEvent, Option.of(layerId.getCurrentLayerId()));
            }
        }

        // just to correctly set event permissions while creating from CalDav or exchange
        if (!newEvent.isFieldSet(EventFields.PERM_ALL)) {
            eventLayerDao.findPrimaryLayerByEventId(eCreatedId).ifPresent(layer -> {
                val eventPatch = new Event();
                eventPatch.setId(eCreatedId);
                eventPatch.setPermAll(layer.getIsEventsClosedByDefault() ? EventActionClass.NONE : EventActionClass.VIEW);
                eventDao.updateEvent(eventPatch);
            });
        }

        return result;
    }

    private ParticipantId getOrganizerId(UidOrResourceId subjectId, ParticipantsOrInvitationsData data) {
        if (data.getOrganizerEmail().isPresent()) {
            return eventInvitationManager.getParticipantIdByEmail(data.getOrganizerEmail().get());
        }
        // XXX: bullshit
        return subjectId.toParticipantId();
    }

    private boolean isParticipant(UidOrResourceId subjectId, EventData data) {
        ParticipantId participantId = subjectId.toParticipantId();
        return eventInvitationManager.getParticipantIdsByEmails(data.getParticipantEmails())
                .exists(t -> participantId.equals(t._2));
    }

    public long createNotChangedRecurrence(PassportUid uid, long masterEventId, Instant recurrenceId, ActionInfo actionInfo) {
        Event master = eventDbManager.getEventByIdForUpdate(masterEventId);
        EventInfo event = eventInfoDbLoader.getEventInfoByEvent(Option.of(uid), master, actionInfo.getActionSource());

        return createNotChangedRecurrence(uid, event, recurrenceId, actionInfo);
    }

    public long createNotChangedRecurrence(
            PassportUid uid, EventInfo master, Instant recurrenceId, ActionInfo actionInfo)
    {
        master.getEvent().validateIsLockedForUpdate();

        Event masterEvent = master.getEvent();

        Validate.none(masterEvent.getRecurrenceId());
        Validate.some(masterEvent.getRepetitionId());
        Validate.isTrue(RepetitionUtils.isValidStart(master.getRepetitionInstanceInfo(), recurrenceId));

        Event eventOverrides = new Event();
        eventOverrides.setRecurrenceId(recurrenceId);
        eventOverrides.setRepetitionIdNull();

        eventOverrides.setStartTs(recurrenceId);
        eventOverrides.setEndTs(recurrenceId.plus(EventRoutines.getInstantInterval(masterEvent).getDuration()));

        long recurrenceEventId = eventDbManager.cloneEventWithDependents(
                master.getEventWithRelations(),
                master.getAttachments(),
                eventOverrides,
                EventAttachmentChangesInfo.NO_CHANGES,
                actionInfo);

        notificationRoutines.recalcNextSendTsForNewRecurrenceOrTail(masterEvent.getId(), recurrenceEventId, actionInfo);
        lastUpdateManager.updateTimestampsAsync(masterEvent.getMainEventId(), actionInfo);

        return recurrenceEventId;
    }

    public CreateInfo createUserOrFeedEvent(
            UidOrResourceId creatorId, EventType eventType, long mainEventId, EventData eventData,
            NotificationsData.Create notificationsData, InvitationProcessingMode invProcessingMode, ActionInfo actionInfo)
    {
        ActorId actorId = ActorId.userOrResource(creatorId);
        boolean isParkingOrApartmentOccupation = getIsParkingOrApartmentOccupation(eventData);

        ParticipantId organizerId = getOrganizerId(creatorId, eventData.getInvData());
        TelemostManager.PatchResult telemost = telemostManager.patchCreateData(eventData, organizerId);

        CreateInfo ci = createEventCommon(creatorId, mainEventId, PassportSid.CALENDAR,
                eventType, eventData, isParkingOrApartmentOccupation, notificationsData, actionInfo);

        EventInvitationResults eventInvitationResults = createInvitationsForNewEvent(
                creatorId, ci.getEventAndRepetition(), eventData.getInvData(),
                isParkingOrApartmentOccupation, actionInfo);

        ListF<EventSendingInfo> sendingInfoList = eventInvitationResults.getSendingInfos();

        RepetitionInstanceInfo createdRepetition = ci.getRepetitionInfo();
        EventWithRelations createdEvent = eventDbManager.getEventWithRelationsByEvent(ci.getEvent());

        sendingInfoList = sendingInfoList.plus(eventInvitationManager.createSendingInfoForResourceSubscribers(
                createdEvent.getParticipants(), createdEvent.getEvent(), MailType.EVENT_INVITATION,
                EventInstanceParameters.fromEvent(createdEvent.getEvent()), Option.empty()));

        ListF<Long> resourceIds = createdEvent.getResourceIds();

        resourceRoutines.lockResourcesByIds(resourceIds);
        resourceScheduleManager.invalidateCachedScheduleForResources(resourceIds);

        ensureExchangeCompatibleIfNeeded(actorId, createdEvent);

        RejectedResources rejectedResources = eventInvitationResults.getRejectedResources()
                .add(resourceAccessRoutines.checkForInaccessibleResources(
                        actorId, createdEvent, createdRepetition,
                        ResourceAccessRoutines.InaccessibilityCheck.onCreate(), actionInfo))
                .add(resourceAccessRoutines.checkForBusyResources(
                        actorId, createdEvent, createdRepetition, Cf.list(), actionInfo));

        if (creatorId.isResource()) {
            Option<NameI18n> reason = rejectedResources.getReason(creatorId.getResourceId());
            Decision decision = reason.isPresent() ? Decision.NO : Decision.YES;
            ewsExportRoutines.setResourceDecisionIfNeeded(
                    creatorId.getResourceId(), createdEvent, ci.getRepetitionInfo(),
                    eventData.getEventUserData().getDecision(), decision, reason, actionInfo);
        }

        repetitionConfirmationManager.rescheduleConfirmation(createdEvent, createdRepetition, actionInfo);

        ListF<EventOnLayerChangeMessageParameters> layerNotifyMails = eventsOnLayerChangeHandler.handleEventCreate(
                creatorId, createdEvent, createdRepetition, actionInfo);

        ListF<Long> layerIds = eventLayerDao.findLayersIdsByEventIds(Cf.list(ci.getEventId()));

        synchronizeWithExchangeOnCreate(createdEvent, createdRepetition, eventData.getExchangeData(), actionInfo);

        if (rejectedResources.isNotEmpty()) {
            ewsExportRoutines.forceUpdateEventForOrganizer(createdEvent, createdRepetition, actionInfo);
        }

        if (creatorId.isUser() && invProcessingMode.isSend()) {
            ListF<EventMessageParameters> mails = eventInvitationManager.createEventInvitationOrCancelMails(
                    ActorId.userOrResource(creatorId), createdEvent, createdRepetition, sendingInfoList, actionInfo);

            eventInvitationManager.sendEventMails(layerNotifyMails.<EventMessageParameters>cast().plus(mails), actionInfo);
        }
        eventsLogger.log(EventChangeLogEvents.created(
                ActorId.userOrResource(creatorId), createdEvent, createdRepetition), actionInfo);

        eventInvitationManager.sendDecisionFixingMailsIfNeeded(createdEvent, createdRepetition, actionInfo);

        if (creatorId.isUser()) {
            fixDecisionInExchangeIfNeeded(
                    creatorId.getUid(), createdEvent, createdRepetition, Option.empty(), actionInfo);
        }

        telemostManager.onEventUpdated(telemost, ci.getEventAndRepetition(), actionInfo);
        lastUpdateManager.updateTimestampsAsync(mainEventId, actionInfo);

        return new CreateInfo(
                ci.getEventAndRepetition(), Option.of(createdEvent.getExternalId()),
                ci.getLayerId(), layerIds, sendingInfoList, layerNotifyMails);
    }

    public void synchronizeWithExchangeOnCreate(
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
            Option<ExchangeData> exchangeDataO, ActionInfo actionInfo)
    {
        if (actionInfo.getActionSource().isFromExchange()) {
            final ExchangeData exchangeData = exchangeDataO.get();
            saveUpdateExchangeId(exchangeData.getSubjectId(), event.getId(), exchangeData.getExchangeId(), actionInfo);
        } else {
            ewsExportRoutines.exportToExchangeIfNeededOnCreate(event, repetitionInfo, actionInfo);
        }
    }

    public void synchronizeWithExchangeOnUpdate(
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition,
            Option<OccurrenceId> occurrenceId, EventChangesInfo eventChangesInfo,
            Option<ExchangeMails> mails, ActionInfo actionInfo)
    {
        if (actionInfo.getActionSource().isFromExchange()) {
            final ExchangeData exchangeData = eventChangesInfo.getExchangeData().get();
            saveUpdateExchangeId(
                    exchangeData.getSubjectId(), updatedEvent.getId(), exchangeData.getExchangeId(), actionInfo);
        } else {
            ewsExportRoutines.exportToExchangeIfNeededOnUpdate(
                    updatedEvent, updatedRepetition, occurrenceId, eventChangesInfo, mails, actionInfo);
        }
    }

    public Option<Tuple2<AvailabilityOverlap, Resource>> findFirstResourceEventIntersectingGivenInterval(
            Option<PassportUid> clientUid, ListF<Long> resourceIds,
            Option<Long> eventId, RepetitionInstanceInfo repetitionInfo,
            boolean useResourceScheduleCache, ActionInfo actionInfo)
    {
        Option<Tuple2<Long, AvailabilityOverlap>> first = findResourcesEventsIntersectingGivenInterval(
                clientUid, resourceIds, eventId, repetitionInfo, useResourceScheduleCache, true, actionInfo).firstO();

        return first.map(t -> Tuple2.tuple(t._2, resourceRoutines.loadById(t._1)));
    }

    public Tuple2List<Long, AvailabilityOverlap> findResourcesEventsIntersectingGivenInterval(
            Option<PassportUid> clientUid, ListF<Long> resourceIds,
            ListF<Long> exceptEventIds, RepetitionInstanceInfo repetitionInfo,
            boolean useResourceScheduleCache, boolean firstOnly, ActionInfo actionInfo)
    {
        Tuple2List<UidOrResourceId, Option<AvailabilityOverlap>> busyOverlappingEvents =
                availRoutines.busyOverlappingEventsInFuture(
                        clientUid, resourceIds.map(UidOrResourceId.resourceF()), actionInfo.getNow(),
                        repetitionInfo, exceptEventIds, useResourceScheduleCache, actionInfo);

        Tuple2List<UidOrResourceId, AvailabilityOverlap> overlaps = Cf2.flatBy2(busyOverlappingEvents)
                .sortedBy2(AvailabilityOverlap.getStartF().andThenNaturalComparator());

        if (firstOnly && overlaps.isNotEmpty()) {
            overlaps = overlaps.take(1);
        }

        return overlaps.map1(UidOrResourceId::getResourceId);
    }

    public void ensureExchangeCompatibleIfNeeded(ActorId subjectId, EventWithRelations event) {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()
                || subjectId.getUidO().exists(u -> !u.isYandexTeamRu()))
        {
            return;
        }

        DateTimeZone tz = event.getTimezone();

        if (event.getMainEvent().getIsExportedWithEws().isSome(true)
            || event.getResources().exists(ResourceInfo.isYaTeamAndSyncWithExchange()))
        {
            if (!WindowsTimeZones.getWinNameByZone(tz).isPresent()) {
                NameI18n message = new NameI18n(
                        "Часовой пояс {} не подходит для экспорта в Exchange",
                        "Timezone {} is not supported for export to Exchange")
                        .replace("{}", NameI18n.constant(tz.getID()));

                throw new ReadableErrorMessageException(message, Situation.EWS_UNSUPPORTED_TIMEZONE);
            }
        }

        if (event.getRecurrenceId().isPresent()) {
            Option<Event> masterEvent = getMainInstOfEvent(event.getEvent());
            if (masterEvent.isPresent()) {

                RepetitionInstanceInfo masterRepetition =
                        repetitionRoutines.getRepetitionInstanceInfoByEventAndTimezone(masterEvent.get(), tz);

                InstantInterval windowInterval = new RecurrenceTimeInfo(
                        event.getRecurrenceId().get(), getInstantInterval(event.getEvent()))
                        .getWindowInterval(tz);

                ListF<RecurrenceTimeInfo> instances = RepetitionUtils
                        .getInstancesInInterval(masterRepetition, windowInterval)
                        .map(i -> new RecurrenceTimeInfo(i.getStart(), i));

                ListF<RecurrenceTimeInfo> recurrences = masterRepetition.getRecurrences()
                        .filter(r -> r.getWindowInterval(tz).overlaps(windowInterval));

                instances.plus(recurrences).sortedBy(RecurrenceTimeInfo::getRecurrenceId).reduceLeft((prev, next) -> {
                    if (!prev.getStartDate(tz).isBefore(next.getStartDate(tz))
                            || prev.getEnd().isAfter(next.getStart()))
                    {
                        RecurrenceTimeInfo time = event.getRecurrenceId().isSome(prev.getRecurrenceId()) ? next : prev;

                        NameI18n message = new NameI18n(
                                "Редактируемый повтор пересекает соседний повтор на {}",
                                "Modified occurrence is crossing or overlapping adjacent occurrence at {}")
                                .replace("{}", NameI18n.constant(time.getStartDate(tz).toString("yyyy-MM-dd")));

                        throw new ReadableErrorMessageException(message, Situation.EWS_OCCURRENCES_OVERLAP);
                    }
                    return next;
                });
            }
        }
    }

    public ListF<ResourceInaccessibility> findInaccessibleResources(
            Option<UserInfo> user, InaccessibleResourcesRequest request)
    {

        ListF<ResourceInfo> resources = request.getResources();
        RepetitionInstanceInfo repetitionInfo = request.getRepetitionInfo();

        Duration eventLength = repetitionInfo.getEventInterval().getDuration();

        if (eventLength.getMillis() <= 0) {
            return resources.map(ResourceInfo::getResource).map(ResourceInaccessibility::zeroLengthEvent);
        }

        Lazy<ListF<Long>> massageResourceIds = Lazy.withSupplier(() ->
                resourceRoutines.findActiveResources(Cf.list(ResourceType.MASSAGE_ROOM)).map(Resource::getId));

        return resources.map(resourceInfo -> {
            Resource resource = resourceInfo.getResource();

            DateTimeZone officeTz = OfficeManager.getOfficeTimeZone(resourceInfo.getOffice());

            boolean canAdmin = user.exists(u -> u.canAdminResource(resource));

            Option<Duration> providedMaxDuration = user.isPresent()
                    ? SpecialResources.getDurationLimit(user.get(), resource)
                    : SpecialResources.getDurationLimit(resource);
            Duration maxDuration = providedMaxDuration
                    .plus1(resource.getType().getMaxAllowedEventDuration()).min();

            if (maxDuration.isShorterThan(eventLength)) {
                return ResourceInaccessibility.tooLongEvent(resource, maxDuration);
            }

            Option<Duration> providedMinDuration = user.isPresent()
                    ? SpecialResources.getMinimalDurationLimit(user.get(), resource)
                    : SpecialResources.getMinimalDurationLimit(resource);
            Duration minDuration = providedMinDuration
                    .plus1(Duration.standardMinutes(1)).max();

            if (minDuration.isLongerThan(eventLength)) {
                return ResourceInaccessibility.tooShortEvent(resource, minDuration);
            }

            if (request.getNowO().exists(repetitionInfo.getEventStart().minus(Duration.standardDays(360))::isAfter)) {
                return ResourceInaccessibility.tooFarEvent(
                        resource, new LocalDate(request.getNowO().get(), officeTz)
                                .toDateTimeAtStartOfDay(officeTz).plusDays(360));
            }

            if (!canAdmin && SpecialResources.isRepetitionUnacceptable(resource) && !repetitionInfo.isEmpty()) {
                return ResourceInaccessibility.repetitionDenied(resource);
            }

            if (request.getNowO().isPresent()) {
                Option<DateTime> maxStart =
                        user.isNotEmpty() ?
                        SpecialResources.getEventMaxStart(user.get(), resourceInfo, request.getNowO().get()) :
                        SpecialResources.getEventMaxStart(resourceInfo, request.getNowO().get());

                if (maxStart.isPresent() && RepetitionUtils.hasInstanceAfter(repetitionInfo, maxStart.get().toInstant())) {
                    return ResourceInaccessibility.tooFarEvent(resource, maxStart.get());
                }
            }

            if (!(request.isVerify() && canAdmin) && request.getNowO().isPresent()) {
                Option<DateInterval> restriction = SpecialResources.getRestrictionDates(resourceInfo).find(r -> {
                    ListF<InstantInterval> intervals = SuggestUtils.crop(Cf.list(r.toInstantInterval(officeTz)),
                            request.getNowO().get(), RepetitionUtils.END_OF_EVERYTHING_TS);

                    return intervals.exists(i -> RepetitionUtils.getIntervals(
                            repetitionInfo, i.getStart(), Option.of(i.getEnd()), true, 1).isNotEmpty());
                });

                if (restriction.isPresent()) {
                    return ResourceInaccessibility.dateRestricted(resource, restriction.get());
                }
            }

            if (request.isVerify() && !canAdmin) {
                DateTime start = repetitionInfo.getEventStart().toDateTime(officeTz);

                Function2B<TimesInUnit, ListF<EventInstanceInterval>> isOverbooked =
                        (u, is) -> u.getUnit().expand(start).exists(i -> is.count(Cf2.f1B(i::overlaps)
                                .compose(EventInstanceInterval::getInterval)) >= u.getTimes());

                Option<InstantInterval> maxInterval = SpecialResources.findMaxEventsLimitCheckInterval(resource, start);

                if (maxInterval.isPresent()) {
                    EventLoadLimits loadLimits = EventLoadLimits.intersectsInterval(maxInterval.get())
                            .withExcludeIds(request.getExcludeEventId());

                    ListF<EventInstanceInterval> events = eventInfoDbLoader
                            .getEventsOnResources(Cf.list(resource.getId()), loadLimits)
                            .flatMap(e -> e.getInstancesInInterval(maxInterval.get()));

                    Option<TimesInUnit> overbooked = SpecialResources.getEventsLimits(resource)
                            .find(l -> isOverbooked.apply(l, events));

                    if (overbooked.isPresent()) {
                        return ResourceInaccessibility.tooManyEvents(resource, overbooked.get());
                    }

                    SetF<Long> userEventIds = user.flatMap(u -> eventDao.findUserCreatedEventIdsByUidAndEventIds(
                            u.getUid(), events.map(EventInstanceInterval::getEventId))).unique();

                    ListF<EventInstanceInterval> userEvents = events
                            .filter(userEventIds.containsF().compose(EventInstanceInterval::getEventId));

                    overbooked = SpecialResources.getEventsLimitsPerUser(resource)
                            .find(l -> isOverbooked.apply(l, userEvents));

                    if (overbooked.isPresent()) {
                        return ResourceInaccessibility.tooManyEventsByUser(resource, overbooked.get());
                    }
                }
            }

            if (request.getUsers().isNotEmpty() && resource.getType() == ResourceType.MASSAGE_ROOM) {
                if (!user.isPresent()) {
                    return ResourceInaccessibility.massageDenied(resource, "Not user");
                }
                if (user.get().isExternalYt()) {
                    return ResourceInaccessibility.massageDenied(resource, "External user");
                }
                if (user.get().isYaMoney()) {
                    return ResourceInaccessibility.massageDenied(resource, "External ya.money user");
                }
                if (!user.get().isInAnyGroup(Group.MASSAGE_ADMIN, Group.SUPER_USER) && request.isVerify()) {
                    Option<EventInstanceInterval> existing = findUsersFirstAttendedEventInstanceOnResourcesInFuture(
                            request.getUsers(), massageResourceIds.get(),
                            request.getNowO().get(), request.getExcludeEventId());

                    if (existing.isPresent()) {
                        return ResourceInaccessibility.massageDenied(resource, "Already occupied " + existing.get());
                    }
                }
            }
            return null;

        }).filterNotNull();
    }

    private Option<EventInstanceInterval> findUsersFirstAttendedEventInstanceOnResourcesInFuture(
            ListF<PassportUid> uids, ListF<Long> resourceIds, Instant now, Option<Long> excludeEventId)
    {
        ListF<Long> excludeEventIds = excludeEventId.flatMap(this::findMasterAndSingleEventIds);
        InfiniteInterval interval = new InfiniteInterval(now, Option.empty());

        Option<EventAndRepetition> first =
                findUsersAttendedEventsWithResourcesStartingIn(uids, resourceIds, interval, excludeEventIds).firstO();

        return first.isPresent()
                ? Option.of(first.get().getInstancesInInterval(interval).first())
                : Option.empty();
    }

    private SetF<ResourceType> findResourceTypes(EventData eventData) {
        ListF<Long> resourceIds = eventInvitationManager.getParticipantIdsByEmails(eventData.getParticipantEmails())
                .get2().filterMap(ParticipantId.getResourceIdIfResourceF());

        return Cf2.flatBy2(resourceRoutines.findResourceTypesByIds(resourceIds)).get2().unique();
    }

    public boolean getIsParkingOrApartmentOccupation(EventData eventData) {
        SetF<ResourceType> resourceTypes = findResourceTypes(eventData);
        boolean isParkingOccupation = resourceTypes.containsTs(ResourceType.PARKING);
        boolean isApartmentOccupation = resourceTypes.containsTs(ResourceType.APARTMENT);
        boolean isHotelOccupation = resourceTypes.containsTs(ResourceType.HOTEL);
        boolean isCampusOccupation = resourceTypes.containsTs(ResourceType.CAMPUS);

        return isParkingOccupation || isApartmentOccupation || isHotelOccupation || isCampusOccupation;
    }

    public ListF<EventInstanceInterval> findNowEventsOnAbsenceLayer(PassportUid uid, Instant now) {
        Option<Layer> layer = layerDao.findAbsenceLayerByUser(uid);

        if (layer.isPresent()) {
            ListF<EventAndRepetition> events = eventInfoDbLoader.getEventsOnLayers(
                    layer.map(Layer.getIdF()), EventLoadLimits.intersectsInterval(now, now));
            return events.flatMap(EventAndRepetition.getInstancesInIntervalF(now, now));

        } else {
            return Cf.list();
        }
    }

    public ListF<EventAndRepetition> findUsersAttendedEventsWithResourcesStartingIn(
            ListF<PassportUid> uids, ListF<Long> resourceIds, InfiniteInterval interval, ListF<Long> excludeEventIds)
    {
        ListF<EventAndRepetition> events = eventInfoDbLoader.getEventsOnResources(
                resourceIds, EventLoadLimits.startsInInterval(interval).withExcludeIds(excludeEventIds));

        ListF<Long> eventIds = eventUserDao.findUsersAttendingEventIds(
                uids, events.map(EventAndRepetition.getEventIdF()));

        return events.filter(EventAndRepetition.getEventIdF().andThen(eventIds.unique().containsF()));
    }

    public ListF<EventWithRelations> findLastUserAttendedEvents(PassportUid uid, int limit, InstantInterval interval) {
        ListF<Long> layerIds = getLayerIds(LayerIdPredicate.allForUser(uid, false));
        ListF<EventAndRepetition> events = eventInfoDbLoader.getEventsOnLayers(
                layerIds, EventLoadLimits.intersectsInterval(interval));

        SqlCondition c = EventUserFields.DECISION.ne(Decision.NO)
                .and(EventUserFields.EVENT_ID.column().inSet(events.map(EventAndRepetition.getEventIdF())))
                .and(EventUserFields.UID.eq(uid))
                .and(EventUserFields.IS_ATTENDEE.eq(true));

        SetF<Long> attendingEventIds = eventUserDao.findEventUsers(c).map(EventUser.getEventIdF()).unique();

        events = events.filter(EventAndRepetition.getEventIdF().andThen(attendingEventIds.containsF()));

        Tuple2List<EventAndRepetition, Instant> withLastStart = events.zipWithFlatMapO(
                EventAndRepetition.getLastInstanceStartInIntervalF(interval));

        events = withLastStart.sortedByDesc(Tuple2.get2F()).get1();

        events = events.stableUniqueBy(EventAndRepetition.getMainEventIdF()).take(limit);

        return eventDbManager.getEventsWithRelationsByEvents(events.map(EventAndRepetition.getEventF()));
    }

    public EventInvitationResults createInvitationsForNewEvent(
            UidOrResourceId creatorId, EventAndRepetition eventAndRepetition,
            ParticipantsOrInvitationsData participantsData,
            boolean isParkingOrApartmentOccupation, ActionInfo actionInfo)
    {
        EventParticipantsChangesInfo eventParticipantsChangesInfo =
                eventInvitationManager.participantsChanges(creatorId.getUidO(), Participants.notMeeting(), participantsData);

        if (creatorId.isUser()) {
            ListF<Email> emails = eventParticipantsChangesInfo.getNewParticipants().get2().map(ParticipantData.getEmailF());
            contactRoutines.exportUserContactsEmails(creatorId.getUid(), emails);
        }

        EventInvitationResults eventInvitationResults = eventInvitationManager.createNewEventInvitations(
                ActorId.userOrResource(creatorId), eventParticipantsChangesInfo.getNewParticipants(),
                eventDbManager.getEventWithRelationsByEvent(eventAndRepetition.getEvent()),
                eventAndRepetition.getRepetitionInfo(), isParkingOrApartmentOccupation,
                UpdateMode.MEETING_CREATION, actionInfo);

        ListF<EventSendingInfo> sendingInfoList = eventInvitationResults.getSendingInfos();

        Event event = eventAndRepetition.getEvent();

        boolean isExportedWithEws =
                mainEventDao.findMainEventById(event.getMainEventId()).getIsExportedWithEws().getOrElse(false);

        boolean sendMail = !isExportedWithEws || isParkingOrApartmentOccupation;

        ActionSource actionSource = actionInfo.getActionSource();

        if (creatorId.isUser() && sendMail && actionSource.isWebOrApi()) {
            if (eventParticipantsChangesInfo.getNewParticipants().get1().containsTs(creatorId.toParticipantId())
                    && passportAuthDomainsHolder.containsYandexTeamRu())
            {
                sendingInfoList = sendingInfoList.plus1(eventInvitationManager
                        .createSendingInfoForEventCreator(creatorId.getUid(), event));

            } else if (eventParticipantsChangesInfo.getNewParticipants().isEmpty()) {
                PassportUid layerOwnerOrCreator = eventDbManager.getPrimaryLayer(event.getId())
                        .map(Layer::getCreatorUid).getOrElse(creatorId.getUid());

                sendingInfoList = sendingInfoList.plus(eventInvitationManager
                        .prepareSendingInfoForOutlookerSubscriber(layerOwnerOrCreator, event));
            }
        }
        return new EventInvitationResults(sendingInfoList, eventInvitationResults.getRejectedResources());
    }

    public CreateInfo createServiceEventIfNotExists(
            PassportUid uid, PassportSid sid, EventData eventData,
            NotificationsData.Create notificationsData, ActionInfo actionInfo)
    {
        String externalId = eventData.getExternalId().get();
        final ListF<Event> existingEvents = eventDao.findServiceEventsByUidSidAndExternalId(uid, sid, externalId);
        if (existingEvents.isNotEmpty()) {
            final String msg =
                "service event exists for uid = " + uid + ", sid = " + sid + ", external id = " + externalId + ". " +
                "Found event ids = " + existingEvents.map(EventFields.ID.getF());
            throw CommandRunException.createSituation(msg, Situation.EVENT_ALREADY_EXISTS);
        }

        return createServiceEvent(uid, sid, eventData, notificationsData, actionInfo);
    }

    public CreateInfo createServiceEvent(
            PassportUid uid, PassportSid sid, EventData eventData,
            NotificationsData.Create notificationsData, ActionInfo actionInfo)
    {
        if (!dbSvcRoutines.exists(sid)) {
            final String msg = "Such SID does not seem to exist in calendar db: " + sid;
            throw new CommandRunException(msg);
        }
        long mainEventId = createMainEvent(eventData.getExternalId().get(), eventData.getTimeZone(), actionInfo);

        CreateInfo ci = createEventCommon(UidOrResourceId.user(uid), mainEventId,
                sid, EventType.SERVICE, eventData, false, notificationsData, actionInfo);

        lastUpdateManager.updateTimestampsAsync(mainEventId, actionInfo);

        if (ControlDataNotification.isSupportedBy(sid)) {
            String eventExtId = mainEventDao.findExternalIdByMainEventId(ci.getEvent().getMainEventId());
            controlDataNotification.send(ci.getEvent(), uid, eventExtId, actionInfo.getNow(), CalendarNotificationType.CREATE);
        }
        return ci;
    }

    /**
     * Attaches given USER event or meeting to the layer specified by information from eDp.
     * @param user user from whose name event-layer creation process is performed
     * @param eventId event to attach
     * @param layerIdO layer to put event instance to (user subject case only)
     * @return id of the layer where event was attached
     */
    public EventAttachedUser attachEventOrMeetingToUser(
            UserInfo user, long eventId, Option<Long> layerIdO,
            Decision decision, EventUser eventUser, NotificationsData.Create notificationsData, ActionInfo actionInfo)
    {
        EventAndRepetition event = eventDbManager.getEventAndRepetitionByIdForUpdate(eventId);
        return attachEventOrMeetingToUser(
                user, event, layerIdO, Option.empty(), decision, eventUser, notificationsData, actionInfo);
    }

    public EventAttachedUser attachEventOrMeetingToUser(
            UserInfo user, EventAndRepetition event, Option<Long> layerIdO, Option<Boolean> isPrimaryInst,
            Decision decision, EventUser eventUser, NotificationsData.Create notificationsData, ActionInfo actionInfo)
    {
        boolean isPrimaryInstance = isPrimaryInst.getOrElse(
                Option.of(eventUser).isMatch(eu -> eu.isFieldSet(EventUserFields.IS_ORGANIZER) && eu.getIsOrganizer()));

        EventAttachedLayerId layerId = createOrUpdateEventLayer(
                user, layerIdO, PassportSid.CALENDAR, event,
                EventType.USER, isPrimaryInstance, actionInfo);

        eventUser = eventUser.copy();
        eventUser.setDecision(decision);
        if (!eventUser.isFieldSet(EventUserFields.AVAILABILITY)) {
            eventUser.setAvailability(Availability.byDecision(decision));
        }

        EventUserUpdate eventUserUpdate = eventUserRoutines.createOrUpdateEventUserAndCreateNotifications(
                user.getUid(), event, eventUser, notificationsData,
                layerId.getCurrentLayerId(), actionInfo);

        return new EventAttachedUser(event.getEvent(), EventFieldsChangesJson.empty(), eventUserUpdate, Option.of(layerId));
    }

    public EventsAttachedUser attachEventOrMeetingsToUserByMainEventId(
            UserInfo userInfo, long mainEventId, Option<Long> layerIdO,
            EventUser eventUserData, NotificationsData.Create notificationsData, ActionInfo actionInfo)
    {
        Validate.isFalse(eventUserData.isAnyFieldSet(EventUserFields.EVENT_ID, EventUserFields.UID));

        ListF<EventAndRepetition> es = eventDbManager.getEventsAndRepetitionsByMainEventIdForUpdate(mainEventId);

        ListF<EventInfo> events = eventInfoDbLoader.getEventInfosByEventsAndRepetitions(
                Option.of(userInfo), es, actionInfo.getActionSource());

        ListF<EventAttachedUser> result = Cf.arrayList();

        val eventAuthInfoById = authorizer.loadEventsInfoForPermsCheck(userInfo, StreamEx.of(events).map(EventInfo::getEventWithRelations).toImmutableList());
        for (EventInfo event : events) {
            val eventId = event.getEventId();
            if (authorizer.canViewEvent(userInfo, eventAuthInfoById.get(eventId), actionInfo.getActionSource())) {
                result.add(attachEventOrMeetingToUser(
                        userInfo, event.getEventAndRepetition(), layerIdO, Option.empty(),
                        Decision.YES, eventUserData, notificationsData, actionInfo));
            }
        }
        lastUpdateManager.updateTimestampsAsync(mainEventId, actionInfo);
        return new EventsAttachedUser(result, Option.of(userInfo.getUid()));
    }

    public EventAttachedUser attachMeetingToOrganizer(PassportUid uid, long eventId, ActionInfo actionInfo) {
        EventUser eventUser = new EventUser();
        eventUser.setUid(uid);
        eventUser.setEventId(eventId);
        eventUser.setIsOrganizer(true);
        return attachEventOrMeetingToUser(userManager.getUserInfo(uid), eventId, Option.empty(), Decision.YES,
                eventUser, new NotificationsData.UseLayerDefaultIfCreate(), actionInfo);
    }

    /*--------------------------------*/
    /*                                */
    /* CREATE EVENT-LAYER, EVENT-USER */
    /*                                */
    /*--------------------------------*/

    public EventLayer createEventLayerData(
            long layerId, PassportUid layerCreatorUid, EventAndRepetition event, ActionInfo actionInfo)
    {
        EventLayer eventLayerData = new EventLayer();
        eventLayerData.setEventId(event.getEventId());
        eventLayerData.setLayerId(layerId);

        EventDbManager.setIndentsData(eventLayerData, event);
        eventLayerData.setLCreatorUid(layerCreatorUid);

        return eventLayerData;
    }

    public EventAttachedLayerId createOrUpdateEventLayer(
            UserInfo user, Option<Long> layerIdO, PassportSid sid,
            EventAndRepetition event, EventType eventType, boolean isPrimaryInstance, ActionInfo actionInfo)
    {
        event.validateIsLockedForUpdate();

        // TODO: ensure that event with id 'eventId' is of type 'eventType'
        final long layerId;
        if (layerIdO.isPresent()) {
            layerId = layerIdO.get();
            checkLayerProperties(eventType, user.getUid(), layerId);
        } else {
            layerId = getOrCreateLayer(eventType, user.getUid(), sid);
            layerUserDao.updateLayerUserSetVisibleInUiByLayerIdAndUid(layerId, user.getUid(), true);
        }
        Layer layer = layerRoutines.getLayerById(layerId);
        PassportUid layerCreatorUid = layer.getCreatorUid();

        EventLayer eventLayerData = createEventLayerData(layerId, layerCreatorUid, event, actionInfo);

        Option<EventLayer> eventLayerO = findOwnerBean(layerCreatorUid, event.getEventId());
        if (!eventLayerO.isPresent() || layerCreatorUid.sameAs(user.getUid())) {
            if (!eventLayerO.isPresent() || eventLayerO.get().getLayerId() != layerId) {
                val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
                authorizer.ensureCanPerformLayerAction(user, layerPermInfo, Optional.empty(), LayerAction.CREATE_EVENT,
                    actionInfo.getActionSource());
            }

            if (!eventLayerO.isPresent()) {
                eventLayerData.setIsPrimaryInst(isPrimaryInstance);
                eventDbManager.saveEventLayer(eventLayerData, actionInfo);

                return EventAttachedLayerId.attached(layerId);

            } else if (!layerIdO.isPresent() || eventLayerO.get().getLayerId() == layerId) {
                eventLayerData.setLayerId(eventLayerO.get().getLayerId());
                eventDbManager.updateEventLayer(eventLayerData, actionInfo);

                return EventAttachedLayerId.alreadyAttached(eventLayerO.get().getLayerId());

            } else {
                EventLayer eventLayer = eventLayerO.get();
                eventDbManager.updateEventLayerByEventIdAndLayerId(
                        eventLayerData, eventLayer.getEventId(), eventLayer.getLayerId(), actionInfo);

                return EventAttachedLayerId.reattached(eventLayer.getLayerId(), layerId);
            }
        } else {
            eventLayerData.setLayerId(eventLayerO.get().getLayerId());
            eventDbManager.updateEventLayer(eventLayerData, actionInfo);

            return EventAttachedLayerId.alreadyAttached(eventLayerO.get().getLayerId());
        }
    }

    public EventResource createEventResourceData(long resourceId, EventAndRepetition event) {
        val eventResource = new EventResource();
        eventResource.setEventId(event.getEventId());
        eventResource.setResourceId(resourceId);

        EventDbManager.setIndentsData(eventResource, event);

        return eventResource;
    }

    public void createEventResource(long resourceId, EventAndRepetition event, boolean isOrganizer, ActionInfo actionInfo) {
        event.validateIsLockedForUpdate();

        val eventResource = createEventResourceData(resourceId, event);

        eventDbManager.saveEventResource(eventResource, actionInfo);
    }

    private long getOrCreateLayer(EventType eventType, PassportUid uid, PassportSid sid) {
        if (eventType == EventType.USER) {
            return layerRoutines.getOrCreateDefaultLayer(uid);
        }
        if (eventType == EventType.SERVICE) {
            return layerRoutines.getOrCreateServiceLayer(uid, sid);
        }
        if (eventType.isAbsence()) {
            return layerRoutines.getOrCreateAbsenceLayer(uid, eventType.toAbsenceType());
        }
        throw new IllegalStateException("don't know how to create layer of type " + eventType);
    }

    private void checkLayerProperties(EventType eventType, PassportUid uid, long layerId) {
        if (eventType == EventType.USER) {
            val layer = layerRoutines.getLayerById(layerId);
            authorizer.ensureLayerType(LayerInfoForPermsCheck.fromLayer(layer), LayerType.USER, LayerType.ABSENCE);
        }
        if (eventType == EventType.FEED) {
            val layer = layerRoutines.getLayerById(layerId);
            authorizer.ensureLayerProperties(LayerInfoForPermsCheck.fromLayer(layer), uid, LayerType.FEED);
        }
    }

    private void updateEventLayersByEvents(
            UserInfo user, ListF<Event> events, long oldLayerId, long newLayerId, ActionInfo actionInfo)
    {
        Validate.notEquals(oldLayerId, newLayerId);
        Validate.forAll(events, Event.getTypeF().andThenEquals(EventType.USER));

        val source = actionInfo.getActionSource();
        authorizer.ensureMoveAction(user, oldLayerId, newLayerId, LayerAction.DETACH_EVENT, source);

        ListF<Long> eventIds = events.map(Event.getIdF());
        ListF<EventLayer> eventLayers = eventLayerDao.findEventLayersByLayerIdAndEventIds(oldLayerId, eventIds);
        archiveManager.storeDeletedEventLayers(eventLayers, actionInfo);
        eventDbManager.removeEventLayersFromCache(eventLayers);

        eventLayerDao.updateIgnoreEventLayerSetLayerIdAndLayerCreatorUidByLayerIdAndEventIds(
                newLayerId, layerRoutines.getLayerById(newLayerId).getCreatorUid(), oldLayerId, eventIds, actionInfo);
        eventLayerDao.deleteEventLayersByLayerIdAndEventIds(oldLayerId, eventIds);

        lastUpdateManager.updateTimestampsAsync(events.map(Event::getMainEventId), actionInfo);
    }

    public void updateEventLayerByMainEventId(
            UserInfo user, long mainEventId, long oldLayerId, long newLayerId, ActionInfo actionInfo)
    {
        ListF<Event> events = eventDao.findEventsByMainId(mainEventId, true);
        updateEventLayersByEvents(user, events, oldLayerId, newLayerId, actionInfo);
    }

    public void updateEventLayerByEventExternalId(
            UserInfo user, String externalId, long oldLayerId, long newLayerId, ActionInfo actionInfo)
    {
        ListF<Event> events = eventDao.findEventsByLayerIdAndExternalIdForUpdate(oldLayerId, externalId);
        updateEventLayersByEvents(user, events, oldLayerId, newLayerId, actionInfo);
    }

    public ModificationInfo deleteServiceEvent(UserInfo user, Event event, ActionSource actionSource) {
        Long eId = event.getId();
        boolean needsSvcNtf = ControlDataNotification.isSupportedBy(event.getSid());
        PassportUid finalUid; // final acting uid (differs from 'uid' for 'needsSvcNtf' case only)
        String eventExtId = mainEventDao.findExternalIdByMainEventId(event.getMainEventId());
        if (needsSvcNtf) {
            finalUid = event.getCreatorUid();
        } else {
            finalUid = user.getUid();

            val eventWithRelations = eventDbManager.getEventWithRelationsById(event.getId());
            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(Optional.of(user), eventWithRelations,
                Optional.empty(), Optional.empty());
            authorizer.ensureCanDeleteEvent(user, eventAuthInfo, actionSource);
        }
        // TODO: should we here check that event instance exists for eStartMs?
        //EventInstanceInfo eventInstanceInfo = getSingleInstance(ctx, finalUid, eStartMs, true, eId, null);
        //Subscription s = eventInstanceInfo.getS();

        // Delete events and related stuff
        ActionInfo actionInfo = new ActionInfo(actionSource, RequestIdStack.current().getOrElse("?"), Instant.now());
        deleteEventsFromDbAndExchange(ActorId.user(user.getUid()), Cf.list(eId), actionInfo, true);
        if (event.getRepetitionId().isPresent()) {
            eventDao.deleteRepetitionsByIds(Cf.list(event.getRepetitionId().get()));
        }

        // Take care (delete if needed) of the possibly empty service layer
        PassportSid sid = event.getSid(); // e.getSid().getOrNull() - it is still valid, though event has been deleted
        String sql = "SELECT id FROM layer WHERE creator_uid = ? AND sid = ?";
        Long layerId = getJdbcTemplate().queryForOption(sql, Long.class, finalUid, sid).getOrNull();
        sql = "SELECT NOT EXISTS (SELECT id FROM event_layer WHERE layer_id = ?)";
        if (getJdbcTemplate().queryForObject(sql, Boolean.class, layerId)) {
            layerDbManager.deleteOnlyLayerWithNotification(layerId);
        }
        // notify services (which want it) about event deletion
        if (needsSvcNtf) { // 'e' is still valid
            // IMPORTANT: service should be notified with original uid
            //            to verify (or to decline) the delete operation
            controlDataNotification.send(event, user.getUid(), eventExtId, new Instant(), CalendarNotificationType.DELETE);
        }
        return ModificationInfo.removed(ModificationScope.SINGLE, Cf.list(eId), Cf.list(eventExtId), Cf.list());
    }

    // https://jira.yandex-team.ru/browse/CAL-3293
    public void deleteEventFromDbAndExchange(long eventId) {
        Event event = eventDao.findEventById(eventId);
        ListF<Long> eventIds = !event.getRecurrenceId().isPresent()
                ? eventDao.findEventIdsByMainEventId(event.getMainEventId())
                : Cf.list(eventId);

        deleteEventsFromDbAndExchange(ActorId.yaCalendar(), eventIds, ActionInfo.adminManager(), true);
    }

    public void deleteEventFromDbAndExchange(ActorId actorId, long eventId, ActionInfo actionInfo) {
        deleteEventsFromDbAndExchange(actorId, Cf.list(eventId), actionInfo, true);
    }

    public void deleteEventsFromDbAndExchange(
            ActorId actorId, ListF<Long> eventIds, ActionInfo actionInfo, boolean deleteMainEventsIfOrphaned)
    {
        deleteEvents(actorId, eventIds, actionInfo, deleteMainEventsIfOrphaned, true);
    }

    public void deleteEvents(
            ActorId actorId, ListF<Long> eventIds, ActionInfo actionInfo,
            boolean deleteMainEventsIfOrphaned, boolean deleteFromExchange)
    {
        ListF<EventAndRepetition> events = eventDbManager.getEventsAndRepetitionsByEventIds(eventIds);

        MapF<Long, String> externalIdByMainId = mainEventDao.findExternalIdsByIds(
                events.map(EventAndRepetition::getMainEventId)).toMap();

        events.forEach(event -> telemostManager.cancelScheduledGeneration(event.getEvent()));

        deleteEventsWithoutLogging(eventIds, actionInfo, deleteMainEventsIfOrphaned, deleteFromExchange);

        events.forEach(event -> externalIdByMainId.getO(event.getMainEventId()).forEach(externalId ->
                eventsLogger.log(EventChangeLogEvents.deleted(actorId, externalId, event), actionInfo)));
    }

    private void deleteEventsWithoutLogging(
            ListF<Long> eventIds, ActionInfo actionInfo,
            boolean deleteMainEventsIfOrphaned, boolean deleteFromExchange)
    {
        if (eventIds.isNotEmpty()) {
            if (deleteFromExchange) {
                ewsExportRoutines.deleteEventsIfNeeded(eventIds, actionInfo);
            }
            archiveManager.storeDeletedEventsWithDependingItems(eventIds, actionInfo);
            eventDbManager.deleteEventsByIds(eventIds, deleteMainEventsIfOrphaned, actionInfo);
        }
    }

    public void deleteEventsFromDbAndExchangeWithoutArchive(ListF<Long> eventIds, ActionInfo actionInfo) {
        if (eventIds.isNotEmpty()) {
            ewsExportRoutines.deleteEventsIfNeeded(eventIds, actionInfo);
            eventDbManager.deleteEventsByIds(eventIds, true, actionInfo);
        }
    }

    public ListF<EventMessageParameters> deleteEvents(
            final Option<UserInfo> clientInfoO, final ListF<Long> eventIds,
            final InvitationProcessingMode invitationProcessingMode, final ActionInfo actionInfo)
    {
        ListF<EventWithRelations> events = eventDbManager.getEventsWithRelationsByIds(eventIds);
        val source = actionInfo.getActionSource();
        for (EventWithRelations event : events) {

            clientInfoO.toOptional()
                .ifPresentOrElse(
                    user -> {
                        val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(Optional.of(user), event,
                            Optional.empty(), Optional.empty());
                        authorizer.ensureCanDeleteEvent(user, eventAuthInfo, source);
                    },
                    () -> {
                        val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(Optional.empty(), event,
                            Optional.empty(), Optional.empty());
                        authorizer.ensureCanDeleteEvent(eventAuthInfo, source);
                    }
                );
        }

        return deleteEventsSafe(clientInfoO.map(ActorId::user), invitationProcessingMode, actionInfo, events, true, true);
    }


    public ListF<EventMessageParameters> deleteEventsSafe(Option<UserInfo> clientInfoO, ListF<Long> eventIds,
                InvitationProcessingMode invitationProcessingMode, ActionInfo actionInfo)
    {
        ListF<EventWithRelations> events = eventDbManager.getEventsWithRelationsByIds(eventIds);

        return deleteEventsSafe(clientInfoO.map(ActorId::user), invitationProcessingMode, actionInfo, events, true, true);
    }

    public ListF<EventMessageParameters> deleteEventsSafe(
            Option<ActorId> actorId, InvitationProcessingMode invitationProcessingMode,
            ActionInfo actionInfo, ListF<EventWithRelations> events, boolean handleSms, boolean deleteFromExchange)
    {
        ListF<EventMessageParameters> res = Cf.list();
        if (actorId.isPresent() && InvitationProcessingMode.SAVE_ATTACH_SEND == invitationProcessingMode) {
            MapF<Long, RepetitionInstanceInfo> repetitionInfoByEventId =
                    repetitionRoutines.getRepetitionInstanceInfos(events);

            ListF<EventSendingInfo> sendingInfos = Cf.arrayList();
            ListF<EventOnLayerChangeMessageParameters> notifyMails = Cf.arrayList();

            for (EventWithRelations event: events) {
                if (event.getEvent().getType() == EventType.USER) {
                    sendingInfos.addAll(cancelMeetingHandler.cancelMeeting(
                            event.getEvent(), actorId.get().getUidO(),
                            event.isParkingOrApartmentOccupation(), event.isExportedWithEws(), actionInfo));

                    notifyMails.addAll(eventsOnLayerChangeHandler.handleEventDelete(
                            actorId.get(),
                            event, repetitionInfoByEventId.getOrThrow(event.getId()),
                            EventInstanceParameters.fromEvent(event.getEvent()), actionInfo));
                }
                eventsLogger.log(EventChangeLogEvents.deleted(actorId.get(),
                        event, repetitionInfoByEventId.getOrThrow(event.getId())), actionInfo);
            }
            if (handleSms && actorId.get().isUser()) {
                for (ListF<EventWithRelations> grouped : events.groupBy(EventWithRelations::getMainEventId).values()) {
                    eventDeletionSmsHandler.handleClosestEventDeletion(actorId.get().getUid(),
                            grouped.zipWith(e -> repetitionInfoByEventId.getOrThrow(e.getId())), actionInfo);
                }
            }
            res = eventInvitationManager.createEventInvitationOrCancelMails(
                    actorId.get(), sendingInfos, actionInfo).plus(notifyMails);
        }

        events.forEach(event -> telemostManager.cancelScheduledGeneration(event.getEvent()));
        deleteEventsWithoutLogging(events.map(EventWithRelations::getId), actionInfo, true, deleteFromExchange);
        return res;
    }

    public void deleteEvent(Option<UserInfo> userInfoO, long eventId,
            InvitationProcessingMode invitationProcessingMode, ActionInfo actionInfo)
    {
        ListF<EventMessageParameters> eventMails =
                deleteEvents(userInfoO, Cf.list(eventId), invitationProcessingMode, actionInfo);
        eventInvitationManager.sendEventMails(eventMails, actionInfo);
    }

    /** ssytnik@: uid should not represent a meeting organizer */
    public EventAttachedUser rejectMeeting(
            long eventId, PassportUid uid, ActionInfo actionInfo, InvitationProcessingMode invitationProcessingMode)
    {
        log.info(LogMarker.EVENT_ID.format(eventId));

        return rejectMeetings(
                Cf.list(eventId), uid, Option.empty(), actionInfo, invitationProcessingMode).events.single();
    }

    public EventsAttachedUser acceptMeetings(
            ListF<Long> eventIds, PassportUid uid, WebReplyData replyData,
            UserParticipantInfo participant,
            ActionInfo actionInfo, InvitationProcessingMode invitationProcessingMode)
    {
        UserInfo user = userManager.getUserInfo(uid);
        EventUser eventUser = new EventUser();
        eventUser.setDecision(replyData.getDecision());
        eventUser.setReason(replyData.getReason());

        if (replyData.getAvailability().isPresent()) {
            eventUser.setAvailability(replyData.getAvailability().get());
        }
        if (participant.isAttendee()) {
            eventUser.setIsAttendee(true);
            eventUser.setIsOptional(participant.isOptional());
        } else {
            eventUser.setDecision(Decision.YES);
        }

        ListF<EventAndRepetition> events = eventDbManager.getEventsAndRepetitionsByEventIdsForUpdate(eventIds);

        ListF<EventAttachedUser> attaches = Cf.arrayList();

        for (EventAndRepetition eventAndRepetition : events) {
            SequenceAndDtStamp sequence = SequenceAndDtStamp.web(eventAndRepetition.getEvent(), actionInfo);
            eventUser.setSequence(sequence.getSequence());
            eventUser.setDtstamp(sequence.getDtStamp());

            attaches.add(attachEventOrMeetingToUser(
                    user, eventAndRepetition, replyData.getLayerId(), Option.empty(),
                    replyData.getDecision(), eventUser, replyData.getNotifications(), actionInfo));
        }
        lastUpdateManager.updateTimestampsAsync(events.map(EventAndRepetition::getMainEventId), actionInfo);

        if (!participant.getUid().isSome(uid)) { // CAL-6760
            eventDao.updateEventsIncrementSequenceByIds(eventIds);
            attaches = attaches.map(EventAttachedUser::withEventSequenceIncremented);
        }

        if (invitationProcessingMode == InvitationProcessingMode.SAVE_ATTACH_SEND) {
            ListF<ReplyMessageParameters> acceptMails = updateDecisionViaEwsOrElseCreateMailIfNeeded(
                    eventIds, uid, replyData.getDecision(), Option.<String>empty(), actionInfo);
            eventInvitationManager.sendEventMails(acceptMails, actionInfo);
        }

        return new EventsAttachedUser(attaches, Option.of(uid));
    }

    public EventsAttachedUser rejectMeetings(
            ListF<Long> eventIds, PassportUid uid, Option<String> reason,
            ActionInfo actionInfo, InvitationProcessingMode invitationProcessingMode)
    {
        eventIds = eventInvitationManager.findNotRejectedFutureEventIds(ParticipantId.yandexUid(uid), eventIds, actionInfo);

        ListF<Event> events = eventDbManager.getEventsByIds(eventIds);

        EventsAttachedUser detach = eventInvitationManager.rejectMeetingsByYandexUser(events, uid, actionInfo);

        lastUpdateManager.updateTimestampsAsync(eventDao.findMainEventIdsByEventIds(eventIds), actionInfo);

        if (invitationProcessingMode != InvitationProcessingMode.SAVE_ATTACH_SEND) return detach;

        ListF<ReplyMessageParameters> rejectMails = updateDecisionViaEwsOrElseCreateMailIfNeeded(
                eventIds, uid, Decision.NO, reason, actionInfo);
        eventInvitationManager.sendEventMails(
                eventsOnLayerChangeHandler.handleEventsAttach(uid, detach.getLayers(), actionInfo), actionInfo);

        eventInvitationManager.sendEventMails(Cf.toList(rejectMails), actionInfo);

        return detach;
    }

    public void fixDecisionInExchangeIfNeeded(
            PassportUid subject, EventWithRelations event, RepetitionInstanceInfo repetition,
            Option<Decision> currentSubjectDecision, ActionInfo actionInfo)
    {
        if (!event.isMeeting()
                || !ewsAutoFixDecisions.get()
                || currentSubjectDecision.getOrElse(Decision.UNDECIDED) != Decision.UNDECIDED
                || event.getOrganizerUidIfMeeting().isSome(subject)
                || !event.userIsAttendee(subject))
        {
            return;
        }

        Option<Decision> decisionToSet = event.findUserEventUser(subject).map(EventUser::getDecision)
                .filter(decision -> decision != Decision.UNDECIDED);

        if (!decisionToSet.isPresent() || !repetition.goesOnAfter(actionInfo.getNow())) {
            return;
        }

        ewsExportRoutines.updateAttendeeDecisionIfNeeded(event.getEvent(), event.getMainEvent(),
                UidOrResourceId.user(subject), decisionToSet.get(), Option.empty(), true, actionInfo);
    }

    public ListF<ReplyMessageParameters> updateDecisionViaEwsOrElseCreateMailIfNeeded(
            ListF<Long> eventIds, PassportUid uid, Decision decision, Option<String> reason, ActionInfo actionInfo)
    {
        ListF<EventInfo> events = eventInfoDbLoader.getEventInfosByIds(
                Option.of(uid), eventIds, actionInfo.getActionSource());

        ListF<EventInfo> participatingMeetings = events.filter(
                event -> event.getEventWithRelations().getParticipants().isMeeting()
                        && event.getEventUser().exists(eu -> eu.getIsAttendee() && !eu.getIsOrganizer()));

        ListF<ReplyMessageParameters> mails = Cf.arrayList();
        ListF<EventInfo> notPastParticipatingMeetings = participatingMeetings
                .filter(e -> e.goesOnAfter(actionInfo.getNow())); // CAL-9659

        for (ListF<EventInfo> meetings : notPastParticipatingMeetings.groupBy(EventInfo::getMainEventId).values()) {
            MapF<Long, Boolean> updatedViaEws = ewsExportRoutines.updateAttendeeDecisionIfNeeded(
                    meetings.map(EventInfo::getEvent), meetings.first().getMainEvent(),
                    UidOrResourceId.user(uid), decision, reason, true, actionInfo);

            ListF<EventInfo> notUpdatedMeetings = !updatedViaEws.containsValueTs(true)
                    ? meetings  // if updating master with recurrences updatedViaEws.size() == 1
                    : meetings.filter(m -> !updatedViaEws.getTs(m.getEventId()));

            if (notUpdatedMeetings.isNotEmpty()) {
                mails.add(eventInvitationManager.createReplyMail(
                        uid, notUpdatedMeetings, decision, actionInfo.getNow()));
            }
        }

        return mails;
    }


    /** deletes or detaches all instances of events found by given external id within a given layer */
    public void deleteOrDetachEventsByExternalId(
            final UserInfo user, final long layerId, final String extId, final ActionInfo actionInfo)
    {
        // We assume that all events of given external_id in a layer are created by the same user (CAL-2404).
        // However, it is still possible that there are both primary and secondary instances in that layer.
        ListF<Event> events = eventDao.findEventsByLayerIdAndExternalId(layerId, extId);
        ListF<EventWithRelations> eventsWithRelations = eventDbManager.getEventsWithRelationsByEvents(events);

        lastUpdateManager.updateMainEventAndLayerTimestamps(events.map(Event.getIdF()), actionInfo);

        final MapF<Long, RepetitionInstanceInfo> repetitionInfoByEventId =
                repetitionRoutines.getRepetitionInstanceInfos(eventsWithRelations);

        final ListF<EventMessageParameters> eventMails = Cf.arrayList();

        //detachEventsInExchange(eventsWithRelations.map(EventWithRelations::getId), user.getUid(),  actionInfo);

        Tuple2<ListF<EventWithRelations>, ListF<EventWithRelations>> partitioned = eventsWithRelations.partition(e -> {
            val primaryLayer = e.getPrimaryLayer();
            return primaryLayer.stream().anyMatch(layer -> layer.getId() == layerId);
        });

        ListF<EventWithRelations> toDelete = partitioned.get1();
        ListF<EventWithRelations> toDetach = partitioned.get2();

        // Primary instances should be deleted, secondary ones should be detached
        toDelete.forEach(e -> {
            eventMails.addAll(deleteEvents(
                    Option.of(user), Cf.list(e.getEvent().getId()),
                    InvitationProcessingMode.SAVE_ATTACH_SEND, actionInfo));
        });

        if (toDetach.isNotEmpty()) {
            ListF<EventAndRepetition> eventAndRepetitions = toDetach.map(event ->
                    new EventAndRepetition(event.getEvent(), repetitionInfoByEventId.getOrThrow(event.getId())));

            ewsExportRoutines.detachEvents(
                    eventAndRepetitions, toDetach.first().getMainEvent(), user.getUid(),  actionInfo);
        }

        toDetach.forEach(e -> {
            // XXX: avoid query
            Option<EventLayer> eventLayer =
                    eventLayerDao.findEventLayerByEventIdAndLayerId(e.getEvent().getId(), layerId);
            if (eventLayer.isPresent()) {
                eventMails.addAll(eventsOnLayerChangeHandler.handleEventChange(
                        UidOrResourceId.user(user.getUid()), e, repetitionInfoByEventId.getOrThrow(e.getId()),
                        LayerIdChangesInfo.removed(Cf.set(eventLayer.get().getLayerId())),
                        EventChangesInfoForMails.EMPTY, actionInfo));

                storeAndDeleteEventLayer(eventLayer.get(), actionInfo);

                eventsLogger.log(EventChangeLogEvents.updated(ActorId.user(user.getUid()),
                        new EventIdLogDataJson(e), EventAttachedLayerId.detached(eventLayer.get().getLayerId())), actionInfo);
            } else {
                log.warn("Event-layer not found by event id: {}, layer id: {}", e.getEvent().getId(), layerId);
            }
        });
        eventInvitationManager.sendEventMails(eventMails, actionInfo);
    }

    public EventsAttachedUser detachEventsFromLayerByMainEventId(UserInfo user, MainEvent mainEvent, long layerId, ActionInfo actionInfo) {
        val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layerRoutines.getLayerById(layerId));
        authorizer.ensureCanPerformLayerAction(user, layerPermInfo, Optional.empty(), LayerAction.DETACH_EVENT,
            actionInfo.getActionSource());

        ListF<Event> events = eventDao.findEventsByMainIds(Cf.list(mainEvent.getId()));
        ListF<Long> eventIds = events.map(Event.getIdF());

        PassportUid layerCreatorUid = layerRoutines.getLayerById(layerId).getCreatorUid();

        ListF<EventUserUpdate> eventUsers = eventUserRoutines.updateEventUsersDecision(
                eventIds, layerCreatorUid, Decision.NO, actionInfo);

        ewsExportRoutines.detachEvents(
                eventDbManager.getEventsAndRepetitionsByEvents(events), mainEvent, layerCreatorUid,  actionInfo);

        lastUpdateManager.updateTimestampsAsync(mainEvent.getId(), actionInfo);
        ListF<EventLayer> eventLayers = eventLayerDao.findEventLayersByLayerIdAndEventIds(layerId, eventIds);

        storeAndDeleteEventLayers(eventLayers, actionInfo);

        return EventsAttachedUser.detached(layerCreatorUid, events, eventUsers, eventLayers);
    }

    public void storeAndDeleteEventLayersByLayerId(long layerId, ActionInfo actionInfo) {
        storeAndDeleteEventLayers(eventLayerDao.findEventLayersByLayerId(layerId), actionInfo);
    }

    public void storeAndDeleteEventLayer(EventLayer eventLayer, ActionInfo actionInfo) {
        storeAndDeleteEventLayers(Cf.list(eventLayer), actionInfo);
    }

    public void storeAndDeleteEventLayers(ListF<EventLayer> eventLayers, ActionInfo actionInfo) {
        archiveManager.storeDeletedEventLayers(eventLayers, actionInfo);
        eventDbManager.removeEventLayersFromCache(eventLayers);
        eventLayerDao.deleteEventLayersByIds(eventLayers.map(EventLayerId::of));
    }

    public void storeAndDeleteEventResource(EventResource eventResource, ActionInfo actionInfo) {
        storeAndDeleteEventResources(Cf.list(eventResource), actionInfo);
    }

    public void storeAndDeleteEventResources(ListF<EventResource> eventResources, ActionInfo actionInfo) {
        archiveManager.storeDeletedEventResources(eventResources, actionInfo);
        eventResourceDao.deleteEventResourcesByIds(eventResources.map(EventResourceId::of));
    }

    public void storeAndDeleteEventUsers(ListF<EventUser> eventUsers, ActionInfo actionInfo) {
        archiveManager.storeDeletedEventUsers(eventUsers, actionInfo);
        notificationDbManager.deleteNotificationsByEventUserIds(eventUsers.map(EventUser::getId));
        eventUserDao.deleteEventUsersByIds(eventUsers.map(EventUser::getId));
    }

    public static InstantInterval getInstantInterval(Event event) {
        return new InstantInterval(event.getStartTs(), event.getEndTs());
    }

    public static Period getPeriod(Event event, DateTimeZone chrono) {
        return new Period(event.getStartTs().getMillis(), event.getEndTs().getMillis(), ISOChronology.getInstance(
                chrono));
    }

    public EventInstanceInfo getSingleInstance(
            Option<PassportUid> uidO, Option<Instant> startO, long eventId, ActionSource actionSource)
    {
        return getSingleInstance(uidO, startO, false, false, eventId, actionSource);
    }

    public EventInstanceInfo getSingleInstance(
            Option<PassportUid> uidO, Option<Instant> startO,
            boolean searchNearStart, boolean lockForUpdate,
            long eventId, ActionSource actionSource)
    {
        Event event = eventDbManager.getEventByIdSafe(eventId, lockForUpdate).getOrThrow(
                CommandRunException.createSituationF("event not found by id " + eventId, Situation.EVENT_NOT_FOUND));

        RepetitionInstanceInfo repetition = repetitionRoutines.getRepetitionInstanceInfoByEvent(event);

        Option<InstantInterval> instanceInterval = Option.empty();
        Option<Instant> recurrenceId = Option.empty();

        if (repetition.isEmpty()) {
            instanceInterval = Option.of(getInstantInterval(event));

        } else if (startO.isPresent() && searchNearStart) {
            ListF<InstantInterval> intervals = RepetitionUtils.getIntervals(
                    repetition, startO.get().minus(Duration.standardHours(12)), Option.empty(), false, 2);

            Comparator<Instant> closerF = AuxDateTime.millisToF(startO.get()).andThenNaturalComparator();
            Option<InstantInterval> interval = intervals.minO(closerF.compose(InstantInterval::getStart));
            Option<Instant> closestId = repetition.getRecurIds().minO(closerF);

            if (interval.isPresent() && !closestId.plus(interval.get().getStart()).minO(closerF).equals(closestId)) {
                instanceInterval = interval;
            } else {
                recurrenceId = closestId;
            }

        } else if (startO.isPresent()) {
            instanceInterval = RepetitionUtils.getInstanceIntervalStartingAt(repetition, startO.get());
            recurrenceId = repetition.getRecurIds().find(startO.containsF());

        } else {
            Instant after = Instant.now().minus(repetition.getEventInterval().getDuration());

            Option<InstantInterval> next = Option.empty();
            Option<Instant> nextRecurrenceId = Option.empty();

            if (searchNearStart) {
                next = RepetitionUtils.getInstanceIntervalStartingAfter(repetition, after);
                nextRecurrenceId = repetition.getRecurIds().filter(after::isBefore).minO();
            }
            if (next.isPresent() && !nextRecurrenceId.exists(next.get().getStart()::isAfter)) {
                instanceInterval = next;

            } else if (nextRecurrenceId.isPresent()) {
                recurrenceId = nextRecurrenceId;

            } else {
                Option<InstantInterval> first = RepetitionUtils.getFirstInstanceInterval(repetition);
                Option<Instant> minRecurrenceId = repetition.getRecurIds().minO();

                if (first.isPresent() && !minRecurrenceId.exists(first.get().getStart()::isAfter)) {
                    instanceInterval = first;
                } else {
                    recurrenceId = minRecurrenceId;
                }
            }
        }

        if (instanceInterval.isPresent()) {
            EventInfo eventInfo = eventInfoDbLoader.getEventInfoByEvent(uidO, event, actionSource);
            return toSingleInstance(uidO, eventInfo, instanceInterval.get());
        }
        if (recurrenceId.isPresent()) {
            RecurrenceIdOrMainEvent id = RecurrenceIdOrMainEvent.recurrenceId(recurrenceId.get());
            Event recurrence = findEventByMainEventIdAndRecurrence(event.getMainEventId(), id, lockForUpdate).get();

            EventInfo eventInfo = eventInfoDbLoader.getEventInfoByEvent(uidO, recurrence, actionSource);
            return toSingleInstance(uidO, eventInfo, getInstantInterval(recurrence));
        }
        throw CommandRunException.createSituation(
                "not a valid start of event instance: " + startO.getOrNull(), Situation.EVENT_NOT_FOUND);
    }

    public EventInstanceInfo getRecurrenceInstanceAsOccurrence(
            Option<PassportUid> uidO, long recurrenceEventId, ActionSource actionSource)
    {
        Event recurrence = eventDbManager.getEventByIdSafe(recurrenceEventId).getOrThrow(
                CommandRunException.createSituationF("event not found by id " + recurrenceEventId, Situation.EVENT_NOT_FOUND));

        Instant recurrenceId = recurrence.getRecurrenceId().getOrThrow("expected recurrence event id");

        Event master = getMainInstOfEvent(recurrence).getOrThrow(CommandRunException.createSituationF(
                "master event not found for recurrence " + recurrenceEventId, Situation.EVENT_NOT_FOUND));

        EventInfo masterEventInfo = eventInfoDbLoader.getEventInfosByEvents(uidO, Cf.list(master), actionSource).single();
        RepetitionInstanceInfo repetitionInfo = masterEventInfo.getRepetitionInstanceInfo();

        if (!RepetitionUtils.isValidStart(repetitionInfo.withoutRecurrence(recurrenceId), recurrenceId)) {
            throw CommandRunException.createSituation(
                    "not a valid start of event instance: " + recurrenceId, Situation.EVENT_NOT_FOUND);
        }
        DateTime recurrenceIdDt = recurrence.getRecurrenceId().get().toDateTime(masterEventInfo.getTimezone());
        Period period = masterEventInfo.getRepetitionInstanceInfo().getEventPeriod();

        return toSingleInstance(uidO, masterEventInfo, new InstantInterval(recurrenceIdDt, recurrenceIdDt.plus(period)));
    }

    private EventInstanceInfo toSingleInstance(
            Option<PassportUid> uidO, EventInfo eventInfo, InstantInterval instanceInterval)
    {
        val event = eventInfo.getEventWithRelations();
        val layerId = uidO.toOptional()
                .flatMap(uid -> findUserLayerWithEvent(uid, event).toOptional())
                .map(EventLayerWithRelations::getLayerId);

        return new EventInstanceInfo(
                instanceInterval, eventInfo.getInfoForPermsCheck(),
                eventInfo.getIndentAndRepetition(), eventInfo.getEvent(),eventInfo.getMainEvent(),
                eventInfo.getRepetitionInfoO(), eventInfo.getEventParticipantsO(),
                eventInfo.getResourcesO(), eventInfo.getEventWithRelationsO(),
                eventInfo.getEventUserWithNotifications(), eventInfo.getAttachmentO(),
                Option.x(layerId), Option.empty(), eventInfo.mayView());
    }

    public EventInstanceForUpdate getSingleEventInstanceForModifierOrCreateRecurrence(
            Option<PassportUid> uidO, long eventId, Instant instanceStart, Option<Long> layerIdO, ActionInfo actionInfo)
    {
        EventInstanceInfo instance = getSingleInstance(
                uidO, Option.of(instanceStart), false, true, eventId, actionInfo.getActionSource());

        if (!instance.getRepInstInfo().isEmpty()) {
            long recurrenceEventId = createNotChangedRecurrence(
                    uidO.getOrElse(instance.getEvent().getCreatorUid()),
                    instance.toEventInfo(), instanceStart, actionInfo);

            return getEventInstanceForModifier(
                    uidO, Option.<Instant>empty(), recurrenceEventId, layerIdO, actionInfo);
        } else {
            return getEventInstanceForModifier(uidO, Option.<Instant>empty(), instance.toEventInfo(), layerIdO);
        }
    }

    public EventInstanceForUpdate getEventInstanceForModifier(
            Option<PassportUid> uidO, Option<Instant> startTsO, long eventId,
            Option<Long> layerIdO, ActionInfo actionInfo)
    {
        Event event = eventDbManager.getEventByIdSafeForUpdate(eventId).getOrThrow(
                CommandRunException.createSituationF("event not found by id " + eventId, Situation.EVENT_NOT_FOUND));

        EventInfo eventInfo = eventInfoDbLoader.getEventInfoByEvent(uidO, event, actionInfo.getActionSource());
        return getEventInstanceForModifier(uidO, startTsO, eventInfo, layerIdO);
    }

    public EventInstanceForUpdate getEventInstanceForModifier(
            Option<PassportUid> uidO, Option<Instant> startTsO, EventInfo eventInfo, Option<Long> layerIdO)
    {
        eventInfo.getEvent().validateIsLockedForUpdate();

        EventWithRelations event = eventInfo.getEventWithRelations();

        Option<InstantInterval> interval = Option.empty();
        if (startTsO.isPresent()) {
            interval = RepetitionUtils.getInstanceIntervalStartingAt(eventInfo.getRepetitionInstanceInfo(), startTsO.get());
            interval = Option.of(interval.getOrThrow(CommandRunException.createSituationF(
                    "not a valid start of event instance: " + startTsO.get(), Situation.EVENT_NOT_FOUND)));
        }
        Option<EventLayer> eventLayer = Option.empty();
        if (layerIdO.isPresent()) {
            eventLayer = event.getEventLayers().find(EventLayer.getLayerIdF().andThenEquals(layerIdO.get()));
        } else if (uidO.isPresent()) {
            eventLayer = findUserLayerWithEvent(uidO.get(), event).map(EventLayerWithRelations::getEventLayer);
        }
        return new EventInstanceForUpdate(
                interval, eventInfo.getInfoForPermsCheck(),
                event, eventInfo.getRepetitionInstanceInfo(),
                eventInfo.getEventUserWithNotifications(), eventLayer, eventInfo.getAttachments());
    }

    /**
     * For the specified user, obtains all logical event instances
     * that appear in given interval, in the sorted order.
     * These logical event instances are computed with the help
     * of two queries (that select physical event instances from the
     * database using certain conditions).
     * The first query pulls out non-repeating / non-regularly repeating events
     * that overlap the interval, while the second query gets all the
     * repeating events (that haven't met due timestamp yet) and applies
     * smart regular repetition algorithm which returns all the
     * logical instances that appear in the interval.
     * All the instances are sorted in the way that comparable EventInstanceInfo specifies.
     * NOTE: overlap == true.
     * @param uidO specifies user (can be null/ANYONE, if layers given)
     * (uid -> looks at -> smb's (e.g. uid's) layers)
     * @param startMs checking interval start (left bound, inclusive)
     * @param endMsO checking interval end (right bound, not inclusive)
     * @param actionSource
     * @return set of EventInfos sorted as its comparable part specifies
     */
    public ListF<EventInstanceInfo> getSortedInstancesIMayView(Option<PassportUid> uidO,
            Instant startMs, Option<Instant> endMsO, LayerIdPredicate layerIds, ActionSource actionSource)
    {
        EventLoadLimits limits = EventLoadLimits.intersectsInterval(new InfiniteInterval(startMs, endMsO));
        return getSortedInstancesIMayView(uidO, EventGetProps.any(), layerIds, limits, actionSource);
    }

    public ListF<EventInstanceInfo> getSortedInstancesIMayView(Option<PassportUid> uidO,
            EventGetProps egp, LayerIdPredicate layerIdPredicate, EventLoadLimits limits, ActionSource actionSource)
    {
        ListF<Long> layerIds = getLayerIds(layerIdPredicate);

        Function<ListF<EventInfo>, ListF<EventInfo>> filterF =
                eis -> eis.filter(e -> e.mayView() && !e.isRejected());

        ListF<EventInfo> events = loadEvents(limits,
                (lims) -> eventInfoDbLoader.getEventInfosOnLayers(uidO, egp, layerIds, lims, actionSource),
                EventInfo::getEventId, filterF);

        return toEventInstanceInfos(events, limits);
    }

    public ListF<EventInstanceInfo> getSortedInstancesIMayView(
            Option<UserInfo> userO, EventGetProps egp, ListF<Layer> layers,
            EventLoadLimits limits, EventsFilter filter, ActionSource actionSource)
    {
        if (!egp.isWithEventParticipants()) {
            ListF<EventIndentIntervalAndPerms> indents = getSortedIndentsFiltered(
                    userO, Either.right(layers), limits, filter, actionSource);

            ListF<EventInfo> eventInfos = eventInfoDbLoader.getEventInfosByIndentsAndPerms(
                    userO, egp, indents.map(EventIndentIntervalAndPerms::getIndentAndRepetitionAndPerms), actionSource);

            MapF<Long, EventInfo> eventInfoById = eventInfos.toMapMappingToKey(EventInfo::getEventId);

            return indents.map(indent -> toEventInstanceInfoFromLayer(
                    indent.getIndentInterval(), eventInfoById.getOrThrow(indent.getEventId())));

        } else {
            return getSortedInstancesFiltered(userO, egp, layers, limits, filter, actionSource);
        }
    }

    public ListF<EventIndentIntervalAndPerms> getSortedIndentsIMayView(
            Option<UserInfo> userO, LayerIdPredicate layerIdPredicate,
            EventLoadLimits limits, EventsFilter filter, ActionSource actionSource)
    {
        return getSortedIndentsFiltered(userO, Either.left(layerIdPredicate), limits, filter, actionSource);
    }

    private ListF<EventInstanceInfo> getSortedInstancesFiltered(
            Option<UserInfo> userO, EventGetProps egp, ListF<Layer> layers,
            EventLoadLimits limits, EventsFilter filter, ActionSource actionSource)
    {
        SetF<Tuple2<Long, Instant>> processedInstances = Cf.hashSet();
        MapF<Long, PassportUid> layerCreatorById = layers.toMap(Layer::getId, Layer::getCreatorUid);
        Function<ListF<EventInfo>, ListF<EventInfo>> filterF = es -> {
            if (filter.mergeLayers()) {
                es = es.filter(e -> processedInstances.add(Tuple2.tuple(e.getEventId(), e.getIndent().getStart())));
            }
            if (userO.isPresent()) {
                if (filter.opaqueOnly()) {
                    es = filterOpaqueOnlyEvents(userO, filter, es);
                } else if (!filter.includeDeclined()) {
                    es = filterOutDeclinedForAuthorized(layerCreatorById, es);
                }
            }
            es = es.filter(e -> e.mayView()
                    || filter.getNoPermsCheckLayerId().isSome(e.getIndent().getLayerOrResourceId()));
            return es;
        };
        ListF<EventInfo> events = loadEvents(limits,
                (lims) -> eventInfoDbLoader.getEventInfosOnLayers(
                        userO.map(UserInfo::getUid), egp, layerCreatorById.keys(), lims, actionSource),
                EventInfo::getEventId, filterF);
        return toEventInstanceInfos(events, limits);
    }

    private ListF<EventInfo> filterOutDeclinedForAuthorized(MapF<Long, PassportUid> layerCreatorById,
                                                            ListF<EventInfo> es) {
        es = es.filterNot(e -> e.getEventParticipants()
                .getEventUser(layerCreatorById.getOrThrow(e.getIndent().getLayerOrResourceId()))
                .exists(eu -> eu.getEventUser().getDecision() == Decision.NO));
        return es;
    }

    private ListF<EventInfo> filterOpaqueOnlyEvents(Option<UserInfo> userO, EventsFilter filter, ListF<EventInfo> es) {
        es = es.filter(e -> {
            Option<EventUserWithRelations> eu = e.getEventParticipants().getEventUser(userO.get().getUid());

            Option<Availability> avail = eu.map(u -> u.getEventUser().getAvailability());
            Option<Decision> decision = eu.map(u -> u.getEventUser().getDecision());

            return avail.exists(a -> a == Availability.BUSY || a == Availability.MAYBE)
                    && (filter.includeDeclined() || !decision.isSome(Decision.NO));
        });
        return es;
    }

    private List<EventIndentAndRepetitionAndPerms> joinInfoForPermsCheck(UserInfo user, List<EventIndentAndRepetition> indents) {
        val ids = StreamEx.of(indents)
            .map(EventIndentAndRepetition::getEventId)
            .collect(CollectorsF.toList());
        val eventsFieldsForPermsCheck = eventDao.findEventsFieldsForPermsCheckByIdsSafe(ids);
        val infos = authorizer.loadInfoForPermsCheck(user, eventsFieldsForPermsCheck, Cf.list());
        val infoByEventId = StreamEx.of(infos).toMap(EventInfoForPermsCheck::getEventId, identity());

        return StreamEx.of(indents)
            .map(i -> new EventIndentAndRepetitionAndPerms(i, requireNonNull(infoByEventId.get(i.getEventId()))))
            .toImmutableList();
    }

    private List<EventIndentAndRepetitionAndPerms> joinInfoForPermsCheck(List<EventIndentAndRepetition> indents) {
        val ids = StreamEx.of(indents)
            .map(EventIndentAndRepetition::getEventId)
            .collect(CollectorsF.toList());
        val eventsFieldsForPermsCheck = eventDao.findEventsFieldsForPermsCheckByIdsSafe(ids);
        val infos = authorizer.loadInfoForPermsCheck(eventsFieldsForPermsCheck, Cf.list());
        val infoByEventId = StreamEx.of(infos).toMap(EventInfoForPermsCheck::getEventId, identity());

        return StreamEx.of(indents)
            .map(i -> new EventIndentAndRepetitionAndPerms(i, requireNonNull(infoByEventId.get(i.getEventId()))))
            .toImmutableList();
    }

    private ListF<EventIndentIntervalAndPerms> getSortedIndentsFiltered(
            Option<UserInfo> userO, Either<LayerIdPredicate, ListF<Layer>> layers,
            EventLoadLimits limits, EventsFilter filter, ActionSource actionSource)
    {
        Function<EventIndentAndRepetition, Long> eventIdF = EventIndentAndRepetition::getEventId;

        ListF<Long> layerIds = layers.fold(this::getLayerIds, ls -> ls.map(Layer::getId));

        Function0<MapF<Long, PassportUid>> layerCreatorById = Cf2.f0(() -> layers.fold(
                ids -> layerDao.findLayerCreatorUids(layerIds).toMap(),
                ls -> ls.toMap(Layer::getId, Layer::getCreatorUid))).memoize();

        SetF<Tuple2<Long, Instant>> processedInstances = Cf.hashSet();

        Function<ListF<EventIndentAndRepetition>, ListF<EventIndentAndRepetitionAndPerms>> filterF = es -> {
            if (es.isNotEmpty() && filter.mergeLayers()) {
                es = es.filter(i -> processedInstances.add(Tuple2.tuple(i.getEventId(), i.getIndent().getStart())));
            }
            if (es.isNotEmpty() && userO.isPresent() && filter.opaqueOnly()) {
                SqlCondition c = EventUserFields.AVAILABILITY.column()
                        .inSet(Cf.list(Availability.BUSY, Availability.MAYBE));

                if (!filter.includeDeclined()) {
                    c = c.and(EventUserFields.DECISION.ne(Decision.NO));
                }
                ListF<Long> matchedIds = eventUserDao.findEventUserEventIds(c
                        .and(EventUserFields.EVENT_ID.column().inSet(es.map(eventIdF)))
                        .and(EventUserFields.UID.eq(userO.get().getUid())));

                es = es.filter(eventIdF.andThen(matchedIds.unique().containsF()));

            } else if (es.isNotEmpty() && userO.isPresent() && !filter.includeDeclined()) {
                SetF<Tuple2<Long, PassportUid>> declinedIds = eventUserDao.findEventUserEventIdsAndUids(
                        EventUserFields.DECISION.eq(Decision.NO)
                                .and(EventUserFields.EVENT_ID.column().inSet(es.map(eventIdF)))
                                .and(EventUserFields.UID.column().inSet(layerCreatorById.apply().values()))).unique();

                es = es.filterNot(e -> declinedIds.containsTs(Tuple2.tuple(
                        e.getEventId(), layerCreatorById.apply().getOrThrow(e.getIndent().getLayerOrResourceId()))));
            }
            if (es.isEmpty()) return Cf.list();


            val copy = Cf.list(es);

            ListF<EventIndentAndRepetitionAndPerms> result = Cf.toList(userO.map(user -> joinInfoForPermsCheck(user, copy))
                    .getOrElse(() -> joinInfoForPermsCheck(copy)));

            if (filter.getNoPermsCheckLayerId().isPresent()) {
                long layerId = filter.getNoPermsCheckLayerId().get();

                Tuple2<ListF<EventIndentAndRepetition>, ListF<EventIndentAndRepetition>> partition = es.partition(
                        i -> layerId == i.getIndent().getLayerOrResourceId());

                if (partition.get2().isNotEmpty()) {
                    SetF<Long> idsCanView = partition.get1().map(eventIdF).unique();

                    result = result.filter(p -> idsCanView.containsTs(p.getEventId())
                            || userO.map(user -> authorizer.canViewEvent(user, p.getPermInfo(), actionSource))
                    .getOrElse(() -> authorizer.canViewEvent(p.getPermInfo(), actionSource)));
                }
            } else if (es.isNotEmpty()) {
                result = result.filter(p -> userO.map(user -> authorizer.canViewEvent(user, p.getPermInfo(), actionSource))
                .getOrElse(() -> authorizer.canViewEvent(p.getPermInfo(), actionSource)));
            }
            return result;
        };
        ListF<EventIndentAndRepetitionAndPerms> events = loadEvents(limits,
                (lims) -> eventInfoDbLoader.getEventIndentsOnLayers(layerIds, lims), eventIdF, filterF);

        ListF<EventIndentIntervalAndPerms> intervals = events
                .flatMap(e -> getIntervals(e.getRepetitionInfo(), limits).map(EventIndentIntervalAndPerms.consF(e)));

        return intervals.takeSorted(
                EventIndentInterval.comparator.compose(EventIndentIntervalAndPerms::getIndentInterval),
                limits.getResultSizeLimit().getOrElse(Integer.MAX_VALUE));
    }

    private <A, B> ListF<B> loadEvents(
            EventLoadLimits limits, Function<EventLoadLimits, ListF<A>> loadF,
            Function<A, Long> eventIdF, Function<ListF<A>, ListF<B>> mapFilterF)
    {
        if (limits.hasResultSizeLimit()) {
            EventLoadLimits resizedLimits = limits.withResultSize(Math.max(limits.getResultSizeLimit().get(), 20));

            ListF<Long> exceptIds = Cf.toArrayList(limits.getExceptIds());
            ListF<B> result = Cf.arrayList();
            ListF<A> events;
            do {
                events = loadF.apply(resizedLimits.withExcludeIds(exceptIds));

                exceptIds.addAll(events.map(eventIdF));
                result.addAll(mapFilterF.apply(events));
            } while (result.size() < limits.getResultSizeLimit().get()
                    && events.size() >= resizedLimits.getResultSizeLimit().get());

            return result;
        } else {
            return mapFilterF.apply(loadF.apply(limits));
        }
    }

    /**
     * Accepts 'overlap' parameter
     */
    public ListF<EventInstanceInfo> getSortedInstancesOnLayer(Option<PassportUid> uidO,
            EventGetProps egp, LayerIdPredicate layerIdPredicate, EventLoadLimits limits, ActionSource actionSource)
    {
        if (!limits.getResultSizeLimit().isPresent() && !limits.getStartsInOrBeforeLimit().isPresent()) {
            throw new IllegalArgumentException("ER.getSortedInstancesInner(): unlimited query w/o endMs requested");
        }
        ListF<Long> layerIds = getLayerIds(layerIdPredicate);
        // Get full information about db-events (as they are, no division into instances)
        ListF<EventInfo> eventInfos = eventInfoDbLoader.getEventInfosOnLayers(
                uidO, egp, layerIds, limits, actionSource);
        // Fill all information about found events (simple or repeating) instances
        // with corresponding event environment (event users, notifications etc.)

        return toEventInstanceInfos(eventInfos, limits);
    }

    /**
     * Accepts 'overlap' parameter
     */
    public ListF<EventInstanceInfo> getSortedInstancesOnResource(Option<PassportUid> uidO,
            EventGetProps egp, ListF<Long> resourceIds, EventLoadLimits limits, ActionSource actionSource)
    {
        if (!limits.getResultSizeLimit().isPresent() && !limits.getStartsInOrBeforeLimit().isPresent()) {
            throw new IllegalArgumentException("ER.getSortedInstancesInner(): unlimited query w/o endMs requested");
        }
        ListF<EventInfo> eventInfos = eventInfoDbLoader.getEventInfosOnResources(
                uidO, egp, resourceIds, limits, actionSource);

        return toEventInstanceInfos(eventInfos, limits);
    }

    private ListF<EventInstanceInfo> toEventInstanceInfos(
            ListF<EventInfo> eventInfos, EventLoadLimits limits)
    {
        ListF<EventInstanceInfo> result = Cf.arrayList();

        for (EventInfo eventInfo : eventInfos) {
            for (InstantInterval eInterval : getIntervals(eventInfo.getRepetitionInstanceInfo(), limits)) {
                result.add(new EventInstanceInfo(
                        eInterval, eventInfo.getInfoForPermsCheck(),
                        eventInfo.getIndentAndRepetition(), eventInfo.getEvent(),
                        eventInfo.getMainEvent(), eventInfo.getRepetitionInfoO(),
                        eventInfo.getEventParticipantsO(), eventInfo.getResourcesO(), eventInfo.getEventWithRelationsO(),
                        eventInfo.getEventUserWithNotifications(), eventInfo.getAttachmentO(),
                        eventInfo.getLayerId(), eventInfo.getResourceId(), eventInfo.mayView()));
            }
        }

        return result.takeSorted(
                new EventInstanceInfoComparator(), limits.getResultSizeLimit().getOrElse(Integer.MAX_VALUE));
    }

    public static EventInstanceInfo toEventInstanceInfoFromLayer(EventIndentInterval indent, EventInfo eventInfo) {
        return new EventInstanceInfo(
                indent.getInterval(), eventInfo.getInfoForPermsCheck(),
                eventInfo.getIndentAndRepetition(), eventInfo.getEvent(), eventInfo.getMainEvent(),
                eventInfo.getRepetitionInfoO(), eventInfo.getEventParticipantsO(), eventInfo.getResourcesO(),
                eventInfo.getEventWithRelationsO(), eventInfo.getEventUserWithNotifications(),
                eventInfo.getAttachmentO(), Option.of(indent.getIndent().getLayerOrResourceId()), Option.empty(),
                eventInfo.mayView());
    }

    private ListF<InstantInterval> getIntervals(RepetitionInstanceInfo repetitionInfo, EventLoadLimits limits) {
        Option<Instant> startMsO = limits.getStartsInOrAfterLimit().orElse(limits.getEndsInOrAfterLimit());
        Option<Instant> endMsO = limits.getStartsInOrBeforeLimit();
        boolean overlap = !limits.getStartsInOrAfterLimit().isPresent();

        int limit = limits.getResultSizeLimit().getOrElse(Integer.MAX_VALUE);

        return RepetitionUtils.getIntervals(
                repetitionInfo, startMsO.getOrElse(repetitionInfo.getEventStart()), endMsO, overlap, limit);
    }

    public Option<EventUser> findEventUser(PassportUid uid, long eventId) {
        return findEventUsersByUidAndEventIds(uid, Cf.set(eventId)).singleO();
    }

    public ListF<EventUser> findEventUsersByUidAndEventIds(PassportUid uid, SetF<Long> eventIds) {
        SqlCondition c = SqlCondition.all(
            EventUserFields.UID.eq(uid.getUid()),
            EventUserFields.EVENT_ID.column().inSet(eventIds)
        );
        return eventUserDao.findEventUsers(c);
    }

    public EventXmlCreationInfo getEventXmlCreationInfo(long eId) {
        Event e = eventDao.findEventById(eId);
        return new EventXmlCreationInfo(e, repetitionRoutines.getRepetitionInstanceInfoByEvent(e));
    }

    // XXX ssytnik: this is strange method.
    public Interval getIntervalByEvent(Event e) {
        DateTimeZone chrono = getEventsTimeZones(Cf.list(e)).single().get2();
        return new Interval(e.getStartTs().getMillis(), e.getEndTs().getMillis(), chrono);
    }

    public DateTimeZone getEventTimeZone(long eventId) {
        return getEventsTimeZones(Cf.list(eventDao.findEventById(eventId))).single().get2();
    }

    public Tuple2List<Long, DateTimeZone> getEventsTimeZones(ListF<Event> events) {
        MapF<Long, DateTimeZone> byMainEventId = mainEventDao
                .findTimezoneIdsByMainEventIds(events.map(Event.getMainEventIdF()))
                .map2(AuxDateTime.getVerifyDateTimeZoneF()).toMap();

        return events.toTuple2List(Event.getIdF(), e -> byMainEventId.getOrThrow(e.getMainEventId()));
    }

    // XXX this is wrong, we need to use event creator uid + note indexing issue
    public Tuple2List<Event, EventLayer> findEventsAndEventLayersByExtId(long layerId, String extId) {
        Validate.isTrue(layerId > 0);
        Validate.notEmpty(extId);

        BeanRowMapper<Event> eventRm = EventHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<EventLayer> eventLayerRm = EventLayerHelper.INSTANCE.offsetRowMapper(eventRm.nextOffset());
        // see https://jira.yandex-team.ru/browse/CAL-1800
        String q =
            "SELECT " + eventRm.columns("e.") + ", " + eventLayerRm.columns("el.") +
            " FROM event e " +
            "INNER JOIN event_layer el ON e.id = el.event_id " +
            "INNER JOIN main_event me ON e.main_event_id = me.id " +
            "WHERE me.external_id = ? AND el.layer_id = ?";
        return getJdbcTemplate().query2(q, eventRm, eventLayerRm, extId, layerId);
    }

    // XXX this is wrong, we need to use event creator uid + note indexing issue
    public Instant findLastModifiedByExtId(long layerId, String extId) {
        Validate.notEmpty(extId);
        String q =
            "SELECT MAX(e.last_update_ts), COUNT(*) " +
            "FROM event e " +
            "INNER JOIN event_layer el ON e.id = el.event_id " +
            "INNER JOIN main_event me ON e.main_event_id = me.id " +
            "WHERE me.external_id = ? and el.layer_id = ?";
        Tuple2<Instant, Integer> r = getJdbcTemplate().queryForObject(
            q, TwoColumnRowMapper.cons(Instant.class, Integer.class), extId, layerId
        );
        Check.C.isTrue(r._2 > 0, "event with id " + extId + " not found on layer " + layerId);
        return r._1;
    }

    public int findEventCount(SqlCondition... conditions) {
        SqlCondition c = SqlCondition.all(conditions);
        return getJdbcTemplate().queryForInt("SELECT COUNT(*) FROM event WHERE " + c.sql(), c.args());
    }

    // XXX fix (redundant event load, join) and merge with findLayerIdByCreatorUidAndEventId
    /**
     * Tries to find any instance (main or recurrence) of the event,
     * which user takes part in. If such instance exists, we identify
     * layer, in which user has this instance, and return it.
     * Here we assume that two conditions apply:
     * 1. all events with the same external_id lie in a SINGLE layer;
     * 2. event can only be found in an only layer: one, created by this user.
     * @return nullable layer id, in which user (uid) has event (eId)
     */
    public Option<Long> getOwnLayerId(PassportUid uid, long eventId) {
        Event event = eventDao.findEventById(eventId);
        return eventInvitationDao.findLayerByUidAndMainEventId(uid, event.getMainEventId())
            .map(LayerFields.ID.getF());
    }

    /*
    public long save(PassportUid uid, Event event, String externalId, ActionInfo actionInfo) {
        long mainEventId = getMainEventIdBySubjectsIdAndExternalIdOrCreateNew(
                UidOrResourceId.user(uid), Cf.<Email>list(), externalId, actionInfo.getNow());
        event.setMainEventId(mainEventId);
        if (event.isFieldSet(EventFields.ID)) {
            eventDbManager.updateEvent(event, actionInfo);
        } else {
            long id = eventDbManager.saveEvent(event);
            event.setId(id);
        }
        return event.getId();
    }
    */

    public static boolean getApplyToFuture(DataProvider dp) {
        DataProvider eDp = dp.getDataProvider("event", true);
        return Binary.parseBoolean(eDp.getText("apply-to-future-events", false));
    }

    public static Instant calculateDueTsFromUntilDate(Instant startTs, LocalDate dueDate, DateTimeZone tz) {
        DateTime startDateTime = new DateTime(startTs, tz);
        DateTime dueDateTime = dueDate.toDateTime(startDateTime.toLocalTime(), tz);
        dueDateTime = dueDateTime.plusDays(1); // due-ts is exclusive in calendar
        return dueDateTime.toInstant();
    }

    public static LocalDate convertDueTsToUntilDate(Instant dueTs, DateTimeZone tz) {
        DateTime dueTsInclusive = new DateTime(dueTs, tz).minusDays(1); // due-ts is exclusive in calendar
        return dueTsInclusive.toLocalDate();
    }

    public static Instant calculateDueTsFromIcsUntilTs(Instant startTs, Instant dueTs, DateTimeZone tz) {
        return calculateDueTsFromUntilDate(startTs, new DateTime(dueTs, tz).toLocalDate(), tz);
    }

    public Option<Event> findEventBySubjectIdAndExternalIdAndRecurrenceId(
            UidOrResourceId subjectId, String externalId, Option<Instant> recurrenceId)
    {
        return findEvent(Cf.list(subjectId), externalId, recurrenceId);
    }

    public Option<Event> findEventByMainEventIdAndRecurrence(
            long mainEventId, RecurrenceIdOrMainEvent recurrenceIdOrMainEvent)
    {
        return findEventByMainEventIdAndRecurrence(mainEventId, recurrenceIdOrMainEvent, false);
    }

    public Option<Event> findEventByMainEventIdAndRecurrence(
            long mainEventId, RecurrenceIdOrMainEvent recurrenceIdOrMainEvent, boolean forUpdate)
    {
        ListF<Event> list = eventDao.findEventByMainIdAndRecurrenceId(mainEventId, recurrenceIdOrMainEvent, forUpdate);
        if (list.size() > 1) {
            log.warn("More than one event found by main event id {} and recurrence id {}", mainEventId, recurrenceIdOrMainEvent);
        }
        return list.firstO();
    }

    public Option<Event> findEventByExternalIdAndRecurrence(String externalId, Option<Instant> recurrenceId) {
        ListF<Event> list = eventDao.findEventsByExternalIdAndRecurrenceId(externalId, recurrenceId);
        if (list.size() > 1) {
            log.warn("More than one event found by external id {} and recurrence id {}", externalId, recurrenceId);
        }
        return list.firstO();
    }

    public Option<Event> findEvent(
            ListF<UidOrResourceId> uids, String externalId, Option<Instant> recurrenceId)
    {
        for (UidOrResourceId uid : uids) {
            final Option<Event> eventO;
            if (recurrenceId.isPresent()) {
                eventO = getRecurrenceEventInstanceBySubjectIdAndExternalId(
                        uid, externalId, recurrenceId.get());
            } else {
                eventO = findMasterEventBySubjectIdAndExternalId(uid, externalId);
            }
            if (eventO.isPresent()) {
                return eventO;
            }
        }
        return Option.empty();
    }

    // XXX ssytnik@: works properly only for organizer's exchangeId.
    // Otherwise, like findEventDeletionTsByExchangeId does, need to
    // look in deleted_* tables and then check if such event exists.
    public Option<Event> findEventByExchangeId(String exchangeId) {
        Option<Long> eventIdO = eventResourceDao.findEventIdByResourceExchangeId(exchangeId);
        if (!eventIdO.isPresent()) {
            eventIdO = eventUserDao.findEventIdByUserExchangeId(exchangeId);
        }
        return eventIdO.map(new Function<Long, Event>() {
            public Event apply(Long eventId) {
                return eventDao.findEventById(eventId);
            }
        });
    }

    public ListF<Long> findRelatedEvents(PassportUid uid) {
        ListF<Long> byCreatorUid = eventDao.findEventsByCreatorUid(uid).map(EventFields.ID.getF());
        ListF<Long> byEventUser = eventUserDao.findEventUsersByUid(uid).map(EventUserFields.EVENT_ID.getF());

        ListF<Long> layerIds = layerDao.findLayersByUid(Cf.list(uid)).map(Layer.getIdF());
        ListF<Long> byEventLayer = eventLayerDao.findEventLayerEventIdsByLayerIds(layerIds);

        ListF<Long> eventIds = Cf.<Long>set()
            .plus(byCreatorUid)
            .plus(byEventUser)
            .plus(byEventLayer)
            .toList()
            ;
        return eventIds;
    }

    // XXX ssytnik@: by the time, it's important to know that event doesn't exist (is everybody care of?)
    public Option<Tuple3<Instant, Long, ActionSource>> findLatestEventDeletionInfoByExchangeId(String exchangeId) {
        Option<DeletedEventResource> deletedEeventResource = deletedEventDao.findLatestDeletedEventResourceByResourceExchangeId(exchangeId);
        if (deletedEeventResource.isPresent()) {
            return Option.of(Tuple3.tuple(deletedEeventResource.get().getDeletionTs(),
                    deletedEeventResource.get().getEventId(), deletedEeventResource.get().getDeletionSource()));
        }
        Option<DeletedEventUser> deletedEventUser = deletedEventDao.findLatestDeletedEventUserByUserExchangeId(exchangeId);
        if (deletedEventUser.isPresent()) {
            return Option.of(Tuple3.tuple(deletedEventUser.get().getDeletionTs(),
                    deletedEventUser.get().getEventId(), deletedEventUser.get().getDeletionSource()));
        }
        return Option.empty();
    }

    @Override
    public PassportUid getCreatorUid(long id) {
        String sql = "SELECT creator_uid FROM event WHERE id = ?";
        return new PassportUid(getJdbcTemplate().queryForOption(sql, Long.class, id).get());
    }

    @Override
    protected Option<EventLayer> findOwnerBean(PassportUid uid, long id) {
        return eventDbManager.getEventLayerForEventAndUser(id, uid);
    }

    // If uid is null (unknown) or uid == eCreatorUid, looks for the primary layer of this event.
    // Otherwise, looks for the layer at which user ('uid') put this MEETING event to.
    // NOTE: this method does not require layer to be of 'user' type.
    public long findLayerIdSafe(Option<PassportUid> uidO, Event e) {
        boolean isCreator = !uidO.isPresent() || e.getCreatorUid().sameAs(uidO.get());
        return findLayerIdSafe(uidO, e.getId(), isCreator);
    }
    // Looks for the layer at which event with given id lies. If fails, attempts to find its primary layer.
    private long findLayerIdSafe(Option<PassportUid> uidO, long eId, boolean isCreator) {
        if (!uidO.isPresent() && !isCreator) {
            throw new IllegalArgumentException("uid cannot be null for non-creator case");
        }

        if (isCreator) {
            val primaryLayerId = eventDbManager.getEventWithRelationsById(eId)
                    .getPrimaryLayer()
                    .map(Layer::getId);
            if (primaryLayerId.isPresent()) {
                return primaryLayerId.get();
            }
        }

        Option<EventLayer> beanO = eventDbManager.getEventLayerForEventAndUser(eId, uidO.get());
        if (beanO.isPresent()) {
            return beanO.get().getLayerId();
        }
        if (isCreator) {
            throw new CommandRunException("isCreator == true");
        }
        // data not found for non-creator user - so, let's find creator data instead
        return findLayerIdSafe(Option.<PassportUid>empty(), eId, true);
    }

    public Option<String> existsExchangeIdBySubjectIdAndEventId(UidOrResourceId subjectId, long eventId) {
        if (subjectId.isResource()) {
            return eventResourceDao
                    .findEventResourceByEventIdAndResourceId(eventId, subjectId.getResourceId())
                    .filterMap(EventResourceFields.EXCHANGE_ID.getF().andThen(
                            (Function<String,Option<String>>) Option::ofNullable));
        } else {
            return eventUserDao
                    .findEventUserByEventIdAndUid(eventId, subjectId.getUid())
                    .filterMap(EventUserFields.EXCHANGE_ID.getF().andThen(Cf2.f(Option::ofNullable)));
        }
    }

    public boolean saveUpdateExchangeId(UidOrResourceId subjectId, long eventId, String exchangeId, ActionInfo actionInfo) {
        if (subjectId.isResource()) {
            return eventResourceDao.saveUpdateExchangeId(subjectId.getResourceId(), eventId, exchangeId, actionInfo);
        } else {
            return eventUserDao.saveUpdateExchangeId(subjectId.getUid(), eventId, exchangeId, actionInfo);
        }
    }

    /**
     * @param currentResources resources of the event before update
     * @param removedParticipants all participants, who was removed during the update
     * @param wasNewResources was added some new resources to the event during the update
     * @return whether event will have attached resources after update (true) or no (false)
     */
    static boolean willHaveResourcesAfterUpdate(List<ResourceInfo> currentResources, List<ParticipantInfo> removedParticipants, boolean wasNewResources) {
        val removedResources = StreamEx.of(removedParticipants)
                .select(ResourceParticipantInfo.class)
                .map(ResourceParticipantInfo::getId)
                .map(ParticipantId::getResourceId)
                .toImmutableSet();
        val hasStillNotRemovedResources = StreamEx.of(currentResources)
                .map(ResourceInfo::getResourceId)
                .anyMatch(id -> !removedResources.contains(id));
        return hasStillNotRemovedResources || wasNewResources;
    }

    public void updateEventCommon(
            Option<PassportUid> clientUid, EventInstanceForUpdate eventInstance,
            EventChangesInfo eventChangesInfo, UpdateMode updateMode, ActionInfo actionInfo)
    {
        long eventId = eventInstance.getEvent().getId();
        EventWithRelations event = eventInstance.getEventWithRelations();

        validateEventStartTsEndTs(eventChangesInfo.getEventChanges());

        Option<TelemostManager.PatchResult> telemost;

        if (eventChangesInfo.wasNonPerUserFieldsChange()) {
            Event eventData = new Event();
            eventData.setId(eventId);
            eventData.setFields(eventChangesInfo.getEventChanges());

            if (passportAuthDomainsHolder.containsYandexTeamRu() && willHaveResourcesAfterUpdate(
                    eventInstance.getEventWithRelations().getResources(),
                    eventChangesInfo.getEventParticipantsChangesInfo().getRemovedParticipants(),
                    eventChangesInfo.getEventParticipantsChangesInfo().wasNewResources())) {
                eventData.setLocation(""); // CAL-6280
            }
            if (eventData.getFieldValueO(EventFields.SEQUENCE)
                    .exists(s -> eventInstance.getEvent().getSequence() > s))
            {
                eventData.unsetField(EventFields.SEQUENCE);
            }

            if (clientUid.isPresent()
                && !actionInfo.getActionSource().isInvitationMailsFree()
                && eventInstance.getEventWithRelations().isMeeting())
            {
                Event data = new Event();
                data.setFields(event.getEvent());
                data.setFields(eventChangesInfo.getEventChanges());

                userManager.checkPublicUserKarma(KarmaCheckAction.update(clientUid.get(), data, actionInfo));
            }

            Repetition repetitionOverrides = eventChangesInfo.getRepetitionChanges();

            if (repetitionOverrides.isNotEmpty()) {
                 Option<Long> newRepetitionIdO;
                 Option<Long> currentRepetitionIdO = eventInstance.getEvent().getRepetitionId();
                if (repetitionOverrides.isFieldSet(RepetitionFields.TYPE) &&
                    repetitionOverrides.getType() == RegularRepetitionRule.NONE)
                {
                    if (currentRepetitionIdO.isPresent()) {
                        eventDao.deleteRepetitionsByIds(Cf.list(currentRepetitionIdO.get()));
                    }
                    newRepetitionIdO = Option.empty();
                } else {
                    newRepetitionIdO = Option.of(
                        repetitionRoutines.createOrUpdateRepetition(currentRepetitionIdO, repetitionOverrides)
                    );
                }
                eventData.setRepetitionId(newRepetitionIdO);
            }
            // rdates && exdates
            repetitionRoutines.updateRdates(eventId, eventChangesInfo.getRdateChangesInfo(), actionInfo);

            updateEventAttachments(eventInstance.getEvent().getId(), eventChangesInfo.getAttachmentChangesInfo());

            telemost = Option.of(telemostManager.patchUpdateData(
                    eventData, eventInstance, eventChangesInfo.getEventParticipantsChangesInfo(), updateMode
            ));
            eventDbManager.updateEvent(eventData, actionInfo);
        } else {
            telemost = Option.empty();
        }

        Option<UserInfo> userO = userManager.getUserInfos(clientUid).singleO();
        val userIsRoomAdmin = userO.isPresent() && authorizer.canAdminSomeEventResources(userO.get(), event.getResourcesForPermsCheck());
        val existsOnUserLayers = clientUid.isPresent() && findUserLayerWithEvent(clientUid.get(), event).isPresent();

        boolean isParticipant = clientUid.isPresent()
                && (eventChangesInfo.getEventParticipantsChangesInfo().wasNewParticipantWithUid(clientUid.get())
                        || event.getParticipants().isParticipantWithInconsistent(clientUid.get()));

        // Common ("level 3") modification: update event_layer/event_user linked to event
        if (clientUid.isPresent()
            && (!userIsRoomAdmin || existsOnUserLayers || isParticipant)
            && !event.isParkingOrApartmentOccupation())
        {
            UserInfo user = userO.get();

            if (eventChangesInfo.eventLayerChanges() && eventInstance.getEventLayer().isPresent()) {
                long oldLayerId = eventInstance.getEventLayer().get().getLayerId();
                long newLayerId = eventChangesInfo.getNewLayerId().get();

                ListF<Long> layerIds = Cf.list(oldLayerId, newLayerId);

                if (actionInfo.getActionSource().isWebOrApi()
                        || layerRoutines.getLayersById(layerIds).stableUniqueBy(Layer::getCreatorUid).size() == 1)
                {
                    updateEventLayerByMainEventId(
                            user, eventInstance.getEvent().getMainEventId(),
                            oldLayerId, newLayerId, actionInfo);
                }
            }

            long eventUserId = eventUserRoutines.createOrUpdateEventUser(
                    clientUid.get(), eventId, eventChangesInfo.getEventUserChanges(), actionInfo);

            Option<Decision> decision = eventChangesInfo
                    .getEventUserChanges().getFieldValueO(EventUserFields.DECISION)
                    .orElse(event.findUserEventUser(clientUid.get()).map(EventUser.getDecisionF()));

            if (!decision.isSome(Decision.NO) && !existsOnUserLayers) {
                createOrUpdateEventLayer(
                        user, eventChangesInfo.getNewLayerId(), event.getEvent().getSid(),
                        eventDbManager.getEventAndRepetitionByIdForUpdate(eventId),
                        event.getEvent().getType(), false, actionInfo);
            }

            notificationDbManager.updateAndRecalcEventNotifications(
                    eventUserId, eventChangesInfo.getNotificationsUpdate(), actionInfo);
        }

        telemost.ifPresent((result) -> telemostManager.onEventUpdated(result, eventId, actionInfo));

        if (eventChangesInfo.timeOrRepetitionChanges()) {
            notificationRoutines.recalcAllNextSendTs(eventId, actionInfo);
        }

        if (eventChangesInfo.wasPerUserFieldsChange()) {
            lastUpdateManager.updateTimestampsAsync(eventInstance.getMainEventId(), actionInfo);
        }
    }

    public UpdateInfo updateEventFromIcsOrExchange(
            UidOrResourceId subjectId, EventData eventData, NotificationsData.Update notificationsUpdate,
            EventInstanceStatusInfo eventInstanceStatusInfo, SequenceAndDtStamp userVersion, ActionInfo actionInfo)
    {
        long eventId = eventData.getEvent().getId();

        EventInstanceForUpdate eventInstance = getEventInstanceForModifier(
                subjectId.getUidO(), Option.empty(), eventId, Option.empty(), actionInfo);
        EventChangesInfo eventChangesInfo = eventChangesFinder.getEventChangesInfo(
                eventInstance, eventData, notificationsUpdate, subjectId.getUidO(), false, false);

        ActionSource actionSource = actionInfo.getActionSource();

        boolean subjectIsEwsOrganizer = eventInstance.getEventWithRelations().isExportedWithEws()
                && (eventInstance.getEventWithRelations().getOrganizerUidIfMeeting().equals(subjectId.getUidO()));

        if (actionSource.isFromExchangeOrMail() && !subjectIsEwsOrganizer) {
            eventChangesInfo = eventChangesInfo.withOnlySubjectMightBeRemoved(subjectId);
        }
        if (actionSource.isFromExchange() && subjectId.isResource()) {
            eventChangesInfo = eventChangesInfo.withoutDeletedExdates();
        }
        if (actionSource.isFromExchange()) {
            removeFutureInstancesIfTimeOrRepetitionChanges(
                    ActorId.userOrResource(subjectId), eventInstance, eventChangesInfo, false, actionInfo);

            val newExdates = StreamEx.of(eventChangesInfo.getRdateChangesInfo().getNewRdates())
                    .remove(Rdate::getIsRdate)
                    .map(Rdate::getStartTs)
                    .collect(CollectorsF.toList());

            ListF<Instant> exdatedRecurrences = eventInstance.getRepetitionInstanceInfo().getRecurrences()
                    .filterMap(rc -> Option.when(newExdates.containsTs(rc.getRecurrenceId()), rc.getRecurrenceId()));

            deleteEvents(ActorId.userOrResource(subjectId), eventDao.findRecurrenceEventsByMainId(
                    eventInstance.getMainEventId(), exdatedRecurrences, false).map(Event::getId), actionInfo, false, false);
        }

        return updateEvent(ActorId.userOrResource(subjectId), eventData.getEventUserData().getDecision(),
                eventInstance, eventChangesInfo, eventInstanceStatusInfo, userVersion, Option.empty(), actionInfo);
    }

    public Tuple2<ListF<Long>, ListF<EventMessageParameters>> removeFutureInstancesIfTimeOrRepetitionChanges(
            ActorId actorId, EventInstanceForUpdate instance,
            EventChangesInfo eventChangesInfo, boolean prepareMails, ActionInfo actionInfo)
    {
        Event event = instance.getEvent();
        if (event.getRecurrenceId().isPresent()) {
            return Tuple2.tuple(Cf.list(), Cf.list());
        }
        Option<Event> master = Option.of(event);

        if (eventChangesInfo.timeChanges()) {
            return removeFutureInstances(actorId, instance, master, Option.empty(), prepareMails, actionInfo);
        }

        if (eventChangesInfo.repetitionChangesIgnoreRdates()) {
            Option<Instant> fromO = !eventChangesInfo.repetitionRuleChanges()
                    ? Option.of(eventChangesInfo.getRepetitionChanges()
                            .getDueTs().getOrElse(RepetitionUtils.END_OF_EVERYTHING_TS))
                    : Option.empty();
            return removeFutureInstances(actorId, instance, master, fromO, prepareMails, actionInfo);
        }

        return Tuple2.tuple(Cf.list(), Cf.list());
    }

    public Tuple2<ListF<Long>, ListF<EventMessageParameters>> removeFutureInstances(
            ActorId actorId, EventInstanceForUpdate instance,
            Option<Event> master, Option<Instant> fromO, boolean prepareMails, ActionInfo actionInfo)
    {
        Event event = instance.getEvent();

        // remove future instances of event
        // 1. remove future RDATEs
        Instant from = fromO.getOrElse(new Instant(0)); // hack
        eventDao.deleteFutureRdates(master.get().getId(), from);

        // 2. remove all events from future with RECURRENCE_ID
        ListF<Long> eventIds = eventDao.findEventIdsByMainEventIdAndRecurrenceIdGe(event.getMainEventId(), from);
        ListF<EventMessageParameters> mails;

        if (prepareMails) {
            ListF<EventWithRelations> events = eventDbManager.getEventsWithRelationsByIds(eventIds);
            mails = deleteEventsSafe(
                    Option.of(actorId), InvitationProcessingMode.SAVE_ATTACH_SEND, actionInfo, events, false, false);
        } else {
            deleteEvents(actorId, eventIds, actionInfo, true, false);
            mails = Cf.list();
        }

        ListF<Long> removedIds = Cf.arrayList();
        removedIds.addAll(eventIds);
        removedIds.add(master.get().getId());

        return Tuple2.tuple(removedIds, mails);
    }

    public UpdateInfo updateEvent(
            ActorId subjectId, Option<Decision> subjectDecision,
            EventInstanceForUpdate eventInstance, EventChangesInfo eventChangesInfo,
            EventInstanceStatusInfo eventInstanceStatusInfo, SequenceAndDtStamp userVersion,
            Option<ExchangeMails> mails, ActionInfo actionInfo)
    {
        long eventId = eventInstance.getEventId();

        updateEventCommon(subjectId.getUidO(), eventInstance, eventChangesInfo, UpdateMode.NORMAL, actionInfo);

        EventParticipantsChangesInfo participantsChangesInfo = eventChangesInfo.getEventParticipantsChangesInfo();

        Event updatedInstance = eventInstance.getEvent().copy();
        updatedInstance.setFields(eventChangesInfo.getEventChanges());

        Option<EventInstanceParameters> exdateInstanceParameters = Option.empty();
        if (eventChangesInfo.isSameWithNewExdate()) { // CAL-6136
            val exdate = eventChangesInfo.getRdateChangesInfo().getNewRdates().get(0).getStartTs();
            Instant endTs = repetitionRoutines.calcEndTs(eventInstance.getEvent(), exdate);

            exdateInstanceParameters = Option.of(new EventInstanceParameters(exdate, endTs, Option.of(exdate)));
        }

        final ListF<EventSendingInfo> sendingInfos;
        RejectedResources rejectedResources = RejectedResources.EMPTY;

        boolean isExportedWithEws = eventInstance.getEventWithRelations().isExportedWithEws();

        if (exdateInstanceParameters.isPresent()) {
            sendingInfos = cancelMeetingHandler.cancelMeeting(
                    eventInstance.getEvent(), subjectId.getUidO(),
                    exdateInstanceParameters.get(),
                    eventInstance.getEventWithRelations().isParkingOrApartmentOccupation(),
                    isExportedWithEws, actionInfo);
        } else {
            EventInvitationResults eventInvitationResults = updateMeetingHandler.handleMeetingUpdate(
                    updatedInstance, subjectId,
                    new ChangedEventInfoForMails(eventInstance, eventChangesInfo.toEventChangesInfoForMails()),
                    eventChangesInfo.getMeetingMailRecipients(),
                    eventInstance.getEventWithRelations().isParkingOrApartmentOccupation(),
                    actionInfo, eventInstanceStatusInfo, UpdateMode.NORMAL, isExportedWithEws);
            sendingInfos = eventInvitationResults.getSendingInfos();
            rejectedResources = eventInvitationResults.getRejectedResources();
        }

        // patched copy-paste from EventInvitationManager.updateUserParticipantDecision()
        if (!eventInstanceStatusInfo.isFromCaldavAlreadyUpdated() && subjectId.isUser()) {
            eventInvitationManager.updateEventUserSequenceAndDtstamp(
                    subjectId.getUid(), eventId, userVersion, actionInfo);
        }

        lastUpdateManager.updateTimestampsAsync(eventInstance.getMainEventId(), actionInfo);

        boolean indentsChanged = eventChangesInfo.timeOrRepetitionChanges();

        boolean checkExchangeCompatibleAndResourcesAvailable =
                indentsChanged || participantsChangesInfo.wasResourcesChange();

        EventAndRepetition updatedEvAndRep = eventDbManager.getEventAndRepetitionById(eventId, indentsChanged);
        EventWithRelations updatedEvent = eventDbManager.getEventWithRelationsByEvent(updatedEvAndRep.getEvent());
        RepetitionInstanceInfo updatedRepetition = updatedEvAndRep.getRepetitionInfo();

        Option<NameI18n> subjectResourceRejectReason = Option.empty();

        if (checkExchangeCompatibleAndResourcesAvailable) {
            resourceRoutines.lockResourcesByIds(indentsChanged
                    ? updatedEvent.getResourceIds()
                    : participantsChangesInfo.getNewResources());

            ensureExchangeCompatibleIfNeeded(subjectId, updatedEvent);

            rejectedResources = rejectedResources
                    .add(resourceAccessRoutines.checkForInaccessibleResources(
                            subjectId, updatedEvent, updatedRepetition,
                            ResourceAccessRoutines.InaccessibilityCheck.onUpdate(eventChangesInfo),
                            actionInfo))
                    .add(resourceAccessRoutines.checkForBusyResources(
                            subjectId, updatedEvent, updatedRepetition, Cf.list(), actionInfo));

            if (subjectId.isResource()) {
                subjectResourceRejectReason = rejectedResources.getReason(subjectId.getResourceId());
            }
        }

        if (subjectId.isResource()) {
            Decision subjectResourceDecision = subjectResourceRejectReason.isPresent() ? Decision.NO : Decision.YES;
            ewsExportRoutines.setResourceDecisionIfNeeded(
                    subjectId.getResourceId(), updatedEvent, updatedRepetition,
                    subjectDecision, subjectResourceDecision, subjectResourceRejectReason, actionInfo);
        }

        if (subjectId.isUser()) {
            fixDecisionInExchangeIfNeeded(
                    subjectId.getUid(), updatedEvent, updatedRepetition, subjectDecision, actionInfo);
        }

        if (indentsChanged) {
            eventDbManager.updateEventLayersAndResourcesIndents(updatedEvAndRep, actionInfo);
        }
        if (eventChangesInfo.wasNonPerUserFieldsChange()) {
            repetitionConfirmationManager.rescheduleConfirmation(
                    updatedEvent, updatedRepetition, actionInfo);
        }
        invalidateResourceScheduleCachesOnUpdate(eventInstance.getEventWithRelations(), participantsChangesInfo);

        final ListF<EventOnLayerChangeMessageParameters> layerNotifyMails;

        if (exdateInstanceParameters.isPresent()) {
            layerNotifyMails = eventsOnLayerChangeHandler.handleEventDelete(
                    subjectId, updatedEvent, updatedRepetition, exdateInstanceParameters.get(), actionInfo);
        } else {
            layerNotifyMails = eventsOnLayerChangeHandler.handleEventUpdate(
                    subjectId, eventInstance, updatedEvent, updatedRepetition,
                    eventChangesInfo.toEventChangesInfoForMails(), actionInfo);
        }

        Option<OccurrenceId> occurrenceId = eventInstance.getEventWithRelations().getOccurrenceId();
        synchronizeWithExchangeOnUpdate(updatedEvent, updatedRepetition, occurrenceId, eventChangesInfo, mails, actionInfo);

        if (rejectedResources.isNotEmpty()) {
            ewsExportRoutines.forceUpdateEventForOrganizer(updatedEvent, updatedRepetition, actionInfo);
        }

        if (eventChangesInfo.timeOrRepetitionRuleOrDueTsChanges()) {
            eventInvitationManager.sendDecisionFixingMailsIfNeeded(
                    updatedEvent, updatedEvAndRep.getRepetitionInfo(), actionInfo);
        }

        eventsLogger.log(EventChangeLogEvents.updated(subjectId,
                eventInstance.getEventWithRelations(), eventInstance.getRepetitionInstanceInfo(),
                updatedEvent, updatedRepetition), actionInfo);

        return new UpdateInfo(
                eventInstance.getEventWithRelations().getLayerIds().plus(updatedEvent.getLayerIds()),
                sendingInfos, layerNotifyMails);
    }

    public void invalidateResourceScheduleCachesOnUpdate(
            EventWithRelations event, EventParticipantsChangesInfo eventParticipantsChangesInfo)
    {
        ListF<Long> resourcesWithEvent = event.getParticipants().getResourceIdsSafeWithInconsistent();

        ListF<Long> newAndRemoved = eventParticipantsChangesInfo.getNewAndRemovedResources();

        resourceScheduleManager.invalidateCachedScheduleForResources(
                resourcesWithEvent.plus(newAndRemoved).stableUnique());
    }

    public void invalidateResourceScheduleCachesOnDelete(EventWithRelations event) {
        ListF<Long> resourceIds = event.getParticipants().getResourceIdsSafeWithInconsistent();

        resourceScheduleManager.invalidateCachedScheduleForResources(resourceIds);
    }


    public boolean eventHasConflicts(PassportUid uid, EventData eventData, ActionInfo actionInfo) {
        ParticipantsData participantsData = eventData.getInvData().getParticipantsDataO()
                .getOrThrow("need to know all resource participants to find conflicts!");
        ListF<ParticipantId> participantIds = participantsData.getParticipantsSafe()
                .map(eventInvitationManager.getParticipantIdByEmailF().compose(ParticipantData.getEmailF()));
        ListF<Long> resourceIds = participantIds.flatMap(ParticipantId.getResourceIdIfResourceF());

        RepetitionInstanceInfo repetitionInfo = RepetitionInstanceInfo.create(
                new InstantInterval(eventData.getEvent().getStartTs(), eventData.getEvent().getEndTs()),
                dateTimeManager.getTimeZoneForUid(uid), Option.of(eventData.getRepetition()));

        return findFirstResourceEventIntersectingGivenInterval(
                Option.of(uid), resourceIds, Option.<Long>empty(), repetitionInfo, true, actionInfo).isPresent();
    }

    public ListF<Long> findMasterAndSingleEventIds(long eventId) {
        return eventDao.findEventIdsByMainEventIds(eventDao.findMainEventIdByEventId(eventId));
    }

    public ListF<Long> findMasterAndSingleEventIds(ListF<Long> eventIds) {
        return eventDao.findEventIdsByMainEventIds(eventDao.findMainEventIdsByEventIds(eventIds));
    }

    public ListF<Long> findEventIdsByLayerIdsAndExternalIds(ListF<Long> layerIds, ListF<String> externalIds) {
        return eventDao.findEventIdsByMainEventIds(mainEventDao.findMainEventIdsOnLayers(layerIds, externalIds));
    }

    public ListF<Long> findMasterAndFutureSingleEventIds(long eventId, ActionInfo actionInfo) {
        return eventDao.findFutureOrMasterEventIdsByMainEventIds(eventDao.findMainEventIdByEventId(eventId), actionInfo);
    }

    public ListF<Long> findMasterAndSingleEventsLayerIds(long eventId) {
        return eventLayerDao.findLayersIdsByEventIds(findMasterAndSingleEventIds(eventId));
    }

    public ListF<Long> findLayerIdsByEventExternalId(String externalId) {
        ListF<MainEvent> mainEvents = mainEventDao.findMainEventsByExternalId(new ExternalId(externalId));
        return eventLayerDao.findLayersIdsByEventIds(
                eventDao.findEventIdsByMainEventIds(mainEvents.map(MainEvent.getIdF())));
    }

    public void saveEventAttachments(long eventId, ListF<EventAttachment> attachments) {
        attachments = attachments.map(EventAttachment.copyF());
        for (EventAttachment attachment : attachments) {
            attachment.setEventId(eventId);
        }
        eventDao.insertEventAttachments(attachments);
    }

    public void updateEventAttachments(long eventId, EventAttachmentChangesInfo changes) {
        if (!changes.getRemoveAttachmentUrls().isEmpty()) {
            eventDao.deleteEventAttachmentsByUrls(eventId, Cf.toList(changes.getRemoveAttachmentUrls()));
        }
        if (!changes.getNewAttachments().isEmpty()) {
            saveEventAttachments(eventId, Cf.toList(changes.getNewAttachments()));
        }
    }

    public Option<EventLayerWithRelations> findUserLayerWithEvent(PassportUid uid, EventWithRelations event) {
        return findUserLayerWithEvent(uid, Option.empty(), event);
    }

    public Option<EventLayerWithRelations> findUserLayerWithEvent(
            PassportUid uid, Option<Long> layerId, EventWithRelations event)
    {
        Option<EventLayerWithRelations> targetEventLayer = Option.empty();

        Lazy<ListF<Long>> userLayerIds = Lazy.withSupplier(
                () -> layerRoutines.getLayerIdsByUid(uid));

        if (layerId.isPresent()) {
            if (event.findLayerById(layerId.get()).exists(l -> l.getCreatorUid().sameAs(uid))) {
                targetEventLayer = Option.of(event.findEventLayerWithRelationsById(layerId.get()).get());

            } else if (userLayerIds.apply().containsTs(layerId.get())) {
                targetEventLayer = event.findEventLayerWithRelationsById(layerId.get());
            }
        }
        if (!targetEventLayer.isPresent()) {
            targetEventLayer = event.findOwnUserLayerWithRelations(uid);
        }
        if (!targetEventLayer.isPresent()) {
            targetEventLayer = userLayerIds.get().iterator().filterMap(event::findEventLayerWithRelationsById).nextO();
        }
        return targetEventLayer;
    }

    public Option<EventLayerWithRelations> findNotDeclinedUserLayerWithEvent(PassportUid uid, EventWithRelations event) {
        Function1B<EventLayerWithRelations> isDeclined =
                el -> event.findUserEventUser(el.getLayerCreatorUid()).exists(eu -> eu.getDecision() == Decision.NO);

        return event.findOwnUserLayerWithRelations(uid).filterNot(isDeclined)
                .orElse(() -> layerRoutines.getLayerIdsByUid(uid).iterator()
                        .filterMap(lid -> event.findEventLayerWithRelationsById(lid).filterNot(isDeclined)).nextO());
    }

    public ListF<Long> getLayerIds(LayerIdPredicate layerIdPredicate) {
        if (layerIdPredicate.isAllForUsers() || layerIdPredicate.isOnlyVisibleInUi()) {
            SqlCondition c = layerIdPredicate.isAllForUsers() ?
                LayerUserFields.UID.column().inSet(layerIdPredicate.getUids()) :
                LayerUserFields.LAYER_ID.column().inSet(layerIdPredicate.getIds());

            if (layerIdPredicate.isOnlyVisibleInUi()) {
                c = c.and(LayerUserFields.IS_VISIBLE_IN_UI.column().eq(true));
            }

            ListF<LayerUser> layerUsers = layerRoutines.getLayerUsers(c);
            return layerUsers.map(LayerUser.getLayerIdF());
        } else {
            return layerIdPredicate.getIds();
        }
    }

    private void validateEventStartTsEndTs(Event event) {
        InstantInterval everything =
                new InstantInterval(RepetitionUtils.START_OF_EVERYTHING_TS, RepetitionUtils.END_OF_EVERYTHING_TS);

        Option<Instant> startTsO = event.getFieldValueO(EventFields.START_TS);
        Option<Instant> endTsO = event.getFieldValueO(EventFields.END_TS);

        Validate.isTrue(startTsO.forAll(everything::contains), "startTs " + startTsO + " unacceptable");
        Validate.isTrue(endTsO.forAll(everything::contains), "endTs " + endTsO + " unacceptable");
    }

    public void setEwsExportRoutinesForTest(EwsExportRoutines ewsExportRoutines) {
        this.ewsExportRoutines = ewsExportRoutines;
    }

    public ListF<Event> findEventsByLayerIdInTimeRange(Long layerId, Instant from, Instant to) {
        return eventDao.findEventsByLayerIdInTimeRange(layerId, from, to);
    }
}
