package ru.yandex.calendar.logic.event;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.generated.EventNotification;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.model.Completion;
import ru.yandex.calendar.logic.event.model.EventUserCreature;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.event.model.EventUserUpdate;
import ru.yandex.calendar.logic.event.model.Priority;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.notification.Channel;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationDao;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.base.UidGen;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class EventUserRoutines {
    private static final Logger logger = LoggerFactory.getLogger(EventUserRoutines.class);

    private static final Duration YAMAIL_NOTIFICATION_OFFSET = Duration.ZERO; // CAL-6223
    private static final Duration MOBILE_NOTIFICATION_OFFSET = Duration.standardMinutes(-5);
    private static final Duration PANEL_NOTIFICATION_OFFSET = Duration.standardMinutes(-15); // PP-61

    @Autowired
    private NotificationRoutines notificationRoutines;
    @Autowired
    private NotificationDao notificationDao;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private SettingsRoutines settingsRoutines;


    private static final EventUser EVENT_USER_DEFAULTS = new EventUser();
    static {
        EVENT_USER_DEFAULTS.setLastSentTsNull();
        EVENT_USER_DEFAULTS.setLastSentSmsIdNull();
        EVENT_USER_DEFAULTS.setTotalSentCount(0);

        EVENT_USER_DEFAULTS.setAvailability(Availability.AVAILABLE);
        EVENT_USER_DEFAULTS.setCompletion(Completion.NOT_APPLICABLE);
        EVENT_USER_DEFAULTS.setPriority(Priority.NORMAL);
        EVENT_USER_DEFAULTS.setDecision(Decision.YES);

        EVENT_USER_DEFAULTS.setIsAttendee(false);
        EVENT_USER_DEFAULTS.setIsOptional(false);
        EVENT_USER_DEFAULTS.setIsOrganizer(false);
        EVENT_USER_DEFAULTS.setIsSubscriber(false);

        EVENT_USER_DEFAULTS.setPrivateTokenNull();
        EVENT_USER_DEFAULTS.setReasonNull();
        EVENT_USER_DEFAULTS.setSequence(0);
        EVENT_USER_DEFAULTS.setDtstampNull();
        EVENT_USER_DEFAULTS.setLastUpdate(new DateTime(2000, 1, 1, 0, 0, DateTimeZone.UTC).toInstant());

        EVENT_USER_DEFAULTS.setExchangeIdNull();

        EVENT_USER_DEFAULTS.setCreationSource(ActionSource.UNKNOWN);
        EVENT_USER_DEFAULTS.setModificationSource(ActionSource.UNKNOWN);
        EVENT_USER_DEFAULTS.setCreationReqIdNull();
        EVENT_USER_DEFAULTS.setModificationReqIdNull();

        EVENT_USER_DEFAULTS.setIcalXMozSnoozeTimeNull();
        EVENT_USER_DEFAULTS.setIcalXMozLastackNull();
    }

    public ListF<EventUserWithRelations> findEventUsersWithRelationsByEventIds(ListF<Long> eventIds, boolean withSubscribers) {
        ListF<EventUserWithRelations> eus = eventUserDao.findEventUsersWithRelationsByEventIds(eventIds, withSubscribers);

        eus.forEach(eu -> settingsRoutines.putInCache(eu.getSettings()));

        return eus;
    }

    public ListF<EventUserWithRelations> findEventUsersWithRelationsByEventIds(ListF<Long> eventIds) {
        return findEventUsersWithRelationsByEventIds(eventIds, true);
    }

    public Option<EventUser> findEventUser(long eventId, PassportUid uid) {
        return eventUserDao.findEventUserByEventIdAndUid(eventId, uid);
    }

    public Option<Decision> findEventUserDecision(PassportUid uid, long eventId) {
        return eventUserDao.findEventUsersByEventIdsAndUid(Cf.list(eventId), uid).singleO().map(EventUser.getDecisionF());
    }

    public EventUserUpdate saveEventUserAndNotification(
            PassportUid uid, long eventId,
            EventUser newEventUser, NotificationsData.Create notificationsData, long layerIdForNotification,
            ActionInfo actionInfo)
    {
        EventAndRepetition event = eventDbManager.getEventAndRepetitionById(eventId);
        return saveEventUserAndNotification(uid, event, newEventUser, notificationsData, layerIdForNotification, actionInfo);
    }

    public EventUserUpdate saveEventUserAndNotification(
            PassportUid uid, EventAndRepetition event,
            EventUser newEventUser, NotificationsData.Create notificationsData, long layerIdForNotification,
            ActionInfo actionInfo)
    {
        ListF<Notification> notifications = getNotifications(uid, notificationsData, layerIdForNotification);

        Validate.isTrue(notificationsData.isCreate());

        return saveEventUser(uid, event, newEventUser, notifications, actionInfo);
    }

    private ListF<Notification> getNotifications(PassportUid uid,
            NotificationsData.Create notificationsData, long layerIdForNotification)
    {
        if (notificationsData.isUseLayerDefaultIfCreate()) {
            return layerRoutines.getNotificationsOrGetDefault(uid, layerIdForNotification);

        } else if (notificationsData.asCreateWithData().getKnownChannels().isPresent()) {
            SetF<Channel> known = notificationsData.asCreateWithData().getKnownChannels().get();

            ListF<Notification> defaults = layerRoutines.getNotificationsOrGetDefault(uid, layerIdForNotification)
                    .filterNot(n -> known.containsTs(n.getChannel()));

            return notificationsData.asCreateWithData().getNotifications().plus(defaults);

        } else {
            return notificationsData.asCreateWithData().getNotifications();
        }
    }

    public long saveEventUser(PassportUid uid, long eventId, EventUser newEventUser,
            ListF<Notification> notifications, ActionInfo actionInfo)
    {
        EventAndRepetition event = eventDbManager.getEventAndRepetitionById(eventId);
        return saveEventUser(uid, event, newEventUser, notifications, actionInfo).cur.get().getId();
    }

    public EventUserCreature eventUserSaveData(
            PassportUid uid, EventAndRepetition event,
            EventUser eventUserData, ListF<Notification> notifications, ActionInfo actionInfo)
    {
        EventUser eventUserFields = new EventUser();
        eventUserFields.setEventId(event.getEventId());
        eventUserFields.setUid(uid);
        eventUserFields.setFields(eventUserData);

        eventUserFields.setCreationSource(actionInfo.getActionSource());
        eventUserFields.setCreationReqId(actionInfo.getRequestIdWithHostId());
        eventUserFields.setFieldValueDefault(EventUserFields.PRIVATE_TOKEN, UidGen.createPrivateToken());

        return new EventUserCreature(Either.left(
                new EventUserData(eventUserFields, plusAutoNotifications(uid, notifications))));
    }

    public EventUserCreature eventUserUpdateData(
            long eventUserId, EventUser newEventUserData, ActionInfo actionInfo)
    {
        EventUser eventUserFields = new EventUser();
        eventUserFields.setId(eventUserId);
        eventUserFields.setFields(newEventUserData);

        eventUserFields.setModificationSource(actionInfo.getActionSource());
        eventUserFields.setModificationReqId(actionInfo.getRequestIdWithHostId());

        return new EventUserCreature(Either.right(eventUserFields));
    }

    public EventUserUpdate saveEventUser(
            PassportUid uid, EventAndRepetition event,
            EventUser eventUserData, ListF<Notification> notifications, ActionInfo actionInfo)
    {
        EventUserCreature creature = eventUserSaveData(uid, event, eventUserData, notifications, actionInfo);
        EventUser eventUserFields = creature.createOrUpdate.getLeft().getEventUser();

        eventUserFields.setId(eventUserDao.saveEventUser(eventUserFields, actionInfo));

        notificationRoutines.recalcAndSaveEventNotifications(
                eventUserFields.getId(), creature.createOrUpdate.getLeft().getNotifications(), event, actionInfo);

        eventUserFields.setFieldDefaults(EVENT_USER_DEFAULTS);

        return new EventUserUpdate(Option.empty(), Option.of(eventUserFields));
    }

    public void updateEventUserAndNotification(long eventUserId, EventUser newEventUserData,
            NotificationsData.Update notifications, ActionInfo actionInfo)
    {
        updateEventUserInner(eventUserId, newEventUserData, actionInfo);

        notificationDbManager.updateAndRecalcEventNotifications(eventUserId, notifications, actionInfo);
    }

    public EventUserUpdate updateEventUser(EventUser oldEventUser, EventUser newEventUserData, ActionInfo actionInfo) {
        EventUser updated = oldEventUser.copy();
        updated.setFields(updateEventUserInner(oldEventUser.getId(), newEventUserData, actionInfo));

        return new EventUserUpdate(Option.of(oldEventUser), Option.of(updated));
    }

    private Tuple2List<MapField<?>, ?> updateEventUserInner(
            long eventUserId, EventUser newEventUserData, ActionInfo actionInfo)
    {
        EventUserCreature creature = eventUserUpdateData(eventUserId, newEventUserData, actionInfo);

        eventUserDao.updateEventUser(creature.createOrUpdate.getRight(), actionInfo);

        return creature.createOrUpdate.getRight().getFieldValues();
    }

    public void updateEventUserAvailability(
            PassportUid uid, long eventId, Availability availability, ActionInfo actionInfo)
    {
        eventUserDao.updateAvailability(eventId, uid, availability, actionInfo);
    }

    public ListF<EventUserUpdate> updateEventUsersDecision(
            ListF<Long> eventIds, PassportUid uid, Decision decision, ActionInfo actionInfo)
    {
        ListF<EventUser> eventUsers = eventUserDao.findEventUsersByEventIdsAndUid(eventIds, uid);

        eventUserDao.updateEventUserDecisionByEventIdsAndUid(
                eventUsers.map(EventUser::getEventId), uid, decision, actionInfo);

        return eventUsers.map(eu -> {
            EventUser updated = eu.copy();
            updated.setDecision(decision);

            updated.setLastUpdate(actionInfo.getNow());
            updated.setModificationSource(actionInfo.getActionSource());
            updated.setModificationReqId(actionInfo.getRequestIdWithHostId());

            return new EventUserUpdate(Option.of(eu), Option.of(updated));
        });
    }

    public EventUserUpdate createOrUpdateEventUserAndCreateNotifications(
            PassportUid uid, EventAndRepetition event, EventUser eventUser,
            NotificationsData.Create notifications, long layerIdForNotification, ActionInfo actionInfo)
    {
        Option<EventUser> oldEventUser = eventUserDao.findEventUserByEventIdAndUid(event.getEventId(), uid);
        if (!oldEventUser.isPresent()) {
            return saveEventUserAndNotification(
                    uid, event, eventUser, notifications, layerIdForNotification, actionInfo);

        } else {
            long eventUserId = oldEventUser.get().getId();
            Tuple2List<MapField<?>, ?> updatedFields = updateEventUserInner(eventUserId, eventUser, actionInfo);

            if (notifications.isCreateWithData()) {
                ListF<Notification> ns = plusAutoNotifications(
                        uid, getNotifications(uid, notifications, layerIdForNotification));

                notificationDbManager.updateAndRecalcEventNotifications(
                        eventUserId, NotificationsData.updateChannels(Cf.set(Channel.values()), ns), actionInfo);
            }

            EventUser updated = oldEventUser.get().copy();
            updated.setFields(updatedFields);
            updated.setFieldDefaults(EVENT_USER_DEFAULTS);

            return new EventUserUpdate(oldEventUser, Option.of(updated));
        }
    }

    public long createOrUpdateEventUser(PassportUid uid, long eventId, EventUser eventUser, ActionInfo actionInfo) {
        EventAndRepetition event = eventDbManager.getEventAndRepetitionById(eventId);
        return createOrUpdateEventUser(uid, event, eventUser, actionInfo);
    }

    public long createOrUpdateEventUser(PassportUid uid, EventAndRepetition event, EventUser eventUser, ActionInfo actionInfo) {
        Option<EventUser> oldEventUser = eventUserDao.findEventUserByEventIdAndUid(event.getEventId(), uid);

        if (!oldEventUser.isPresent()) {
            return saveEventUser(uid, event, eventUser, Cf.list(), actionInfo).cur.get().getId();

        } else {
            return updateEventUser(oldEventUser.get(), eventUser, actionInfo).cur.get().getId();
        }
    }

    public ListF<EventUserWithRelations> filterNotDismissedNotDeclinedEventUsersHaveSmsNotification(
            CollectionF<EventUserWithRelations> eventUsers)
    {
        ListF<Long> eventUserIds = eventUsers.map(EventUserWithRelations::getId);

        SetF<Long> smsNotificationEventUserIds = notificationDao
                .findEventNotificationsByEventUserIdsAndChannel(eventUserIds, Channel.SMS)
                .map(EventNotification::getEventUserId).unique();

        eventUsers = eventUsers.filter(eu -> smsNotificationEventUserIds.containsTs(eu.getId()));

        eventUsers = eventUsers.filter(eu -> eu.getEventUser().getDecision() != Decision.NO);

        eventUsers = eventUsers.filterNot(eu -> eu.getSettingsYt().exists(SettingsYt::getIsDismissed));

        return eventUsers.toList();
    }

    private ListF<Notification> plusAutoNotifications(PassportUid uid, ListF<Notification> notifications) {
        return notifications
                .plus(Option.when(uid.isYandexTeamRu(), new Notification(Channel.XIVA, YAMAIL_NOTIFICATION_OFFSET)))
                .plus(Option.when(!uid.isYandexTeamRu(), new Notification(Channel.PANEL, PANEL_NOTIFICATION_OFFSET)))
                .plus(Option.when(uid.isYandexTeamRu(), new Notification(Channel.MOBILE, MOBILE_NOTIFICATION_OFFSET)));
    }

    public static Notification autoPanelNotification() {
        return new Notification(Channel.PANEL, PANEL_NOTIFICATION_OFFSET);
    }
} //~
