package ru.yandex.calendar.logic.event;

import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.ews.exp.OccurrenceId;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventInvitation;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.event.repetition.RecurrenceTimeInfo;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.participant.ExternalUserParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.ResourceInfoForPermsCheck;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.mail.micronaut.common.Lazy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;

@Slf4j
public class EventWithRelations {
    private final EventTime time;
    private final Event event;
    private final MainEvent mainEvent;
    private final EventParticipants eventParticipants;
    private final EventLayers eventLayers;

    public EventWithRelations(
            Event event, MainEvent mainEvent,
            EventParticipants eventParticipants,
            EventLayers eventLayers)
    {
        this.time = new EventTime(event.getStartTs(), event.getEndTs(), event.getIsAllDay(), mainEvent.getTimezoneId());
        this.event = event;
        this.mainEvent = mainEvent;
        this.eventParticipants = eventParticipants;
        this.eventLayers = eventLayers;
    }

    public Event getEvent() {
        return event;
    }

    public long getId() {
        return event.getId();
    }

    public Instant getStartTs() {
        return time.getStartTs();
    }

    public Instant getEndTs() {
        return time.getEndTs();
    }

    public EventInterval getEventInterval(InstantInterval interval) {
        return time.getEventInterval(interval);
    }

    public EventDateTime getEventDateTime(Instant startOrEnd) {
        return time.getEventDateTime(startOrEnd);
    }

    public MainEvent getMainEvent() {
        return mainEvent;
    }

    public CollectionF<EventUserWithRelations> getEventUsersWithRelation() {
        return eventParticipants.getEventUsers();
    }

    public ListF<EventUser> getEventUsers() {
        return getEventUsersWithRelation().map(EventUserWithRelations::getEventUser);
    }

    public ListF<EventLayerWithRelations> getEventLayersWithRelations() {
        return eventLayers.getLayersWithRelations();
    }

    public ListF<Layer> getLayers() {
        return eventLayers.getLayers();
    }

    public ListF<EventLayer> getEventLayers() {
        return eventLayers.getEventLayers();
    }

    public ListF<EventInvitation> getInvitations() {
        return eventParticipants.getInvitations();
    }

    public ListF<EventResource> getEventResources() {
        return eventParticipants.getEventResources();
    }

    public long getMainEventId() {
        return event.getMainEventId();
    }

    public String getExternalId() {
        return mainEvent.getExternalId();
    }

    public Option<Instant> getRecurrenceId() {
        return event.getRecurrenceId();
    }

    public boolean isMeeting() {
        return getParticipants().isMeeting();
    }

    public Participants getParticipants() {
        return eventParticipants.getParticipants();
    }

    public EventTime getTime() {
        return time;
    }

    public DateTimeZone getTimezone() {
        return time.getTz();
    }

    public UidOrResourceId getOrganizerIfMeetingOrElseCreator() {
        Participants participants = getParticipants();
        if (participants.isMeeting()) {
            ParticipantInfo organizer = participants.getOrganizer();
            if (organizer.isResource()) {
                return UidOrResourceId.resource(((ResourceParticipantInfo) organizer).getResourceId());
            } else if (organizer.getUid().isPresent()) {
                return UidOrResourceId.user(organizer.getUid().get());
            }
        }
        return UidOrResourceId.user(getEvent().getCreatorUid());
    }

    public Option<Email> getOrganizerEmailIfMeeting() {
        Participants participants = getParticipants();
        return participants.isMeeting() ? Option.of(participants.getOrganizer().getEmail()) : Option.empty();
    }

    public Option<PassportUid> getOrganizerUidIfMeeting() {
        Participants participants = getParticipants();
        return participants.isMeeting() ? participants.getOrganizer().getUid() : Option.empty();
    }

    public Option<Email> getOrganizerIfMeetingOrElseCreatorEmail() {
        Participants participants = getParticipants();
        if (participants.isMeeting()) {
            return Option.of(participants.getOrganizer().getEmail());
        } else if (participants.isNotMeetingStrict()) {
            Option<EventUserWithRelations> eventUserO = findUserEventUserWithRelations(getEvent().getCreatorUid());
            return !eventUserO.isPresent() ? Option.empty() : Option.of(eventUserO.get().getSettings().getEmail());
        } else {
            throw new IllegalStateException("Not supported participants type: " + participants);
        }
    }

    public Option<String> getOrganizerIfMeetingOrElseCreatorEventExchangeId() {
        Participants participants = getParticipants();
        if (participants.isMeeting()) {
            return participants.getOrganizer().getExchangeId();
        } else if (participants.isNotMeetingStrict()) {
            return findUserEventUserWithRelations(getEvent().getCreatorUid())
                    .filterMap(eu -> eu.getEventUser().getExchangeId());
        } else {
            throw new IllegalStateException("Not supported participants type: " + participants);
        }
    }

    public Option<ParticipantId> getOrganizerIdSafe() {
        return getParticipants().getOrganizerIdSafe();
    }

    public boolean isExportedWithEws() {
        return getMainEvent().getIsExportedWithEws().getOrElse(false);
    }

    public boolean ownerIs(PassportUid uid) {
        Participants participants = getParticipants();
        if (participants.isMeeting()) {
            if (participants.getOrganizer().isResource()) {
                return getEvent().getCreatorUid().sameAs(uid);
            }

            return participants.isOrganizer(uid);
        }

        if (getEvent().getCreatorUid().sameAs(uid))
            return true;

        return participants.isOrganizerWithInconsistent(uid);
    }

    private final Lazy<Optional<Layer>> primaryLayer = new Lazy<>();

    public Optional<Layer> getPrimaryLayer() {
        return primaryLayer.getOrCompute(this::computePrimaryLayer);
    }

    public Optional<PassportUid> getPrimaryLayerCreatorUid() {
        return getPrimaryLayer().map(Layer::getCreatorUid);
    }

    public Option<EventLayerWithRelations> findEventLayerWithRelationsById(long layerId) {
        return eventLayers.findLayerWithRelationsById(layerId);
    }

    public Option<Layer> findLayerById(long layerId) {
        return findEventLayerWithRelationsById(layerId).map(EventLayerWithRelations::getLayer);
    }

    private Optional<Layer> computePrimaryLayer() {
        Participants participants = getParticipants();

        Option<EventLayerWithRelations> r = eventLayers.getLayersWithRelations()
                .find(el -> el.getEventLayer().getIsPrimaryInst());

        if (r.isPresent()) {
            return Optional.of(r.get().getLayer());
        }

        Option<ParticipantId> organizerId = participants.getOrganizerIdSafe();

        if (organizerId.isPresent() && organizerId.get().isYandexUser()) {
            Option<EventLayerWithRelations> layerO = eventLayers.findLayerWithRelationsByUid(organizerId.get().getUid());

            if (layerO.isPresent()) {
                return Optional.of(layerO.get().getLayer());
            }
        }

        if (eventLayers.isEmpty()) {
            log.warn("Event {} has no layers", event.getId());
        }

        return Optional.empty();
    }

    public boolean existsOnLayerWithId(long layerId) {
        return findLayerById(layerId).isPresent();
    }

    public boolean existsEventLayerForUser(PassportUid layerOwnerUid) {
        return findOwnUserLayerWithRelations(layerOwnerUid).isPresent();
    }

    public boolean isApartmentOccupation() {
        return getResourceTypes().containsTs(ResourceType.APARTMENT);
    }

    public boolean isHotelOccupation() {
        return getResourceTypes().containsTs(ResourceType.HOTEL);
    }

    public boolean isParkingOccupation() {
        return getResourceTypes().containsTs(ResourceType.PARKING);
    }

    public boolean isCampusOccupation() {
        return getResourceTypes().containsTs(ResourceType.CAMPUS);
    }

    public boolean isParkingOrApartmentOccupation() {
        return isParkingOccupation() || isApartmentOccupation() || isHotelOccupation() || isCampusOccupation();
    }

    public boolean isMultiOffice() {
        return getResources().map(ResourceInfo.officeIdF()).unique().size() > 1;
    }

    public boolean organizerLetToEditAnyMeeting() {
        Option<ParticipantInfo> organizer = eventParticipants.getParticipants().getOrganizerSafe();
        return organizer.isPresent() && organizer.get().letParticipantsEdit();
    }

    public EventParticipants getEventParticipants() {
        return eventParticipants;
    }

    public ListF<ResourceParticipantInfo> getResourceParticipantsYaTeamAndSyncWithExchange() {
        return eventParticipants.getParticipants().getResourceParticipantsSafeWithInconsistent()
                .filter(ResourceParticipantInfo.isYaTeamAndSyncWithExchangeF());
    }

    public EventWithRelations withRecurrenceTimeInfo(RecurrenceTimeInfo info) {
        Event event = getEvent().copy();
        event.setRepetitionIdNull();
        event.setRecurrenceId(info.getRecurrenceId());

        event.setStartTs(info.getInterval().getStart());
        event.setEndTs(info.getInterval().getEnd());

        return new EventWithRelations(event, mainEvent, eventParticipants, eventLayers);
    }

    public EventWithRelations withInterval(InstantInterval interval) {
        Event event = getEvent().copy();

        event.setStartTs(interval.getStart());
        event.setEndTs(interval.getEnd());

        return new EventWithRelations(event, mainEvent, eventParticipants, eventLayers);
    }

    public EventWithRelations withoutResources(ListF<Long> resourceIds) {
        if (resourceIds.isEmpty()) return this;

        return new EventWithRelations(event, mainEvent, eventParticipants.withoutResources(resourceIds), eventLayers);
    }

    public ListF<ResourceInfo> getResources() {
        return eventParticipants.getResources();
    }

    public ListF<ResourceType> getResourceTypes() {
        return getResources().map(ResourceInfo.resourceF().andThen(Resource.getTypeF()));
    }

    private ListF<ResourceInfoForPermsCheck> resourcesForPermsCheck;

    public ListF<ResourceInfoForPermsCheck> getResourcesForPermsCheck() {
        if (resourcesForPermsCheck == null) {
            resourcesForPermsCheck = getResources().map(r -> ResourceInfoForPermsCheck.fromResource(r.getResource()));
        }
        return resourcesForPermsCheck;
    }

    public ListF<Long> getResourceIds() {
        return getResources().map(ResourceInfo.resourceIdF());
    }

    public ListF<Long> getLayerIds() {
        return eventLayers.getLayerIds();
    }

    public ListF<PassportUid> getParticipantAndSubscriberUids() {
        return getParticipants().getParticipantUidsSafe();
    }

    public ListF<PassportUid> getDeclinedUids() {
        return getEventUsers().filter(EventUser.getDecisionF().andThenEquals(Decision.NO)).map(EventUser::getUid);
    }

    public Option<EventLayerWithRelations> findOwnUserLayerWithRelations(PassportUid uid) {
        return eventLayers.findLayerWithRelationsByUid(uid);
    }

    public Option<Layer> findOwnUserLayer(PassportUid uid) {
        return findOwnUserLayerWithRelations(uid).map(EventLayerWithRelations::getLayer);
    }

    public Option<EventLayer> findOwnEventLayer(PassportUid uid) {
        return findOwnUserLayerWithRelations(uid).map(EventLayerWithRelations::getEventLayer);
    }

    public Option<EventInvitation> findInvitation(Email email) {
        return getParticipants().getByIdSafeWithInconsistent(ParticipantId.invitationIdForExternalUser(email))
                .<ExternalUserParticipantInfo>uncheckedCast().map(ExternalUserParticipantInfo::getInvitation);
    }

    public boolean userIsAttendee(PassportUid uid) {
        return eventParticipants.userIsAttendeeWithInconsistent(uid);
    }

    public Option<EventUser> findUserEventUser(PassportUid uid) {
        return findUserEventUserWithRelations(uid).map(EventUserWithRelations::getEventUser);
    }

    public Option<Decision> findUserEventUserDecision(PassportUid uid) {
        return findUserEventUserWithRelations(uid).map(eu -> eu.getEventUser().getDecision());
    }

    public Option<EventUserWithRelations> findUserEventUserWithRelations(PassportUid uid) {
        return eventParticipants.getEventUser(uid);
    }

    public Option<OccurrenceId> getOccurrenceId() {
        return getRecurrenceId().map(recurrenceId ->
                new OccurrenceId(getExternalId(), new InstantInterval(getStartTs(), getEndTs()), recurrenceId));
    }

    public long getEventDurationInMillis() {
        return getEndTs().getMillis() - getStartTs().getMillis();
    }
} //~
