package ru.yandex.calendar.logic.event;

import java.util.Optional;
import java.util.Set;

import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerUserWithRelations;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sending.param.EventOnLayerChangeMessageParameters;
import ru.yandex.calendar.logic.sending.param.Sender;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.inside.passport.PassportUid;

import static java.util.Objects.requireNonNull;

public class EventsOnLayerChangeHandler {
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private RepetitionRoutines repetitionRoutines;


    public ListF<EventOnLayerChangeMessageParameters> handleEventCreate(
            UidOrResourceId creatorId,
            EventWithRelations createdEvent, RepetitionInstanceInfo createdRepetition, ActionInfo actionInfo)
    {
        return handleEventChange(
                creatorId, createdEvent, createdRepetition, EventInstanceParameters.fromEvent(createdEvent.getEvent()),
                LayerIdChangesInfo.changes(Cf.list(), createdEvent.getLayerIds()),
                EventChangesInfoForMails.EMPTY, actionInfo);
    }

    public ListF<EventOnLayerChangeMessageParameters> handleEventUpdate(
            ActorId updaterId, EventInstanceForUpdate instance,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updateRepetition,
            EventChangesInfoForMails changesInfo, ActionInfo actionInfo)
    {
        LayerIdChangesInfo layerIdChanges = LayerIdChangesInfo.changes(
                instance.getEventWithRelations().getLayerIds(), updatedEvent.getLayerIds());

        return handleEventChange(
                updaterId, updatedEvent, updateRepetition,
                EventInstanceParameters.fromEvent(updatedEvent.getEvent()),
                layerIdChanges, changesInfo, actionInfo);
    }

    public ListF<EventOnLayerChangeMessageParameters> handleEventDelete(
            UidOrResourceId actorId, EventWithRelations event, RepetitionInstanceInfo repetition,
            EventInstanceParameters instance, ActionInfo actionInfo)
    {
        return handleEventDelete(ActorId.userOrResource(actorId), event, repetition, instance, actionInfo);
    }

    public ListF<EventOnLayerChangeMessageParameters> handleEventDelete(
            ActorId actorId, EventWithRelations event, RepetitionInstanceInfo repetition,
            EventInstanceParameters instance, ActionInfo actionInfo)
    {
        return handleEventChange(
                actorId, event, repetition, instance,
                LayerIdChangesInfo.changes(event.getLayerIds(), Cf.list()),
                EventChangesInfoForMails.EMPTY, actionInfo);
    }

    public ListF<EventOnLayerChangeMessageParameters> handleEventChange(
            UidOrResourceId actorId, EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition,
            LayerIdChangesInfo layerIdChanges, EventChangesInfoForMails changesInfo, ActionInfo actionInfo)
    {
        return handleEventChange(
                actorId, updatedEvent, updatedRepetition,
                EventInstanceParameters.fromEvent(updatedEvent.getEvent()),
                layerIdChanges, changesInfo, actionInfo);
    }

    public ListF<EventOnLayerChangeMessageParameters> handleEventChange(
            UidOrResourceId actorId,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition, EventInstanceParameters instance,
            LayerIdChangesInfo layerIdChanges, EventChangesInfoForMails changesInfo, ActionInfo actionInfo)
    {
        return handleEventChange(
                ActorId.userOrResource(actorId), updatedEvent, updatedRepetition, instance,
                layerIdChanges, changesInfo, actionInfo);
    }

    public ListF<EventOnLayerChangeMessageParameters> handleEventChange(
        ActorId actorId,
        EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition, EventInstanceParameters instance,
        LayerIdChangesInfo layerIdChanges, EventChangesInfoForMails changesInfo, ActionInfo actionInfo)
    {
        ListF<PassportUid> filterUids = changesInfo.getParticipantsChanges().getNewAndRemovedUids()
                .plus(updatedEvent.getParticipantAndSubscriberUids())
                .plus(actorId.getUidO())
                .plus(updatedEvent.getDeclinedUids());

        ListF<LayerUser> notifyLayerUsers = layerRoutines.getLayerUsersIsNotifyChangesByLayerIds(layerIdChanges.getAll())
                .filter(LayerUser.getUidF().andThen(filterUids.unique().containsF().notF()));

        if (notifyLayerUsers.isEmpty()) return Cf.list();

        ListF<LayerUser> actorLayerUsers = layerRoutines.getLayerUsersByUids(actorId.getUidO());

        MapF<Long, Layer> layerById = layerRoutines.getLayersById(
                notifyLayerUsers.plus(actorLayerUsers).map(LayerUser.getLayerIdF())).toMapMappingToKey(Layer.getIdF());
        Function<LayerUser, LayerUserWithRelations> joinLayerF = Tuple2
                .join(Function.identityF(), LayerUser.getLayerIdF().andThen(Cf2.f(layerById::getOrThrow)))
                .andThen(layerRoutines.createLayerUserWithRelationsF(Option.empty()).asFunction());

        ListF<PassportUid> notifyUids = notifyLayerUsers.map(LayerUser::getUid);

        MapF<PassportUid, UserInfo> notifyUserInfoByUid = userManager
                .getUserInfos(notifyUids).toMapMappingToKey(UserInfo.getUidF());

        MapF<PassportUid, SettingsInfo> notifyUserSettingsByUid = settingsRoutines
                .getSettingsByUidBatch(notifyUids);

        ListF<Long> actorInvitedLayerIds = actorLayerUsers.map(joinLayerF)
                .filterNot(LayerUserWithRelations::isAnonymousSharing)
                .map(LayerUserWithRelations::getLayerId);

        Sender actorSender = eventInvitationManager.getSender(actorId);
        ListF<EventOnLayerChangeMessageParameters> result = Cf.arrayList();

        val notifyUserLayersSharingByUid = authorizer.loadLayerSharing(Set.copyOf(notifyUids));
        for (LayerUserWithRelations layerUser : notifyLayerUsers.map(joinLayerF)) {
            val layer = layerUser.getLayer();
            val user = layerUser.getLayerUser();
            val userInfo = notifyUserInfoByUid.getOrThrow(user.getUid());
            val layersSharing = requireNonNull(notifyUserLayersSharingByUid.get(user.getUid()));

            Option<SettingsInfo> settings = notifyUserSettingsByUid.getO(user.getUid());
            Option<Sender> sender = Option.when(
                    actorInvitedLayerIds.containsTs(layer.getId()) && !layerUser.isAnonymousSharing(), actorSender);

            Option<MailType> mailType;

            if (layerIdChanges.getAdded().containsTs(user.getLayerId())) {
                mailType = Option.of(MailType.EVENT_ON_LAYER_ADDED);
            } else if (layerIdChanges.getRemoved().containsTs(user.getLayerId())) {
                mailType = Option.of(MailType.EVENT_ON_LAYER_REMOVED);
            } else {
                mailType = Option.when(!changesInfo.isEmpty(), MailType.EVENT_ON_LAYER_UPDATED);
            }

            if (mailType.isPresent() && settings.isPresent()) {
                final boolean canView;
                if (mailType.isSome(MailType.EVENT_ON_LAYER_REMOVED)) {
                    val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(Optional.of(userInfo), updatedEvent,
                        Optional.of(layer), Optional.of(layersSharing));
                    canView = authorizer.canViewEvent(userInfo, eventAuthInfo, actionInfo.getActionSource());
                } else {
                    val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(Optional.of(userInfo), updatedEvent,
                        Optional.empty(), Optional.of(layersSharing));
                    canView = authorizer.canViewEvent(userInfo, eventAuthInfo, actionInfo.getActionSource());
                }

                if (canView) {
                    result.add(eventInvitationManager.createEventOnLayerChangeMail(
                        sender, user, layerUser.getEvaluatedLayerName(),
                        settings.get(), updatedEvent, updatedRepetition, instance, mailType.get(),
                        Option.when(mailType.isSome(MailType.EVENT_ON_LAYER_UPDATED), changesInfo), actionInfo));
                }
            }
        }
        return result;
    }

    public ListF<EventOnLayerChangeMessageParameters> handleEventsAttach(
            PassportUid uid, EventsAttachedLayerIds attachedLayerIds, ActionInfo actionInfo)
    {
        Tuple2List<Event, EventAttachedLayerId> attachedIds = attachedLayerIds.getEvents();
        ListF<Event> events = attachedIds.filterBy2(EventAttachedLayerId.wasAttachOrDetachF())
                .map(Tuple2.get1F());

        ListF<EventWithRelations> eventsWr = eventDbManager.getEventsWithRelationsByEvents(events);
        MapF<Long, EventWithRelations> eventWrById = eventsWr.toMapMappingToKey(EventWithRelations::getId);
        MapF<Long, RepetitionInstanceInfo> repetitionByEventId = repetitionRoutines.getRepetitionInstanceInfos(eventsWr);

        Function1B<Event> isRecurrenceF = e -> e.getRecurrenceId().isPresent();
        SetF<Long> masterEventIds = events.filter(isRecurrenceF.notF()).map(Event.getIdF()).unique();
        SetF<Long> recurrenceEventIds = events.filter(isRecurrenceF).map(Event.getIdF()).unique();

        ListF<EventOnLayerChangeMessageParameters> mails = Cf.arrayList();

        for (Tuple2<Long, EventAttachedLayerId> master
                : attachedIds.map1(Event.getIdF()).filterBy1(masterEventIds.containsF()))
        {
            ListF<Long> attached = master.get2().getAttachedId();
            ListF<Long> detached = master.get2().getDetachedId();

            mails.addAll(handleEventChange(
                    UidOrResourceId.user(uid),
                    eventWrById.getOrThrow(master.get1()), repetitionByEventId.getOrThrow(master.get1()),
                    EventInstanceParameters.fromEvent(eventWrById.getOrThrow(master.get1()).getEvent()),
                    new LayerIdChangesInfo(attached.unique(), Cf.set(), detached.unique()),
                    EventChangesInfoForMails.EMPTY, actionInfo));
        }

        SetF<Long> masterNotifiedLayerIds = mails.map(EventOnLayerChangeMessageParameters.getLayerIdF()).unique();

        for (Tuple2<Long, EventAttachedLayerId> recurrence
                : attachedIds.map1(Event.getIdF()).filterBy1(recurrenceEventIds.containsF()))
        {
            SetF<Long> attached = recurrence.get2().getAttachedId().unique().minus(masterNotifiedLayerIds);
            SetF<Long> detached = recurrence.get2().getDetachedId().unique().minus(masterNotifiedLayerIds);

            mails.addAll(handleEventChange(
                    UidOrResourceId.user(uid),
                    eventWrById.getOrThrow(recurrence.get1()), repetitionByEventId.getOrThrow(recurrence.get1()),
                    EventInstanceParameters.fromEvent(eventWrById.getOrThrow(recurrence.get1()).getEvent()),
                    new LayerIdChangesInfo(attached, Cf.<Long>set(), detached),
                    EventChangesInfoForMails.EMPTY, actionInfo));
        }
        return mails;
    }
}
