package ru.yandex.calendar.logic.event;

import org.jdom.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.web.ModificationScope;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.util.xml.CalendarXmlizer;

/**
 * Event modification (update or deletion) information
 * @author ssytnik
 */
public class ModificationInfo {
    // XXX: create separate lists for updated, deleted, created events
    private final ListF<Long> eventIds; // ids of updated / deleted events
    private final ListF<String> externalIds;
    private final Option<EventAndRelations> newEvent;
    private final Option<EventAndRelations> updatedEvent;
    private final ModificationScope scope;
    private final boolean wasBroughtBack; // identifies if event was brought back to series

    private final ListF<EventMessageParameters> eventMails;

    private ModificationInfo(
            ModificationScope direction, ListF<Long> eIds, ListF<String> externalIds,
            Option<EventAndRelations> newEvent, Option<EventAndRelations> updatedEvent,
            boolean wasBroughtBack, ListF<EventMessageParameters> eventMails)
    {
        this.eventIds = eIds;
        this.externalIds = externalIds;
        this.newEvent = newEvent;
        this.updatedEvent = updatedEvent;
        this.scope = direction;
        this.wasBroughtBack = wasBroughtBack;
        this.eventMails = eventMails;
    }

    public static ModificationInfo updated(
            Option<EventAndRelations> updatedEvent, ModificationScope direction,
            ListF<Long> modifiedEventIds, ListF<String> modifiedExternalIds,
            Option<EventAndRelations> newEvent, ListF<EventMessageParameters> eventMails)
    {
        return new ModificationInfo(direction, modifiedEventIds, modifiedExternalIds,
                newEvent, updatedEvent, false, eventMails);
    }

    public static ModificationInfo removed(
            ModificationScope direction, ListF<Long> modifiedEventIds,
            ListF<String> modifiedExternalIds, ListF<EventMessageParameters> eventMails)
    {
        return new ModificationInfo(direction, modifiedEventIds, modifiedExternalIds,
                Option.empty(), Option.empty(), false, eventMails);
    }

    public static ModificationInfo broughtBack(ListF<Long> modifiedEventIds, ListF<String> modifiedExternalIds) {
        return new ModificationInfo(
                ModificationScope.SINGLE, modifiedEventIds, modifiedExternalIds,
                Option.empty(), Option.empty(), true, Cf.list());
    }

    public ModificationInfo plusMails(ListF<EventMessageParameters> mails) {
        return new ModificationInfo(scope, eventIds, externalIds, newEvent, updatedEvent, wasBroughtBack, eventMails.plus(mails));
    }

    public ListF<Long> getEventIds() {
        return eventIds;
    }

    public ListF<String> getExternalIds() {
        return externalIds;
    }

    public Option<Long> getNewEventId() {
        return newEvent.map(EventAndRelations::getEventId);
    }

    public Option<EventAndRepetition> getNewEvent() {
        return newEvent.map(EventAndRelations::getEventAndRepetition);
    }

    public Option<EventAndRelations> getNewEventAndRelations() {
        return newEvent;
    }

    public Option<EventAndRepetition> getUpdatedEvent() {
        return updatedEvent.map(EventAndRelations::getEventAndRepetition);
    }

    public boolean wasBroughtBack() {
        return wasBroughtBack;
    }

    public ListF<EventMessageParameters> getEventMails() {
        return eventMails;
    }

    public ModificationScope getScope() {
        return scope;
    }

    // element +this{fields}
    public void appendXmlTo(Element e, boolean outSplitInfo) {
        CalendarXmlizer.appendElmColl(e, null, "id", eventIds); // updated / deleted ids
        CalendarXmlizer.appendElm(e, "scope", scope.name().toLowerCase());
        if (wasBroughtBack) {
            CalendarXmlizer.appendElm(e, "was-brought-back", wasBroughtBack);
        }
    }
}
