package ru.yandex.calendar.logic.event;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Hours;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.web.cmd.RequestDataConverter;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventInvitationData;
import ru.yandex.calendar.logic.event.model.EventInvitationUpdateData;
import ru.yandex.calendar.logic.event.model.EventInvitationsData;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.event.model.ParticipantsOrInvitationsData;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.sharing.perm.EventActionClass;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.data.DataProvider;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.HumanReadableTimeParser;
import ru.yandex.calendar.util.dates.HumanReadableTimeParser.Result;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author akirakozov
 */
public class RequestEventDataConverter extends AbstractEventDataConverter {
    private static final Duration DEFAULT_EVENT_DURATION = Hours.ONE.toStandardDuration();
    private final DateTimeZone dataTz;
    private final DataProvider eDp;
    private final DataProvider uDp;

    public RequestEventDataConverter(DateTimeZone dataTz, DateTimeZone eventTz, DataProvider eDp, DataProvider uDp) {
        super(eventTz);
        Validate.notNull(eDp);
        this.dataTz = dataTz;
        this.eDp = eDp;
        this.uDp = uDp;
    }

    public RequestEventDataConverter(DateTimeZone dataTz, DateTimeZone eventTz, DataProvider dp) {
        this(dataTz, eventTz, dp.getDataProvider("event", false), dp.getDataProvider("user", false));
    }

    protected EventData doConvert() {
        EventData eventData = convertEventData();
        Instant startTs = eventData.getEvent().getStartTs();
        eventData.setRepetition(convertRepetition(startTs));
        eventData.setEventUserData(convertEventUserData(eventData.getEvent()));
        eventData.setInvData(convertInvData());
        eventData.getRdates().addAll(convertRDateData());
        return eventData;
    }

    private EventData convertEventData() {
        EventData eventData = new EventData();
        String idStr = eDp.getText(dpName(EventFields.ID), false);
        if (StringUtils.isNotEmpty(idStr)) {
            eventData.getEvent().setId(Long.parseLong(idStr));
        }
        String extIdStr = eDp.getText(dpName(MainEventFields.EXTERNAL_ID), false);
        if (StringUtils.isNotEmpty(extIdStr)) {
            eventData.setExternalId(Option.of(extIdStr));
        }
        String mainEventIdStr = eDp.getText(dpName(EventFields.MAIN_EVENT_ID), false);
        if (StringUtils.isNotEmpty(mainEventIdStr)) {
            eventData.getEvent().setMainEventId(Long.parseLong(mainEventIdStr));
        }
        String recurIdStr = eDp.getText(dpName(EventFields.RECURRENCE_ID), false);
        if (StringUtils.isNotEmpty(recurIdStr)) {
            eventData.getEvent().setRecurrenceId(DateTimeFormatter.parseTimestampFromMachinesSafe(recurIdStr, dataTz).get());
        }

        eventData.setInstanceStartTs(DateTimeFormatter.toNullableTimestampUnsafe(
                eDp.getText("ts", false), DateTimeZone.UTC));

        String name = eDp.getText(dpName(EventFields.NAME), false);
        String isAllDayStr = eDp.getText(dpName(EventFields.IS_ALL_DAY), false);
        boolean isManuallySetToAllDay = Binary.parseBoolean(isAllDayStr);
        boolean isAllDay = isManuallySetToAllDay;

        String startTsStr = eDp.getText("start-ts", true);
        String endTsStr = eDp.getText("end-ts", true);
        boolean timeIsSet = (hasTime(startTsStr) && hasTime(endTsStr)) || isManuallySetToAllDay;

        Instant startTs;
        Instant endTs;

        if (timeIsSet) {
            startTs = AuxDateTime.toInstantIgnoreGap(LocalDateTime.parse(startTsStr), isAllDay ? eventTz : dataTz);
            endTs = AuxDateTime.toInstantIgnoreGap(LocalDateTime.parse(endTsStr), isAllDay ? eventTz : dataTz);
        } else {
            startTs = parseTimestamp(startTsStr, dataTz);

            Result result = HumanReadableTimeParser.parse(name);
            if (result.gotSomething()) {
                LocalTime startTime = result.getStartTime();
                startTs = startTs.plus(startTime.getMillisOfDay());
                endTs = startTs.plus(DEFAULT_EVENT_DURATION);
                name = result.getRest();
            } else {
                // parsing time from name failed, consider event all-day
                endTs = new DateTime(startTs, eventTz).plusDays(1).toInstant();
                isAllDay = true;
            }
        }

        eventData.getEvent().setStartTs(startTs);
        eventData.getEvent().setEndTs(endTs);
        eventData.getEvent().setIsAllDay(isAllDay);
        eventData.setTimeZone(eventTz);

        if (StringUtils.isNotEmpty(name)) {
            eventData.getEvent().setName(name);
        }

        final String location = getDataProviderField(EventFields.LOCATION);
        if (StringUtils.isNotEmpty(location)) {
            eventData.getEvent().setLocation(location);
        }
        final String description = getDataProviderField(EventFields.DESCRIPTION);
        if (StringUtils.isNotEmpty(description)) {
            eventData.getEvent().setDescription(description);
        }
        final String url = getDataProviderField(EventFields.URL);
        if (StringUtils.isNotEmpty(url)) {
            eventData.getEvent().setUrl(url);
        }

        String permAllStr = getDataProviderField(EventFields.PERM_ALL);
        if (StringUtils.isNotEmpty(permAllStr)) {
            eventData.getEvent().setPermAll(EventActionClass.R.valueOf(permAllStr));
        }
        String permParticipantsStr = getDataProviderField(EventFields.PERM_PARTICIPANTS);
        if (StringUtils.isNotEmpty(permParticipantsStr)) {
            eventData.getEvent().setPermParticipants(EventActionClass.R.valueOf(permParticipantsStr));
        }
        String participantsInviteStr = getDataProviderField(EventFields.PARTICIPANTS_INVITE);
        if (participantsInviteStr != null) {
            eventData.getEvent().setParticipantsInvite(Binary.parseBoolean(participantsInviteStr));
        }

        DataProvider lDp = eDp.getDataProvider("layer", false);
        if (lDp != null) {
            final String layerId = lDp.getText("layer-id", false);
            if (StringUtils.isNotEmpty(layerId)) {
                eventData.setLayerId(Cf.Long.parse(layerId));
            }
            final String previousLayerId = lDp.getText("cur-layer-id", false);
            if (StringUtils.isNotEmpty(previousLayerId)) {
                eventData.setPrevLayerId(Cf.Long.parse(previousLayerId));
            }
        }

        DataProvider attachmentsDp = eDp.getDataProvider("attachments", false);
        if (attachmentsDp != null) {
            ListF<EventAttachment> attachments = Cf.arrayList();
            for (DataProvider attachmentDp : attachmentsDp.getDataProviders("attachment")) {
                EventAttachment attachment = new EventAttachment();
                attachment.setUrl(attachmentDp.getText("id", true));
                attachment.setFilename(attachmentDp.getText("filename", true));
                attachments.add(attachment);
            }
            eventData.setAttachments(attachments);
        }

        return eventData;
    }

    private String dpName(MapField<?> externalId) {
        return RequestDataConverter.dpName(externalId);
    }

    private String getDataProviderField(MapField<?> field) {
        return eDp.getText(dpName(field), false);
    }

    private ParticipantsOrInvitationsData convertInvData() {
        Option<Email> orgEmail = Option.ofNullable(eDp.getText("organizer", false)).map(Email.consF());

        if (StringUtils.isNotEmpty(eDp.getText("attendees", false))) {
            ListF<EventInvitationData> invitations = Cf.arrayList();
            for (String attendeeStr: eDp.getText("attendees", false).split("\\s*,\\s*")) {
                int pos = attendeeStr.indexOf(';');
                if (pos >= 0) {
                    invitations.add(new EventInvitationData(
                            new Email(attendeeStr.substring(0, pos).toLowerCase()),
                            Option.of(attendeeStr.substring(pos + 1)), false));
                } else {
                    invitations.add(new EventInvitationData(new Email(attendeeStr.toLowerCase())));
                }
            }
            return ParticipantsOrInvitationsData.eventInvitationData(new EventInvitationsData(orgEmail, invitations));
        } else {
            String invEmailsStr = eDp.getText("invitations", false);
            String remEmailsStr = eDp.getText("rem-invitations", false);

            ListF<Email> newEmails = AuxColl.splitToUniqueEmailArray(invEmailsStr, false);
            ListF<Email> remEmails = AuxColl.splitToUniqueEmailArray(remEmailsStr, false);
            return ParticipantsOrInvitationsData.eventInvitationUpdateData(
                    new EventInvitationUpdateData(orgEmail, newEmails, remEmails));
        }
    }

    private Repetition convertRepetition(Instant startTs) {
        DataProvider rDp = eDp.getDataProvider("repetition", false);
        return RequestDataConverter.convertRepetition(eventTz, rDp, startTs);
    }

    private ListF<Rdate> convertRDateData() {
        ListF<Rdate> res = Cf.arrayList();
        convertRDateDataInner(dataTz, false, Option.ofNullable(eDp.getText("exdates_start_ts", false)), Option.<String>empty(), res);
        convertRDateDataInner(
                dataTz, true, Option.ofNullable(eDp.getText("rdates_start_ts", false)),
                Option.ofNullable(eDp.getText("rdates_end_ts", false)), res);
        return res;
    }

    private void convertRDateDataInner(
            DateTimeZone tz, boolean isRdate, Option<String> strStarts,
            Option<String> strEnds, ListF<Rdate> resList) {
        boolean isPeriodType = strEnds.isPresent() && !strEnds.get().isEmpty();
        if (!isRdate && isPeriodType) {
            String msg = "EXDATE type must be DATE or DATETIME";
            throw new IllegalArgumentException(msg);
        }
        ListF<String> startRdatesTs = Cf.x(StringUtils.split(strStarts.getOrElse(""), ","));

        ListF<String> endRdatesTs = Cf.x(StringUtils.split(strEnds.getOrElse(""), ","));

        if (AuxColl.isSet(startRdatesTs)) {
            if (startRdatesTs.length() < endRdatesTs.length()) {
                String msg = "Not corresponding length of start&end arrays";
                throw new IllegalArgumentException(msg);
            }
            for (int i = 0; i < startRdatesTs.length(); i++) {
                Rdate rdate = new Rdate();
                Instant endTs = isPeriodType && i < endRdatesTs.length() ?
                        DateTimeFormatter.parseTimestampFromMachinesSafe(endRdatesTs.get(i), tz).getOrNull() : null;
                rdate.setEndTs(endTs);
                rdate.setStartTs(DateTimeFormatter.parseTimestampFromMachinesSafe(startRdatesTs.get(i), tz).getOrNull());
                rdate.setIsRdate(isRdate);
                resList.add(rdate);
            }
        }
    }

    private EventUserData convertEventUserData(Event event) {
        if (uDp == null) {
            return new EventUserData();
        }
        ListF<Notification> notifications = RequestDataConverter.convertNotifications(
                uDp.getDataProvider("notification", false));

        if (Binary.parseBoolean(eDp.getText("is_smart_date", false)) && !event.getIsAllDay()) { // CAL-6539
            final DateTime eventStart = event.getStartTs().toDateTime(eventTz);
            final LocalDateTime localMidnight = eventStart.withTimeAtStartOfDay().toLocalDateTime();

            notifications = notifications.map(new Function<Notification, Notification>() {
                public Notification apply(Notification n) {
                    DateTime reminder = AuxDateTime.toDateTimeIgnoreGap(localMidnight.plus(n.getOffset()), eventTz);
                    Duration offset = new Duration(eventStart, reminder);

                    return new Notification(n.getChannel(), offset.getMillis() > 0 ? Duration.standardMinutes(-15) : offset);
                }
            });
        }
        return new EventUserData(RequestDataConverter.convertEventUser(uDp), notifications);
    }

    public static EventData convertAndValidate(DateTimeZone dataTz, DateTimeZone eventTz, DataProvider dp) {
        RequestEventDataConverter conv = new RequestEventDataConverter(dataTz, eventTz, dp);
        return conv.convert();
    }

    public static EventData convert(DateTimeZone dataTz, DateTimeZone eventTz, DataProvider eDp, DataProvider uDp) {
        RequestEventDataConverter conv = new RequestEventDataConverter(dataTz, eventTz, eDp, uDp);
        return conv.convert();
    }

    public static Instant getStartTimestamp(DataProvider eDp, DateTimeZone tz) {
        String value = eDp.getText("start-ts", false);
        if (StringUtils.isNotEmpty(value)) {
            return parseTimestamp(value, tz);
        }
        throw new CommandRunException("start-ts parameter required");
    }

    private static Instant parseTimestamp(String value, DateTimeZone tz) {
        return DateTimeFormatter.parseTimestampFromMachinesSafe(value, tz)
            .getOrThrow("failed to parse ", value);
    }

    private boolean hasTime(String tsStr) {
        return DateTimeFormatter.timestampFromMachinesHasTimePart(tsStr);
    }
}
