package ru.yandex.calendar.logic.event.archive;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.DeletedEvent;
import ru.yandex.calendar.logic.beans.generated.DeletedEventLayer;
import ru.yandex.calendar.logic.beans.generated.DeletedEventResource;
import ru.yandex.calendar.logic.beans.generated.DeletedEventUser;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.misc.lang.Check;

/**
 * @author akirakozov
 */
public class ArchiveManager {

    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private DeletedEventDao deletedEventDao;

    private static final MapObjectCopier<Event, DeletedEvent> eventCopier = new MapObjectCopier<Event, DeletedEvent>(Event.class, DeletedEvent.class);
    private static final MapObjectCopier<EventUser, DeletedEventUser> eventUserCopier = new MapObjectCopier<EventUser, DeletedEventUser>(EventUser.class, DeletedEventUser.class);
    private static final MapObjectCopier<EventLayer, DeletedEventLayer> eventLayerCopier = new MapObjectCopier<EventLayer, DeletedEventLayer>(EventLayer.class, DeletedEventLayer.class);
    private static final MapObjectCopier<EventResource, DeletedEventResource> eventResourceCopier = new MapObjectCopier<EventResource, DeletedEventResource>(EventResource.class, DeletedEventResource.class);

    static {
        Check.C.isTrue(eventCopier.isNotEmpty());
        Check.C.isTrue(eventUserCopier.isNotEmpty());
        Check.C.isTrue(eventLayerCopier.isNotEmpty());
        Check.C.isTrue(eventResourceCopier.isNotEmpty());
    }

    public static Function2<Event, String, DeletedEvent> toDeletedEventF(
            final ActionInfo actionInfo)
    {
        return new Function2<Event, String, DeletedEvent>() {
            public DeletedEvent apply(Event event, String externalId) {
                DeletedEvent deletedEvent = new DeletedEvent();
                eventCopier.copy(event, deletedEvent);
                deletedEvent.setExternalId(externalId);
                deletedEvent.setDeletionSource(actionInfo.getActionSource());
                deletedEvent.setDeletionTs(actionInfo.getNow());
                deletedEvent.setDeletionReqId(actionInfo.getRequestIdWithHostId());
                return deletedEvent;
            }
        };
    }

    public static Function<EventUser, DeletedEventUser> toDeletedEventUserF(
            final ActionInfo actionInfo)
    {
        return new Function<EventUser, DeletedEventUser>() {
            public DeletedEventUser apply(EventUser eventUser) {
                DeletedEventUser deletedEventUser = new DeletedEventUser();
                eventUserCopier.copy(eventUser, deletedEventUser);
                deletedEventUser.setDeletionSource(actionInfo.getActionSource());
                deletedEventUser.setDeletionTs(actionInfo.getNow());
                deletedEventUser.setDeletionReqId(actionInfo.getRequestIdWithHostId());
                return deletedEventUser;
            }
        };
    }

    public static Function<EventLayer, DeletedEventLayer> toDeletedEventLayerF(
            final ActionInfo actionInfo)
    {
        return new Function<EventLayer, DeletedEventLayer>() {
            public DeletedEventLayer apply(EventLayer eventLayer) {
                DeletedEventLayer deletedEventLayer = new DeletedEventLayer();
                eventLayerCopier.copy(eventLayer, deletedEventLayer);
                deletedEventLayer.setDeletionSource(actionInfo.getActionSource());
                deletedEventLayer.setDeletionTs(actionInfo.getNow());
                deletedEventLayer.setDeletionReqId(actionInfo.getRequestIdWithHostId());
                return deletedEventLayer;
            }
        };
    }

    public static Function<EventResource, DeletedEventResource> toDeletedEventResourceF(
            final ActionInfo actionInfo)
    {
        return new Function<EventResource, DeletedEventResource>() {
            public DeletedEventResource apply(EventResource eventResource) {
                DeletedEventResource deletedEventResource = new DeletedEventResource();
                eventResourceCopier.copy(eventResource, deletedEventResource);
                deletedEventResource.setDeletionSource(actionInfo.getActionSource());
                deletedEventResource.setDeletionTs(actionInfo.getNow());
                deletedEventResource.setDeletionReqId(actionInfo.getRequestIdWithHostId());
                return deletedEventResource;
            }
        };
    }

    public void storeDeletedEvents(Tuple2List<Event, String> events, ActionInfo actionInfo) {
        deletedEventDao.saveDeletedEvents(events.map(toDeletedEventF(actionInfo)));
    }

    public void storeDeletedEventUsers(ListF<EventUser> eventUsers, ActionInfo actionInfo) {
        deletedEventDao.saveDeletedEventUsers(eventUsers.map(toDeletedEventUserF(actionInfo)));
    }

    public void storeDeletedEventLayers(ListF<EventLayer> eventLayers, ActionInfo actionInfo) {
        deletedEventDao.saveDeletedEventLayers(eventLayers.map(toDeletedEventLayerF(actionInfo)));
    }

    public void storeDeletedEventResources(ListF<EventResource> eventResources, ActionInfo actionInfo) {
        deletedEventDao.saveDeletedEventResources(eventResources.map(toDeletedEventResourceF(actionInfo)));
    }

    public void storeDeletedEventsWithDependingItems(ListF<Long> eventIds, ActionInfo actionInfo) {
        storeDeletedEvents(eventDao.findEventsWithExternalIdByEventIds(eventIds), actionInfo);
        storeDeletedEventUsers(eventUserDao.findEventUsersByEventIds(eventIds), actionInfo);
        storeDeletedEventLayers(eventLayerDao.findEventLayersByEventIds(eventIds), actionInfo);
        storeDeletedEventResources(eventResourceDao.findEventResourcesByEventIds(eventIds), actionInfo);
    }
}
