package ru.yandex.calendar.logic.event.archive;

import java.util.Optional;

import lombok.val;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.DeletedEvent;
import ru.yandex.calendar.logic.beans.generated.DeletedEventFields;
import ru.yandex.calendar.logic.beans.generated.DeletedEventHelper;
import ru.yandex.calendar.logic.beans.generated.DeletedEventLayer;
import ru.yandex.calendar.logic.beans.generated.DeletedEventLayerFields;
import ru.yandex.calendar.logic.beans.generated.DeletedEventResource;
import ru.yandex.calendar.logic.beans.generated.DeletedEventResourceFields;
import ru.yandex.calendar.logic.beans.generated.DeletedEventUser;
import ru.yandex.calendar.logic.beans.generated.DeletedEventUserFields;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;

public class DeletedEventDao extends CalendarJdbcDaoSupport {
    @Autowired
    private GenericBeanDao genericBeanDao;

    @RunWithRandomTest
    public Option<DeletedEventResource> findLatestDeletedEventResourceByResourceExchangeId(String exchangeId) {
        String q = "SELECT * FROM deleted_event_resource WHERE exchange_id = ? ORDER BY deletion_ts DESC LIMIT 1";
        return getJdbcTemplate().queryForOption(q, DeletedEventResource.class, exchangeId);
    }

    @RunWithRandomTest
    public Option<DeletedEventUser> findLatestDeletedEventUserByUserExchangeId(String exchangeId) {
        String q = "SELECT * FROM deleted_event_user WHERE exchange_id = ? ORDER BY deletion_ts DESC LIMIT 1";
        return getJdbcTemplate().queryForOption(q, DeletedEventUser.class, exchangeId);
    }

    @RunWithRandomTest
    public ListF<DeletedEvent> findDeletedEventById(long eventId) {
        String q = "SELECT * FROM deleted_event WHERE id = ?";
        return getJdbcTemplate().queryForList(q, DeletedEvent.class, eventId);
    }

    @RunWithRandomTest
    public ListF<DeletedEvent> findDeletedEventsByIds(ListF<Long> eventIds) {
        String q = "SELECT * FROM deleted_event WHERE id " + SqlQueryUtils.inSet(eventIds.stableUnique());

        if (skipQuery(eventIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, DeletedEvent.class);
    }

    @RunWithRandomTest
    public Tuple2List<String, Instant> findDeletedEventExternalIdsAndTiestampsByIds(ListF<Long> eventIds) {
        SqlCondition c = DeletedEventFields.ID.column().inSet(eventIds);

        String q = "SELECT DISTINCT external_id, deletion_ts FROM deleted_event" + c.whereSql();

        if (skipQuery(eventIds, q, c.args())) {
            return Tuple2List.tuple2List();
        }
        return getJdbcTemplate().queryForList2(q, String.class, Instant.class, c.args());
    }

    @RunWithRandomTest
    public ListF<DeletedEventLayer> findDeletedEventLayerByLayerIdDeletedSince(long layerId, Instant since) {
        String q = "SELECT * FROM deleted_event_layer WHERE layer_id = ? AND deletion_ts >= ?";

        return getJdbcTemplate().queryForList(q, DeletedEventLayer.class, layerId, since);
    }

    @RunWithRandomTest
    public ListF<Long> findEventIdsByEventLayerIdsDeletedAfter(ListF<Long> layerIds, Instant after) {
        SqlCondition c = DeletedEventLayerFields.LAYER_ID.column().inSet(layerIds)
                .and(DeletedEventLayerFields.DELETION_TS.gt(after));

        String q = "SELECT DISTINCT event_id FROM deleted_event_layer" + c.whereSql();

        return skipQuery(layerIds, q, c.args()) ? Cf.list() : getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public Tuple2List<Long, DeletedEvent> findEventsDeletedAfterByLayerIds(ListF<Long> layerIds, Instant after) {
        SqlCondition layerC = DeletedEventLayerFields.LAYER_ID.column().inSet(layerIds);
        SqlCondition eventC = DeletedEventFields.DELETION_TS.gt(after);

        RowMapper<Long> lidRm = (rs, num) -> rs.getLong(1);
        BeanRowMapper<DeletedEvent> eRm = DeletedEventHelper.INSTANCE.offsetRowMapper(1);

        String q = "SELECT el.layer_id, " + eRm.columns("e.") + " FROM deleted_event_layer el" +
                " INNER JOIN deleted_event e ON e.id = el.event_id" +
                " WHERE (" + layerC.sqlForTable("el") + ")" + eventC.andSqlForTable("e");

        if (skipQuery(layerC.and(eventC), q, layerC.args(), eventC.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(q, lidRm, eRm, layerC.args(), eventC.args());
    }

    public void saveDeletedEvents(ListF<DeletedEvent> deletedEvents) {
        genericBeanDao.insertBeans(deletedEvents, DeletedEventFields.OBJECT_DESCRIPTION.getFields());
    }

    public void saveDeletedEventUsers(ListF<DeletedEventUser> deletedEventUsers) {
        genericBeanDao.insertBeans(deletedEventUsers, DeletedEventUserFields.OBJECT_DESCRIPTION.getFields());
    }

    public void saveDeletedEventLayers(ListF<DeletedEventLayer> deletedEventLayers) {
        genericBeanDao.insertBeans(deletedEventLayers, DeletedEventLayerFields.OBJECT_DESCRIPTION.getFields());
    }

    public void saveDeletedEventResources(ListF<DeletedEventResource> deletedEventResources) {
        genericBeanDao.insertBeans(deletedEventResources, DeletedEventResourceFields.OBJECT_DESCRIPTION.getFields());
    }

    @RunWithRandomTest
    public ListF<DeletedEventResource> findDeletedEventResourcesByDeletedEventExternalId(String externalId) {
        String q = "SELECT r.* FROM deleted_event_resource r INNER JOIN deleted_event e ON r.event_id = e.id WHERE e.external_id = ?";
        return getJdbcTemplate().queryForList(q, DeletedEventResource.class, externalId);
    }

    @RunWithRandomTest
    public boolean findDeletedMasterExistsByExternalIdAndEventUserUids(
            String externalId, ListF<PassportUid> uids)
    {
        SqlCondition euC = DeletedEventUserFields.UID.column().inSet(uids);

        SqlCondition eC = DeletedEventFields.RECURRENCE_ID.column().isNull()
                .and(DeletedEventFields.EXTERNAL_ID.eq(externalId));

        String q = "SELECT EXISTS (SELECT 1 FROM deleted_event e, deleted_event_user eu"
                + " WHERE eu.event_id = e.id" + eC.andSqlForTable("e") + euC.andSqlForTable("eu") + ")";

        if (skipQuery(uids, q, eC.args(), euC.args())) return false;

        return getJdbcTemplate().queryForObject(q, Boolean.class, eC.args(), euC.args());
    }

    @RunWithRandomTest
    public boolean findDeletedMasterExistsByExternalIdAndResourceIds(
            String externalId, ListF<Long> resourceIds)
    {
        SqlCondition erC = DeletedEventResourceFields.RESOURCE_ID.column().inSet(resourceIds);

        SqlCondition eC = DeletedEventFields.RECURRENCE_ID.column().isNull()
                .and(DeletedEventFields.EXTERNAL_ID.eq(externalId));

        String q = "SELECT EXISTS (SELECT 1 FROM deleted_event e, deleted_event_resource er"
                + " WHERE er.event_id = e.id" + eC.andSqlForTable("e") + erC.andSqlForTable("er") + ")";

        if (skipQuery(resourceIds, q, eC.args(), erC.args())) return false;

        return getJdbcTemplate().queryForObject(q, Boolean.class, eC.args(), erC.args());
    }

    public boolean existsDeletedEvent(String externalId, Optional<Instant> recurrenceId) {
        val ec = DeletedEventFields.EXTERNAL_ID.eq(externalId).and(
                recurrenceId.map(r -> DeletedEventFields.RECURRENCE_ID.eq(r)).orElseGet(() -> DeletedEventFields.RECURRENCE_ID.column().isNull()));
        val q = "SELECT EXISTS (select 1 FROM deleted_event e WHERE " + ec.sqlForTable("e") + ")";
        return getJdbcTemplate().queryForObject(q, Boolean.class, ec.args());
    }

    @RunWithRandomTest
    public ListF<DeletedEventResource> findEventResourcesByEventExternalIdDeletedNotAfter(String externalId, Instant to) {
        String q = "SELECT r.* FROM deleted_event_resource r" +
                " INNER JOIN deleted_event e ON r.event_id = e.id" +
                " WHERE e.external_id = ? AND r.deletion_ts <= ?";
        return getJdbcTemplate().queryForList(q, DeletedEventResource.class, externalId, to);
    }

    @RunWithRandomTest
    public ListF<DeletedEventResource> findEventResourcesByEventIdsDeletedNotAfter(ListF<Long> eventIds, Instant to) {
        SqlCondition c = DeletedEventResourceFields.EVENT_ID.column().inSet(eventIds)
                .and(DeletedEventResourceFields.DELETION_TS.le(to));
        String q = "SELECT * FROM deleted_event_resource" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, DeletedEventResource.class, c.args());
    }

} //~
