package ru.yandex.calendar.logic.event.avail;

import lombok.val;
import one.util.streamex.StreamEx;
import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.frontend.web.cmd.run.api.ResourcesForStaffComparator;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceReservation;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventActions;
import ru.yandex.calendar.logic.event.EventDateTime;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetition;
import ru.yandex.calendar.logic.event.repetition.IntersectingIntervals;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceSet;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerUserWithRelations;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationInfo;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationManager;
import ru.yandex.calendar.logic.resource.schedule.EventIdOrReservationInterval;
import ru.yandex.calendar.logic.resource.schedule.ResourceDaySchedule;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.MoscowTime;
import ru.yandex.misc.xml.jdom.JdomUtils;
import ru.yandex.misc.xml.stream.XmlWriter;
import ru.yandex.misc.xml.stream.builder.XmlBuilder;
import ru.yandex.misc.xml.stream.builder.XmlBuilders;

/**
 * User availability / business stuff
 */
public class AvailRoutines {
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private ResourceReservationManager resourceReservationManager;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private DateTimeManager dateTimeManager;


    public AvailabilityIntervals getAvailabilityIntervalsForStaff(
            Email clientEmail, Email subjectEmail, Instant start, Instant end, ActionInfo actionInfo)
    {
        PassportUid subjectUid = userManager.getUidByEmail(subjectEmail).getOrThrow("unknown user" + subjectEmail);
        PassportUid clientUid = userManager.getUidByEmail(clientEmail).getOrThrow("unknown user" + clientEmail);

        AvailabilityRequest req = AvailabilityRequest.interval(start, end).includeEventsInfo().excludeAbsencesEvents();
        return getAvailabilityIntervals(clientUid, SubjectId.uid(subjectUid), req, actionInfo).getIntervals();
    }

    public AvailabilityIntervalsOrRefusal getAvailabilityIntervals(
            PassportUid clientUid, SubjectId subject, AvailabilityRequest request, ActionInfo actionInfo)
    {
        if (subject.isEmail()) {
            return getAvailabilityIntervalssByEmails(clientUid, Cf.list(subject.getEmail()), request, actionInfo)
                    .single().get2();
        } else {
            return getAvailabilityIntervalss(clientUid, Cf.list(subject.getUidOrResourceId()), request, actionInfo)
                    .single().getIntervalsOrRefusal();
        }
    }

    public Tuple2List<Email, AvailabilityIntervalsOrRefusal> getAvailabilityIntervalssByEmails(
            PassportUid clientUid, ListF<Email> emails, AvailabilityRequest request, ActionInfo actionInfo)
    {
        Tuple2List<Email, Option<UidOrResourceId>> subjects = eventInvitationManager.getSubjectIdsByEmails(emails);

        Tuple2List<Email, UidOrResourceId> knownSubjects = Cf2.flatBy2(subjects);
        ListF<Email> unknownEmails = subjects.filterBy2Not(Option::isPresent).get1();

        MapF<UidOrResourceId, AvailabilityIntervalsOrRefusal> resultBySubjectId =
                getAvailabilityIntervalss(clientUid, knownSubjects.get2(), request, actionInfo).toMap(
                        UserOrResourceAvailabilityIntervals.getUserOrResourceF(),
                        UserOrResourceAvailabilityIntervals.getIntervalsOrRefusalF());

        AvailabilityIntervalsOrRefusal unknown =
                AvailabilityIntervalsOrRefusal.refusal(AvailabilityQueryRefusalReason.UNKNOWN_USER);

        return knownSubjects.map2(resultBySubjectId::getOrThrow)
                .plus(unknownEmails.zipWith(Function.constF(unknown)));
    }

    public ListF<UserOrResourceAvailabilityIntervals> getAvailabilityIntervalss(
            PassportUid clientUid, ListF<UidOrResourceId> subjectIds,
            AvailabilityRequest request, ActionInfo actionInfo)
    {
        Tuple2<ListF<UidOrResourceId>, ListF<UidOrResourceId>> t = subjectIds.partition(UidOrResourceId.isUserF());
        ListF<PassportUid> uids = t._1.map(UidOrResourceId.getUidF());
        ListF<Resource> resources = resourceDao.findResourcesByIds(t._2.map(UidOrResourceId.getResourceIdF()));

        return getAvailabilityIntervalss(clientUid, uids, resources, request, actionInfo);
    }

    public ListF<UserOrResourceAvailabilityIntervals> getAvailabilityIntervalss(
            PassportUid clientUid, ListF<PassportUid> uids, ListF<Resource> resources,
            AvailabilityRequest request, ActionInfo actionInfo)
    {
        val clientInfo = userManager.getUserInfo(clientUid);

        final Function1B<UidOrResourceId> canViewAvailabilityF = request.isWithPermsCheck()
                ? authorizer.canViewAvailability(clientInfo, uids, resources).getAvailable()::contains
                : Function1B.trueF();

        val subjects = uids.map(UidOrResourceId.userF())
                .plus(resources.map(Resource.getIdF().andThen(UidOrResourceId.resourceF())));

        val canView = subjects.filter(canViewAvailabilityF);
        val canNotView = subjects.filter(canViewAvailabilityF.notF());

        val availabilities = getSubjectsEventsAvailabilities(Option.of(clientUid), canView, request, actionInfo);

        return toSubjectsAvailabilityIntervals(clientInfo, availabilities, request, actionInfo.getActionSource())
                .plus(canNotView.map(UserOrResourceAvailabilityIntervals.refusalF(AvailabilityQueryRefusalReason.PERMISSION_DENIED)));
    }

    public ListF<AvailabilityOverlapOrRefusal> findFirstAvailabilityOverlaps(
            PassportUid clientUid, ListF<UidOrResourceId> subjectIds,
            AvailabilityRequest request, ActionInfo actionInfo)
    {
        val clientInfo = userManager.getUserInfo(clientUid);

        val participants = authorizer.canViewAvailability(clientInfo, subjectIds);

        val denied = StreamEx.of(participants.getUnavailable())
                .map(s -> AvailabilityOverlapOrRefusal.refusal(s, AvailabilityQueryRefusalReason.PERMISSION_DENIED))
                .toImmutableList();

        val overlaps = findFirstAvailabilityOverlaps(
                Option.of(clientUid), Cf.toList(participants.getAvailable()), request, actionInfo);

        return Cf.toList(denied).plus(overlaps.map(AvailabilityOverlapOrRefusal::overlap));
    }

    public Option<AvailabilityOverlap> busyOverlappingEventsInFuture(
            Option<PassportUid> clientUid, UidOrResourceId subjectId, Instant checkPeriodStart,
            RepetitionInstanceInfo repetition, ListF<Long> exceptEventIds,
            boolean useResourceScheduleCache, ActionInfo actionInfo)
    {
        return busyOverlappingEventsInFuture(
                clientUid, Cf.list(subjectId), checkPeriodStart,
                repetition, exceptEventIds, useResourceScheduleCache, actionInfo).single().get2();
    }

    public Tuple2List<UidOrResourceId, Option<AvailabilityOverlap>> busyOverlappingEventsInFuture(
            Option<PassportUid> clientUid, ListF<UidOrResourceId> subjectIds, Instant checkPeriodStart,
            RepetitionInstanceInfo repetition, ListF<Long> exceptEventIds,
            boolean useResourceScheduleCache, ActionInfo actionInfo)
    {
        RepetitionInstanceSet repetitions = RepetitionInstanceSet.boundedByMaxCheckPeriod(repetition, checkPeriodStart);

        AvailabilityRequest request = AvailabilityRequest.intervals(repetitions)
                .withBusyOnly(true)
                .excludeEventIds(exceptEventIds)
                .withUseResourceCheduleCache(useResourceScheduleCache);

        return findFirstAvailabilityOverlaps(clientUid, subjectIds, request, actionInfo);
    }

    private Tuple2List<UidOrResourceId, ListF<EventAvailability>> getSubjectsEventsAvailabilities(
            Option<PassportUid> uid, ListF<UidOrResourceId> subjects, AvailabilityRequest request, ActionInfo actionInfo)
    {
        if (request.getIntervals().isEmpty()) return subjects.zipWith(Function.constF(Cf.list()));

        ListF<PassportUid> uids = subjects.filterMap(UidOrResourceId.getUidOF());
        ListF<Long> resourceIds = subjects.filterMap(UidOrResourceId.getResourceIdOF());

        ListF<EventAvailability> availabilities = getUsersEventsAvailabilities(uids, request, actionInfo)
                .plus(getResourcesEventsAvailabilities(uid, resourceIds, request, actionInfo));

        MapF<UidOrResourceId, ListF<EventAvailability>> availsBySubject =
                availabilities.groupBy(EventAvailability::getSubject);

        return subjects.zipWith(s -> availsBySubject.getOrElse(s, Cf.list()));
    }

    private ListF<EventAvailability> getUsersEventsAvailabilities(
            ListF<PassportUid> uids, AvailabilityRequest request, ActionInfo actionInfo)
    {
        ListF<LayerUserWithRelations> allLayerUsers = layerRoutines.getLayerUsersWithRelationsByUids(uids);

        ListF<LayerUserWithRelations> layerUsersWithRelations = selectBusyingLayerUsers(allLayerUsers, request);
        ListF<Long> layerIds = layerUsersWithRelations.map(LayerUserWithRelations::getLayerId);

        ListF<EventIndentAndRepetition> events = eventInfoDbLoader.getEventIndentsOnLayers(
                layerIds, getLimits(request));

        events = events.filter(EventIndentAndRepetition.hasZeroLengthF().notF());

        Tuple2List<PassportUid, Long> userLayerIds = layerUsersWithRelations.toTuple2List(
                lu -> lu.getLayerUser().getUid(), LayerUserWithRelations::getLayerId);

        return getAvailabilitiesFromEventUsers(userLayerIds, events, request.isBusyOnly());
    }

    private ListF<EventAvailability> getAvailabilitiesFromEventUsers(
            Tuple2List<PassportUid, Long> userLayerIds, ListF<EventIndentAndRepetition> events, boolean busyOnly)
    {
        MapF<Long, ListF<PassportUid>> uidsByLayerId = userLayerIds.groupBy2();

        Function<EventIndentAndRepetition, ListF<Tuple3<Long, PassportUid, EventIndentAndRepetition>>> spawnF = e -> {
            ListF<PassportUid> uids = uidsByLayerId.getOrElse(e.getIndent().getLayerOrResourceId(), Cf.list());
            return uids.map(uid -> Tuple3.tuple(e.getEventId(), uid, e));
        };

        MapF<Tuple2<Long, PassportUid>, EventIndentAndRepetition> eventByIdAndUid =
                events.flatMap(spawnF).toMap(Tuple3.get12F(), Tuple3.get3F());

        ListF<Availability> avails = Option.when(!busyOnly, Availability.MAYBE).plus1(Availability.BUSY);

        SqlCondition c = EventUserFields.AVAILABILITY.column().inSet(avails)
                .and(EventUserFields.DECISION.ne(Decision.NO))
                .and(EventUserFields.UID.column().inSet(userLayerIds.get1().unique()))
                .and(EventUserFields.EVENT_ID.column().inSet(events.map(EventIndentAndRepetition::getEventId).unique()));

        return eventUserDao.findEventUsers(c).filterMap(eu -> {
            Option<EventIndentAndRepetition> event = eventByIdAndUid
                    .getO(Tuple2.tuple(eu.getEventId(), eu.getUid()));

            return event.map(e -> EventAvailability.eventAndRepetition(
                    event.get(), UidOrResourceId.user(eu.getUid()), eu.getAvailability()));
        });
    }

    private ListF<EventAvailability> getResourcesEventsAvailabilities(
            Option<PassportUid> uid, ListF<Long> resourceIds, AvailabilityRequest request, ActionInfo actionInfo)
    {
        if (request.isUseResourceScheduleCache()) {
            return getResourcesEventsAvailabilitiesFromScheduleCache(uid, resourceIds, request, actionInfo);
        }
        ListF<EventIndentAndRepetition> events = eventInfoDbLoader.getEventIndentsOnResources(
                resourceIds, getLimits(request));

        events = events.filter(EventIndentAndRepetition.hasZeroLengthF().notF());

        ListF<EventAvailability> eventAvailabilities = events.map(event -> EventAvailability.eventAndRepetition(
                event, UidOrResourceId.resource(event.getIndent().getLayerOrResourceId()), Availability.BUSY));

        ListF<ResourceReservationInfo> reservations = resourceReservationManager
                .findReservations(resourceIds, request.getInterval(), uid, actionInfo);

        reservations = reservations.filter(ResourceReservationInfo.hasZeroLengthF().notF());

        ListF<EventAvailability> reservationAvailabilities = reservations
                .map(rr -> EventAvailability.reservationAndRepetition(rr, rr.getResourceId(), Availability.BUSY));

        return eventAvailabilities.plus(reservationAvailabilities);
    }

    private ListF<EventAvailability> getResourcesEventsAvailabilitiesFromScheduleCache(
            Option<PassportUid> uid, ListF<Long> resourceIds, AvailabilityRequest request, ActionInfo actionInfo)
    {
        DateTimeZone tz = uid.map(dateTimeManager.getTimeZoneForUidF()).getOrElse(MoscowTime.TZ);
        RepetitionInstanceSet repetitions = request.getIntervals();

        RepetitionInstanceSet everyDaysSet = RepetitionInstanceSet.fromSuccessiveIntervals(
                AuxDateTime.splitByDays(repetitions.getFirstStart(), repetitions.getLastEnd(), tz));

        ListF<LocalDate> days = repetitions.overlap(everyDaysSet)
                .map(i -> new LocalDate(i.getSecondInterval().getStart(), tz));

        ListF<ResourceDaySchedule> schedules = resourceScheduleManager.getResourceScheduleDataForDays(
                uid, resourceIds, days, tz, request.getExcludeEventIds(), actionInfo);

        SetF<Tuple3<InstantInterval, Either<Long, Long>, Long>> keys = Cf.hashSetWithExpectedSize(
                schedules.iterator().map(s -> s.getSchedule().getIntervals().size()).sum(Cf.Integer));

        return schedules.iterator().<EventAvailability>flatMap(s -> s.getSchedule().getIntervals().iterator()
                .filter(i -> request.getIntervals().overlaps(i.getInterval()))
                .filter(i -> keys.add(Tuple3.tuple(i.getInterval(), i.getEventIdOrReservationId(), s.getResourceId())))
                .map(i -> EventAvailability.eventIdOrReservationInstance(i, s.getResourceId(), Availability.BUSY))
        ).toList();
    }

    private EventLoadLimits getLimits(AvailabilityRequest request) {
        EventLoadLimits limits = EventLoadLimits.intersectsInterval(request.getInterval());
        return request.getExcludeEventIds().isEmpty()
                ? limits
                : limits.withExcludeIds(eventRoutines.findMasterAndSingleEventIds(request.getExcludeEventIds()));
    }

    private ListF<LayerUserWithRelations> selectBusyingLayerUsers(
            ListF<LayerUserWithRelations> layerUsers, AvailabilityRequest request)
    {
        if (!request.isWithAbsencesEvents()) {
            layerUsers = layerUsers.filterNot(lu -> lu.getLayer().getType().isAbsence());
        }
        if (!request.isWithNonAbsencesEvents()) {
            layerUsers = layerUsers.filter(lu -> lu.getLayer().getType().isAbsence());
        }
        return layerUsers.filter(LayerUserWithRelations::affectsAvailability);
    }

    private ListF<UserOrResourceAvailabilityIntervals> toSubjectsAvailabilityIntervals(
            UserInfo clientInfo, Tuple2List<UidOrResourceId, ListF<EventAvailability>> subjectsAvailabilities,
            AvailabilityRequest request, ActionSource actionSource)
    {
        ListF<EventAvailability> availabilities = subjectsAvailabilities.get2().flatten();
        MapF<Long, AvailabilityEventInfo> eventInfoById = loadEventInfos(
                clientInfo, request, availabilities, actionSource);

        MapF<PassportUid, SettingsInfo> reservationCreatorByUid = settingsRoutines.getSettingsByUidBatch(
                availabilities.filterMap(EventAvailability::getReservationCreatorUid));

        Comparator<AvailabilityInterval> comparator = AvailabilityInterval.comparator;

        ListF<UserOrResourceAvailabilityIntervals> intervalss = Cf.arrayList();
        for (Tuple2<UidOrResourceId, ListF<EventAvailability>> t : subjectsAvailabilities) {
            UidOrResourceId subjectId = t.get1();

            ListF<AvailabilityInterval> intervals = Cf.arrayList();
            for (EventAvailability avail : t.get2()) {
                Option<Long> eventId = avail.getEventId();

                ListF<InstantInterval> instances = avail.getOverlappingIntervals(request.getIntervals());

                AvailabilityEventInfo eventInfo;
                if (eventId.isPresent()) {
                    if (!eventInfoById.containsKeyTs(eventId.get())) continue;

                    eventInfo = eventInfoById.getOrThrow(eventId.get());
                } else {
                    SettingsInfo organizer = reservationCreatorByUid.getOrThrow(avail.getReservationCreatorUid().get());
                    eventInfo = AvailabilityEventInfo.empty().withName("").withOrganizer(organizer);
                }
                if (subjectId.isResource()) {
                    eventInfo = eventInfo.withoutResources();
                }
                Object debugInfo = avail.getDebugInfo();

                for (InstantInterval instance : instances) {
                    Option<EventInterval> eventInterval = Option.empty();
                    if (eventInfo.getName().isPresent() && avail.getEventAndRepetition().isPresent()) {
                        eventInterval = Option.of(avail.getEventAndRepetition().get().getEventInterval(instance));

                    } else if (eventInfo.getName().isPresent()) {
                        eventInterval = Option.of(new EventInterval(instance.getStart(),
                                EventDateTime.dateTime(instance.getStart()),
                                EventDateTime.dateTime(instance.getEnd())));
                    }
                    intervals.add(new AvailabilityInterval(
                            instance, avail.getAvailability(), eventInfo.withInterval(eventInterval), debugInfo));
                }
            }
            AvailabilityIntervals i = new AvailabilityIntervals(
                    intervals.sorted(comparator), request.getInterval().getStart(), request.getInterval().getEnd());
            intervalss.add(UserOrResourceAvailabilityIntervals.intervals(subjectId, i));
        }
        return intervalss;
    }

    private MapF<Long, AvailabilityEventInfo> loadEventInfos(
            UserInfo clientInfo, AvailabilityRequest request,
            ListF<EventAvailability> availabilities, ActionSource actionSource)
    {
        ListF<EventIndentAndRepetition> indents = availabilities.filterMap(EventAvailability::getEventAndRepetition);
        ListF<Long> eventIds = availabilities.filterMap(
                a -> a.getEventId().isPresent() && !a.getEvent().isPresent() ? a.getEventId() : Option.empty());

        EventGetProps egp = EventGetProps.none();

        egp = egp.loadEventFields(Cf.<MapField<?>>list()
                .plus(Option.when(request.isWithEventsNames(), EventFields.TYPE))
                .plus(Option.when(request.isWithEventsNames(), EventFields.NAME))
                .plus(Option.when(request.isWithEventsResources(), EventFields.LOCATION))
                .plus(Option.when(request.isWithActions(), EventFields.SEQUENCE)));

        if (request.isWithActions()) {
            egp = egp.loadEventWithRelationsWithSequenceNums();
        }

        if (request.isWithEventsParticipants() || request.isWithEventsOrganizers()) {
            egp = egp.loadEventParticipants();
        }
        if (request.isWithEventsResources()) {
            egp = egp.loadEventResources();
        }

        MapF<Long, EventInfo> eventInfoById = Cf.hashMap();
        Function1V<EventInfo> putF = ei -> eventInfoById.put(ei.getEventId(), ei);

        if (indents.isNotEmpty() && request.isWithAnyEventInfo()) {
            eventInfoDbLoader.getEventInfosByIndents(Option.of(clientInfo), egp, indents, actionSource).forEach(putF);
        }
        eventIds = eventIds.filterNot(eventInfoById::containsKeyTs);

        if (eventIds.isNotEmpty() && request.isWithAnyEventInfo()) {
            eventInfoDbLoader.getEventInfosByIdsSafe(Option.of(clientInfo), egp, eventIds, actionSource).forEach(putF);
        }

        boolean needPermsCheck = request.isWithPermsCheck()
                && (request.isWithEventsNames() || egp.isWithEventParticipants() || egp.isWithEventResources());

        MapF<Long, AvailabilityEventInfo> result = Cf.hashMap();

        for (EventAvailability availability : availabilities.filter(EventAvailability::hasEventId)) {
            long eventId = availability.getEventId().get();
            boolean isNotExternalResource = availability.getSubject().isResource() && !clientInfo.isExternalYt();

            Option<EventInfo> eventO;

            if (request.isWithAnyEventInfo()) {
                eventO = eventInfoById.getO(eventId);
                if (!eventO.isPresent()) {
                    continue;
                }
            } else {
                eventO = Option.empty();
            }

            boolean canView = !needPermsCheck || eventO.exists(e -> authorizer.canViewEvent(
                    clientInfo, e.getInfoForPermsCheck(), actionSource));

            Option<String> name = eventO.flatMapO(e ->
                    Option.when(canView && request.isWithEventsNames(), () -> e.getEvent().getName()));

            Option<EventType> type = eventO.flatMapO(e ->
                    Option.when(canView && request.isWithEventsNames(), () -> e.getEvent().getType()));

            Option<ParticipantInfo> organizer = (isNotExternalResource || canView) && request.isWithEventsOrganizers()
                    ? eventO.flatMapO(e -> e.getParticipants().getOrganizerSafe())
                    : Option.empty();

            ListF<ResourceInfo> resources = canView && request.isWithEventsResources()
                    ? eventO.flatMap(EventInfo::getResources)
                    : Cf.list();

            Option<String> location = canView && request.isWithEventsResources()
                    ? eventO.flatMapO(e -> StringUtils.notEmptyO(e.getEvent().getLocation()))
                    : Option.empty();

            Option<Participants> participants = canView && request.isWithEventsParticipants()
                    ? eventO.map(EventInfo::getParticipants)
                    : Option.empty();
            Option<EventActions> actions = request.isWithActions() && canView && eventInfoById.getO(eventId).isPresent()
                    ? Option.of(eventRoutines.getEventActions(clientInfo, eventInfoById.getO(eventId).get().getEventWithRelations(), Option.empty(), ActionSource.WEB))
                    : Option.empty();

            result.put(eventId, new AvailabilityEventInfo(
                    Option.of(eventId), name, Option.empty(), type,
                    organizer.filterMap(ParticipantInfo.getSettingsIfUserF()),
                    resources,
                    location,
                    participants,
                    actions, eventO.isPresent() && request.isWithActions() ? eventO.get().getEventFieldValueO(EventFields.SEQUENCE) : Option.empty()));
        }
        return result;
    }

    private Tuple2List<UidOrResourceId, Option<AvailabilityOverlap>> findFirstAvailabilityOverlaps(
            Option<PassportUid> uid, ListF<UidOrResourceId> subjectIds,
            AvailabilityRequest request, ActionInfo actionInfo)
    {
        RepetitionInstanceSet repetitions = request.getIntervals();

        if (repetitions.isEmpty()) return subjectIds.zipWith(Function.constF(Option.<AvailabilityOverlap>empty()));

        Tuple2List<UidOrResourceId, ListF<EventAvailability>> subjectsAvailabilities =
                getSubjectsEventsAvailabilities(uid, subjectIds, request, actionInfo);

        Tuple2List<UidOrResourceId, Option<AvailabilityOverlap>> overlaps = Tuple2List.arrayList();

        for (Tuple2<UidOrResourceId, ListF<EventAvailability>> subjectAvailabilities : subjectsAvailabilities) {
            Option<AvailabilityOverlap> firstMostBusy = Option.empty();
            Instant upperSearchBound = repetitions.getLastEnd();

            for (EventAvailability avail : subjectAvailabilities._2) {
                if (firstMostBusy.exists(o -> o.getAvailability().isMoreBusy(avail.getAvailability()))) {
                    continue;
                }
                RepetitionInstanceSet availRepetitions = new RepetitionInstanceSet(
                        avail.getRepetitionInfo(), repetitions.getFirstStart(), upperSearchBound);

                Option<IntersectingIntervals> intervals = repetitions.findFirstOverlap(availRepetitions);
                Either<Long, ResourceReservation> eventIdOrReservation = avail.getEventIdOrReservation();

                if (intervals.isPresent()) {
                    AvailabilityOverlap candidate = new AvailabilityOverlap(avail.getAvailability(),
                            new EventIdOrReservationInterval(eventIdOrReservation, intervals.get().getSecondInterval()),
                            intervals.get().getFirstInterval());

                    if (!firstMostBusy.isPresent() || candidate.isMoreBusyOrSameButStartsBefore(firstMostBusy.get())) {
                        firstMostBusy = Option.of(candidate);
                    }
                    if (candidate.getAvailability() == Availability.BUSY) {
                        upperSearchBound = candidate.getStart();
                    }
                }
            }
            overlaps.add(subjectAvailabilities._1, firstMostBusy);
        }
        return overlaps;
    }

    public static Element xmlizeAvailability(
            ListF<AvailabilityInterval> intervals, DateTimeZone clientTimezone, Option<Office> office)
    {
        XmlBuilder xb = XmlBuilders.defaultBuilder();
        xmlizeAvailability(intervals, clientTimezone, office, xb);
        return JdomUtils.I.readRootElement(xb.toValidXmlString());
    }

    public static void xmlizeAvailability(
            ListF<AvailabilityInterval> intervals, DateTimeZone clientTimezone, Option<Office> office, XmlWriter xw)
    {
        xw.startElement("events");
        for (AvailabilityInterval i : intervals) {

            xw.startElement("event");

            DateTimeFormatter formatter = ISODateTimeFormat.dateTime().withZone(clientTimezone);
            xw.textElement("start", formatter.print(i.getInterval().getStart()));
            xw.addComment(ObjectUtils.toString(i.getDebugInfo()).replaceAll("--", "+"));
            xw.textElement("end", formatter.print(i.getInterval().getEnd()));
            if (i.getEventName().isPresent()) {
                xw.textElement("name", i.getEventName().get());
            }

            ListF<ResourceInfo> resources = i.getResources().toList();

            if (office.isPresent()) {
                resources = resources.sorted(new ResourcesForStaffComparator(office.get()));
            }

            for (ResourceInfo resource : resources) {
                if (resource.getName().isPresent() || resource.getResource().getExchangeName().isPresent()) {
                    xw.emptyElement("resource");
                    xw.addAttribute("name", resource.getName().getOrElse(""));
                    xw.addAttribute("id", resource.getResource().getExchangeName().getOrElse(""));
                }
            }

            xw.endElement();

        }
        xw.endElement();
    }

} //~
