package ru.yandex.calendar.logic.event.avail;

import net.fortuna.ical4j.model.parameter.FbType;
import net.fortuna.ical4j.model.property.Transp;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.ObjectUtils;

/**
 * @author akirakozov
 * @author Sergey Shinderuk
 */
public enum Availability implements Comparable<Availability>, StringEnum {
    AVAILABLE(FbType.FREE, Option.of(Transp.TRANSPARENT)),
    MAYBE(FbType.BUSY_TENTATIVE, Option.<Transp>empty()),
    BUSY(FbType.BUSY, Option.of(Transp.OPAQUE)),
    ;

    public static final StringEnumResolver<Availability> R = StringEnumResolver.r(Availability.class);

    private final FbType fbType;
    private final Option<Transp> transp;

    private Availability(FbType fbType, Option<Transp> transp) {
        this.fbType = fbType;
        this.transp = transp;
    }

    public boolean isMoreBusy(Availability other) {
        return this.compareTo(other) > 0;
    }

    public static Comparator<Availability> moreBusyFirstComparator() {
        return Comparator.<Availability>naturalComparator().reversed();
    }

    public Availability max(Availability other) { return ObjectUtils.max(this, other); }

    public static Availability find(String s) {
        return R.valueOf(s);
    }

    public String toDbValue() {
        return CamelWords.parse(this.name()).toDbName();
    }

    @Override
    public String value() {
        return toDbValue();
    }

    public FbType toFbType() {
        return fbType;
    }

    public Option<Transp> toTransp() {
        return transp;
    }

    private static final MapF<Transp, Availability> byTransp = Cf.list(values())
            .zipWithFlatMapO(transpF())
            .invert()
            .toMap();

    public static Availability byIcalTransparency(Transp transparency) {
        return byTransp.getOrThrow(transparency);
    }

    public static Availability byDecision(Decision decision) {
        if (decision == Decision.YES) return BUSY;
        if (decision == Decision.MAYBE) return MAYBE;
        if (decision == Decision.NO) return AVAILABLE;
        if (decision == Decision.UNDECIDED) return MAYBE;
        throw new AssertionError("unexpected decision: " + decision);
    }

    private static Function<Availability, Option<Transp>> transpF() {
        return new Function<Availability, Option<Transp>>() {
            public Option<Transp> apply(Availability availability) {
                return availability.transp;
            }
        };
    }

}
