package ru.yandex.calendar.logic.event.avail;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.event.EventActions;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.inside.passport.PassportUid;

/**
 * @author dbrylev
 */
public class AvailabilityEventInfo {
    private final Option<Long> eventId;
    private final Option<String> name;
    private final Option<EventInterval> interval;
    private final Option<EventType> type;
    private final Option<SettingsInfo> organizer;
    private final Option<String> location;
    private final ListF<ResourceInfo> resources;
    private final Option<Participants> participants;
    private final Option<EventActions> eventActions;
    private final Option<Integer> sequence;

    public AvailabilityEventInfo(
            Option<Long> eventId, Option<String> name,
            Option<EventInterval> interval, Option<EventType> type,
            Option<SettingsInfo> organizer,
            ListF<ResourceInfo> resources, Option<String> location,
            Option<Participants> participants,
            Option<EventActions> eventActions,
            Option<Integer> sequence)
    {
        this.eventId = eventId;
        this.name = name;
        this.interval = interval;
        this.type = type;
        this.organizer = organizer;
        this.resources = resources;
        this.location = location;
        this.participants = participants;
        this.eventActions = eventActions;
        this.sequence = sequence;
    }

    public static AvailabilityEventInfo empty() {
        return new AvailabilityEventInfo(
                Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Cf.list(), Option.empty(), Option.empty(), Option.empty(), Option.empty());
    }

    public static AvailabilityEventInfo resources(ListF<ResourceInfo> resources) {
        return new AvailabilityEventInfo(
                Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), resources, Option.empty(), Option.empty(), Option.empty(), Option.empty());
    }

    public AvailabilityEventInfo withInterval(Option<EventInterval> interval) {
        return new AvailabilityEventInfo(eventId, name, interval, type, organizer, resources, location, participants,
                eventActions, sequence);
    }

    public AvailabilityEventInfo withoutResources() {
        return new AvailabilityEventInfo(eventId, name, interval, type, organizer, Cf.list(), location, participants,
                eventActions, sequence);
    }

    public AvailabilityEventInfo withOrganizer(SettingsInfo organizer) {
        return new AvailabilityEventInfo(eventId, name, interval, type, Option.of(organizer), resources, location, participants,
                eventActions, sequence);
    }

    public AvailabilityEventInfo withName(String name) {
        return new AvailabilityEventInfo(eventId, Option.of(name), interval, type, organizer, resources, location, participants,
                eventActions, sequence);
    }

    public AvailabilityEventInfo withEventActions(EventActions actions) {
        return new AvailabilityEventInfo(eventId, name, interval, type, organizer, resources, location, participants,
                Option.of(actions), sequence);
    }

    public boolean isEmpty() {
        return !name.isPresent() && !organizer.isPresent() && resources.isEmpty() && !participants.isPresent();
    }

    public Option<PassportUid> getOrganizerUid() {
        return organizer.map(SettingsInfo.getUidF());
    }

    public ListF<Long> getResourceIds() {
        return resources.map(ResourceInfo.resourceIdF());
    }

    public Option<Long> getEventId() {
        return eventId;
    }

    public Option<EventInterval> getInterval() {
        return interval;
    }

    public Option<Instant> getInstanceStartTs() {
        return interval.map(EventInterval::getInstanceStart);
    }

    public Option<String> getName() {
        return name;
    }

    public Option<EventType> getType() {
        return type;
    }

    public Option<SettingsInfo> getOrganizer() {
        return organizer;
    }

    public ListF<ResourceInfo> getResources() {
        return resources;
    }

    public Option<String> getLocation() {
        return location;
    }

    public Option<Participants> getParticipants() {
        return participants;
    }

    public Option<EventActions> getActions() {
        return eventActions;
    }

    public Option<Integer> getSequenceId() {
        return sequence;
    }
}
