package ru.yandex.calendar.logic.event.avail;

import javax.annotation.Nullable;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.event.EventActions;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.IntervalComparator;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.util.dates.InstantIntervalSet;
import ru.yandex.commune.mapObject.MapObject;
import ru.yandex.commune.util.serialize.reflect.ReflectionToMultilineStringSerializeMe;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author ssytnik
 */
@ReflectionToMultilineStringSerializeMe
public class AvailabilityInterval {
    public static final Comparator<AvailabilityInterval> comparator =
            IntervalComparator.INSTANCE.compose(AvailabilityInterval::getInterval);

    final InstantInterval interval;
    final Availability availability;

    final AvailabilityEventInfo eventInfo;
    final @Nullable Object debugInfo;

    public AvailabilityInterval(
            InstantInterval interval, Availability availability,
            AvailabilityEventInfo eventInfo, @Nullable Object debugInfo)
    {
        Validate.isTrue(availability.isMoreBusy(Availability.AVAILABLE));
        this.interval = interval;
        this.availability = availability;
        this.eventInfo = eventInfo;
        this.debugInfo = debugInfo;
    }

    @Override
    public String toString() {
        return "(" + interval + "," + availability + "," + eventInfo.getName() + ",...)";
    }

    public InstantInterval getInterval() {
        return interval;
    }

    public Availability getAvailability() {
        return availability;
    }

    public Option<Long> getEventId() {
        return eventInfo.getEventId();
    }

    public Option<String> getEventName() {
        return eventInfo.getName();
    }

    public Option<EventType> getEventType() {
        return eventInfo.getType();
    }

    public Option<Instant> getInstanceStartTs() {
        return eventInfo.getInstanceStartTs();
    }

    public Option<EventInterval> getEventInterval() {
        return eventInfo.getInterval();
    }

    public Option<Boolean> getIsAllDay() {
        return eventInfo.getInterval().map(i -> i.getStart().isDate());
    }

    public Option<SettingsInfo> getOrganizer() {
        return eventInfo.getOrganizer();
    }

    public ListF<ResourceInfo> getResources() {
        return eventInfo.getResources();
    }

    public Option<String> getEventLocation() {
        return eventInfo.getLocation();
    }

    public Option<Participants> getParticipants() {
        return eventInfo.getParticipants();
    }

    public Option<EventActions> getActions() {
        return eventInfo.getActions();
    }

    public Option<Integer> getSequenceId() {
        return eventInfo.getSequenceId();
    }

    @Nullable
    public Object getDebugInfo() {
        return debugInfo;
    }

    public AvailabilityInterval merge(AvailabilityInterval that) {
        Validate.V.equals(this.getInterval(), that.getInterval());
        Validate.V.none(this.getEventName());
        Validate.V.none(that.getEventName());
        Validate.V.none(this.getParticipants());
        Validate.V.none(that.getParticipants());

        ListF<ResourceInfo> resources = this.getResources().plus(that.getResources())
                .stableUniqueBy(ResourceInfo.resourceIdF());

        return new AvailabilityInterval(
                this.getInterval(), ObjectUtils.max(this.getAvailability(), that.getAvailability()),
                AvailabilityEventInfo.resources(resources), Tuple2.tuple(this.getDebugInfo(), that.getDebugInfo()));
    }

    public static Function2<AvailabilityInterval, AvailabilityInterval, AvailabilityInterval> mergeF() {
        return new Function2<AvailabilityInterval, AvailabilityInterval, AvailabilityInterval>() {
            public AvailabilityInterval apply(AvailabilityInterval thizz, AvailabilityInterval thadd) {
                return thizz.merge(thadd);
            }
        };
    }

    public boolean hasSameAvailability(AvailabilityInterval other) {
        return availability == other.availability;
    }


    private SetF<Long> resourceIdsCache;

    private SetF<Long> getResourceIds() {
        if (resourceIdsCache == null) {
            resourceIdsCache = getResources().map(ResourceInfo.resourceIdF()).unique();
        }
        return resourceIdsCache;
    }

    public boolean hasSameResources(AvailabilityInterval other) {
        return getResourceIds().equals(other.getResourceIds());
    }

    public AvailabilityInterval withInterval(InstantInterval newI) {
        return new AvailabilityInterval(newI, availability, eventInfo, debugInfo);
    }

    public Function<InstantInterval, AvailabilityInterval> withIntervalF() {
        return new Function<InstantInterval, AvailabilityInterval>() {
            public AvailabilityInterval apply(InstantInterval newI) {
                return AvailabilityInterval.this.withInterval(newI);
            }
        };
    }

    public static final Function<AvailabilityInterval, InstantInterval> intervalF = new Function<AvailabilityInterval, InstantInterval>() {
        public InstantInterval apply(AvailabilityInterval availInterval) {
            return availInterval.getInterval();
        }
    };

    public static final Function<AvailabilityInterval, Option<String>> eventNameOF = AvailabilityInterval::getEventName;

    public static Function<AvailabilityInterval, Instant> startF = new Function<AvailabilityInterval, Instant>() {
        public Instant apply(AvailabilityInterval availInterval) {
            return availInterval.getInterval().getStart();
        }
    };

    public static Function<AvailabilityInterval, AvailabilityInterval> withAvailabilityF(final Availability availability) {
        return new Function<AvailabilityInterval, AvailabilityInterval>() {
            public AvailabilityInterval apply(AvailabilityInterval availInterval) {
                return new AvailabilityInterval(
                        availInterval.interval, availability, availInterval.eventInfo, availInterval.debugInfo);
            }
        };
    }

    public static Function1B<AvailabilityInterval> isMoreBusyF(final Availability availability) {
        return new Function1B<AvailabilityInterval>() {
            public boolean apply(AvailabilityInterval availInterval) {
                return availInterval.getAvailability().isMoreBusy(availability);
            }
        };
    }

    public ListF<AvailabilityInterval> minusInterval(InstantInterval intervalToCut) {
        return InstantIntervalSet.minus(getInterval(), intervalToCut).getIntervals().map(withIntervalF());
    }

    public AvailabilityInterval withStart(Instant newStart) {
        return new AvailabilityInterval(
                interval.withStart(newStart), availability, eventInfo, debugInfo);
    }

    public AvailabilityInterval withEnd(Instant newEnd) {
        return new AvailabilityInterval(
                interval.withEnd(newEnd), availability, eventInfo, debugInfo);
    }

    public static Function<EventInstanceInfo, AvailabilityInterval> fromEventInstanceInfoF() {
        return ei -> {
            MapObject debugInfo = ei.getEvent().getFields(EventFields.ID, EventFields.MAIN_EVENT_ID);

            AvailabilityEventInfo eventInfo = new AvailabilityEventInfo(
                    Option.of(ei.getEventId()),
                    Option.of(ei.getEvent().getName()),
                    Option.of(ei.getEventInterval()),
                    Option.of(ei.getEvent().getType()),
                    ei.getEventWithRelations().getParticipants().getOrganizerSafe().filterMap(ParticipantInfo.getSettingsIfUserF()),
                    ei.getEventWithRelations().getResources(),
                    StringUtils.notEmptyO(ei.getEvent().getLocation()),
                    Option.of(ei.getEventWithRelations().getParticipants()),
                    Option.empty(), Option.when(ei.getEventUser().isPresent(), ei.getEventUser().get().getSequence()));

            return new AvailabilityInterval(
                    new InstantInterval(ei.getEvent().getStartTs(), ei.getEvent().getEndTs()),
                    ei.getEventUserWithNotifications().get().getEventUser().getAvailability(), eventInfo, debugInfo);
        };
    }

    public static Function<AvailabilityInterval, Availability> getAvailabilityF() {
        return new Function<AvailabilityInterval, Availability>() {
            public Availability apply(AvailabilityInterval availabilityInterval) {
                return availabilityInterval.getAvailability();
            }
        };
    }

    public static Function<AvailabilityInterval, Instant> getEndF() {
        return intervalF.andThen(InstantInterval::getEnd);
    }
}
