package ru.yandex.calendar.logic.event.avail;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author gutman
 */
public class AvailabilityIntervalSet {
    private final ListF<AvailabilityInterval> intervals;

    public AvailabilityIntervalSet(ListF<AvailabilityInterval> intervals) {
        for (Tuple2<AvailabilityInterval, AvailabilityInterval> t : intervals.zip(intervals.drop(1))) {
            Validate.V.isTrue(!t._1.getInterval().getEnd().isAfter(t._2.getInterval().getEnd()));
            if (t._1.hasSameAvailability(t._2) && t._1.hasSameResources(t._2)) {
                Validate.V.isTrue(t._1.getInterval().getEnd().isBefore(t._2.getInterval().getStart()));
            }
        }
        this.intervals = intervals;
    }

    public AvailabilityIntervalSet() {
        this(Cf.<AvailabilityInterval>list());
    }

    public static ListF<AvailabilityInterval> createAvailabilityTimeline(ListF<AvailabilityInterval> intervals) {
        Validate.V.forAll(intervals, AvailabilityInterval.eventNameOF.andThen(Cf2.f1B(n -> !n.isPresent())));
        return intervals.foldLeft(new AvailabilityIntervalSet(), new Function2<AvailabilityIntervalSet, AvailabilityInterval, AvailabilityIntervalSet>() {
            public AvailabilityIntervalSet apply(AvailabilityIntervalSet set, AvailabilityInterval interval) {
                return set.plus(new AvailabilityIntervalSet(Cf.list(interval)));
            }
        }).intervals;
    }

    public AvailabilityIntervalSet plus(AvailabilityIntervalSet set) {
        AvailabilityIntervalSet pureAddition = set.minus(this);
        AvailabilityIntervalSet pureKeep = this.minus(set);

        ListF<AvailabilityInterval> intersection = intersect(set).map(AvailabilityInterval.mergeF());

        ListF<AvailabilityInterval> i = pureAddition.intervals.plus(pureKeep.intervals).plus(intersection).sortedBy(AvailabilityInterval.startF);

        return new AvailabilityIntervalSet(Cf2.merge(i, new Function2<AvailabilityInterval, AvailabilityInterval, ListF<AvailabilityInterval>>() {
            public ListF<AvailabilityInterval> apply(AvailabilityInterval i1, AvailabilityInterval i2) {
                if (i1.getInterval().abuts(i2.getInterval()) && i1.hasSameAvailability(i2) && i1.hasSameResources(i2)) {
                    return Cf.list(i1.withEnd(i2.getInterval().getEnd()));
                } else {
                    return Cf.list(i1, i2);
                }
            }
        }));
    }

    private AvailabilityIntervalSet minus1(final AvailabilityInterval sub) {
        return new AvailabilityIntervalSet(intervals.flatMap(new Function<AvailabilityInterval, ListF<AvailabilityInterval>>() {
            public ListF<AvailabilityInterval> apply(AvailabilityInterval interval) {
                return interval.minusInterval(sub.getInterval());
            }
        }));
    }

    private AvailabilityIntervalSet minus(AvailabilityIntervalSet sub) {
        return intervals.foldLeft(sub, new Function2<AvailabilityIntervalSet, AvailabilityInterval, AvailabilityIntervalSet>() {
            public AvailabilityIntervalSet apply(AvailabilityIntervalSet set, AvailabilityInterval interval) {
                return set.minus1(interval);
            }
        });
    }

    private Tuple2List<AvailabilityInterval, AvailabilityInterval> intersect(final AvailabilityIntervalSet that) {
        AvailabilityIntervalSet newThis = this;
        AvailabilityIntervalSet newThat = that;

        for (;;) {
            if (newThis.isEmpty() || newThat.isEmpty()) {
                return Cf.Tuple2List.cons();
            }

            if (newThis.getStart().equals(newThat.getStart())) {
                break;
            }

            Instant newStart = ObjectUtils.max(newThis.getStart(), newThat.getStart());
            newThis = newThis.dropBefore(newStart);
            newThat = newThat.dropBefore(newStart);
        }

        Instant newFirstEnd = ObjectUtils.min(newThis.intervals.first().getInterval().getEnd(), newThat.intervals.first().getInterval().getEnd());

        AvailabilityInterval newThisNewFirst = newThis.intervals.first().withEnd(newFirstEnd);
        AvailabilityInterval newThatNewFirst = newThat.intervals.first().withEnd(newFirstEnd);

        return Cf.Tuple2List.fromPairs(newThisNewFirst, newThatNewFirst).plus(newThis.dropBefore(newFirstEnd).intersect(newThat.dropBefore(newFirstEnd)));
    }

    private Instant getStart() {
        return this.intervals.first().getInterval().getStart();
    }

    private AvailabilityIntervalSet dropBefore(Instant instant) {
        if (isEmpty()) {
            return this;
        }
        if (intervals.first().getInterval().getStart().isAfter(instant)) {
            return this;
        }
        if (intervals.first().getInterval().getEnd().isAfter(instant)) {
            return new AvailabilityIntervalSet(Cf.list(intervals.first().withStart(instant)).plus(intervals.drop(1)));
        }
        return new AvailabilityIntervalSet(intervals.drop(1)).dropBefore(instant);
    }

    private boolean isEmpty() {
        return intervals.isEmpty();
    }

}
