package ru.yandex.calendar.logic.event.avail;

import java.util.LinkedHashMap;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 */
@ru.yandex.commune.util.serialize.reflect.ReflectionToMultilineStringSerializeMe
public class AvailabilityIntervals {
    private final ListF<AvailabilityInterval> availIntervals;
    private final Instant start;
    private final Instant end;

    public AvailabilityIntervals(ListF<AvailabilityInterval> availIntervals, Instant start, Instant end) {
        Validate.V.forAll(availIntervals, AvailabilityInterval.isMoreBusyF(Availability.AVAILABLE));

        this.availIntervals = availIntervals;
        this.start = start;
        this.end = end;
    }

    public AvailabilityIntervals withAvailability(Availability availability) {
        return new AvailabilityIntervals(availIntervals.map(AvailabilityInterval.withAvailabilityF(availability)), start, end);
    }

    public AvailabilityIntervals bounded() {
        return new AvailabilityIntervals(getBounded(availIntervals), start, end);
    }

    public ListF<AvailabilityInterval> mergedByDays(DateTimeZone tz) {
        MapF<LocalDate, ListF<AvailabilityInterval>> byDays =
                Cf.<LocalDate, ListF<AvailabilityInterval>>x(new LinkedHashMap<>());

        for (AvailabilityInterval availInterval : availIntervals) {
            for (InstantInterval interval : AuxDateTime.splitByDays(availInterval.interval, tz)) {
                LocalDate date = new LocalDate(interval.getStart(), tz);
                byDays.getOrElseUpdate(date, Cf::arrayList).add(availInterval.withInterval(interval));
            }
        }
        ListF<AvailabilityInterval> result = Cf.arrayList();
        byDays.forEach((date, intervals) -> {
            Tuple2<ListF<AvailabilityInterval>, ListF<AvailabilityInterval>> namedAllDayAndNot =
                    intervals.partition(i -> i.getEventName().isPresent() && i.getIsAllDay().isSome(true)); // CAL-7343

            result.addAll(namedAllDayAndNot.get1());
            result.addAll(merged(namedAllDayAndNot.get2(), true));
        });

        return result;
    }

    public AvailabilityIntervals forTz(DateTimeZone tz) {
        Tuple2<ListF<AvailabilityInterval>, ListF<AvailabilityInterval>> allDayAndNot =
                availIntervals.partition(ai -> ai.getEventInterval().exists(ei -> ei.getStart().isDate()));

        if (allDayAndNot.get1().isEmpty()) return this;

        ListF<AvailabilityInterval> allDays = allDayAndNot.get1()
                .map(ai -> ai.withInterval(ai.getEventInterval().get().toInstantInterval(tz)));

        return new AvailabilityIntervals(
                allDays.plus(allDayAndNot.get2()).sorted(AvailabilityInterval.comparator), start, end);
    }

    private static ListF<AvailabilityInterval> merged(
            ListF<AvailabilityInterval> intervals, boolean smoothUnnamed)
    {
        Tuple2<ListF<AvailabilityInterval>, ListF<AvailabilityInterval>> namedUnnamed =
                intervals.partition(a -> a.getEventName().isPresent());

        ListF<AvailabilityInterval> named = namedUnnamed._1;
        ListF<AvailabilityInterval> unnamed = namedUnnamed._2;
        ListF<AvailabilityInterval> unnamedSuccessive = createSuccessiveAvailIntervals(unnamed, smoothUnnamed);

        for (AvailabilityInterval availInterval : named) {
            unnamedSuccessive = cutOutInterval(unnamedSuccessive, availInterval.getInterval());
        }
        return unnamedSuccessive.plus(named).sorted(AvailabilityInterval.comparator);
    }

    public ListF<AvailabilityInterval> merged() {
        return merged(availIntervals, false);
    }

    public ListF<AvailabilityInterval> unmerged() {
        return availIntervals;
    }

    public ListF<InstantInterval> getIntervals() {
        return availIntervals.map(AvailabilityInterval.intervalF);
    }

    private static ListF<AvailabilityInterval> cutOutInterval(ListF<AvailabilityInterval> successiveIntervals, final InstantInterval intervalToCut) {
        return successiveIntervals.flatMap(new Function<AvailabilityInterval, ListF<AvailabilityInterval>>() {
            public ListF<AvailabilityInterval> apply(AvailabilityInterval availInterval) {
                return availInterval.minusInterval(intervalToCut);
            }
        });
    }

    private ListF<AvailabilityInterval> getBounded(ListF<AvailabilityInterval> availIntervals) {
        InstantInterval bound = new InstantInterval(start.getMillis(), end.getMillis());
        ListF<AvailabilityInterval> replacements = Cf.arrayList();
        for (AvailabilityInterval ai : availIntervals) {
            InstantInterval i = ai.getInterval();
            if (bound.contains(i)) {
                replacements.add(ai);
            } else {
                if (bound.overlaps(i)) {
                    InstantInterval replacement = bound.overlap(i);
                    replacements.add(ai.withInterval(replacement));
                }
            }
        }
        return replacements;
    }

    public static ListF<AvailabilityInterval> createSuccessiveAvailIntervals(
            ListF<AvailabilityInterval> intervals, boolean smooth)
    {
        if (smooth) {
            return Cf2.merge(intervals.sortedBy(AvailabilityInterval.startF), (first, second) -> {
                InstantInterval one = first.interval;
                InstantInterval two = second.interval;

                if (!one.getEnd().isBefore(two.getStart())) {
                    return Cf.list(new AvailabilityInterval(
                            one.withEnd(ObjectUtils.max(one.getEnd(), two.getEnd())),
                            ObjectUtils.max(first.availability, second.availability),
                            AvailabilityEventInfo.empty(), null));
                }
                return Cf.list(first, second);
            });
        }
        return AvailabilityIntervalSet.createAvailabilityTimeline(intervals);
    }

    public static Function<AvailabilityIntervals, Option<Availability>> getMostBusyAvailabilityF() {
        return new Function<AvailabilityIntervals, Option<Availability>>() {
            public Option<Availability> apply(AvailabilityIntervals availabilityIntervals) {
                return availabilityIntervals.availIntervals.map(AvailabilityInterval.getAvailabilityF()).maxO();
            }
        };
    }

    public static Function<AvailabilityIntervals, ListF<InstantInterval>> getIntervalsF() {
        return new Function<AvailabilityIntervals, ListF<InstantInterval>>() {
            public ListF<InstantInterval> apply(AvailabilityIntervals is) {
                return is.getIntervals();
            }
        };
    }

}
