package ru.yandex.calendar.logic.event.avail;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.ResourceReservation;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.resource.schedule.EventIdOrReservationInterval;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class AvailabilityOverlap {
    private final Availability availability;
    private final InstantInterval overlap;

    private final EventIdOrReservationInterval eventInterval;
    private final InstantInterval instanceInterval;

    public AvailabilityOverlap(
            Availability availability, EventIdOrReservationInterval eventInterval, InstantInterval instanceInterval)
    {
        this.availability = availability;
        this.overlap = RepetitionUtils.overlapStartsAbutInclusive(
                eventInterval.getInterval(), instanceInterval).getOrThrow("No overlap");
        this.eventInterval = eventInterval;
        this.instanceInterval = instanceInterval;
    }

    public boolean isMoreBusy(AvailabilityOverlap that) {
        return this.availability.isMoreBusy(that.availability);
    }

    public boolean startsBefore(AvailabilityOverlap that) {
        return this.getStart().isBefore(that.getStart());
    }

    public boolean isMoreBusyOrSameButStartsBefore(AvailabilityOverlap that) {
        return this.isMoreBusy(that) || !that.isMoreBusy(this) && this.startsBefore(that);
    }

    public Either<Long, ResourceReservation> getEventIdOrReservation() {
        return eventInterval.getEventIdOrReservation();
    }

    public Availability getAvailability() {
        return availability;
    }

    public InstantInterval getOverlap() {
        return overlap;
    }

    public InstantInterval getInstanceInterval() {
        return instanceInterval;
    }

    public Instant getStart() {
        return overlap.getStart();
    }

    public LocalDate getStartDate(DateTimeZone tz) {
        return new LocalDate(getStart(), tz);
    }

    public static Function<AvailabilityOverlap, Instant> getStartF() {
        return new Function<AvailabilityOverlap, Instant>() {
            public Instant apply(AvailabilityOverlap o) {
                return o.getStart();
            }
        };
    }
}
