package ru.yandex.calendar.logic.event.avail;

import org.joda.time.ReadableInstant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceSet;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Daniel Brylev
 */
public class AvailabilityRequest {
    private final RepetitionInstanceSet intervals;
    private final boolean withPermsCheck;
    private final boolean withEventsNames;
    private final boolean withEventsOrganizers;
    private final boolean withEventsResources;
    private final boolean withEventsParticipants;
    private final boolean withAbsencesEvents;
    private final boolean withNonAbsencesEvents;
    private final boolean withActions;
    private final ListF<Long> excludeEventIds;
    private final boolean useResourceScheduleCache;
    private final boolean busyOnly;

    private AvailabilityRequest(
            RepetitionInstanceSet intervals, boolean withPermsCheck,
            boolean withEventsNames, boolean withEventsOrganizers, boolean withEventsResources, boolean withEventsParticipants,
            boolean withAbsencesEvents, boolean withNonAbsencesEvents,
            ListF<Long> excludeEventIds, boolean useResourceScheduleCache, boolean busyOnly, boolean withActions)
    {
        this.intervals = intervals;
        this.withPermsCheck = withPermsCheck;
        this.withEventsNames = withEventsNames;
        this.withEventsResources = withEventsResources;
        this.withEventsParticipants = withEventsParticipants;
        this.withEventsOrganizers = withEventsOrganizers;
        this.withAbsencesEvents = withAbsencesEvents;
        this.withNonAbsencesEvents = withNonAbsencesEvents;
        this.excludeEventIds = excludeEventIds;
        this.useResourceScheduleCache = useResourceScheduleCache;
        this.busyOnly = busyOnly;
        this.withActions = withActions;
    }

    public static AvailabilityRequest intervals(RepetitionInstanceSet intervals) {
        return new AvailabilityRequest(intervals, true, false, false, false, false, true, true, Option.empty(), true, false, false);
    }

    public static AvailabilityRequest interval(InstantInterval interval) {
        return intervals(RepetitionInstanceSet.fromSuccessiveIntervals(Cf.list(interval)));
    }

    public static AvailabilityRequest interval(ReadableInstant start, ReadableInstant end) {
        return interval(new InstantInterval(start, end));
    }

    public AvailabilityRequest excludeEventId(Option<Long> eventId) {
        return excludeEventIds(eventId);
    }

    public AvailabilityRequest excludeEventIds(ListF<Long> eventIds) {
        Builder b = builder();
        b.excludeEventIds = eventIds;
        return b.build();
    }


    public AvailabilityRequest includeEventsNames() {
        Builder b = builder();
        b.withEventsNames = true;
        return b.build();
    }

    public AvailabilityRequest includeActions() {
        Builder b = builder();
        b.withActions = true;
        return b.build();
    }

    public AvailabilityRequest includeEventsOrganizers() {
        Builder b = builder();
        b.withEventsOrganizers = true;
        return b.build();
    }

    public AvailabilityRequest includeEventsResources() {
        Builder b = builder();
        b.withEventsResources = true;
        return b.build();
    }

    public AvailabilityRequest includeEventsParticipants() {
        Builder b = builder();
        b.withEventsParticipants = true;
        return b.build();
    }

    public AvailabilityRequest withoutPermsCheck() {
        Builder b = builder();
        b.withPermsCheck = false;
        return b.build();
    }

    public AvailabilityRequest excludeAbsencesEvents() {
        Builder b = builder();
        b.withAbsencesEvents = false;
        return b.build();
    }

    public AvailabilityRequest excludeNonAbsencesEvents() {
        Builder b = builder();
        b.withNonAbsencesEvents = false;
        return b.build();
    }

    public AvailabilityRequest withBusyOnly(boolean busyOnly) {
        Builder b = builder();
        b.busyOnly = busyOnly;
        return b.build();
    }

    public AvailabilityRequest withUseResourceCheduleCache(boolean use) {
        Builder b = builder();
        b.useResourceScheduleCache = use;
        return b.build();
    }

    public AvailabilityRequest includeEventsNamesWithoutPermsCheck() {
        return includeEventsNames().withoutPermsCheck();
    }

    public AvailabilityRequest includeEventsInfo() {
        return includeEventsNames().includeEventsOrganizers().includeEventsResources().includeEventsParticipants();
    }

    public InstantInterval getInterval() { return new InstantInterval(intervals.getFirstStart(), intervals.getLastEnd()); }

    public RepetitionInstanceSet getIntervals() {
        return intervals;
    }

    public boolean isWithEventsNames() {
        return withEventsNames;
    }

    public boolean isWithEventsOrganizers() {
        return withEventsOrganizers;
    }

    public boolean isWithEventsResources() {
        return withEventsResources;
    }

    public boolean isWithEventsParticipants() {
        return withEventsParticipants;
    }

    public boolean isWithPermsCheck() {
        return withPermsCheck;
    }

    public boolean isWithAbsencesEvents() {
        return withAbsencesEvents;
    }

    public boolean isWithNonAbsencesEvents() {
        return withNonAbsencesEvents;
    }

    public ListF<Long> getExcludeEventIds() {
        return excludeEventIds;
    }

    public boolean isUseResourceScheduleCache() {
        return useResourceScheduleCache;
    }

    public boolean isBusyOnly() {
        return busyOnly;
    }

    public boolean isWithActions() {
        return withActions;
    }

    public boolean isWithAnyEventInfo() {
        return withEventsNames || withEventsOrganizers || withEventsResources || withEventsParticipants;
    }

    private Builder builder() {
        return new Builder(
                intervals, withPermsCheck,
                withEventsNames, withEventsOrganizers, withEventsResources, withEventsParticipants,
                withAbsencesEvents, withNonAbsencesEvents, excludeEventIds, useResourceScheduleCache,
                busyOnly, withActions);
    }

    private static class Builder {
        RepetitionInstanceSet intervals;
        boolean withPermsCheck;
        boolean withEventsNames;
        boolean withEventsOrganizers;
        boolean withEventsResources;
        boolean withEventsParticipants;
        boolean withAbsencesEvents;
        boolean withNonAbsencesEvents;
        boolean withActions;
        ListF<Long> excludeEventIds;
        boolean useResourceScheduleCache;
        boolean busyOnly;

        Builder(
                RepetitionInstanceSet intervals, boolean withPermsCheck,
                boolean withEventsNames, boolean withEventsOrganizers,
                boolean withEventsResources, boolean withEventsParticipants,
                boolean withAbsencesEvents, boolean withNonAbsencesEvents, ListF<Long> excludeEventIds,
                boolean useResourceScheduleCache, boolean busyOnly, boolean withActions)
        {
            this.intervals = intervals;
            this.withPermsCheck = withPermsCheck;
            this.withEventsNames = withEventsNames;
            this.withEventsOrganizers = withEventsOrganizers;
            this.withEventsResources = withEventsResources;
            this.withEventsParticipants = withEventsParticipants;
            this.withAbsencesEvents = withAbsencesEvents;
            this.withNonAbsencesEvents = withNonAbsencesEvents;
            this.excludeEventIds = excludeEventIds;
            this.useResourceScheduleCache = useResourceScheduleCache;
            this.busyOnly = busyOnly;
            this.withActions = withActions;
        }

        AvailabilityRequest build() {
            return new AvailabilityRequest(
                    intervals, withPermsCheck,
                    withEventsNames, withEventsOrganizers, withEventsResources, withEventsParticipants,
                    withAbsencesEvents, withNonAbsencesEvents, excludeEventIds, useResourceScheduleCache,
                    busyOnly, withActions);
        }
    }
}
