package ru.yandex.calendar.logic.event.avail;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.ResourceReservation;
import ru.yandex.calendar.logic.beans.generated.ResourceReservationFields;
import ru.yandex.calendar.logic.event.EventIndent;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetition;
import ru.yandex.calendar.logic.event.repetition.IntersectingIntervals;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceSet;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationInfo;
import ru.yandex.calendar.logic.resource.schedule.EventIdOrReservationInterval;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public abstract class EventAvailability {
    private final UidOrResourceId subject;
    private final Availability availability;

    private EventAvailability(UidOrResourceId subject, Availability availability) {
        this.subject = subject;
        this.availability = availability;
    }

    public Availability getAvailability() {
        return availability;
    }

    public UidOrResourceId getSubject() {
        return subject;
    }

    public abstract RepetitionInstanceInfo getRepetitionInfo();

    public ListF<InstantInterval> getOverlappingIntervals(RepetitionInstanceSet repetitionSet) {
        return repetitionSet.overlap(getRepetitionInfo()).map(IntersectingIntervals::getSecondInterval);
    }

    public boolean hasEventId() {
        return getEventId().isPresent();
    }

    public abstract Option<EventIndent> getEvent();

    public abstract Option<Long> getEventId();

    public abstract Option<ResourceReservation> getReservation();

    public Option<EventIndentAndRepetition> getEventAndRepetition() {
        return getEvent().map(Cf2.f2(EventIndentAndRepetition::new).bind2(getRepetitionInfo()));
    }

    public Option<PassportUid> getReservationCreatorUid() {
        return getReservation().map(ResourceReservation.getCreatorUidF());
    }

    public abstract Object getDebugInfo();



    public static EventAndRepeat eventAndRepetition(
            EventIndentAndRepetition event, UidOrResourceId subject, Availability availability)
    {
        return new EventAndRepeat(event, subject, availability);
    }

    public static EventOrReservationInstance eventIdOrReservationInstance(
            EventIdOrReservationInterval interval, long resourceId, Availability availability)
    {
        return new EventOrReservationInstance(interval, UidOrResourceId.resource(resourceId), availability);
    }

    public static ReservationAndRepetition reservationAndRepetition(
            ResourceReservationInfo reservation, long resourceId, Availability availability)
    {
        return new ReservationAndRepetition(reservation, UidOrResourceId.resource(resourceId), availability);
    }

    public Either<Long, ResourceReservation> getEventIdOrReservation() {
        return Either.fromOptions(getEventId(), getReservation());
    }

    public static class EventAndRepeat extends EventAvailability {
        private final EventIndentAndRepetition event;

        public EventAndRepeat(EventIndentAndRepetition event, UidOrResourceId subject, Availability availability) {
            super(subject, availability);
            this.event = event;
        }

        @Override
        public RepetitionInstanceInfo getRepetitionInfo() {
            return event.getRepetitionInfo();
        }

        @Override
        public Option<EventIndent> getEvent() {
            return Option.of(event.getIndent());
        }

        @Override
        public Option<Long> getEventId() {
            return Option.of(event.getIndent().getEventId());
        }

        @Override
        public Option<ResourceReservation> getReservation() {
            return Option.empty();
        }

        @Override
        public Object getDebugInfo() {
            Event info = new Event();
            info.setId(event.getIndent().getEventId());
            info.setStartTs(event.getIndent().getStart());
            info.setEndTs(event.getIndent().getEnd());
            info.setMainEventId(event.getIndent().getMainEventId());

            return info;
        }
    }

    public static class ReservationAndRepetition extends EventAvailability {
        private final ResourceReservationInfo reservation;

        public ReservationAndRepetition(
                ResourceReservationInfo reservation, UidOrResourceId subject, Availability availability)
        {
            super(subject, availability);
            this.reservation = reservation;
        }

        @Override
        public RepetitionInstanceInfo getRepetitionInfo() {
            return reservation.getRepetitionInfo();
        }

        @Override
        public Option<EventIndent> getEvent() {
            return Option.empty();
        }

        @Override
        public Option<Long> getEventId() {
            return Option.empty();
        }

        @Override
        public Option<ResourceReservation> getReservation() {
            return Option.of(reservation.getReservation());
        }

        @Override
        public Object getDebugInfo() {
            return reservation.getReservation().getFields(
                    );
        }
    }

    public static class EventOrReservationInstance extends EventAvailability {
        private final EventIdOrReservationInterval interval;

        public EventOrReservationInstance(
                EventIdOrReservationInterval interval, UidOrResourceId subject, Availability availability)
        {
            super(subject, availability);
            this.interval = interval;
        }

        @Override
        public RepetitionInstanceInfo getRepetitionInfo() {
            return RepetitionInstanceInfo.noRepetition(interval.getInterval());
        }

        @Override
        public Option<EventIndent> getEvent() {
            return Option.empty();
        }

        @Override
        public Option<Long> getEventId() {
            return interval.getEventId();
        }

        @Override
        public Option<ResourceReservation> getReservation() {
            return interval.getReservation();
        }

        @Override
        public Object getDebugInfo() {
            return interval.isEvent()
                    ? LogMarker.EVENT_ID.format(interval.getEventId().get())
                    : interval.getReservation().get().getFields(
                            ResourceReservationFields.RESERVATION_ID, ResourceReservationFields.START_TS);
        }
    }
}
