package ru.yandex.calendar.logic.event.avail.absence;

import java.util.Optional;

import javax.inject.Inject;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.frontend.ews.exp.EwsExportRoutines;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventHelper;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.EventChangesInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventUserRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.log.change.EventChangesJson;
import ru.yandex.calendar.logic.log.change.changes.EventFieldsChangesJson;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.Staff;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.login.PassportLogin;
import ru.yandex.misc.log.reqid.RequestIdStack;

import static ru.yandex.calendar.logic.event.avail.absence.AbsenceType.ABSENCE;
import static ru.yandex.calendar.logic.event.avail.absence.AbsenceType.DUTY;
import static ru.yandex.calendar.logic.event.avail.absence.AbsenceType.OFFICE_WORK;
import static ru.yandex.calendar.logic.event.avail.absence.AbsenceType.TRIP;
import static ru.yandex.calendar.logic.event.avail.absence.AbsenceType.WORK_FROM_HOME;

public class AbsenceUpdater {
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(AbsenceUpdater.class);

    @Autowired
    private UserManager userManger;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private EwsExportRoutines ewsExportRoutines;
    @Autowired
    private LastUpdateManager lastUpdateManager;

    @Autowired
    private Staff staff;

    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Inject
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private EventsLogger eventsLogger;

    public void update() {
        DateTime midnight = DateTime.now(DateTimeZone.UTC).withTimeAtStartOfDay();
        DateTime nextYearMidnight = midnight.plusYears(1);

        updateFromTime(staff.getAbsences(midnight.toInstant(), nextYearMidnight.toInstant()), midnight.toInstant());
    }

    void updateFromTime(ListF<Absence> absences, Instant time) {
        MapF<PassportLogin, ListF<Absence>> absenceByLogin = absences.groupBy(Absence::getLogin);

        ListF<Long> toRemoveIds = Cf.toArrayList(
                eventDao.findAllNonFinishedAbsenceEventIds(time));

        Tuple2List<Long, Participants> eventsWithParticipants =
                eventInvitationManager.getParticipantsByEventIds(toRemoveIds);

        Tuple2<Tuple2List<Long,Participants>,Tuple2List<Long,Participants>> x =
                eventsWithParticipants.partitionBy2(Participants::isMeetingOrInconsistent);

        ListF<Long> toRemoveMeetings = x._1.get1();
        ListF<Long> toRemoveNonMeetings = x._2.get1();

        if (toRemoveMeetings.isNotEmpty()) {
            log.warn("Meetings on absence layer: {}", toRemoveMeetings);
        }

        ActionInfo actionInfo = new ActionInfo(ActionSource.STAFF, RequestIdStack.current().orElse("?"), new Instant());
        for (PassportLogin login : absenceByLogin.keySet()) {
            final Optional<PassportUid> uidO = userManger.getYtUserUidByLogin(login.getRawValue());

            if (uidO.isPresent()) {
                ListF<Long> updatedEventIds = updateForUser(uidO.get(), absenceByLogin.getTs(login), actionInfo);

                lastUpdateManager.updateMainEventAndLayerTimestamps(updatedEventIds, actionInfo);

                toRemoveNonMeetings = toRemoveNonMeetings.filter(updatedEventIds.containsF().notF());
            } else {
                log.warn("Uid not found by login {}, can't do anything", login);
            }
        }
        if (toRemoveIds.isNotEmpty()) {
            eventRoutines.deleteEventsFromDbAndExchange(ActorId.yaCalendar(), toRemoveNonMeetings, actionInfo, true);
        }
    }

    private ListF<Long> updateForUser(PassportUid uid, ListF<Absence> absences, ActionInfo actionInfo) {
        ListF<Long> updatedEventIds = Cf.arrayList();
        DateTimeZone tz = dateTimeManager.getTimeZoneForUid(uid);
        Language lang = settingsRoutines.getLanguage(uid);

        long layerId = layerRoutines.getOrCreateAndGetAbsenceLayerId(uid);
        for (Absence absence : absences) {
            String externalId = absence.getExternalId() + "gap.yandex-team.ru";

            Option<Event> eventO = eventDao.findEventsByLayerIdAndExternalIdForUpdate(layerId, externalId).singleO();

            if (eventO.isPresent()) {
                final Event event = eventO.get();
                final long eventId = event.getId();

                DateTimeZone eventTz = eventRoutines.getEventTimeZone(eventId);

                Event eventFields = new Event();
                eventFields.setId(eventId);
                eventFields.setName(absence.getEventName(lang));
                eventFields.setDescription(absence.getComment());
                eventFields.setStartTs(absence.getStart(eventTz));
                eventFields.setEndTs(absence.getEnd(eventTz));
                eventFields.setIsAllDay(absence.isFullDay());
                eventFields.setModificationSource(ActionSource.STAFF);
                eventFields.setModificationReqId(actionInfo.getRequestIdWithHostId());
                eventFields.setType(resolveEventType(absence));
                eventDbManager.updateEvent(eventFields, actionInfo);

                EventAndRepetition eventAndRepetition = eventDbManager.getEventAndRepetitionByEvent(event);
                eventDbManager.updateEventLayersAndResourcesIndents(eventAndRepetition, actionInfo);
                eventUserRoutines.updateEventUserAvailability(uid, eventId, resolveAvailability(absence), actionInfo);

                updatedEventIds.add(eventFields.getId());

                Event updated = event.copy();
                updated.setFields(eventFields);

                if (eventDbManager.getIsExportedWithEws(event)) {
                    Event eventChanges = EventHelper.INSTANCE.findChanges(event, updated).getFieldsIfSet(
                            EventFields.NAME, EventFields.DESCRIPTION, EventFields.START_TS,
                            EventFields.END_TS, EventFields.IS_ALL_DAY);

                    EventChangesInfo.EventChangesInfoFactory factory = new EventChangesInfo.EventChangesInfoFactory();
                    factory.setEventChanges(eventChanges);
                    EventChangesInfo eventChangesInfo = factory.create();

                    ewsExportRoutines.exportToExchangeIfNeededOnUpdate(
                            updated.getId(), eventChangesInfo, Option.empty(), actionInfo);
                }

                eventsLogger.log(EventChangeLogEvents.updated(
                        ActorId.yaCalendar(), new EventIdLogDataJson(externalId, event),
                        EventChangesJson.empty().withEvent(EventFieldsChangesJson.of(Optional.of(event), updated))), actionInfo);

            } else {
                EventData eventData = createEventData(layerId, absence, externalId, lang, tz);
                long mainEventId = eventRoutines.createMainEvent(uid, eventData, actionInfo);


                eventRoutines.createUserOrFeedEvent(
                        UidOrResourceId.user(uid), resolveEventType(absence), mainEventId, eventData,
                        NotificationsData.create(Cf.list()),
                        InvitationProcessingMode.SAVE_ONLY, actionInfo);
            }
        }
        return updatedEventIds;
    }

    private EventData createEventData(long layerId, Absence absence, String extId, Language lang, DateTimeZone tz) {
        EventData eventData = new EventData();
        eventData.getEvent().setName(absence.getEventName(lang));
        eventData.getEvent().setDescription(absence.getComment());
        eventData.getEvent().setStartTs(absence.getStart(tz));
        eventData.getEvent().setEndTs(absence.getEnd(tz));
        eventData.getEvent().setIsAllDay(absence.isFullDay());
        eventData.getEvent().setType(resolveEventType(absence));
        eventData.setLayerId(layerId);
        eventData.setEventUserData(createEventUserData(absence));
        eventData.setExternalId(Option.of(extId));
        eventData.setTimeZone(tz);
        return eventData;
    }

    private EventType resolveEventType(Absence absence) {
        AbsenceType absenceType = absence.getType();
        switch (absenceType) {
            case TRIP: return EventType.TRIP;
            case ILLNESS: return EventType.ILLNESS;
            case ABSENCE: case PAID_DAY_OFF: return EventType.ABSENCE;
            case LEARNING: return EventType.LEARNING;
            case VACATION: return EventType.VACATION;
            case CONFERENCE_TRIP: return EventType.CONFERENCE_TRIP;
            case CONFERENCE: return EventType.CONFERENCE;
            case MATERNITY: return EventType.MATERNITY;
            case DUTY: return EventType.DUTY;
            case WORK_FROM_HOME: return EventType.REMOTE_WORK;
            case OFFICE_WORK: return EventType.OFFICE_WORK;
            default: return EventType.USER;
        }
    }

    private static Availability resolveAvailability(Absence absence) {
        if (absence.getType() == OFFICE_WORK || absence.getType() == WORK_FROM_HOME) {
            return Availability.AVAILABLE;
        }

        if ((absence.getType() == ABSENCE || absence.getType() == DUTY) && absence.isWorkInAbsence()) {
            return Availability.AVAILABLE;
        }

        // CAL-4598
        if (absence.getType() == TRIP) {
            return Availability.AVAILABLE;
        }

        return Availability.BUSY;
    }

    private static EventUserData createEventUserData(Absence absence) {
        final EventUser eventUser = new EventUser();
        eventUser.setAvailability(resolveAvailability(absence));
        return new EventUserData(eventUser, Cf.list());
    }

    public void setEwsExportRoutinesForTest(EwsExportRoutines ewsExportRoutines) {
        this.ewsExportRoutines = ewsExportRoutines;
    }
}
