package ru.yandex.calendar.logic.event.dao;

import java.util.NoSuchElementException;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3List;
import ru.yandex.calendar.logic.beans.Bean;
import ru.yandex.calendar.logic.beans.BeanHelper;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventAttachmentFields;
import ru.yandex.calendar.logic.beans.generated.EventAttachmentHelper;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventHelper;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserHelper;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.RdateFields;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.beans.generated.RepetitionHelper;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.RecurrenceIdOrMainEvent;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.repetition.RecurrenceTimeInfo;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.sharing.perm.EventActionClass;
import ru.yandex.calendar.logic.sharing.perm.EventFieldsForPermsCheck;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.calendar.util.db.FieldsAssignment;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.mapObject.db.MapObjectFieldsRowMapper;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.annotation.SampleValue;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlPart;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.db.resultSet.ResultSetUtils;
import ru.yandex.misc.db.resultSet.Tuple2RowMapper;

public class EventDao extends CalendarJdbcDaoSupport {
    @Autowired
    private GenericBeanDao genericBeanDao;

    public void updateEvent(Event event) {
        genericBeanDao.updateBean(event);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateEventLastUpdateTsAndEventModificationInfo(long eventId, Instant lastUpdateTs, ActionInfo actionInfo) {
        Event temp = new Event();
        temp.setLastUpdateTs(lastUpdateTs);
        temp.setModificationSource(actionInfo.getActionSource());
        temp.setModificationReqId(actionInfo.getRequestIdWithHostId());
        temp.setId(eventId);
        updateEvent(temp);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateEventModificationInfo(long eventId, ActionInfo actionInfo) {
        Event temp = new Event();
        temp.setModificationSource(actionInfo.getActionSource());
        temp.setModificationReqId(actionInfo.getRequestIdWithHostId());
        temp.setId(eventId);
        genericBeanDao.updateBean(temp);
    }

    @RunWithRandomTest(possible=IncorrectResultSizeDataAccessException.class)
    public PassportUid findCreatorUidByEventId(long eventId) {
        return findCreatorUidsByEventIds(Cf.list(eventId)).single().get2();
    }

    @RunWithRandomTest(possible=IncorrectResultSizeDataAccessException.class)
    public Tuple2List<Long, PassportUid> findCreatorUidsByEventIds(ListF<Long> eventIds) {
        String q = "SELECT id, creator_uid FROM event WHERE id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return Tuple2List.tuple2List();

        MapF<Long, PassportUid> x = getJdbcTemplate().queryForList2(q, Long.class, Long.class)
                .map2(PassportUid::cons).toMap();
        try {
            return eventIds.zipWith(x::getOrThrow);
        } catch (NoSuchElementException e) {
            throw new IncorrectResultSizeDataAccessException(eventIds.size(), x.size());
        }
    }

    public void updateEventIncrementSequenceById(long eventId) {
        updateEventsIncrementSequenceByIds(Cf.list(eventId));
    }

    @RunWithRandomTest
    public void updateEventsIncrementSequenceByIds(ListF<Long> eventIds) {
        String q = "UPDATE event SET sequence = sequence + 1 WHERE id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public String findExternalIdByEventId(long eventId) {
        // XXX possible deadlock
        String q = "SELECT external_id FROM main_event WHERE id = " +
                "(SELECT main_event_id FROM event e WHERE e.id = ?)";
        return getJdbcTemplate().queryForObject(q, String.class, eventId);
    }

    /**
     * @see LayerDao#findLayerById(long)
     */
    public Event findEventById(long id) {
        return findEventById(id, false);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Event findEventById(long id, boolean forUpdate) {
        String q = "SELECT * FROM event WHERE id = ?" + forUpdateSql(forUpdate);
        return lockedForUpdate(Option.of(getJdbcTemplate().queryForObject(q, Event.class, id)), forUpdate).single();
    }

    /**
     * @see LayerDao#findLayersByIds(ListF)
     */
    public ListF<Event> findEventsByIds(ListF<Long> ids) {
        return findEventsByIds(ids, false);
    }

    @RunWithRandomTest(possible=IncorrectResultSizeDataAccessException.class)
    public ListF<Event> findEventsByIds(ListF<Long> ids, boolean forUpdate) {
        SqlCondition c = EventFields.ID.column().inSet(ids);
        String q = "SELECT * FROM event" + c.whereSql() + forUpdateSql(forUpdate);

        if (skipQuery(c, q, c.args())) return Cf.list();

        ListF<Event> events = lockedForUpdate(getJdbcTemplate().queryForList(q, Event.class, c.args()), forUpdate);

        MapF<Long, Event> eventsById = events.toMapMappingToKey(Event::getId);
        try {
            return ids.map(eventsById::getOrThrow);
        } catch (NoSuchElementException e) {
            throw new IncorrectResultSizeDataAccessException("events not found by some of ids " + ids, ids.size(), eventsById.size());
        }
    }

    @RunWithRandomTest
    public ListF<Long> lockEventsByIds(ListF<Long> eventIds) {
        SqlCondition c = EventFields.ID.column().inSet(eventIds);
        String q = "SELECT id FROM event" + c.whereSql() + " FOR UPDATE";

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public ListF<Long> lockEventsByLayerId(long layerId) {
        String nested = "SELECT e.id FROM event e"
                + " INNER JOIN event_layer el ON e.id = el.event_id"
                + " WHERE el.layer_id = ?";

        String query = "SELECT id FROM event WHERE id IN (" + nested + ") FOR UPDATE";
        return getJdbcTemplate().queryForList(query, Long.class, layerId);
    }

    public ListF<Event> findEventsByIdsSafe(ListF<Long> ids) {
        return findEventsByIdsSafe(ids, false);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByIdsSafe(ListF<Long> ids, boolean forUpdate) {
        return findEvents(forUpdate, EventFields.ID.column().inSet(ids));
    }

    @RunWithRandomTest
    public Tuple2List<Long, String> findEventsNameByIdsSafe(ListF<Long> ids) {
        SqlCondition c = EventFields.ID.column().inSet(ids);
        String q = "SELECT id, name FROM event" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(q,
                EventFields.ID.getDatabaser().rowMapper(),
                EventFields.NAME.getDatabaser().rowMapper(), c.args());
    }

    @RunWithRandomTest
    public Tuple3List<Long, String, String> findEventsNameAndLocationByIdsSafe(ListF<Long> ids) {
        SqlCondition c = EventFields.ID.column().inSet(ids);
        String q = "SELECT id, name, location FROM event" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Tuple3List.tuple3List();

        return getJdbcTemplate().query3(q,
                EventFields.ID.getDatabaser().rowMapper(),
                EventFields.NAME.getDatabaser().rowMapper(),
                EventFields.LOCATION.getDatabaser().rowMapper(), c.args());
    }

    public ListF<Event> findEventsFieldsByIdsSafe(ListF<Long> ids, ListF<MapField<?>> fields) {
        ListF<MapField<?>> extFields = fields.plus(EventFields.ID).stableUnique();

        SqlCondition c = EventFields.ID.column().inSet(ids);
        String q = "SELECT " + BeanHelper.columns(extFields, "") + " FROM event" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().query(q, BeanHelper.fieldsRowMapper(Event.class, extFields, ""), c.args());
    }

    @RunWithRandomTest
    public ListF<EventFieldsForPermsCheck> findEventsFieldsForPermsCheckByIdsSafe(ListF<Long> ids) {
        return findEventsFieldsByIdsSafe(ids, EventFieldsForPermsCheck.FIELDS).map(EventFieldsForPermsCheck::fromEvent);
    }

    @RunWithRandomTest
    public boolean findEventExistsById(long eventId) {
        return getJdbcTemplate().queryForOption("SELECT 1 FROM event WHERE id = ?", Integer.class, eventId).isPresent();
    }

    private ListF<Event> findEvents(
            ListF<? extends MapField<?>> onlyFields, boolean forUpdate, SqlCondition... conditions)
    {
        SqlCondition c = SqlCondition.all(conditions);
        // XXX: list instead of *
        String q = "SELECT * FROM event" + c.whereSql() + forUpdateSql(forUpdate);

        if (skipQuery(c, q, c.args())) return Cf.list();

        MapObjectFieldsRowMapper<Event> rm = new MapObjectFieldsRowMapper<>(EventFields.OBJECT_DESCRIPTION, onlyFields);
        return lockedForUpdate(getJdbcTemplate().query(q, rm, c.args()), forUpdate);
    }

    private ListF<Event> findEvents(boolean forUpdate, SqlCondition... conditions) {
        return findEvents(EventFields.OBJECT_DESCRIPTION.getFields(), forUpdate, conditions);
    }

    @RunWithRandomTest
    public ListF<Event> findEvents(SqlCondition... conditions) {
        return findEvents(false, conditions);
    }

    public ListF<Event> findEventsByMainId(long mainEventId) {
        return findEventsByMainId(mainEventId, false);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByMainId(long mainEventId, boolean forUpdate) {
        return findEvents(forUpdate, EventFields.MAIN_EVENT_ID.eq(mainEventId));
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByMainIds(ListF<Long> mainEventId) {
        return findEvents(EventFields.MAIN_EVENT_ID.column().inSet(mainEventId));
    }

    @RunWithRandomTest(possible = EmptyResultDataAccessException.class)
    public Event findEventByEventUserId(long eventUserId) {
        String sql = "SELECT e.* FROM event_user eu" +
                " INNER JOIN event e ON e.id = eu.event_id" +
                " WHERE eu.id = ?";
        return getJdbcTemplate().queryForObject(sql, Event.class, eventUserId);
    }

    @RunWithRandomTest
    public ListF<Event> findUserOrganizedEvents(PassportUid uid) {
        String q = "SELECT e.* FROM event_user eu" +
                " INNER JOIN event e ON e.id = eu.event_id" +
                " WHERE eu.is_organizer = TRUE and eu.uid = ?";
        return getJdbcTemplate().queryForList(q, Event.class, uid);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByLayerId(long layerId) {
        String sql = "SELECT e.* FROM event e" +
            " INNER JOIN event_layer el ON el.event_id = e.id" +
            " WHERE el.layer_id = ?";
        return getJdbcTemplate().queryForList(sql, Event.class, layerId);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByLayerIdInTimeRange(long layerId, Instant from, Instant to) {
        String sql = "SELECT e.* FROM event e" +
                " INNER JOIN event_layer el ON el.event_id = e.id" +
                " WHERE el.layer_id = ? AND NOT (e.start_ts >= ? OR e.end_ts <= ?)";
        return getJdbcTemplate().queryForList(sql, Event.class, layerId, to, from);
    }

    @RunWithRandomTest
    public ListF<Event> findMasterEventByMainId(long mainEventId) {
        return findMasterEventByMainId(mainEventId, false);
    }

    @RunWithRandomTest
    public ListF<Event> findMasterEventByMainId(long mainEventId, boolean forUpdate) {
        String q = "SELECT * FROM event WHERE main_event_id = ? AND recurrence_id IS NULL" + forUpdateSql(forUpdate);
        return lockedForUpdate(getJdbcTemplate().queryForList(q, Event.class, mainEventId), forUpdate);
    }

    @RunWithRandomTest
    public Option<Event> findMasterEventByEventId(long eventId) {
        // XXX possible deadlock
        String q = "SELECT * FROM event WHERE recurrence_id IS NULL AND main_event_id =" +
                " (SELECT main_event_id FROM event WHERE id = ?)";
        return getJdbcTemplate().queryForOption(q, Event.class, eventId);
    }

    @RunWithRandomTest
    public Option<Long> findMainEventIdByEventId(long eventId) {
        return findMainEventIdsByEventIds(Cf.list(eventId)).singleO();
    }

    @RunWithRandomTest
    public ListF<Long> findMainEventIdsByEventIds(ListF<Long> eventIds) {
        SqlCondition c = EventFields.ID.column().inSet(eventIds);
        String q = "SELECT main_event_id FROM event" + c.whereSql();

        return skipQuery(c, q, c.args()) ? Cf.list() : getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public ListF<Event> findRecurrenceEventByMainId(long mainEventId, Instant recurrenceId) {
        return findRecurrenceEventsByMainId(mainEventId, Cf.list(recurrenceId), false);
    }

    @RunWithRandomTest
    public ListF<Event> findRecurrenceEventsByMainId(long mainEventId, ListF<Instant> recurrenceIds, boolean forUpdate) {
        SqlCondition c = EventFields.MAIN_EVENT_ID.eq(mainEventId)
                .and(EventFields.RECURRENCE_ID.column().inSet(recurrenceIds));

        String q = "SELECT * FROM event" + c.whereSql() + forUpdateSql(forUpdate);

        if (skipQuery(c, q, c.args())) return Cf.list();

        return lockedForUpdate(getJdbcTemplate().queryForList(q, Event.class, c.args()), forUpdate);
    }

    public ListF<Event> findEventByMainIdAndRecurrenceId(long mainId, RecurrenceIdOrMainEvent id, boolean forUpdate) {
        if (id.getRecurrenceId().isPresent()) {
            return findRecurrenceEventsByMainId(mainId, id.getRecurrenceId(), forUpdate);
        } else {
            return findMasterEventByMainId(mainId, forUpdate);
        }
    }

    @RunWithRandomTest
    public ListF<Event> findMasterEventsWithPossibleIds(ListF<Long> ids) {
        String q = "SELECT * FROM event WHERE recurrence_id IS NULL AND id " + SqlQueryUtils.inSet(ids);

        if (skipQuery(ids, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Event.class);
    }

    @RunWithRandomTest
    public ListF<Event> findRecurrenceEventsWithPossibleIds(ListF<Long> ids) {
        String q = "SELECT * FROM event WHERE recurrence_id IS NOT NULL AND id " + SqlQueryUtils.inSet(ids);

        if (skipQuery(ids, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Event.class);
    }

    @RunWithRandomTest
    public ListF<Event> findRecurrenceEventsByMainId(long mainEventId) {
        SqlCondition c = EventFields.RECURRENCE_ID.column().isNotNull().and(EventFields.MAIN_EVENT_ID.eq(mainEventId));
        return getJdbcTemplate().queryForList("SELECT * FROM event" + c.whereSql(), Event.class, c.args());
    }

    @RunWithRandomTest
    public ListF<Long> findEventIdsByMainEventId(long mainEventId) {
        return findEventIdsByMainEventIds(Cf.list(mainEventId));
    }

    @RunWithRandomTest
    public ListF<Long> findEventIdsByMainEventIds(ListF<Long> mainEventIds) {
        return findEvents(Cf.list(EventFields.ID), false, EventFields.MAIN_EVENT_ID.column().inSet(mainEventIds))
            .map(EventFields.ID.getF());
    }

    @RunWithRandomTest
    public ListF<Long> findFutureOrMasterEventIdsByMainEventIds(ListF<Long> mainEventIds, ActionInfo actionInfo) {
        SqlCondition c = EventFields.MAIN_EVENT_ID.column().inSet(mainEventIds)
                .and(EventFields.RECURRENCE_ID.column().isNull()
                        .or(EventFields.END_TS.column().gt(actionInfo.getNow())));

        String q = "SELECT id FROM event" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public ListF<Event> findFutureRecurrencesForUpdate(long mainEventId, ActionInfo actionInfo) {
         return findEvents(true, EventFields.MAIN_EVENT_ID.eq(mainEventId)
                 .and(EventFields.RECURRENCE_ID.column().isNotNull())
                 .and(EventFields.END_TS.column().gt(actionInfo.getNow())));
    }

    @RunWithRandomTest
    public ListF<Long> findEventIdsByMainEventIdAndRecurrenceIdGe(long mainEventId, Instant start) {
        SqlCondition c = EventFields.MAIN_EVENT_ID.eq(mainEventId).and(EventFields.RECURRENCE_ID.ge(start));

        String q = "SELECT id FROM event" + c.whereSql();
        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByMainEventIdAndStartTsIn(long mainEventId, Instant from, Instant to) {
        return genericBeanDao.loadBeans(EventHelper.INSTANCE, EventFields.MAIN_EVENT_ID.eq(mainEventId)
                .and(EventFields.START_TS.ge(from).and(EventFields.START_TS.lt(to))));
    }

    @RunWithRandomTest
    public ListF<Long> findEventInstanceIdsWithRepetition(long mainEventId) {
        String q = "SELECT id FROM event WHERE recurrence_id IS NOT NULL AND main_event_id = ?";
        return getJdbcTemplate().queryForList(q, Long.class, mainEventId);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Tuple2<Event, Option<Repetition>> findEventWithRepetitionByEventId(long eventId) {
        BeanRowMapper<Event> erm = EventHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<Repetition> rrm = RepetitionHelper.INSTANCE.offsetRowMapper(erm.nextOffset());

        String q = "SELECT " + erm.columns("e.") + ", " + rrm.columns("r.") +
                " FROM event e LEFT JOIN repetition r ON e.repetition_id = r.id WHERE e.id = ?";

        Tuple2RowMapper<Event, Option<Repetition>> rm = Tuple2RowMapper.rowMapper(
                erm, rrm.filterFieldNotNull(RepetitionFields.ID));
        return getJdbcTemplate().queryForObject(q, rm, eventId);
    }

    @RunWithRandomTest
    public int findEventCount(SqlCondition... conditions) {
        SqlCondition c = SqlCondition.all(conditions);
        String q = "SELECT COUNT(*) FROM event WHERE " + c.sql();
        return getJdbcTemplate().queryForInt(q, c.args());
    }

    @RunWithRandomTest
    public int findRdateCount() {
        String q = "SELECT COUNT(*) FROM rdate";
        return getJdbcTemplate().queryForInt(q);
    }

    @RunWithRandomTest
    public ListF<Rdate> findRdates(SqlCondition... conditions) {
        SqlCondition c = SqlCondition.all(conditions);
        String q = "SELECT * FROM rdate WHERE " + c.sql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Rdate.class, c.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Rdate findRdateById(long id) {
        String q = "SELECT * FROM rdate WHERE id = ?";
        return getJdbcTemplate().queryForObject(q, Rdate.class, id);
    }

    @RunWithRandomTest
    public ListF<Rdate> findRdatesByEventId(long eventId) {
        return findRdatesByEventIds(Cf.list(eventId));
    }

    @RunWithRandomTest
    public ListF<Rdate> findRdateRdatesByEventId(long eventId) {
        String q = "SELECT * FROM rdate WHERE event_id = ? AND is_rdate = TRUE";
        return getJdbcTemplate().queryForList(q, Rdate.class, eventId);
    }

    @RunWithRandomTest
    public ListF<Rdate> findExdateRdatesByEventId(long eventId) {
        String q = "SELECT * FROM rdate WHERE event_id = ? AND is_rdate = FALSE";
        return getJdbcTemplate().queryForList(q, Rdate.class, eventId);
    }

    @RunWithRandomTest
    public boolean findRdatesExistByEventId(long eventId) {
        return findRdatesByEventId(eventId).isNotEmpty();
    }

    @RunWithRandomTest
    public ListF<Rdate> findRdatesByEventIds(ListF<Long> eventIds) {
        return findRdates(RdateFields.EVENT_ID.column().inSet(eventIds));
    }

    @RunWithRandomTest
    public ListF<Rdate> findExdatesByEventId(long eventId) {
        return findRdates(RdateFields.IS_RDATE.eq(false).and(RdateFields.EVENT_ID.eq(eventId)));
    }

    @RunWithRandomTest
    public void deleteFutureRdates(long eventId, Instant start) {
        String q = "DELETE FROM rdate WHERE start_ts >= ? AND event_id = ?";
        getJdbcTemplate().update(q, start, eventId);
    }

    @RunWithRandomTest
    public void deleteRdateBy(boolean isRdate, long eventId, Instant instant) {
        String q = "DELETE FROM rdate WHERE is_rdate = ? AND event_id = ? AND start_ts = ?";
        getJdbcTemplate().update(q, isRdate, eventId, instant);
    }

    @RunWithRandomTest
    public void deleteRdatesByEventIds(ListF<Long> eventIds) {
        String q = "DELETE FROM rdate WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return;

        getJdbcTemplate().update(q);
    }


    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public String findExternalIdByMainEventId(long id) {
        String q = "SELECT external_id FROM main_event WHERE id = ?";
        return getJdbcTemplate().queryForObject(q, String.class, id);
    }

    public long saveEvent(Event eventTemplate, ActionInfo actionInfo) {
        Event tmp = eventTemplate.copy();

        tmp.setCreationReqId(actionInfo.getRequestIdWithHostId());
        tmp.setCreationSource(actionInfo.getActionSource());

        tmp.setFieldValueDefault(EventFields.CREATION_TS, actionInfo.getNow());
        tmp.setFieldValueDefault(EventFields.LAST_UPDATE_TS, actionInfo.getNow());

        return genericBeanDao.insertBeanGetGeneratedKey(tmp);
    }

    @RunWithRandomTest(possible=DataIntegrityViolationException.class)
    public void deleteEventsByIds(ListF<Long> eventIds) {
        SqlCondition inCondition = EventFields.ID.column().inSet(eventIds);
        String q = "DELETE FROM event WHERE " + inCondition.sql();

        if (skipQuery(inCondition, q, inCondition.args())) return;

        getJdbcTemplate().update(q, inCondition.args());
    }

    @RunWithRandomTest
    public ListF<Event> findAbsenceEventsByUserId(PassportUid uid) {
        String q =
            "SELECT e.* FROM event e" +
            " INNER JOIN event_layer el ON el.event_id = e.id" +
            " INNER JOIN layer l ON l.id = el.layer_id" +
            " WHERE l.type = ? AND e.creator_uid = ?";
        return getJdbcTemplate().queryForList(q, Event.class, LayerType.ABSENCE, uid);
    }

    @RunWithRandomTest
    public ListF<Long> findAllNonFinishedAbsenceEventIds(Instant finishTs) {
        String q =
            "SELECT e.id FROM event e" +
            " INNER JOIN event_layer el ON el.event_id = e.id" +
            " INNER JOIN layer l ON l.id = el.layer_id" +
            " WHERE l.type = ? AND e.end_ts >= ?";
        return getJdbcTemplate().queryForList(q, Long.class, LayerType.ABSENCE, finishTs);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByLayerIdAndExternalId(long layerId, String externalId) {
        String q =
            "SELECT e.*" +
            " FROM event e" +
            " INNER JOIN event_layer el ON e.id = el.event_id" +
            " INNER JOIN main_event me ON e.main_event_id = me.id" +
            " WHERE me.external_id = ? AND el.layer_id = ?";
        return getJdbcTemplate().queryForList(q, Event.class, externalId, layerId);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByLayerIdAndExternalIdForUpdate(long layerId, String externalId) {
        String nested = "SELECT e.id FROM event e"
                + " INNER JOIN event_layer el ON e.id = el.event_id"
                + " INNER JOIN main_event me ON e.main_event_id = me.id"
                + " WHERE me.external_id = ? AND el.layer_id = ?";

        String query = "SELECT * FROM event WHERE id IN (" + nested + ") FOR UPDATE";
        return lockedForUpdate(getJdbcTemplate().queryForList(query, Event.class, externalId, layerId), true);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByCreatorUid(PassportUid creatorUid) {
        String q = "SELECT * FROM event WHERE creator_uid = ?";
        return getJdbcTemplate().queryForList(q, Event.class, creatorUid);
    }

    @RunWithRandomTest
    public ListF<Long> findUserCreatedEventIdsByUidAndEventIds(PassportUid uid, ListF<Long> eventIds) {
        SqlCondition c = EventFields.CREATOR_UID.eq(uid);
        c = c.and(EventFields.ID.column().inSet(eventIds));

        String q = "SELECT id FROM event" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public ListF<Long> findUserCreatedEventIdsWithoutOrganizer(PassportUid uid, ListF<Long> eventIds) {
        String q = "SELECT e.id FROM event e" +
                " LEFT JOIN event_user eu ON eu.event_id = e.id AND eu.is_organizer = TRUE" +
                " LEFT JOIN event_invitation ei ON ei.event_id = e.id AND ei.is_organizer = TRUE" +
                " WHERE eu.event_id IS NULL AND ei.event_id IS NULL" +
                " AND e.creator_uid = ? AND e.id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, uid)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, uid);
    }

    @RunWithRandomTest
    public ListF<Long> findUserCreatedEventIdsWithoutOtherUserAndResource(PassportUid uid, ListF<Long> eventIds) {
        String q = "SELECT e.id FROM event e"
                + " LEFT JOIN event_user eu ON eu.event_id = e.id AND eu.uid != ?"
                + " LEFT JOIN event_resource er ON er.event_id = e.id"
                + " WHERE eu.event_id IS NULL AND er.event_id IS NULL AND e.id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, uid)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, uid);
    }

    @RunWithRandomTest
    public Tuple2List<Event, String> findEventsWithExternalIdByEventIds(
            ListF<Long> eventIds)
    {
        final SqlCondition idsCondition = EventFields.ID.column().inSet(eventIds);

        BeanRowMapper<Event> eRm = EventHelper.INSTANCE.offsetRowMapper(0);
        RowMapper<String> extIdRm = ResultSetUtils.getColumnRowMapper(String.class, "ext_id");

        String sql = "SELECT " + eRm.columns("e.") + ", me.external_id ext_id" +
                " FROM event e" +
                " INNER JOIN main_event me ON e.main_event_id = me.id" +
                " WHERE " + idsCondition.sqlForTable("e");

        if (skipQuery(idsCondition, sql, idsCondition.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(sql, eRm, extIdRm, idsCondition.args());
    }

    @RunWithRandomTest
    public ListF<Long> findRepetitionIdsByEventIds(ListF<Long> eventIds) {
        SqlCondition inCondition = EventFields.ID.column().inSet(eventIds);
        String q = "SELECT repetition_id FROM event WHERE repetition_id IS NOT NULL " + inCondition.andSql();

        if (skipQuery(inCondition, q, inCondition.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, inCondition.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Repetition findRepetitionById(long id) {
        String q = "SELECT * FROM repetition WHERE id = ?";
        return getJdbcTemplate().queryForObject(q, Repetition.class, id);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public ListF<Repetition> findRepetitionsByIds(ListF<Long> ids) {
        String q = "SELECT * FROM repetition WHERE id " + SqlQueryUtils.inSet(ids);

        if (skipQuery(ids, q)) return Cf.list();

        MapF<Long, Repetition> repetitionsById = getJdbcTemplate().queryForList(q, Repetition.class).toMapMappingToKey(RepetitionFields.ID.getF());
        try {
            return ids.map(repetitionsById::getOrThrow);
        } catch (NoSuchElementException e) {
            throw new EmptyResultDataAccessException("some of repetitions not found by ids: " + ids, ids.size());
        }
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateRepetitionDueTs(long id, Instant dueTs) {
        String q = "UPDATE repetition SET due_ts = ? WHERE id = ?";
        getJdbcTemplate().updateRow(q, dueTs, id);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public Instant findLastUpdateTsByEventId(long eventId) {
        String q = "SELECT last_update_ts FROM event WHERE id = ?";
        return getJdbcTemplate().queryForObject(q, Instant.class, eventId);
    }

    public long saveRepetition(Repetition newRepetition) {
        return genericBeanDao.insertBeanGetGeneratedKey(newRepetition);
    }

    public void updateRepetition(Repetition repetition) {
        genericBeanDao.updateBean(repetition);
    }

    @RunWithRandomTest
    public void deleteRepetitionsByIds(ListF<Long> id) {
        String q = "DELETE FROM repetition WHERE id " + SqlQueryUtils.inSet(id);

        if (skipQuery(id, q)) return;

        getJdbcTemplate().update(q);
    }

    public long saveRdate(Rdate rdate) {
        return genericBeanDao.insertBeanGetGeneratedKey(rdate);
    }

    @RunWithRandomTest
    public Tuple2List<String, Instant> getDuplicateMeetingKeys() {
        String q =
            "SELECT me.external_id, e.recurrence_id" +
            " FROM event e" +
            " INNER JOIN main_event me ON e.main_event_id = me.id" +
            " GROUP BY me.external_id, e.recurrence_id HAVING COUNT(1) > 1";
        return getJdbcTemplate().queryForList2(q, String.class, Instant.class);
    }

    @RunWithRandomTest
    public ListF<Instant> getDuplicateMeetingKeysByExternalId(String externalId) {
        String q =
            "SELECT recurrence_id" +
            " FROM event e" +
            " WHERE main_event_id IN (SELECT id FROM main_event WHERE external_id = ?)" +
            " GROUP BY recurrence_id HAVING COUNT(1) > 1";
        return getJdbcTemplate().queryForList(q, Instant.class, externalId);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByExternalIdAndRecurrenceId(String externalId, Option<Instant> recurrenceId) {
        SqlCondition recurrenceIdCond = recurrenceId.isPresent() ?
                EventFields.RECURRENCE_ID.eq(recurrenceId.get()) :
                EventFields.RECURRENCE_ID.column().isNull();
        String q =
            "SELECT e.*" +
            " FROM event e" +
            " INNER JOIN main_event me ON e.main_event_id = me.id" +
            " WHERE me.external_id = ? " + recurrenceIdCond.andSqlForTable("e");
        return getJdbcTemplate().queryForList(q, Event.class, externalId, recurrenceId);
    }

    @RunWithRandomTest
    public Tuple2List<Event, EventUser> findEventsWithWrongCreator() {
        BeanRowMapper<Event> eRm = EventHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<EventUser> euRm = EventUserHelper.INSTANCE.offsetRowMapper(eRm.nextOffset());

        String q = "SELECT " + eRm.columns("e.") + ", " + euRm.columns("eu.") +
                " FROM event e INNER JOIN event_user eu ON e.id = eu.event_id" +
                " WHERE eu.is_organizer AND eu.uid != e.creator_uid";
        return Tuple2List.tuple2List(getJdbcTemplate().query(q, Tuple2RowMapper.rowMapper(eRm, euRm)));
    }

    @RunWithRandomTest
    public ListF<Long> findEventsWithMultipleOrganizers() {
        String q = "SELECT event_id FROM event_user WHERE is_organizer GROUP BY event_id HAVING COUNT(event_id) > 1";
        return getJdbcTemplate().queryForList(q, Long.class);
    }

    public ListF<Event> findEventsWithoutEventLayers() {
        String q = "SELECT * FROM event WHERE id NOT IN (SELECT event_id FROM event_layer)";
        return getJdbcTemplate().queryForList(q, Event.class);
    }

    @RunWithRandomTest
    public ListF<Event> findServiceEventsByUidSidAndExternalId(PassportUid uid, PassportSid sid, String externalId) {
        String q =
            "SELECT e.* FROM main_event me INNER JOIN event e ON e.main_event_id = me.id " +
            "WHERE me.external_id = ? AND e.creator_uid = ? AND e.sid = ? AND e.type = ?";
        return getJdbcTemplate().queryForList(q, Event.class, externalId, uid, sid, EventType.SERVICE);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateEventPermAll(long eventId, @SampleValue({"none", "view"}) EventActionClass permAll) {
        String q = "UPDATE event SET perm_all = ? WHERE id = ?";
        getJdbcTemplate().updateRow(q, permAll, eventId);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateEventsPermAll(ListF<Long> eventIds, @SampleValue({"none", "view"}) EventActionClass permAll) {
        String q = "UPDATE event SET perm_all = ? WHERE id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, permAll)) return;

        getJdbcTemplate().update(q, permAll);
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByCreationTsInterval(Instant minCreationTs, Instant maxCreationTs) {
        String q = "SELECT * FROM event WHERE creation_ts > ? AND creation_ts < ?";
        return getJdbcTemplate().queryForList(q, Event.class, minCreationTs, maxCreationTs);
    }

    @RunWithRandomTest
    public void updateEventSetCreatorUidByIds(ListF<Long> eventIds, PassportUid newCreator, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event SET creator_uid = ?," + info.sql() + " WHERE id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, newCreator, info.args())) return;

        getJdbcTemplate().update(q, newCreator, info.args());
    }

    private static String forUpdateSql(boolean forUpdate) {
        return forUpdate ? " FOR UPDATE" : "";
    }

    private static <T extends Bean> ListF<T> lockedForUpdate(ListF<T> beans, boolean forUpdate) {
        if (forUpdate) {
            beans.forEach(Bean::setLockedForUpdate);
        }
        return beans;
    }

    private static SqlPart modificationInfoAssignment(ActionInfo actionInfo) {
        return new FieldsAssignment(Tuple2List.<String, Object>fromPairs(
                "modification_source", actionInfo.getActionSource(),
                "modification_req_id", actionInfo.getRequestIdWithHostId(),
                "last_update_ts", actionInfo.getNow()));
    }

    @RunWithRandomTest
    public Tuple2List<Long, RecurrenceTimeInfo> findRecurrenceInstantInfosByMainEventIds(
            ListF<Long> mainEventIds, Option<SqlCondition> additionalCondition)
    {
        SqlCondition condition = EventFields.RECURRENCE_ID.column().isNotNull()
                .and(EventFields.MAIN_EVENT_ID.column().inSet(mainEventIds))
                .and(additionalCondition);

        String q = "SELECT main_event_id, recurrence_id, start_ts, end_ts FROM event WHERE " + condition.sql();

        if (skipQuery(condition, q, condition.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate()
                .queryForList4(q, Long.class, Instant.class, Instant.class, Instant.class, condition.args())
                .toTuple2List(t -> t._1, t -> new RecurrenceTimeInfo(t._2, t._3, t._4));
    }

    @RunWithRandomTest
    public ListF<Long> filterMasterAndFutureRecurrences(ListF<Long> eventIds, ActionInfo actionInfo) {
        SqlCondition filterCondition =
                EventFields.RECURRENCE_ID.column().isNull().or(EventFields.END_TS.column().gt(actionInfo.getNow()));
        SqlCondition condition = EventFields.ID.column().inSet(eventIds).and(filterCondition);
        String q = "SELECT id FROM event WHERE" + condition.sql();

        if (skipQuery(eventIds, q, condition.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, condition.args());
    }

    public void insertEventAttachments(ListF<EventAttachment> attachments) {
        if (attachments.isNotEmpty()) {
            genericBeanDao.insertBeans(attachments, attachments.first().getSetFields());
        }
    }

    @RunWithRandomTest
    public void deleteEventAttachmentsByUrls(long eventId, ListF<String> eventAttachmentUrls) {
        String q = "DELETE FROM event_attachment WHERE event_id = ? AND url " + SqlQueryUtils.inSet(eventAttachmentUrls);

        if (skipQuery(eventAttachmentUrls, q)) return;

        getJdbcTemplate().update(q, eventId);
    }

    @RunWithRandomTest
    public ListF<EventAttachment> findEventAttachmentsByEventId(long eventId) {
        return findEventAttachmentsByEventIds(Cf.list(eventId)).single().get2();
    }

    @RunWithRandomTest
    public Tuple2List<Long, ListF<EventAttachment>> findEventAttachmentsByEventIds(ListF<Long> eventIds) {
        ListF<EventAttachment> attachments = genericBeanDao.loadBeansByField(
                EventAttachmentHelper.INSTANCE,
                EventAttachmentFields.EVENT_ID,
                eventIds);
        MapF<Long, ListF<EventAttachment>> map = attachments.groupBy(EventAttachment::getEventId);

        return eventIds.zipWith(id -> map.getOrElse(id, Cf.list()));
    }

    @RunWithRandomTest
    public void deleteEventAttachmentsByEventIds(ListF<Long> eventIds) {
        String q = "DELETE FROM event_attachment WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return;

        getJdbcTemplate().update(q);
    }
} //~
