package ru.yandex.calendar.logic.event.dao;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.EventLayer;
import ru.yandex.calendar.logic.beans.generated.EventLayerFields;
import ru.yandex.calendar.logic.beans.generated.EventLayerHelper;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerHelper;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventLayerId;
import ru.yandex.calendar.logic.event.EventLayerWithRelations;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.db.resultSet.ResultSetUtils;

/**
 * @author akirakozov
 */
public class EventLayerDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;


    @RunWithRandomTest
    public Option<EventLayer> findEventLayerByEventIdAndLayerId(long eventId, long layerId) {
        String q = "SELECT * FROM event_layer WHERE event_id = ? AND layer_id = ?";
        return getJdbcTemplate().queryForOption(q, EventLayer.class, eventId, layerId);
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayersByEventId(long eventId) {
        String q = "SELECT * FROM event_layer WHERE event_id = ?";
        return getJdbcTemplate().queryForList(q, EventLayer.class, eventId);
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayersByEventIds(ListF<Long> eventIds) {
        return genericBeanDao.loadBeans(EventLayerHelper.INSTANCE, EventLayerFields.EVENT_ID.column().inSet(eventIds));
    }

    @RunWithRandomTest
    public ListF<Long> findLayersIdsByEventIds(ListF<Long> eventIds) {
        SqlCondition c = EventLayerFields.EVENT_ID.column().inSet(eventIds);
        String q = "SELECT DISTINCT layer_id FROM event_layer" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayersByLayerIdAndEventIds(long layerId, ListF<Long> eventIds) {
        String q = "SELECT * FROM event_layer WHERE layer_id = ? AND event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, layerId)) return Cf.list();

        return getJdbcTemplate().queryForList(q, EventLayer.class, layerId);
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayersByLayerId(long layerId) {
        String q = "SELECT * FROM event_layer WHERE layer_id = ?";
        return getJdbcTemplate().queryForList(q, EventLayer.class, layerId);
    }

    @RunWithRandomTest
    public Option<EventLayer> findEventLayerByEventIdAndLayerCreatorUid(long eventId, PassportUid layerCreatorUid) {
        String q = "SELECT * FROM event_layer WHERE event_id = ? AND l_creator_uid = ?";
        return getJdbcTemplate().queryForOption(q, EventLayer.class, eventId, layerCreatorUid);
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayersByEventIdAndLayerCreatorUids(
            long eventId, ListF<PassportUid> layerCreatorUids)
    {
        String q = "SELECT * FROM event_layer" +
                " WHERE event_id = ? AND l_creator_uid " + SqlQueryUtils.inSet(layerCreatorUids);

        if (skipQuery(layerCreatorUids, q, eventId)) return Cf.list();

        return getJdbcTemplate().queryForOption(q, EventLayer.class, eventId);
    }

    @RunWithRandomTest
    public ListF<PassportUid> findUsersIdsWithLayersWithEvent(long eventId) {
        String q = "SELECT lu.uid" +
            " FROM layer_user lu" +
            " INNER JOIN event_layer el ON lu.layer_id = el.layer_id" +
            " WHERE el.event_id = ?";
        return getJdbcTemplate().queryForList(q, Long.class, eventId).map(PassportUid::cons);
    }

    private ListF<EventLayerWithRelations> findEventLayersWithRelations(SqlCondition c) {
        BeanRowMapper<EventLayer> elRm = EventLayerHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<Layer> lRm = LayerHelper.INSTANCE.offsetRowMapper(elRm.nextOffset());

        String q = "SELECT " + elRm.columns("el.") + ", " + lRm.columns("l.") +
                " FROM event_layer el" +
                " INNER JOIN layer l ON el.layer_id = l.id" +
                " WHERE " + c.sqlForTable("el");

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().query2(q, elRm, lRm, c.args()).map(EventLayerWithRelations::new);
    }

    @RunWithRandomTest
    public ListF<EventLayerWithRelations> findEventLayersWithRelations() {
        return findEventLayersWithRelations(SqlCondition.trueCondition());
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayers() {
        String q = "SELECT * FROM event_layer";
        return getJdbcTemplate().queryForList(q, EventLayer.class);
    }

    @RunWithRandomTest
    public ListF<EventLayerWithRelations> findEventLayersWithRelationsByEventIds(ListF<Long> eventIds, Option<Long> layerIdO) {
        SqlCondition inSetCondition = EventLayerFields.EVENT_ID.column().inSet(eventIds);
        if (layerIdO.isPresent()) {
            inSetCondition = inSetCondition.and(EventLayerFields.LAYER_ID.column().eq(layerIdO.get()));
        }
        return findEventLayersWithRelations(inSetCondition);
    }

    /** @deprecated remove, use {@link #findEventLayersWithRelationsByEventIdsAndLayerCreatorUid(ListF, PassportUid)} */
    @RunWithRandomTest
    @Deprecated
    public Option<Tuple2<EventLayer, Layer>> findEventLayerWithLayerByEventIdAndLayerCreatorUid(long eventId, PassportUid layerCreatorUid) {
        BeanRowMapper<EventLayer> elRm = EventLayerHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<Layer> lRm = LayerHelper.INSTANCE.offsetRowMapper(elRm.nextOffset());

        String q = "SELECT " + elRm.columns("el.") + ", " + lRm.columns("l.") + " FROM event_layer el" +
                " INNER JOIN layer l ON el.layer_id = l.id" +
                " WHERE el.event_id = ? AND el.l_creator_uid = ?";
        Tuple2List<EventLayer, Layer> r = getJdbcTemplate().query2(q, elRm, lRm, eventId, layerCreatorUid);
        // XXX: move warning outside
        if (r.size() > 1) {
            logger.warn("Database inconsistent: event is on more then one user layer: " + r);
        }
        return r.firstO();
    }

    @RunWithRandomTest
    public ListF<EventLayerWithRelations> findEventLayersWithRelationsByEventIdsAndLayerCreatorUid(ListF<Long> eventIds, PassportUid uid) {
        final SqlCondition inSetCondition =
            EventLayerFields.EVENT_ID.column().inSet(eventIds)
                .and(EventLayerFields.L_CREATOR_UID.eq(uid));
        return findEventLayersWithRelations(inSetCondition);
    }

    @RunWithRandomTest
    public ListF<EventLayerWithRelations> findEventLayersWithRelationsByEventIdAndLayerCreatorUid(long eventId, PassportUid uid) {
        return findEventLayersWithRelationsByEventIdsAndLayerCreatorUid(Cf.list(eventId), uid);
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayersByEventIdsAndLayerCreatorUid(ListF<Long> eventIds, PassportUid uid) {
        return findEventLayersByEventIdsAndLayerCreatorUids(eventIds, Cf.list(uid));
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayersByEventIdsAndLayerCreatorUids(ListF<Long> eventIds, ListF<PassportUid> uids) {
        SqlCondition c = EventLayerFields.EVENT_ID.column().inSet(eventIds)
                .and(EventLayerFields.L_CREATOR_UID.column().inSet(uids));

        String q = "SELECT * FROM event_layer" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, EventLayer.class, c.args());
    }

    @RunWithRandomTest
    public ListF<EventLayer> findEventLayersByLayerCreatorUid(PassportUid uid) {
        String q = "SELECT * FROM event_layer WHERE l_creator_uid = ?";
        return getJdbcTemplate().queryForList(q, EventLayer.class, uid);
    }

    @RunWithRandomTest
    public ListF<Long> findEventLayerEventIdsByLayerId(Long layerId) {
        String sql = "SELECT event_id FROM event_layer WHERE layer_id = ?";
        return getJdbcTemplate().queryForList(sql, Long.class, layerId);
    }

    @RunWithRandomTest
    public ListF<Long> findEventLayerEventIdsByLayerIds(ListF<Long> layerIds) {
        String sql = "SELECT event_id FROM event_layer WHERE layer_id " + SqlQueryUtils.inSet(layerIds);

        if (skipQuery(layerIds, sql)) return Cf.list();

        return getJdbcTemplate().queryForList(sql, Long.class).stableUnique();
    }

    @RunWithRandomTest
    public ListF<Long> findEventLayerEventIdsByLayerIdsAndEventIds(ListF<Long> layerIds, ListF<Long> eventIds) {
        String q  = "SELECT event_id FROM event_layer" +
                " WHERE layer_id " + SqlQueryUtils.inSet(layerIds) + " AND event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, layerIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class).stableUnique();
    }

    @RunWithRandomTest
    public ListF<EventLayerId> findEventLayerEventIdsAndLayerIds(ListF<Long> eventIds, ListF<Long> layerIds) {
        SqlCondition c = EventLayerFields.EVENT_ID.column().inSet(eventIds)
                .and(EventLayerFields.LAYER_ID.column().inSet(layerIds));

        String q  = "SELECT event_id, layer_id FROM event_layer" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().query(q, (rs, rn) -> new EventLayerId(rs.getLong(1), rs.getLong(2)), c.args());
    }

    @RunWithRandomTest
    public Tuple2List<Long, LayerInfoForPermsCheck> findPrimaryLayersForPersCheckByEventIds(ListF<Long> eventIds) {
        SqlCondition c = EventLayerFields.EVENT_ID.column().inSet(eventIds)
                .and(EventLayerFields.IS_PRIMARY_INST.eq(true));

        String q = "SELECT el.event_id, " + LayerInfoForPermsCheck.columns("l.") +
                " FROM layer l INNER JOIN event_layer el ON el.layer_id = l.id" + c.whereSqlForTable("el");

        if (skipQuery(c, q, c.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(q,
                ResultSetUtils.getColumnRowMapper(Long.class, 1),
                LayerInfoForPermsCheck.rowMapper(1), c.args());
    }

    @RunWithRandomTest
    public Option<Layer> findPrimaryLayerByEventId(long eventId) {
        SqlCondition c = EventLayerFields.EVENT_ID.eq(eventId)
                .and(EventLayerFields.IS_PRIMARY_INST.eq(true));

        String q = "SELECT l.* " +
                " FROM layer l INNER JOIN event_layer el ON el.layer_id = l.id" + c.whereSqlForTable("el");

        if (skipQuery(c, q, c.args())) return Option.empty();

        return getJdbcTemplate().queryForOption(q, Layer.class, c.args());
    }

    @RunWithRandomTest
    public Tuple2List<Long, LayerInfoForPermsCheck> findOrganizerLayersByEventIdsAndUids(
            Tuple2List<Long, PassportUid> eventIdsUids)
    {
        SqlCondition c = EventLayerFields.EVENT_ID.column().inSet(eventIdsUids.get1())
                .and(EventLayerFields.L_CREATOR_UID.column().inSet(eventIdsUids.get2()));

        String q = "SELECT el.event_id, " + LayerInfoForPermsCheck.columns("l.") + " FROM layer l" +
                " INNER JOIN event_layer el ON el.layer_id = l.id" +
                " INNER JOIN event_user eu ON eu.event_id = el.event_id AND eu.is_organizer = TRUE" +
                " WHERE el.l_creator_uid = eu.uid AND " + c.sqlForTable("el");

        if (skipQuery(eventIdsUids, q, c.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(q,
                ResultSetUtils.getColumnRowMapper(Long.class, 1),
                LayerInfoForPermsCheck.rowMapper(1), c.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void deleteEventLayerById(EventLayerId eventLayerId) {
        genericBeanDao.deleteBeanById(EventLayerHelper.INSTANCE, eventLayerId.asTuple());
    }

    @RunWithRandomTest
    public void deleteEventLayersByIds(ListF<EventLayerId> eventLayerIds) {
        genericBeanDao.deleteBeans(EventLayerHelper.INSTANCE, EventLayerId.inSet(eventLayerIds));
    }

    @RunWithRandomTest
    public void deleteEventLayersByEventIds(ListF<Long> eventIds) {
        String q = "DELETE FROM event_layer WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest
    public void deleteEventLayersByLayerIds(ListF<Long> layerId) {
        String q = "DELETE FROM event_layer WHERE layer_id " + SqlQueryUtils.inSet(layerId);

        if (skipQuery(layerId, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest
    public void deleteEventLayersByLayerIdsAndEventIds(ListF<Long> layerIds, ListF<Long> eventIds) {
        genericBeanDao.deleteBeans(EventLayerHelper.INSTANCE,
                EventLayerFields.LAYER_ID.column().inSet(layerIds)
                        .and(EventLayerFields.EVENT_ID.column().inSet(eventIds)));
    }

    @RunWithRandomTest
    public void deleteEventLayersByLayerIdAndEventIds(long layerId, ListF<Long> eventIds) {
        String q = "DELETE FROM event_layer WHERE layer_id = ? AND event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, layerId)) return;

        getJdbcTemplate().update(q, layerId);
    }

    @RunWithRandomTest(possible= IncorrectResultSizeDataAccessException.class)
    public void deleteEventLayerByEventIdAndLayerCreatorUid(long eventId, PassportUid layerCreatorUid) {
        String q = "DELETE FROM event_layer WHERE event_id = ? AND l_creator_uid = ?";
        getJdbcTemplate().updateRow(q, eventId, layerCreatorUid);
    }

    @RunWithRandomTest
    public void deleteNotPrimaryEventLayersByLayerIds(ListF<Long> layerIds) {
        String q = "DELETE FROM event_layer WHERE is_primary_inst = FALSE AND layer_id " + SqlQueryUtils.inSet(layerIds);

        if (skipQuery(layerIds, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest
    public void updateEventLayersSetLayerIdAndLayerCreatorUidByIds(
            ListF<EventLayerId> eventLayerIds, long newLayerId, PassportUid newCreator, ActionInfo actionInfo)
    {
        SqlCondition c = EventLayerId.inSet(eventLayerIds);
        String q = "UPDATE event_layer"
                + " SET layer_id = ?, l_creator_uid = ?, modification_source = ?, modification_req_id = ?"
                + " WHERE " + c.sql();

        ListF<?> args = Cf.list(newLayerId, newCreator,
                actionInfo.getActionSource(), actionInfo.getRequestIdWithHostId(), c.args());

        if (skipQuery(c, q, args)) return;

        getJdbcTemplate().update(q, args);
    }

    @RunWithRandomTest
    public void updateIgnoreEventLayerSetLayerIdAndLayerCreatorUidByLayerId(
            long newLayerId, PassportUid newLayerCreator, long oldLayerId, ActionInfo actionInfo)
    {
        SqlCondition condition = EventLayerFields.LAYER_ID.eq(oldLayerId);
        updateIgnoreEventLayerSetLayerIdAndLayerCreatorUid(newLayerId, newLayerCreator, condition, actionInfo);
    }

    @RunWithRandomTest
    public void updateIgnoreEventLayerSetLayerIdAndLayerCreatorUidByLayerIdAndEventIds(
            long newLayerId, PassportUid newLayerCreator, long oldLayerId, ListF<Long> eventIds, ActionInfo actionInfo)
    {
        SqlCondition condition = EventLayerFields.LAYER_ID.eq(oldLayerId)
                .and(EventLayerFields.EVENT_ID.column().inSet(eventIds));

        updateIgnoreEventLayerSetLayerIdAndLayerCreatorUid(newLayerId, newLayerCreator, condition, actionInfo);
    }

    private void updateIgnoreEventLayerSetLayerIdAndLayerCreatorUid(
            long newLayerId, PassportUid newLayerCreator, SqlCondition eventLayerCondition, ActionInfo actionInfo)
    {
        String sql = "UPDATE event_layer el"
                + " SET layer_id = ?, l_creator_uid = ?, modification_source = ?, modification_req_id = ?"
                + " WHERE (" + eventLayerCondition.sql() + ")"
                + " AND NOT EXISTS (SELECT 1 FROM event_layer WHERE layer_id = ? AND event_id = el.event_id)";

        ListF<?> args = Cf.list(newLayerId, newLayerCreator,
                actionInfo.getActionSource(), actionInfo.getRequestIdWithHostId(),
                eventLayerCondition.args(), newLayerId);

        if (skipQuery(eventLayerCondition, sql, args)) return;

        getJdbcTemplate().update(sql, args);
    }

    @RunWithRandomTest
    public void updateNotIsPrimaryInstByEventId(long eventId) {
        getJdbcTemplate().update("UPDATE event_layer SET is_primary_inst = FALSE WHERE event_id = ?", eventId);
    }

    @RunWithRandomTest
    public boolean findIfEventLayerExistsByUidAndEventId(PassportUid uid, long eventId) {
        String sql = "SELECT COUNT(*)" +
                " FROM event_layer el INNER JOIN layer l ON el.layer_id = l.id" +
                " WHERE el.event_id = ? AND l.creator_uid = ?";
        return getJdbcTemplate().queryForInt(sql, eventId, uid) > 0;
    }

    @RunWithRandomTest
    public ListF<Long> findEventIdsByLayerIdAndUserOrganizerId(
            long layerId, PassportUid organizerUid)
    {
        String q =
            "SELECT el.event_id FROM event_layer el" +
            " INNER JOIN event_user eu ON el.event_id = eu.event_id" +
            " WHERE el.layer_id = ? AND eu.uid = ? AND eu.is_organizer = TRUE";
        return getJdbcTemplate().queryForList(q, Long.class, layerId, organizerUid);
    }

    @RunWithRandomTest
    public ListF<Long> findEventIdsWithOnlyOneEventLayer(ListF<Long> eventIds) {
        SqlCondition c = EventLayerFields.EVENT_ID.column().inSet(eventIds);
        String q =
            "SELECT event_id FROM event_layer " +
            "WHERE " + c.sql() + " " +
            "GROUP BY event_id HAVING COUNT(1) = 1";

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    public void saveEventLayer(EventLayer eventLayer) {
        genericBeanDao.insertBean(eventLayer);
    }

    public void saveEventLayersBatch(ListF<EventLayer> eventLayers) {
        genericBeanDao.insertBeans(eventLayers);
    }

    public void updateEventLayerByEventIdAndLayerId(EventLayer updateData, long eventId, long layerId) {
        genericBeanDao.updateRow(updateData, EventLayerFields.EVENT_ID.eq(eventId)
                .and(EventLayerFields.LAYER_ID.eq(layerId)));
    }

    public void updateEventLayersByEventId(EventLayer updateData, long eventId) {
        genericBeanDao.updateBeans(updateData, EventLayerFields.EVENT_ID.eq(eventId));
    }

    @RunWithRandomTest
    public ListF<Long> findEventIdsWithoutEventUserByLayerIdAndUid(
            final PassportUid uid, final long layerId)
    {
        final String sql3 =
                "SELECT el.event_id " +
                "FROM event_layer el " +
                "WHERE " +
                    "NOT EXISTS " +
                        "(SELECT eu.id FROM event_user eu " +
                        "WHERE eu.event_id = el.event_id AND eu.uid = ?) " +
                    "AND el.layer_id = ?";
        return getJdbcTemplate().queryForList(sql3, Long.class, uid, layerId);
    }
}
