package ru.yandex.calendar.logic.event.dao;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.EventLayerFields;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventResourceFields;
import ru.yandex.calendar.logic.beans.generated.EventResourceHelper;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventResourceId;
import ru.yandex.calendar.logic.sharing.perm.ResourceInfoForPermsCheck;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.db.resultSet.ResultSetUtils;

public class EventResourceDao extends CalendarJdbcDaoSupport {
    @Autowired
    private GenericBeanDao genericBeanDao;

    @RunWithRandomTest
    public Option<EventResource> findEventResourceByEventIdAndResourceId(
            long eventId, long resourceId)
    {
        String q = "SELECT * FROM event_resource WHERE event_id = ? AND resource_id = ?";
        return getJdbcTemplate().queryForOption(q, EventResource.class, eventId, resourceId);
    }

    @RunWithRandomTest
    public ListF<Long> findEventResourceEventIdsByResourceId(long resourceId) {
        String q = "SELECT event_id FROM event_resource WHERE resource_id = ?";
        return getJdbcTemplate().queryForList(q, Long.class, resourceId);
    }

    @RunWithRandomTest
    public ListF<Long> findEventResourceEventIdsByResourceIds(ListF<Long> resourceIds) {
        String q = "SELECT event_id FROM event_resource WHERE resource_id " + SqlQueryUtils.inSet(resourceIds);

        if (skipQuery(resourceIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class);
    }

    @RunWithRandomTest
    public ListF<Long> findEventResourceIdsByEventIds(ListF<Long> eventIds) {
        String q = "SELECT resource_id FROM event_resource WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class);
    }

    @RunWithRandomTest
    public ListF<EventResource> findEventResourcesByEventIds(ListF<Long> eventIds) {
        String q = "SELECT * FROM event_resource WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, EventResource.class);
    }

    @RunWithRandomTest
    public ListF<EventResource> findEventResourcesByEventId(long eventId) {
        return findEventResourcesByEventIds(Cf.list(eventId));
    }

    @RunWithRandomTest
    public ListF<EventResource> findEventResources() {
        String q = "SELECT * FROM event_resource";
        return getJdbcTemplate().queryForList(q, EventResource.class);
    }

    @RunWithRandomTest
    public ListF<EventResource> findEventResourcesByEventIdsAndResourceIds(ListF<Long> eventIds, ListF<Long> resourceIds) {
        String q = "SELECT * FROM event_resource" +
                " WHERE event_id " + SqlQueryUtils.inSet(eventIds) +
                " AND resource_id " + SqlQueryUtils.inSet(resourceIds);

        if (skipQuery(eventIds, resourceIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, EventResource.class);
    }

    @RunWithRandomTest
    public Tuple2List<Long, ResourceInfoForPermsCheck> findResourcesForPermsCheckByEventIds(ListF<Long> eventIds) {
        SqlCondition c = EventResourceFields.EVENT_ID.column().inSet(eventIds);

        String q = "SELECT er.event_id, " + ResourceInfoForPermsCheck.columns("r.") +
                " FROM event_resource er INNER JOIN resource r ON r.id = er.resource_id" + c.whereSqlForTable("er");

        if (skipQuery(c, q, c.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().query2(q,
                ResultSetUtils.getColumnRowMapper(Long.class, 1),
                ResourceInfoForPermsCheck.rowMapper(1), c.args());
    }

    public void saveEventResource(EventResource eventResource) {
        genericBeanDao.insertBean(eventResource);
    }

    public void saveEventResourcesBatch(ListF<EventResource> eventResources) {
        genericBeanDao.insertBeans(eventResources);
    }

    public void updateEventResourcesByEventId(EventResource updateData, long eventId) {
        genericBeanDao.updateBeans(updateData, EventResourceFields.EVENT_ID.eq(eventId));
    }

    @RunWithRandomTest
    public Option<Long> findEventIdByResourceExchangeId(String exchangeId) {
        String q = "SELECT event_id FROM event_resource WHERE exchange_id = ?";
        return getJdbcTemplate().queryForOption(q, Long.class, exchangeId);
    }

    @RunWithRandomTest
    public boolean saveUpdateExchangeId(long resourceId, long eventId, String exchangeId, ActionInfo actionInfo) {
        String q = "UPDATE event_resource SET exchange_id = ?, modification_source = ?, modification_req_id = ?" +
                " WHERE resource_id = ? AND event_id = ?";
        return getJdbcTemplate().update(
                q, exchangeId, actionInfo.getActionSource(), actionInfo.getRequestIdWithHostId(), resourceId, eventId) > 0;
    }

    @RunWithRandomTest
    public ListF<String> findExchangeIds(ListF<Long> eventIds) {
        SqlCondition c = EventLayerFields.EVENT_ID.column().inSet(eventIds);
        String q =
            "SELECT exchange_id FROM event_resource" +
            " WHERE exchange_id IS NOT NULL " + c.andSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, String.class, c.args());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void deleteEventResourceById(EventResourceId eventResourceId) {
        genericBeanDao.deleteBeanById(EventResourceHelper.INSTANCE, eventResourceId.asTuple());
    }

    @RunWithRandomTest
    public void deleteEventResourcesByEventIds(ListF<Long> eventIds) {
        String q = "DELETE FROM event_resource WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return;

        getJdbcTemplate().update(q);
    }

    public void deleteEventResourcesByIds(ListF<EventResourceId> eventResourceIds) {
        genericBeanDao.deleteBeans(EventResourceHelper.INSTANCE, EventResourceId.inSet(eventResourceIds));
    }

    @RunWithRandomTest
    public void deleteEventResourceByResourceId(ListF<Long> resourceIds) {
        String q = "DELETE FROM event_resource WHERE resource_id " + SqlQueryUtils.inSet(resourceIds);

        if (skipQuery(resourceIds, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest
    public int deleteEventResourcesByDomain(PassportDomain domain) {
        String q = "DELETE FROM event_resource WHERE resource_id IN (SELECT id FROM resource WHERE domain = ?)";
        return getJdbcTemplate().update(q, domain.getDomain().getDomain());
    }
}
