package ru.yandex.calendar.logic.event.dao;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventHelper;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.EventUserHelper;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.MainEventHelper;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventUserWithRelations;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.calendar.util.db.FieldsAssignment;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.db.q.SqlPart;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.db.resultSet.Tuple3RowMapper;

/**
 * @author akirakozov
 */
public class EventUserDao extends CalendarJdbcDaoSupport {

    private static EventUser withLastUpdateAndDecisionSourceIfNeeded(EventUser eventUser, ActionInfo actionInfo) {
        EventUser newEventUser = eventUser.clone();
        if (!newEventUser.isFieldSet(EventUserFields.LAST_UPDATE)) {
            newEventUser.setLastUpdate(actionInfo.getNow());
        }
        if (eventUser.isFieldSet(EventUserFields.DECISION)) {
            newEventUser.setDecisionSource(actionInfo.getActionSource());
        }
        return newEventUser;
    }

    @Autowired
    private GenericBeanDao genericBeanDao;

    @RunWithRandomTest
    public int updateEventUserDecisionByEventIdAndUserId(
            long eventId, PassportUid uid, Decision decision, ActionInfo actionInfo)
    {
        return updateEventUserDecisionByEventIdsAndUid(Cf.list(eventId), uid, decision, actionInfo);
    }

    @RunWithRandomTest
    public int updateEventUserDecisionByEventIdsAndUid(
            ListF<Long> eventIds, PassportUid uid, Decision decision, ActionInfo actionInfo)
    {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET decision = ?, decision_source = ?, " + info.sql() +
                " WHERE uid = ? AND event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, decision, info.args(), uid)) return 0;

        return getJdbcTemplate().update(q, decision, actionInfo.getActionSource(), info.args(), uid);
    }

    @RunWithRandomTest
    public int updateEventUserReasonByEventIdsAndUserId(
            ListF<Long> eventIds, PassportUid uid, String reason, ActionInfo actionInfo)
    {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET reason = ?, " + info.sql() +
                " WHERE uid = ? AND event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q, reason, info.args(), uid)) return 0;

        return getJdbcTemplate().update(q, reason, info.args(), uid);
    }


    @RunWithRandomTest
    public void updateDtstampAndCopySequenceForWeb(long eventId, PassportUid uid, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user eu SET sequence = e.sequence, dtstamp = ?, " +  info.sql() +
            " FROM event e WHERE e.id = eu.event_id AND eu.event_id = ? AND eu.uid = ?";
        getJdbcTemplate().update(q, actionInfo.getNow(), info.args(), eventId, uid);
    }

    @RunWithRandomTest
    public void updateSequenceAndDtstamp(long eventId, PassportUid uid, int sequence, Instant dtstamp, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user eu SET sequence = ?, dtstamp = ?, " +  info.sql() +
            " WHERE eu.event_id = ? AND eu.uid = ?";
        getJdbcTemplate().update(q, sequence, dtstamp, info.args(), eventId, uid);
    }

    @RunWithRandomTest
    public void updateAvailability(long eventId, PassportUid uid, Availability availability, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET availability = ?, " +  info.sql() + " WHERE event_id = ? AND uid = ?";
        getJdbcTemplate().update(q, availability, info.args(), eventId, uid);
    }

    @RunWithRandomTest
    public int updateEventUserDecisionByEventUserId(long eventUserId, Decision decision, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET decision = ?, decision_source = ?," + info.sql() + " WHERE id = ?";
        return getJdbcTemplate().update(q, decision, actionInfo.getActionSource(), info.args(), eventUserId);
    }

    @RunWithRandomTest
    public void updateEventUserSetNotAttendeeAndNotOrganizerByEventIdAndUserId(
            long eventId, PassportUid uid, ActionInfo actionInfo)
    {
        updateEventUserSetIsAttendeeAndIsOrganizerByEventIdAndUid(eventId, uid, false, false, actionInfo);
    }

    @RunWithRandomTest
    public void updateEventUserSetIsAttendeeAndIsOrganizerByEventIdAndUid(
            long eventId, PassportUid uid, boolean isAttendee, boolean isOrganizer, ActionInfo actionInfo)
    {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET is_attendee = ?, is_organizer = ?, " + info.sql() +
                " WHERE event_id = ? AND uid = ?";
        getJdbcTemplate().update(q, isAttendee, isOrganizer, info.args(), eventId, uid);
    }

    @RunWithRandomTest
    public int updateEventUserIsAttendeeByEventIdAndUserId(long eventId, PassportUid uid, boolean isAttendee, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET is_attendee = ?, " + info.sql() + " WHERE event_id = ? AND uid = ?";
        return getJdbcTemplate().update(q, isAttendee, info.args(), eventId, uid);
    }

    @RunWithRandomTest
    public int updateEventUserSetOrganizerByEventIdAndUid(long eventId, PassportUid uid, boolean isOrganizer, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET is_organizer = ?, " + info.sql() +
                " WHERE event_id = ? AND uid = ?";
        return getJdbcTemplate().update(q, isOrganizer, info.args(), eventId, uid);
    }

    @RunWithRandomTest
    public void updateEventUserSetIsOptionalByEventIdAndUid(long eventId, PassportUid uid, boolean isOptional, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET is_optional = ?, " + info.sql() +
                " WHERE event_id = ? AND uid = ?";
        getJdbcTemplate().update(q, isOptional, info.args(), eventId, uid);
    }

    public void updateEventUser(EventUser eventUser, ActionInfo actionInfo) {
        genericBeanDao.updateBean(withLastUpdateAndDecisionSourceIfNeeded(eventUser, actionInfo));
    }

    @RunWithRandomTest
    public boolean saveUpdateExchangeId(PassportUid uid, long eventId, String exchangeId, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET exchange_id = ?, " + info.sql() + " WHERE uid = ? AND event_id = ?";
        return getJdbcTemplate().update(q, exchangeId, info.args(), uid, eventId) > 0;
    }

    @RunWithRandomTest
    public void deleteEventUserForEventAndUser(long eventId, PassportUid uid) {
        String q = "DELETE FROM event_user WHERE event_id = ? AND uid = ?";
        getJdbcTemplate().update(q, eventId, uid);
    }

    @RunWithRandomTest
    public void deleteEventUsersByEventIds(ListF<Long> eventIds) {
        String q = "DELETE FROM event_user WHERE event_id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return;

        getJdbcTemplate().update(q);
    }

    public void deleteEventUsersByIds(ListF<Long> ids) {
        String q = "DELETE FROM event_user WHERE id " + SqlQueryUtils.inSet(ids);

        if (skipQuery(ids, q)) return;

        getJdbcTemplate().update(q);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public EventUser findEventUserById(long id) {
        String q = "SELECT * FROM event_user WHERE id = ?";
        return getJdbcTemplate().queryForObject(q, EventUser.class, id);
    }

    @RunWithRandomTest
    public Option<EventUser> findEventUserByExchangeId(String exchangeId) {
        String q = "SELECT * FROM event_user WHERE exchange_id = ?";
        return getJdbcTemplate().queryForOption(q, EventUser.class, exchangeId);
    }

    @RunWithRandomTest
    public Option<Tuple3<EventUser, Event, MainEvent>> findEventUserAndJunkByExchangeId(String exchangeId) {
        BeanRowMapper<EventUser> euRm = EventUserHelper.INSTANCE.offsetRowMapper(0);
        BeanRowMapper<Event> eRm = EventHelper.INSTANCE.offsetRowMapper(euRm.nextOffset());
        BeanRowMapper<MainEvent> meRm = MainEventHelper.INSTANCE.offsetRowMapper(eRm.nextOffset());

        String q = "SELECT " + Cf.list(euRm.columns("eu."), eRm.columns("e."), meRm.columns("me.")).mkString(", ") +
                " FROM event_user eu" +
                " INNER JOIN event e ON eu.event_id = e.id" +
                " INNER JOIN main_event me ON me.id = e.main_event_id" +
                " WHERE eu.exchange_id = ?";

        return getJdbcTemplate().queryForOption(q, Tuple3RowMapper.rowMapper(euRm, eRm, meRm), exchangeId);
    }

    @RunWithRandomTest
    public ListF<EventUser> findEventUsersByEventId(long eventId) {
        return findEventUsers(EventUserFields.EVENT_ID.eq(eventId));
    }

    @RunWithRandomTest
    public ListF<EventUser> findEventUsersByEventIds(ListF<Long> eventIds) {
        return findEventUsers(EventUserFields.EVENT_ID.column().inSet(eventIds));
    }

    @RunWithRandomTest
    public ListF<EventUser> findEventUsersByIds(ListF<Long> eventUserIds) {
        return findEventUsers(EventUserFields.ID.column().inSet(eventUserIds));
    }

    @RunWithRandomTest
    public ListF<EventUserWithRelations> findEventUsersWithRelationsByEventIds(ListF<Long> eventIds,
                                                                               boolean withSubscribers) {
        BeanRowMapper<EventUser> euRm = EventUserHelper.INSTANCE.offsetRowMapper(0);
        RowMapper<SettingsInfo> sRm = SettingsInfo.rowMapper(euRm.nextOffset());

        String q = "SELECT " + euRm.columns("eu.") + ", " + SettingsInfo.columns("s.", "syt.") +
                " FROM event_user eu" +
                " INNER JOIN settings s ON eu.uid = s.uid" +
                " LEFT JOIN settings_yt syt ON syt.uid = s.uid" +
                " WHERE eu.event_id " + SqlQueryUtils.inSet(eventIds) +
                (withSubscribers ? "" : " AND (eu.is_subscriber = FALSE or eu.is_attendee = TRUE or eu.is_organizer = TRUE)");

        if (skipQuery(eventIds, q)) return Cf.list();

        return getJdbcTemplate().query2(q, euRm, sRm).map(EventUserWithRelations::new);
    }

    @RunWithRandomTest
    public ListF<EventUserWithRelations> findEventUsersWithRelationsByEventIds(ListF<Long> eventIds) {
        return findEventUsersWithRelationsByEventIds(eventIds, true);
    }

    @RunWithRandomTest
    public ListF<EventUser> findEventUsersByEventIdsAndUid(ListF<Long> eventIds, PassportUid uid) {
        SqlCondition c = EventUserFields.EVENT_ID.column().inSet(eventIds).and(EventUserFields.UID.eq(uid));
        return findEventUsers(c);
    }

    @RunWithRandomTest
    public ListF<EventUser> findEventUsersByEventIdsAndUids(ListF<Long> eventIds, ListF<PassportUid> uids) {
        SqlCondition c = EventUserFields.EVENT_ID.column().inSet(eventIds)
                .and(EventUserFields.UID.column().inSet(uids));
        return findEventUsers(c);
    }

    @RunWithRandomTest
    public Option<EventUser> findEventUserByEventIdAndUid(long eventId, PassportUid uid) {
        String q = "SELECT * FROM event_user WHERE event_id = ? AND uid = ?";
        return getJdbcTemplate().queryForOption(q, EventUser.class, eventId, uid);
    }

    @RunWithRandomTest
    public Option<EventUser> findEventUserByPrivateToken(String privateToken) {
        String q = "SELECT * FROM event_user WHERE private_token = ?";
        return getJdbcTemplate().queryForOption(q, EventUser.class, privateToken);
    }

    // XXX: make private
    @RunWithRandomTest
    public ListF<EventUser> findEventUsers(SqlCondition... conditions) {
        return findEventUsers(SqlCondition.all(conditions), SqlOrder.unordered(), SqlLimits.all());
    }

    @RunWithRandomTest
    public ListF<EventUser> findEventUsers(SqlCondition condition, SqlOrder order, SqlLimits limits) {
        String q = "SELECT * FROM event_user" +
                " WHERE " + condition.sql() +
                " " + order.toSql() +
                " " + limits.toMysqlLimits();

        if (skipQuery(condition, q, condition.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, EventUser.class, condition.args());
    }

    @RunWithRandomTest
    public ListF<Long> findEventUserEventIds(SqlCondition condition) {
        String q = "SELECT event_id FROM event_user" + condition.whereSql();

        if (skipQuery(condition, q, condition.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, condition.args());
    }

    @RunWithRandomTest
    public Tuple2List<Long, PassportUid> findEventUserEventIdsAndUids(SqlCondition condition) {
        return getJdbcTemplate().query2(
                "SELECT event_id, uid FROM event_user" + condition.whereSql(),
                (rs, rn) -> rs.getLong(1), (rs, rn) -> PassportUid.cons(rs.getLong(2)), condition.args());
    }

    @RunWithRandomTest
    public ListF<EventUser> findEventUsersByUid(PassportUid uid) {
        String q = "SELECT * FROM event_user WHERE uid = ?";
        return getJdbcTemplate().queryForList(q, EventUser.class, uid);
    }

    @RunWithRandomTest
    public Option<Long> findEventIdByUserExchangeId(String exchangeId) {
        String q = "SELECT event_id FROM event_user WHERE exchange_id = ?";
        return getJdbcTemplate().queryForOption(q, Long.class, exchangeId);
    }

    @RunWithRandomTest
    public Option<PassportUid> findOrganizerByEventId(long eventId) {
        return findOrganizersByEventIds(Cf.list(eventId)).single().get2();
    }

    @RunWithRandomTest
    public Tuple2List<Long, Option<PassportUid>> findOrganizersByEventIds(ListF<Long> eventIds) {
        SqlCondition c = EventUserFields.IS_ORGANIZER.eq(true).and(EventUserFields.EVENT_ID.column().inSet(eventIds));
        String q = "SELECT event_id, uid FROM event_user" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Tuple2List.tuple2List();

        MapF<Long, PassportUid> x = getJdbcTemplate().queryForList2(q, Long.class, Long.class, c.args())
                .toMap(Tuple2.get1F(), t -> PassportUid.cons(t.get2()));

        return eventIds.zipWith(x::getO);
    }

    @RunWithRandomTest
    public ListF<Long> findUserOrganizedEventIdsByUidAndEventIds(PassportUid uid, ListF<Long> eventIds) {
        SqlCondition c = EventUserFields.UID.eq(uid);
        c = c.and(EventUserFields.IS_ORGANIZER.eq(true));
        c = c.and(EventUserFields.EVENT_ID.column().inSet(eventIds));

        String q = "SELECT event_id FROM event_user" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public ListF<Long> findUsersAttendingEventIds(ListF<PassportUid> uids, ListF<Long> eventIds) {
        SqlCondition c = EventUserFields.UID.column().inSet(uids)
                .and(EventUserFields.EVENT_ID.column().inSet(eventIds))
                .and(EventUserFields.IS_ATTENDEE.eq(true).or(EventUserFields.IS_ORGANIZER.eq(true)));

        String q = "SELECT event_id FROM event_user" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, c.args());
    }

    @RunWithRandomTest
    public Tuple2List<PassportUid, ListF<Event>> findUserOrganizedEventsByUids(
            ListF<PassportUid> uids, SqlCondition eventCondition)
    {
        SqlCondition eventUserCondition = EventUserFields.UID.column().inSet(uids);
        eventUserCondition = eventUserCondition.and(EventUserFields.IS_ORGANIZER.eq(true));

        RowMapper<PassportUid> uidRm = (rs, rowNum) -> PassportUid.cons(rs.getLong(1));
        BeanRowMapper<Event> eRm = EventHelper.INSTANCE.offsetRowMapper(1);

        String q = "SELECT eu.uid, " + eRm.columns("e.") + " FROM event_user eu" +
                " INNER JOIN event e ON e.id = eu.event_id" +
                " WHERE " + eventCondition.sqlForTable("e") + " AND " + eventUserCondition.sqlForTable("eu");

        MapF<PassportUid, ListF<Event>> eventsByUid = getJdbcTemplate()
                .query2(q, uidRm, eRm, eventCondition.args(), eventUserCondition.args())
                .groupBy1();

        return uids.zipWith(u -> eventsByUid.getOrElse(u, Cf.list()));
    }

    @RunWithRandomTest
    public ListF<Event> findEventsByEventUsers(SqlCondition eventCondition, SqlCondition eventUserCondition) {
        String q = "SELECT DISTINCT e.* FROM event_user eu" +
                " INNER JOIN event e ON e.id = eu.event_id" +
                " WHERE " + eventCondition.sqlForTable("e") + " AND " + eventUserCondition.sqlForTable("eu");
        return getJdbcTemplate().queryForList(q, Event.class, eventCondition.args(), eventUserCondition.args());
    }

    public long saveEventUser(EventUser eventUser, ActionInfo actionInfo) {
        return genericBeanDao.insertBeanGetGeneratedKey(withLastUpdateAndDecisionSourceIfNeeded(eventUser, actionInfo));
    }

    public ListF<Long> saveEventUsersBatch(ListF<EventUser> eventUsers, ActionInfo actionInfo) {
        return genericBeanDao.insertBeansBatchGetGeneratedKeys(
                eventUsers.map(eu -> withLastUpdateAndDecisionSourceIfNeeded(eu, actionInfo)));
    }

    public void updateEventUsersBatch(ListF<EventUser> eventUsers, ActionInfo actionInfo) {
        genericBeanDao.updateBeans(eventUsers.map(eu -> withLastUpdateAndDecisionSourceIfNeeded(eu, actionInfo)));
    }

    @RunWithRandomTest
    public ListF<EventUser> findOrganizersWithoutEventLayers() {
        String q = "SELECT eu.* FROM event_user eu WHERE is_organizer AND NOT EXISTS " +
                "(SELECT * FROM event_layer el INNER JOIN layer_user lu ON el.layer_id = lu.layer_id " +
                "WHERE el.event_id = eu.event_id AND lu.uid = eu.uid)";
        return getJdbcTemplate().queryForList(q, EventUser.class);
    }

    @RunWithRandomTest
    public void updateEventUserAvailabilityByEventIdAndUserId(
            long eventId, PassportUid uid, Availability availability, ActionInfo actionInfo)
    {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET availability = ?, " + info.sql() + " WHERE event_id = ? AND uid = ?";
        getJdbcTemplate().update(q, availability, info.args(), eventId, uid);
    }

    @RunWithRandomTest
    public void updateEventUsersSetOwnerUidByIds(ListF<Long> eventUserIds, PassportUid newOwner, ActionInfo actionInfo) {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET uid = ?, " + info.sql() + " WHERE id " + SqlQueryUtils.inSet(eventUserIds);

        if (skipQuery(eventUserIds, q, newOwner, info.args())) return;

        getJdbcTemplate().update(q, newOwner, info.args());
    }

    @RunWithRandomTest
    public void updateEventUsersSetIsOrganizerAndIsAttendeeByIds(
            ListF<Long> eventUserIds, boolean isOrganizer, boolean isAttendee, ActionInfo actionInfo)
    {
        SqlPart info = modificationInfoAssignment(actionInfo);
        String q = "UPDATE event_user SET is_organizer = ?, is_attendee = ?, " + info.sql() +
                " WHERE id " + SqlQueryUtils.inSet(eventUserIds);

        if (skipQuery(eventUserIds, q, isOrganizer, isAttendee, info.args())) return;

        getJdbcTemplate().update(q, isOrganizer, isAttendee, info.args());
    }

    @RunWithRandomTest
    public ListF<Long> findBigMeetings(Long usersCountLimit, Instant maxEventStartTs) {
        String q = "SELECT event_id " +
                "FROM (SELECT event_id, COUNT(*) AS users_count FROM event_user GROUP BY event_id HAVING COUNT(*) > ?" +
                ") AS sub " +
                "JOIN event ON event.id = event_id " +
                "WHERE event.recurrence_id IS NOT NULL " +
                "AND event.start_ts < ? " +
                "ORDER BY users_count DESC;";
        return getJdbcTemplate().queryForList(q, Long.class, usersCountLimit, maxEventStartTs);
    }

    private static SqlPart modificationInfoAssignment(ActionInfo actionInfo) {
        return new FieldsAssignment(Tuple2List.<String, Object>fromPairs(
                "modification_source", actionInfo.getActionSource(),
                "modification_req_id", actionInfo.getRequestIdWithHostId(),
                "last_update", actionInfo.getNow()));
    }
}
