package ru.yandex.calendar.logic.event.dao;

import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.beans.BeanHelper;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.EventInvitationFields;
import ru.yandex.calendar.logic.beans.generated.EventLayerFields;
import ru.yandex.calendar.logic.beans.generated.EventResourceFields;
import ru.yandex.calendar.logic.beans.generated.EventTimezoneInfo;
import ru.yandex.calendar.logic.beans.generated.EventTimezoneInfoFields;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.event.ExternalId;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.mapObject.MapObjectDescription;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.email.Email;

import static java.util.Collections.emptyMap;

public class MainEventDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;

    @RunWithRandomTest
    public long saveMainEvent(ExternalId extId, DateTimeZone eventTz, Instant now) {
        return saveMainEvents(Cf.list(extId), eventTz, now).single()._2;
    }

    @RunWithRandomTest
    public Tuple2List<String, Long> saveMainEvents(
            final ListF<ExternalId> externalIds, final DateTimeZone eventsTz, final Instant now)
    {
        ListF<MainEvent> datas = externalIds.map(extId -> {
            MainEvent me = new MainEvent();
            me.setExternalId(extId.getRaw());
            me.setExternalIdNormalized(extId.getNormalized());
            me.setLastUpdateTs(now);
            me.setTimezoneId(eventsTz.getID());

            return me;
        });
        return datas
                .zip(genericBeanDao.insertBeansBatchGetGeneratedKeys(datas))
                .toTuple2List((me, id) -> Tuple2.tuple(me.getExternalId(), id));
    }

    @RunWithRandomTest
    public long saveMainEvent(ExternalId externalId, DateTimeZone eventTz, Instant now, boolean isExportedWithEws) {
        MainEvent mainEvent = new MainEvent();
        mainEvent.setExternalId(externalId.getRaw());
        mainEvent.setExternalIdNormalized(externalId.getNormalized());
        mainEvent.setLastUpdateTs(now);
        mainEvent.setTimezoneId(eventTz.getID());
        mainEvent.setIsExportedWithEws(isExportedWithEws);

        return genericBeanDao.insertBeanGetGeneratedKey(mainEvent);
    }

    public void updateMainEvent(MainEvent mainEvent) {
        genericBeanDao.updateBean(mainEvent);
    }

    @RunWithRandomTest
    public void updateMainEventLastUpdateTsByMainEventIds(ListF<Long> mainEventIds, Instant now) {
        String q = "UPDATE main_event SET last_update_ts = ? WHERE id " + SqlQueryUtils.inSet(mainEventIds);

        if (skipQuery(mainEventIds, q, now)) return;

        getJdbcTemplate().update(q, now);
    }

    @RunWithRandomTest
    public void updateStaleLastUpdateTsByIds(ListF<Long> mainEventIds, Instant ts) {
        SqlCondition c = MainEventFields.ID.column().inSet(mainEventIds)
                .and(MainEventFields.LAST_UPDATE_TS.lt(ts));

        String q = "UPDATE main_event SET last_update_ts = ?" + c.whereSql();

        if (skipQuery(c, q, ts, c.args())) return;

        getJdbcTemplate().update(q, ts, c.args());
    }

    @RunWithRandomTest
    public void updateMainEventLastUpdateTsByLayerId(long layerId, Instant now) {
        String q = "UPDATE main_event me" +
            " SET last_update_ts = ?" +
            " FROM event e" +
            " INNER JOIN event_layer el ON el.event_id = e.id AND el.layer_id = ?" +
            " WHERE e.main_event_id = me.id";
        getJdbcTemplate().update(q, now, layerId);
    }

    @RunWithRandomTest
    public void updateMainEventLastUpdateTsByLayerIdsAndTimezoneIdsForEventsOccurredAfter(
            ListF<Long> layerIds, ListF<String> timezoneIds, Instant after, Instant now)
    {
        String q = "UPDATE main_event me" +
                " SET last_update_ts = ?" +
                " FROM event e" +
                " INNER JOIN event_layer el ON el.event_id = e.id" +
                " LEFT JOIN repetition r ON r.id = e.repetition_id" +
                " WHERE el.layer_id " + SqlQueryUtils.inSet(layerIds) +
                " AND me.timezone_id " + SqlQueryUtils.inSet(timezoneIds) +
                " AND (e.end_ts > ? OR (r.id IS NOT NULL AND r.due_ts IS NULL OR r.due_ts > ?))" +
                " AND e.main_event_id = me.id";

        if (skipQuery(layerIds, timezoneIds, q, now, after, after)) return;

        getJdbcTemplate().update(q, now, after, after);
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsOnLayer(long layerId, ListF<ExternalId> externalIds) {
        SqlCondition c = MainEventFields.EXTERNAL_ID.column().inSet(externalIds.map(ExternalId.getRawF()))
                .or(MainEventFields.EXTERNAL_ID_NORMALIZED.column().inSet(externalIds.map(ExternalId.getNormalizedF())));
        return findMainEventsOnLayerWithCondition(layerId, c);
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsOnLayer(long layerId) {
        return findMainEventsOnLayerWithCondition(layerId, SqlCondition.trueCondition());
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsOnLayerModifiedSince(long layerId, Instant since) {
        return findMainEventsOnLayerWithCondition(layerId, MainEventFields.LAST_UPDATE_TS.ge(since));
    }

    @RunWithRandomTest
    private ListF<MainEvent> findMainEventsOnLayerWithCondition(long layerId, SqlCondition mainEventCondition) {
        String q = "SELECT DISTINCT me.*" +
            " FROM event e" +
            " INNER JOIN event_layer el ON e.id = el.event_id" +
            " INNER JOIN main_event me ON e.main_event_id = me.id" +
            " WHERE el.layer_id = ? " + mainEventCondition.andSqlForTable("me");

        if (skipQuery(mainEventCondition, q, layerId, mainEventCondition.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, MainEvent.class, layerId, mainEventCondition.args());
    }

    @RunWithRandomTest
    public ListF<Long> findMainEventIdsOnLayers(ListF<Long> layerIds, ListF<String> externalIds) {
        SqlCondition elC = EventLayerFields.LAYER_ID.column().inSet(layerIds);
        SqlCondition meC = MainEventFields.EXTERNAL_ID.column().inSet(externalIds);

        String q = "SELECT DISTINCT me.id"
                + " FROM main_event me, event e, event_layer el"
                + " WHERE me.id = e.main_event_id AND e.id = el.event_id"
                + meC.andSqlForTable("me") + elC.andSqlForTable("el");

        if (skipQuery(layerIds, externalIds, q, meC.args(), elC.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, Long.class, meC.args(), elC.args());
    }

    @RunWithRandomTest
    public Tuple2List<String, Instant> findMainEventExternalIdsAndTimestampsOnLayersUpdatedAfter(
            ListF<Long> layerIds, Instant after)
    {
        SqlCondition layerC = EventLayerFields.LAYER_ID.column().inSet(layerIds);
        SqlCondition mainC = MainEventFields.LAST_UPDATE_TS.column().gt(after);

        String q = "SELECT DISTINCT external_id, main_event.last_update_ts" +
                " FROM main_event" +
                " JOIN event ON main_event.id = event.main_event_id" +
                " JOIN event_layer el on event.id = el.event_id" +
                " WHERE " + layerC.sql() + " AND main_event.last_update_ts > ?";

        if (skipQuery(layerC.and(mainC), q, layerC.args(), mainC.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().queryForList2(q, String.class, Instant.class, layerC.args(), mainC.args());
    }

    @RunWithRandomTest
    public Tuple2List<Long, Long> findMainEventIdsOnLayersUpdatedAfter(ListF<Long> layerIds, Instant after) {
        SqlCondition layerC = EventLayerFields.LAYER_ID.column().inSet(layerIds);
        SqlCondition mainC = MainEventFields.LAST_UPDATE_TS.column().gt(after);

        String q = "SELECT DISTINCT el.layer_id, me.id" +
                " FROM event e" +
                " INNER JOIN event_layer el ON el.event_id = e.id" +
                " INNER JOIN main_event me ON me.id = e.main_event_id" +
                " WHERE (" + layerC.sqlForTable("el") + ")" + mainC.andSqlForTable("me");

        if (skipQuery(layerC.and(mainC), q, layerC.args(), mainC.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().queryForList2(q, Long.class, Long.class, layerC.args(), mainC.args());
    }

    @RunWithRandomTest
    public ListF<Long> findEventIdsByExternalId(ExternalId externalId) {
        String q = "SELECT e.id FROM event e" +
                " INNER JOIN main_event me ON me.id = e.main_event_id" +
                " WHERE (me.external_id = ? OR me.external_id_normalized = ?)";

        return getJdbcTemplate().queryForList(q, Long.class, externalId.getRaw(), externalId.getNormalized());
    }

    private ListF<MainEvent> findMainEventsByEventRelations(MapObjectDescription desc, SqlCondition cond) {
        String q = "SELECT DISTINCT me.*" +
                " FROM " + desc.getTableName() + " er" +
                " INNER JOIN event e ON e.id = er.event_id" +
                " INNER JOIN main_event me ON me.id = e.main_event_id" +
                cond.whereSqlForTable("er");

        if (skipQuery(cond, q, cond.args())) return Cf.list();

        return getJdbcTemplate().queryForList(q, MainEvent.class, cond.args());
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsByEventUsers(ListF<PassportUid> uids, ListF<Long> eventIds) {
        return findMainEventsByEventRelations(
                EventUserFields.OBJECT_DESCRIPTION,
                EventUserFields.UID.column().inSet(uids)
                        .and(EventUserFields.EVENT_ID.column().inSet(eventIds)));
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsByLayerIds(ListF<Long> layerIds, ListF<Long> eventIds) {
        return findMainEventsByEventRelations(
                EventLayerFields.OBJECT_DESCRIPTION,
                EventLayerFields.LAYER_ID.column().inSet(layerIds)
                        .and(EventLayerFields.EVENT_ID.column().inSet(eventIds)));
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsByInvitationEmails(ListF<Email> emails, ListF<Long> eventIds) {
        return findMainEventsByEventRelations(
                EventInvitationFields.OBJECT_DESCRIPTION,
                EventInvitationFields.EMAIL.column().inSet(emails)
                        .and(EventInvitationFields.EVENT_ID.column().inSet(eventIds)));
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsByResourceIds(ListF<Long> resourceIds, ListF<Long> eventIds) {
        return findMainEventsByEventRelations(
                EventResourceFields.OBJECT_DESCRIPTION,
                EventResourceFields.RESOURCE_ID.column().inSet(resourceIds)
                        .and(EventResourceFields.EVENT_ID.column().inSet(eventIds)));
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public MainEvent findMainEventById(long id) {
        String q = "SELECT * FROM main_event WHERE id = ?";
        return getJdbcTemplate().queryForObject(q, MainEvent.class, id);
    }

    @RunWithRandomTest
    public Option<MainEvent> findMainEventOById(long id) {
        String q = "SELECT * FROM main_event WHERE id = ?";
        return getJdbcTemplate().queryForOption(q, MainEvent.class, id);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public MainEvent findMainEventByEventId(long eventId) {
        // XXX possible deadlock
        String q = "SELECT * FROM main_event WHERE id =" +
                " (SELECT main_event_id FROM event WHERE id = ?)";
        return getJdbcTemplate().queryForObject(q, MainEvent.class, eventId);
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsByEventIds(ListF<Long> eventIds) {
        String q = "SELECT DISTINCT me.* FROM main_event me" +
                " INNER JOIN event e ON e.main_event_id = me.id" +
                " WHERE e.id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return Cf.list();

        return getJdbcTemplate().queryForList(q, MainEvent.class);
    }

    public ListF<String> findExternalIdsByEventIds(ListF<Long> eventIds) {
        return findMainEventsByEventIds(eventIds).map(MainEvent.getExternalIdF()).stableUnique();
    }

    public Map<Long, String> mapEventIdsToExternalIds(List<Long> eventIds) {
        val q = "SELECT DISTINCT e.id, me.external_id FROM main_event me" +
                " INNER JOIN event e ON e.main_event_id = me.id" +
                " WHERE e.id " + SqlQueryUtils.inSet(Cf.toList(eventIds));

        if (skipQuery(eventIds, q)) return emptyMap();

        return StreamEx.of(getJdbcTemplate().queryForList2(q, Long.class, String.class))
                .toMap(Tuple2::get1, Tuple2::get2);
    }

    @RunWithRandomTest
    public Tuple2List<String, Instant> findExternalIdsAndTimestampsByEventIds(ListF<Long> eventIds) {
        String q = "SELECT DISTINCT me.external_id, me.last_update_ts FROM main_event me" +
                " INNER JOIN event e ON e.main_event_id = me.id" +
                " WHERE e.id " + SqlQueryUtils.inSet(eventIds);

        if (skipQuery(eventIds, q)) return Tuple2List.tuple2List();

        return getJdbcTemplate().queryForList2(q, String.class, Instant.class);
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEvents() {
        String q = "SELECT * FROM main_event";
        return getJdbcTemplate().queryForList(q, MainEvent.class);
    }

    public ListF<MainEvent> findMainEventsFieldsByIdsSafe(ListF<Long> ids, ListF<MapField<?>> fields) {
        ListF<MapField<?>> extFields = fields.plus(MainEventFields.ID).stableUnique();

        SqlCondition c = MainEventFields.ID.column().inSet(ids);
        String q = "SELECT " + BeanHelper.columns(extFields, "") + " FROM main_event" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Cf.list();

        return getJdbcTemplate().query(q, BeanHelper.fieldsRowMapper(MainEvent.class, extFields, ""), c.args());
    }

    @RunWithRandomTest(possible=IncorrectResultSizeDataAccessException.class)
    public ListF<MainEvent> findMainEventsByIds(ListF<Long> ids) {
        String q = "SELECT * FROM main_event WHERE id " + SqlQueryUtils.inSet(ids);

        if (skipQuery(ids, q)) return Cf.list();

        MapF<Long, MainEvent> mainEvents =
            getJdbcTemplate().queryForList(q, MainEvent.class)
                .toMapMappingToKey(MainEventFields.ID.getF());
        try {
            return ids.map(mainEvents::getOrThrow);
        } catch (NoSuchElementException e) {
            throw new IncorrectResultSizeDataAccessException(ids.size(), mainEvents.size());
        }
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsByExternalId(ExternalId externalId) {
        String q = "SELECT * FROM main_event WHERE (external_id = ? OR external_id_normalized = ?)";
        return getJdbcTemplate().queryForList(q, MainEvent.class, externalId.getRaw(), externalId.getNormalized());
    }

    @RunWithRandomTest
    public ListF<MainEvent> findMainEventsWithoutEvents() {
        String q = "SELECT * FROM main_event me" +
            " WHERE NOT EXISTS(SELECT * FROM event WHERE main_event_id = me.id)";
        return getJdbcTemplate().queryForList(q, MainEvent.class);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public String findExternalIdByMainEventId(long id) {
        String q = "SELECT external_id FROM main_event WHERE id = ?";
        return getJdbcTemplate().queryForObject(q, String.class, id);
    }

    @RunWithRandomTest(possible=IncorrectResultSizeDataAccessException.class)
    public Tuple2List<Long, String> findExternalIdsByIds(ListF<Long> mainEventIds) {
        SqlCondition c = MainEventFields.ID.column().inSet(mainEventIds);
        String q = "SELECT id, external_id FROM main_event" + c.whereSql();

        if (skipQuery(c, q, c.args())) return Tuple2List.tuple2List();

        return getJdbcTemplate().queryForList2(q, Long.class, String.class, c.args());
    }

    public void deleteMainEventById(long id) {
        String q = "DELETE FROM main_event WHERE id = ?";
        getJdbcTemplate().update(q, id);
    }

    @RunWithRandomTest
    public void deleteMainEventsIfOrphaned(ListF<Long> mainEventIds) {
        String q = "DELETE FROM main_event me" +
                " WHERE me.id " + SqlQueryUtils.inSet(mainEventIds) +
                " AND NOT EXISTS (SELECT 1 FROM event e WHERE e.main_event_id = me.id)";

        if (skipQuery(mainEventIds, q)) return;

        getJdbcTemplate().update(q);
    }

    public void insertEventTimezoneInfo(EventTimezoneInfo info) {
        genericBeanDao.insertBean(info);
    }

    @RunWithRandomTest
    public void deleteTimezoneInfosByMainEventIdsIfOrphaned(CollectionF<Long> mainEventIds) {
        SqlCondition c = EventTimezoneInfoFields.MAIN_EVENT_ID.column().inSet(mainEventIds);
        String q = "DELETE FROM event_timezone_info etz"
                + c.whereSqlForTable("etz")
                + " AND NOT EXISTS (SELECT 1 FROM event e WHERE e.main_event_id = etz.main_event_id)";

        if (skipQuery(c, q, c.args())) return;

        getJdbcTemplate().update(q, c.args());
    }

    @RunWithRandomTest
    public Tuple2List<Long, String> findTimezoneIdsByMainEventIds(ListF<Long> mainEventIds) {
        String q = "SELECT id, timezone_id FROM main_event" +
                " WHERE timezone_id IS NOT NULL AND id " + SqlQueryUtils.inSet(mainEventIds);

        if (skipQuery(mainEventIds, q)) return Tuple2List.tuple2List();

        return getJdbcTemplate().queryForList2(q, Long.class, String.class);
    }

    public void updateTimezoneIdById(long mainEventId, String timezoneId) {
        String q = "UPDATE main_event SET timezone_id = ? WHERE id = ?";
        getJdbcTemplate().updateRow(q, timezoneId, mainEventId);
    }

    // for tests
    @RunWithRandomTest
    public int findCountOfMainIds(ExternalId externalId) {
        String q = "SELECT COUNT(*) FROM main_event WHERE (external_id = ? OR external_id_normalized = ?)";
        return getJdbcTemplate().queryForInt(q, externalId.getRaw(), externalId.getNormalized());
    }

    public void deleteMainEventByExternalId(ExternalId externalId) {
        String q = "DELETE FROM main_event WHERE external_id = ? OR external_id_normalized = ?";
        getJdbcTemplate().update(q, externalId.getRaw(), externalId.getNormalized());
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateIsExportedWithEwsById(long mainEventId, boolean isExportedWithEws) {
        String q = "UPDATE main_event SET is_exported_with_ews = ? WHERE id = ?";
        getJdbcTemplate().updateRow(q, isExportedWithEws, mainEventId);
    }

}
