package ru.yandex.calendar.logic.event.grid;

import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;

import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.xml.CalendarXmlizer;

/**
 * For given xml element and day, can append 'grayed' and 'weekno' attributes.
 * BEWARE: in {@link #isGrayed(LocalDate)}, parameter should increment SEQUENTIALLY
 * (as 'dayno' does), so that 'weekno' will correspond to it.
 * @author ssytnik
 */
public abstract class GrayedWeekNoAppender {
    private int dayno = 0;

    // Sets 'grayed', 'weekno' attributes and returns isGrayed
    public boolean append(Element e, LocalDate localDate) {
        boolean grayed = isGrayed(localDate);
        CalendarXmlizer.setAttr(e, "grayed", grayed, null);
        CalendarXmlizer.setAttr(e, "weekno", (dayno++ / 7 + 1), null);
        return grayed;
    }

    // ABSTRACT //

    public abstract boolean isGrayed(LocalDate localDate);

    // STATIC //

    public static GrayedWeekNoAppender createDay(final LocalDate showDate) {
        return new GrayedWeekNoAppender() {
            public boolean isGrayed(LocalDate localDate) {
                return !showDate.equals(localDate);
            }
        };
    }

    public static GrayedWeekNoAppender createWeek(LocalDate showDate, DayOfWeek startWeekday, final DateTimeZone tz) {
        final long weekStartMs = DateTimeFormatter.getViewTypeBound(tz, showDate, ViewType.WEEK, startWeekday, true).getMillis();
        final long weekEndMs = DateTimeFormatter.getViewTypeBound(tz, showDate, ViewType.WEEK, startWeekday, false).getMillis();
        return new GrayedWeekNoAppender() {
            public boolean isGrayed(LocalDate localDate) {
                final long localDateMillis = localDate.toDateTimeAtStartOfDay(tz).getMillis();
                return (localDateMillis < weekStartMs || weekEndMs <= localDateMillis); // [rd1; rd2)
            }
        };
    }

    public static GrayedWeekNoAppender createMonth(final LocalDate showDate) {
        return new GrayedWeekNoAppender() {
            public boolean isGrayed(LocalDate localDate) {
                return localDate.getYear() != showDate.getYear() || localDate.getMonthOfYear() != showDate.getMonthOfYear();
            }
        };
    }
}
