package ru.yandex.calendar.logic.event.grid;

import org.joda.time.DateTimeFieldType;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.validation.RequestValidator.IAtomicValidator;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;
import ru.yandex.misc.lang.CamelWords;

/**
 * Specifies type of a grid which user looks at
 * @author ssytnik
 */
public enum ViewType implements StringEnum {
    DAY,
    WEEK,
    MONTH,
    GREY_DAY,
    GREY_WEEK,
    GREY_MONTH,
    ;

    public static final StringEnumResolver<ViewType> R = StringEnumResolver.r(ViewType.class);

    public static final IAtomicValidator AV = new IAtomicValidator() {
        public void validate(String name, String value) {
            try {
                R.valueOf(value);
            } // does not accept null
            catch (RuntimeException e) {
                RequestValidator.addInvalidParameterCode(name, value);
            }
        }
    }; // AV ends

    public static ViewType getDefault() {
        return WEEK;
    }
    public String toDbValue() {
        return CamelWords.parse(this.name()).toDbName();
    }

    public static Option<ViewType> fromValueSafe(String value) {
        return Option.ofNullable(value).filterMap(ViewType.R::fromValueO);
    }

    @Override
    public String value() {
        return toDbValue();
    }

    public GrayedWeekNoAppender createGrayedWeekNoAppender(
            LocalDate showDate, DayOfWeek startWeekday, DateTimeZone tz)
    {
        switch (this) {
        case DAY: case GREY_DAY: return GrayedWeekNoAppender.createDay(showDate);
        case WEEK: case GREY_WEEK: return GrayedWeekNoAppender.createWeek(showDate, startWeekday, tz);
        case MONTH: case GREY_MONTH: return GrayedWeekNoAppender.createMonth(showDate);
        default: throw new IllegalStateException("Unknown viewType: " + this);
        }
    }

    public DateTimeFieldType getDateTimeFieldType() {
        switch (this) {
        case DAY: case GREY_DAY: return DateTimeFieldType.dayOfYear();
        case WEEK: case GREY_WEEK: return DateTimeFieldType.weekOfWeekyear();
        case MONTH: case GREY_MONTH: return DateTimeFieldType.monthOfYear();
        default: throw new IllegalStateException("Unknown viewType: " + this);
        }
    }

    public boolean isGrey() {
        return this == GREY_DAY || this == GREY_WEEK || this == GREY_MONTH;
    }
}
