package ru.yandex.calendar.logic.event.model;

import lombok.Getter;
import lombok.Setter;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.frontend.ews.ExchangeData;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;

public class EventData {
    @Getter
    @Setter
    private Event event = new Event();
    @Getter
    @Setter
    private Repetition repetition = RepetitionRoutines.createNoneRepetition();
    @Getter
    @Setter
    private EventUserData eventUserData = new EventUserData();

    @Getter private Option<Instant> instanceStartTs = Option.empty();

    @Getter private Option<Long> layerId = Option.empty();
    @Getter private Option<Long> prevLayerId = Option.empty();

    /** @deprecated external id is a property of main event, not event */
    @Deprecated
    @Getter
    @Setter
    private Option<String> externalId = Option.empty();

    // InvUpdateData contains emails of new invited guests and about
    // emails of removed guests. InvData contains full information about invitations.
    // InvUpdateData is used for make-up.
    // InvData is used for ics import and import events from exchange.
    private ParticipantsOrInvitationsData participantsData = ParticipantsOrInvitationsData
            .participantsData(ParticipantsData.notMeeting());
    @Getter
    @Setter
    private ListF<Rdate> rdates = Cf.arrayList();
    // for exchange only
    @Getter
    @Setter
    private Option<ExchangeData> exchangeData = Option.empty();

    @Getter private Option<ListF<EventAttachment>> attachmentsO = Option.empty();

    private Option<DateTimeZone> timeZone = Option.empty();

    public EventData copy() {
        EventData copy = new EventData();
        copy.event = this.event.copy();
        copy.repetition = this.repetition.copy();
        copy.eventUserData = new EventUserData(eventUserData.getEventUser().copy(), eventUserData.getNotifications());
        copy.instanceStartTs = this.instanceStartTs;
        copy.layerId = this.layerId;
        copy.prevLayerId = this.prevLayerId;
        copy.externalId = this.externalId;
        copy.participantsData = this.participantsData;
        copy.exchangeData = this.exchangeData;
        copy.attachmentsO = this.attachmentsO;
        copy.timeZone = this.timeZone;
        return copy;
    }

    public ListF<Email> getParticipantEmails() {
        return participantsData.getParticipantEmails();
    }

    public ListF<ParticipantData> getParticipantsSafe() {
        return participantsData.getParticipantsDataO().map(ParticipantsData::getParticipantsSafe).getOrElse(Cf::list);
    }

    public MapF<Email, Decision> getParticipantsDecisionsSafe() {
        return getParticipantsSafe().toMap(ParticipantData::getEmail, ParticipantData::getDecision);
    }

    public Option<Email> getOrganizerEmail() {
        return participantsData.getOrganizerEmail();
    }

    public ListF<Rdate> findExdates() {
        return rdates.filter(Function1B.wrap(Rdate.getIsRdateF()).notF());
    }

    public void excludeExdates(SetF<Instant> exdates) {
        setRdates(rdates.filter(Function1B.wrap(Rdate.getIsRdateF()).notF()
                .andF(Rdate.getStartTsF().andThen(exdates.containsF().notF()))));
    }

    public void setAttachments(ListF<EventAttachment> attachments) {
        attachmentsO = Option.of(attachments);
    }

    public void setLayerId(Option<Long> layerId) {
        this.layerId = layerId;
    }

    public void setLayerId(long layerId) {
        this.layerId = Option.of(layerId);
    }

    public void setPrevLayerId(long prevLayerId) {
        this.prevLayerId = Option.of(prevLayerId);
    }

    public void setInstanceStartTs(Option<Instant> instanceStartTs) {
        this.instanceStartTs = instanceStartTs;
    }

    public void setInstanceStartTs(Instant instanceStartTs) {
        setInstanceStartTs(Option.ofNullable(instanceStartTs));
    }

    public DateTimeZone getTimeZone() {
        return timeZone.getOrThrow("timezone was not specified");
    }

    public void setInvData(ParticipantsData participantsData) {
        this.participantsData = ParticipantsOrInvitationsData.participantsData(participantsData);
    }

    public void setInvData(EventInvitationUpdateData eventInvitationUpdateData) {
        this.participantsData = ParticipantsOrInvitationsData.eventInvitationUpdateData(eventInvitationUpdateData);
    }

    public void setInvData(EventInvitationsData eventInvitationsData) {
        this.participantsData = ParticipantsOrInvitationsData.eventInvitationData(eventInvitationsData);
    }

    public void setInvData(Email attendee, Email... attendees) {
        setInvData(Option.empty(), Cf.list(attendee).plus(attendees).toArray(Email.class));
    }

    public void setInvData(Option<Email> organizer, Email... attendees) {
        this.participantsData = ParticipantsOrInvitationsData.eventInvitationData(
                new EventInvitationsData(organizer, Cf.x(attendees).map(EventInvitationData.consWithNoNameF())));
    }

    public void setInvData(ParticipantsOrInvitationsData participantsOrInvitationsData) {
        this.participantsData = participantsOrInvitationsData;
    }

    public ParticipantsOrInvitationsData getInvData() {
        return participantsData;
    }

    public void setParticipantsData(ParticipantsOrInvitationsData participantsData) {
        this.participantsData = participantsData;
    }

    public void setTimeZone(DateTimeZone timeZone) {
        this.timeZone = Option.of(timeZone);
    }

    public Option<InstantInterval> getInterval() {
        return event.isAllFieldSet(EventFields.START_TS, EventFields.END_TS)
                ? Option.of(new InstantInterval(getEvent().getStartTs(), getEvent().getEndTs()))
                : Option.empty();
    }

    public EventData withEventIdAndInstanceStart(long eventId, Instant start) {
        EventData data = this.copy();
        data.getEvent().setId(eventId);
        data.setInstanceStartTs(start);

        return data;
    }
}
