package ru.yandex.calendar.logic.event.repetition;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Daniel Brylev
 */
public class EventAndRepetition {

    private final Event event;
    private final RepetitionInstanceInfo repetitionInfo;

    public EventAndRepetition(Event event, RepetitionInstanceInfo repetitionInfo) {
        Validate.equals(new InstantInterval(event.getStartTs(), event.getEndTs()), repetitionInfo.getEventInterval());
        this.event = event;
        this.repetitionInfo = repetitionInfo;
    }

    public Event getEvent() {
        return event;
    }

    public RepetitionInstanceInfo getRepetitionInfo() {
        return repetitionInfo;
    }

    public void validateIsLockedForUpdate() {
        event.validateIsLockedForUpdate();
    }

    public long getEventId() {
        return event.getId();
    }

    public long getMainEventId() {
        return event.getMainEventId();
    }

    public boolean isValidStart(Instant start) {
        return RepetitionUtils.isValidStart(repetitionInfo, start);
    }

    public ListF<InstantInterval> getInstanceIntervalsInInterval(InstantInterval interval) {
        return RepetitionUtils.getInstancesInInterval(repetitionInfo, interval);
    }

    public Option<InstantInterval> getFirstInstanceIntervalStartingAfter(Instant after) {
        return RepetitionUtils.getInstanceIntervalStartingAfter(repetitionInfo, after);
    }

    public Option<InstantInterval> getFirstInstanceIntervalEndingAfter(Instant after) {
        return RepetitionUtils.getInstanceIntervalEndingAfter(repetitionInfo, after);
    }

    public Option<Instant> getLastInstanceStartInInterval(InstantInterval interval) {
        return getInstanceIntervalsInInterval(interval).lastO().map(InstantInterval::getStart);
    }

    public EventInterval getEventInterval(InstantInterval interval) {
        return new EventInstanceInterval(this, interval).getEventInterval();
    }

    public ListF<EventInstanceInterval> getInstancesInInterval(InstantInterval interval) {
        return getInstancesInInterval(new InfiniteInterval(interval.getStart(), Option.of(interval.getEnd())));
    }

    public ListF<EventInstanceInterval> getInstancesInInterval(InfiniteInterval interval) {
        return RepetitionUtils
                .getIntervals(repetitionInfo, interval.getStart(), interval.getEnd(), true, Integer.MAX_VALUE)
                .map(EventInstanceInterval.consF(this));
    }

    public Option<EventInstanceInterval> getClosestInterval(Instant instant) {
        return RepetitionUtils.getClosestInstanceInterval(repetitionInfo, instant)
                .map(EventInstanceInterval.consF(this));
    }

    public Option<EventInstanceInterval> getClosestIntervalWithRecurrence(Instant instant) {
        return RepetitionUtils.getClosestInstanceIntervalWithRecurrence(repetitionInfo, instant)
                .map(EventInstanceInterval.consF(this));
    }

    public boolean goesOnAfter(Instant ts) {
        return repetitionInfo.goesOnAfter(ts);
    }

    public static Function1B<EventAndRepetition> hasZeroLengthF() {
        return new Function1B<EventAndRepetition>() {
            public boolean apply(EventAndRepetition e) {
                return e.getRepetitionInfo().getEventInterval().isEmpty();
            }
        };
    }

    public static Function2<Event, RepetitionInstanceInfo, EventAndRepetition> consF() {
        return new Function2<Event, RepetitionInstanceInfo, EventAndRepetition>() {
            public EventAndRepetition apply(Event e, RepetitionInstanceInfo r) {
                return new EventAndRepetition(e, r);
            }
        };
    }

    public static Function<EventAndRepetition, ListF<EventInstanceInterval>> getInstancesInIntervalF(Instant start, Instant end) {
        return getInstancesInIntervalF(new InstantInterval(start, end));
    }

    public static Function<EventAndRepetition, ListF<EventInstanceInterval>>
    getInstancesInIntervalF(InstantInterval interval) {
        return getInstancesInIntervalF(new InfiniteInterval(interval.getStart(), Option.of(interval.getEnd())));
    }

    public static Function<EventAndRepetition, ListF<EventInstanceInterval>>
    getInstancesInIntervalF(final InfiniteInterval interval) {
        return new Function<EventAndRepetition, ListF<EventInstanceInterval>>() {
            public ListF<EventInstanceInterval> apply(EventAndRepetition e) {
                return e.getInstancesInInterval(interval);
            }
        };
    }

    public static Function<EventAndRepetition, ListF<EventInstanceInterval>>
    getInstancesStartingInIntervalF(InstantInterval interval) {
        return e -> RepetitionUtils.getInstanceIntervalsStartingIn(e.getRepetitionInfo(), interval)
                .map(EventInstanceInterval.consF(e));
    }

    public static Function<EventAndRepetition, Long> getEventIdF() {
        return getEventF().andThen(Event.getIdF());
    }

    public static Function<EventAndRepetition, Long> getMainEventIdF() {
        return new Function<EventAndRepetition, Long>() {
            public Long apply(EventAndRepetition e) {
                return e.getMainEventId();
            }
        };
    }

    public static Function<EventAndRepetition, Event> getEventF() {
        return new Function<EventAndRepetition, Event>() {
            public Event apply(EventAndRepetition e) {
                return e.getEvent();
            }
        };
    }

    public static Function<EventAndRepetition, RepetitionInstanceInfo> getRepetitionInfoF() {
        return new Function<EventAndRepetition, RepetitionInstanceInfo>() {
            public RepetitionInstanceInfo apply(EventAndRepetition e) {
                return e.getRepetitionInfo();
            }
        };
    }

    public static Function<EventAndRepetition, Option<Repetition>> getRepetitionF() {
        return new Function<EventAndRepetition, Option<Repetition>>() {
            public Option<Repetition> apply(EventAndRepetition e) {
                return e.getRepetitionInfo().getRepetition();
            }
        };
    }

    public static Function<EventAndRepetition, Option<Instant>> getLastInstanceStartInIntervalF(
            final InstantInterval interval) {
        return e -> e.getLastInstanceStartInInterval(interval);
    }

    public static Function<EventAndRepetition, Option<Instant>> getFirstInstanceStartAfterF(final Instant after) {
        return new Function<EventAndRepetition, Option<Instant>>() {
            public Option<Instant> apply(EventAndRepetition e) {
                return e.getFirstInstanceIntervalStartingAfter(after).map(InstantInterval::getStart);
            }
        };
    }

    public boolean containsEventIntervalOf(EventAndRepetition anotherEvent) {
        return repetitionInfo.containsInterval(anotherEvent.getRepetitionInfo().getEventInterval());
    }
}
