package ru.yandex.calendar.logic.event.repetition;

import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.event.EventDateTime;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.IntervalComparator;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Daniel Brylev
 */
public class EventInstanceInterval {

    public static final Comparator<EventInstanceInterval> comparator =
            EventInstanceInterval.getIntervalF().andThen(IntervalComparator.INSTANCE)
                    .thenComparing(EventInstanceInterval.getEventIdF().andThenNaturalComparator());

    private final EventAndRepetition eventAndRepetition;
    private final InstantInterval interval;

    public EventInstanceInterval(EventAndRepetition eventAndRepetition, InstantInterval interval) {
        this.eventAndRepetition = eventAndRepetition;
        this.interval = interval;
    }

    public Event getEvent() {
        return eventAndRepetition.getEvent();
    }

    public long getEventId() {
        return eventAndRepetition.getEventId();
    }

    public InstantInterval getInterval() {
        return interval;
    }

    public EventInterval getEventInterval() {
        return new EventInterval(
                interval.getStart(),
                getEventDateTime(interval.getStart()),
                getEventDateTime(interval.getEnd()));
    }

    private EventDateTime getEventDateTime(Instant startOrEnd) {
        return eventAndRepetition.getEvent().getIsAllDay()
                ? EventDateTime.date(new LocalDate(startOrEnd, eventAndRepetition.getRepetitionInfo().getTz()))
                : EventDateTime.dateTime(startOrEnd);
    }

    public Instant getStart() {
        return interval.getStart();
    }

    public static Function<EventInstanceInterval, Event> getEventF() {
        return new Function<EventInstanceInterval, Event>() {
            public Event apply(EventInstanceInterval e) {
                return e.getEvent();
            }
        };
    }

    public static Function<EventInstanceInterval, Long> getEventIdF() {
        return getEventF().andThen(Event.getIdF());
    }

    public static Function<EventInstanceInterval, InstantInterval> getIntervalF() {
        return new Function<EventInstanceInterval, InstantInterval>() {
            public InstantInterval apply(EventInstanceInterval e) {
                return e.getInterval();
            }
        };
    }

    public static Function<InstantInterval, EventInstanceInterval> consF(EventAndRepetition event) {
        return interval1 -> new EventInstanceInterval(event, interval1);
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + "[" + getEventId() + "," + getInterval() + "]";
    }
}
