package ru.yandex.calendar.logic.event.repetition;

import java.util.function.Function;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.Period;
import org.joda.time.chrono.ISOChronology;

import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author yashunsky
 */
public class RecurrenceTimeInfo {
    private Instant recurrenceId;
    private InstantInterval interval;

    public RecurrenceTimeInfo(Instant recurrenceId, InstantInterval interval) {
        this.recurrenceId = recurrenceId;
        this.interval = interval;
    }

    public RecurrenceTimeInfo(Instant recurrenceId, Instant startTs, Instant endTs) {
        this(recurrenceId, new InstantInterval(startTs, endTs));
    }

    public static RecurrenceTimeInfo fromEvent(Event event) {
        return new RecurrenceTimeInfo(
                event.getRecurrenceId().getOrThrow("Not a recurring event"), event.getStartTs(), event.getEndTs());
    }


    public RecurrenceTimeInfo mapTz(Function<Instant, Instant> changeTz) {
        return new RecurrenceTimeInfo(
                changeTz.apply(recurrenceId), changeTz.apply(interval.getStart()), changeTz.apply(interval.getEnd()));
    }

    public Instant getRecurrenceId() {
        return recurrenceId;
    }

    public InstantInterval getInterval() {
        return interval;
    }

    public InstantInterval getWindowInterval(DateTimeZone tz) {
        return AuxDateTime.expandToDaysInterval(
                ObjectUtils.min(recurrenceId, getStart()),
                ObjectUtils.max(recurrenceId, getEnd()), tz);
    }

    public Instant getStart() {
        return interval.getStart();
    }

    public Instant getEnd() {
        return interval.getEnd();
    }

    public LocalDate getStartDate(DateTimeZone tz) {
        return new LocalDate(getStart(), tz);
    }

    public Period getPeriod(DateTimeZone tz) {
        return new Period(interval.getStartMillis(), interval.getEndMillis(), ISOChronology.getInstance(tz));
    }
}
